# Copyright (C) 2000-2001 The OpenRPG Project
#
#        openrpg-dev@lists.sourceforge.net
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# --
#
# File: inputValidator.py
# Author: Greg Copeland
# Maintainer:
#
# Description: Contains simple input validators to help reduce the amount of
# user input generated text.
#

__version__ = "$Id: inputValidator.py,v 1.4 2005/06/05 20:12:40 digitalxero Exp $"


##
## Module Loading
##
from wxPython.wx import *
import string


##
## Text Only input (no numbers allowed)
##
class TextOnlyValidator( wxPyValidator ):
    def __init__( self ):
        wxPyValidator.__init__( self )
        EVT_CHAR( self, self.onChar )



    def Clone( self ):
        return TextOnlyValidator()



    def Validate( self, win ):
        tc = self.GetWindow()
        val = tc.GetValue()

        retVal = true
        for x in val:
            if x not in string.letters:
                retVal = false
                break

        return retVal



    def onChar( self, event ):
        key = event.KeyCode()
        if key < WXK_SPACE or key == WXK_DELETE or key > 255:
            event.Skip()

        elif chr(key) in string.letters:
            event.Skip()

        else:
            if not wxValidator_IsSilent():
                wxBell()

        # Returning without calling even.  Skip eats the event before it
        # gets to the text control
        return



##
## Number Only input (no text allowed)
##
class NumberOnlyValidator( wxPyValidator ):
    def __init__( self ):
        wxPyValidator.__init__( self )
        EVT_CHAR( self, self.onChar )



    def Clone( self ):
        return NumberOnlyValidator()



    def Validate( self, win ):
        tc = self.GetWindow()
        val = tc.GetValue()

        retVal = true
        for x in val:
            if x not in string.digits:
                retVal = false
                break

        return retVal



    def onChar( self, event ):
        key = event.KeyCode()
        if key < WXK_SPACE or key == WXK_DELETE or key > 255:
            event.Skip()

        elif chr(key) in string.digits:
            event.Skip()

        else:
            if not wxValidator_IsSilent():
                wxBell()

        # Returning without calling even.  Skip eats the event before it
        # gets to the text control
        return






##
## Math Only input (no text allowed, only numbers of math symbols)
##
class MathOnlyValidator( wxPyValidator ):
    def __init__( self ):
        wxPyValidator.__init__( self )

        # Build it as part of the class and not per Validate() call
        self.allowedInput = "0123456789()*/+-<>"
        EVT_CHAR( self, self.onChar )



    def Clone( self ):
        return MathOnlyValidator()



    def Validate( self, win ):
        tc = self.GetWindow()
        val = tc.GetValue()

        retVal = true
        for x in val:
            if x not in self.allowedInput:
                retVal = false
                break

        return retVal



    def onChar( self, event ):
        key = event.KeyCode()
        if key < WXK_SPACE or key == WXK_DELETE or key > 255:
            event.Skip()

        elif chr(key) in self.allowedInput:
            event.Skip()

        else:
            if not wxValidator_IsSilent():
                wxBell()

        # Returning without calling even.  Skip eats the event before it
        # gets to the text control
        return






##
## Text and number input but DO NOT allow ANY HTML type input (no numbers allowed)
##
class NoHTMLValidator( wxPyValidator ):
    def __init__( self ):
        wxPyValidator.__init__( self )

        # Build it as part of the class and not per Validate() call
        self.allowedInput = " 1234567890!@#$%^&*()_-+=`~abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ,[]{}|;:'\",./?\\"
        EVT_CHAR( self, self.onChar )



    def Clone( self ):
        return NoHTMLValidator()



    def Validate( self, win ):
        tc = self.GetWindow()
        val = tc.GetValue()

        retVal = true
        for x in val:
            if x not in self.allowedInput:
                retVal = false
                break

        return retVal



    def onChar( self, event ):
        key = event.KeyCode()
        if key < WXK_SPACE or key == WXK_DELETE or key > 255:
            event.Skip()

        elif chr(key) in self.allowedInput:
            event.Skip()

        else:
            if not wxValidator_IsSilent():
                wxBell()

        # Returning without calling even.  Skip eats the event before it
        # gets to the text control
        return
