/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
-----------------------------------------
file	:	SAMLAuthorizationDecisionStatement.cpp
project	:	Policy Server
date	:	
author	:	andy
rights	:	(c) Parthenon Computing
notes	:	 extension to openSAML C++ library (based on Java class)
--------------------------------------------
*/

#include "internal.h"

using namespace saml;
using namespace std;

SAMLAuthorizationDecisionStatement::SAMLAuthorizationDecisionStatement(
    SAMLSubject* subject,
    const XMLCh* resource,
    const XMLCh* decision,
    const Iterator<SAMLAction*>& actions,
    SAMLEvidence* evidence
    ) : SAMLSubjectStatement(subject), m_resource(XML::assign(resource)), m_decision(XML::assign(decision)), m_evidence(NULL)
{
    RTTI(SAMLAuthorizationDecisionStatement);

    while(actions.hasNext())
        m_actions.push_back(static_cast<SAMLAction*>(actions.next()->setParent(this)));

    if(evidence) {
        evidence->setParent(this);
        m_evidence = evidence;
    }
}

SAMLAuthorizationDecisionStatement::SAMLAuthorizationDecisionStatement(DOMElement* e)
  : SAMLSubjectStatement(e), m_resource(NULL), m_decision(NULL), m_evidence(NULL)
{
    RTTI(SAMLAuthorizationDecisionStatement);
    fromDOM(e);
}

SAMLAuthorizationDecisionStatement::SAMLAuthorizationDecisionStatement(std::istream& in)
  : SAMLSubjectStatement(in), m_resource(NULL), m_decision(NULL), m_evidence(NULL)
{
    RTTI(SAMLAuthorizationDecisionStatement);
    fromDOM(m_document->getDocumentElement());
}

SAMLAuthorizationDecisionStatement::~SAMLAuthorizationDecisionStatement()
{
    if(m_bOwnStrings) {
        XMLString::release(&m_resource);
        XMLString::release(&m_decision);
    }
    for (vector<SAMLAction*>::const_iterator i=m_actions.begin(); i!=m_actions.end(); i++)
        delete (*i);
    delete m_evidence;
}

void SAMLAuthorizationDecisionStatement::ownStrings()
{
    if (!m_bOwnStrings) {
        m_resource=XML::assign(m_resource);
        m_decision=XML::assign(m_decision);
        m_bOwnStrings=true;
    }
}

void SAMLAuthorizationDecisionStatement::fromDOM(DOMElement* e)
{
    if (SAMLConfig::getConfig().strict_dom_checking) {
        if (XMLString::compareString(XML::SAML_NS,e->getNamespaceURI()))
            throw MalformedException("SAMLAuthorizationDecisionStatement::fromDOM() missing saml namespace on root element");
    
        if (XMLString::compareString(L(AuthorizationDecisionStatement),e->getLocalName())) {
            auto_ptr<saml::QName> type(saml::QName::getQNameAttribute(e,XML::XSI_NS,L(type)));
            if ((XMLString::compareString(L(Statement),e->getLocalName()) && XMLString::compareString(L(SubjectStatement),e->getLocalName())) ||
                !type.get() || XMLString::compareString(XML::SAML_NS,type->getNamespaceURI()) ||
                XMLString::compareString(L(AuthorizationDecisionStatementType),type->getLocalName()))
                throw MalformedException("SAMLAuthorizationDecisionStatement::fromDOM() requires saml:AuthorizationDecisionStatement at root");
        }
    }

    // attributes: resource, decision
    m_resource = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(Resource)));
    m_decision = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(Decision)));

    // the remaining top-level children (subject has been deserialized in the base
    // class) are either actions or evidence
    DOMElement* n=XML::getFirstChildElement(e);
    while(n) {
        if (XML::isElementNamed(n,XML::SAML_NS,L(Action))) {
            SAMLAction* a=new SAMLAction(n);
            m_actions.push_back(static_cast<SAMLAction*>(a->setParent(this)));
        }
        else if (XML::isElementNamed(n,XML::SAML_NS,L(Evidence))) {
            m_evidence = new SAMLEvidence(n);
            m_evidence->setParent(this);
        }
        n=XML::getNextSiblingElement(n);
    }

    checkValidity();
}

void SAMLAuthorizationDecisionStatement::setResource(const XMLCh* resource)
{
    if (XML::isEmpty(resource))
        throw SAMLException("resource cannot be null or empty");
        
    if (m_bOwnStrings)
        XMLString::release(&m_resource);
    else {
        m_resource=NULL;
        ownStrings();
    }
    m_resource=XML::assign(resource);
    setDirty();
}

void SAMLAuthorizationDecisionStatement::setDecision(const XMLCh* decision)
{
    if (XML::isEmpty(decision))
        throw SAMLException("decision cannot be null or empty");
        
    if (m_bOwnStrings)
        XMLString::release(&m_decision);
    else {
        m_decision=NULL;
        ownStrings();
    }
    m_decision=XML::assign(decision);
    setDirty();
}

void SAMLAuthorizationDecisionStatement::setActions(const Iterator<SAMLAction*>& actions)
{
    while (m_actions.size())
        removeAction(0);
    while (actions.hasNext())
        addAction(actions.next());
}

void SAMLAuthorizationDecisionStatement::addAction(SAMLAction* action)
{
    if (action) {
        action->setParent(this);
        m_actions.push_back(action);
        ownStrings();
        setDirty();
    }
    else
        throw SAMLException("action cannot be null");
}

void SAMLAuthorizationDecisionStatement::removeAction(unsigned long index)
{
    SAMLAction* kill=m_actions[index];
    m_actions.erase(m_actions.begin()+index);
    delete kill;
    ownStrings();
    setDirty();
}

void SAMLAuthorizationDecisionStatement::setEvidence(SAMLEvidence* evidence)
{
    delete m_evidence;
    m_evidence=NULL;
    if (evidence)
        m_evidence=static_cast<SAMLEvidence*>(evidence->setParent(this));
    ownStrings();
    setDirty();
}

DOMElement* SAMLAuthorizationDecisionStatement::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* s=doc->createElementNS(XML::SAML_NS,L(AuthorizationDecisionStatement));
    if (xmlns)
        s->setAttributeNS(XML::XMLNS_NS,L(xmlns),XML::SAML_NS);
    return s;
}

DOMNode* SAMLAuthorizationDecisionStatement::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLSubjectStatement::toDOM(doc, xmlns);
    DOMElement* s=static_cast<DOMElement*>(m_root);
    doc=s->getOwnerDocument();

    // Construct an AuthorizationDecisionStatement
    if (m_bDirty) {
        s->setAttributeNS(NULL,L(Resource),m_resource);
        s->setAttributeNS(NULL,L(Decision),m_decision);
    
        for (vector<SAMLAction*>::const_iterator i=m_actions.begin(); i!=m_actions.end(); i++)
            s->appendChild((*i)->toDOM(doc,false));
    
        if(m_evidence)
            s->appendChild(m_evidence->toDOM(doc,false));
    
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(s,XML::SAML_NS);
    }

    return m_root;
}

void SAMLAuthorizationDecisionStatement::checkValidity() const
{
    SAMLSubjectStatement::checkValidity();
    if (XML::isEmpty(m_resource) || XML::isEmpty(m_decision) || m_actions.empty())
        throw MalformedException("SAMLAuthorizationDecisionStatement invalid, requires a resource, a decision and at least one action");
}

SAMLObject* SAMLAuthorizationDecisionStatement::clone() const
{
    return new SAMLAuthorizationDecisionStatement(
        static_cast<SAMLSubject*>(m_subject->clone()),
        m_resource,
        m_decision,
        getActions().clone(), 
        m_evidence ? static_cast<SAMLEvidence*>(m_evidence->clone()) : NULL
        );
}
