
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gui_utils.c 1555 2006-11-15 09:04:35Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "environment.h"
#include "filelist_menu.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "lang.h"
#include "logger.h"
#include "main_menu.h"
#include "otk.h"
#include "oxine.h"
#include "playback_menu.h"
#include "playlist_menu.h"
#include "playlist_m3u.h"
#include "scheduler.h"

extern oxine_t *oxine;

static void repaint_message_dialog (void *oxine_p);

void
current_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    assert (oxine->current_menu);
    oxine->current_menu (oxine);
}


void
playback_ended_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    assert (oxine->current_menu);
    assert (oxine->playback_ended_menu);

    if ((oxine->current_menu == hide_user_interface)
        || (oxine->current_menu == repaint_message_dialog)
        || (oxine->current_menu == show_playback_menu_cb)
        || (oxine->current_menu == show_playback_info_cb)
        || (oxine->current_menu == show_playback_controls_cb)
        || (oxine->current_menu == show_playback_settings_p1_cb)
        || (oxine->current_menu == show_playback_settings_p2_cb)) {
        oxine->playback_ended_menu (oxine);
    } else {
        oxine->current_menu (oxine);
    }
}


void
backto_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    assert (oxine->backto_menu);
    oxine->backto_menu (oxine);
}


void
show_user_interface (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    assert (oxine->current_menu);
    assert (oxine->backto_menu);
    assert (oxine->playback_ended_menu);

    otk_draw (oxine->otk);

    oxine->user_interface_is_visible = true;
    odk_set_forward_events_to_xine (oxine->odk, false);
}


void
hide_user_interface (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    cancel_job (oxine->otk_clear_job);
    oxine->otk_clear_job = 0;

    otk_clear (oxine->otk);

    oxine->user_interface_is_visible = false;
    oxine->playback_controls_are_visible = false;
    oxine->stream_parameter_is_visible = false;

#ifdef HAVE_VDR
    if (odk_current_is_dvd (oxine->odk)
        || odk_current_is_vdr (oxine->odk))
#else
    if (odk_current_is_dvd (oxine->odk))
#endif
        odk_set_forward_events_to_xine (oxine->odk, true);
    else
        odk_set_forward_events_to_xine (oxine->odk, false);

    oxine->current_menu = hide_user_interface;
}


static void
set_time_string (void *oxine_p, otk_widget_t * widget)
{
    char time_str[640];

    time_t current = time (NULL);
    struct tm *brokentime = localtime (&current);
    strftime (time_str, 639, "%X", brokentime);
    otk_label_set_text (widget, time_str);
}


static void
show_clock (oxine_t * oxine)
{
    otk_widget_t *time = otk_label_new (oxine->otk, 780, 40, 780,
                                        OTK_ALIGN_RIGHT | OTK_ALIGN_VCENTER,
                                        "");
    set_time_string (oxine, time);
    otk_label_set_upcall (time, set_time_string, oxine);
    otk_widget_set_update (time, true);
}


otk_widget_t *
create_new_window (bool persistent, bool with_clock)
{
    otk_widget_t *w = otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);

    if (persistent)
        otk_window_keep (w, true);

    if (with_clock)
        show_clock (oxine);

    return w;
}


void
show_menu_background (const char *mrl)
{
    if (odk_current_is_playback_mode (oxine->odk)) {
        odk_play_background_stream (oxine->odk, mrl);
    } else {
        char *current = odk_current_get_mrl (oxine->odk);
        if (!current || !mrl || (strcmp (mrl, current) != 0)) {
            if (mrl && odk_play (oxine->odk, mrl, NULL, ODK_MODE_LOGO)) {
                return;
            }
            if (odk_play (oxine->odk, OXINE_BACKGROUNDS "/mainmenu.png",
                          NULL, ODK_MODE_LOGO)) {
                return;
            }
            fatal (_("Could not open or play background, leaving oxine!"));
            info (_("Please make sure, that your version of xine-lib "
                    "can play PNG files."));
            abort ();
        }
    }
}


#define STREAM_INFO_POS_X (20)
#define STREAM_INFO_POS_Y (20)
#define STREAM_INFO_WIDTH (800 - 40)
#define STREAM_INFO_ALIGN (OTK_ALIGN_LEFT | OTK_ALIGN_TOP)


static int
position_get (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    return odk_get_stream_param (oxine->odk, ODK_PARAM_POSITION);
}


static int
volume_get (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    return odk_get_stream_param (oxine->odk, ODK_PARAM_AUDIO_VOLUME);
}


void
show_stream_parameter (oxine_t * oxine, odk_stream_param_t param)
{
    if (!((oxine->current_menu == hide_user_interface)
          || (oxine->current_menu == show_playback_info_cb))) {
        current_menu_cb (oxine);
        return;
    }

    hide_user_interface (oxine);
    create_new_window (false, false);

    if ((param == ODK_PARAM_POSITION) || (param == ODK_PARAM_SPEED)) {
        otk_widget_t *s =
            otk_slider_new (oxine->otk, 50, 535, 730, 40, 0, 20, false,
                            OTK_SLIDER_HORIZONTAL, NULL, false, false,
                            0, 0, 100, 5, position_get, oxine, NULL, NULL);
        otk_widget_set_update (s, true);
        otk_widget_set_enabled (s, false);

        char *media_pplay_label = NULL;
        if (odk_get_stream_param (oxine->odk, ODK_PARAM_SPEED) ==
            ODK_SPEED_PAUSE) {
            media_pplay_label = "<";
        } else {
            media_pplay_label = ">";
        }
        otk_widget_t *b = otk_text_button_new (oxine->otk, 20, 535, 40, 40,
                                               media_pplay_label, NULL, NULL);
        otk_widget_set_selectable (b, OTK_SELECTABLE_NONE);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }

    else if ((param == ODK_PARAM_AUDIO_VOLUME)
             || (param == ODK_PARAM_AUDIO_MUTE)) {
        otk_widget_t *s =
            otk_slider_new (oxine->otk, 50, 535, 730, 40, 0, 20, false,
                            OTK_SLIDER_HORIZONTAL, NULL, false, false,
                            0, 0, 100, 5, volume_get, oxine, NULL, NULL);
        otk_widget_set_update (s, true);
        otk_widget_set_enabled (s, false);

        uint8_t *volume_button_bitmap = NULL;
        if (odk_get_stream_param (oxine->odk, ODK_PARAM_AUDIO_MUTE)) {
            volume_button_bitmap = odk_get_bitmap (BITMAP_MUTE);
        } else {
            volume_button_bitmap = odk_get_bitmap (BITMAP_VOLUME);
        }
        otk_widget_t *b = otk_bitmap_button_new (oxine->otk, 20, 535, 40, 40,
                                                 volume_button_bitmap, NULL,
                                                 NULL);
        otk_widget_set_selectable (b, OTK_SELECTABLE_NONE);
    }

    else {
        char msg[256];
        char *param_str = odk_get_stream_param_as_string (oxine->odk, param);

        if (param == ODK_PARAM_SPEED) {
            snprintf (msg, 256, "%s", param_str);
        }

        else {
            char *param_name = odk_get_stream_param_name (oxine->odk, param);
            snprintf (msg, 256, "%s: %s", param_name, param_str);
            ho_free (param_name);
        }

        ho_free (param_str);

        otk_label_new (oxine->otk, STREAM_INFO_POS_X, STREAM_INFO_POS_Y,
                       STREAM_INFO_WIDTH, STREAM_INFO_ALIGN, msg);
    }

    otk_draw (oxine->otk);
    oxine->otk_clear_job = schedule_job (6000, hide_user_interface, oxine);
    oxine->stream_parameter_is_visible = true;
}


static void
repaint_message_dialog (void *oxine_p)
{
    show_message_dialog (oxine->current_dialog_data.ok_cb,
                         oxine->current_dialog_data.ok_cb_data,
                         oxine->current_dialog_data.cancel_cb,
                         oxine->current_dialog_data.cancel_cb_data,
                         oxine->current_dialog_data.type,
                         oxine->current_dialog_data.background_mrl,
                         oxine->current_dialog_data.msg);
}


void
show_message_dialog (otk_cb_t ok_cb, void *ok_cb_data,
                     otk_cb_t cancel_cb, void *cancel_cb_data,
                     message_dialog_type_t type, const char *background_mrl,
                     const char *format, ...)
{
    char msg[1024];
    va_list args;
    va_start (args, format);
    vsnprintf (msg, 1024, format, args);
    va_end (args);

    ho_free (oxine->current_dialog_data.background_mrl);
    ho_free (oxine->current_dialog_data.msg);

    oxine->current_dialog_data.ok_cb = ok_cb;
    oxine->current_dialog_data.ok_cb_data = ok_cb_data;
    oxine->current_dialog_data.cancel_cb = cancel_cb;
    oxine->current_dialog_data.cancel_cb_data = cancel_cb_data;
    oxine->current_dialog_data.type = type;
    if (background_mrl)
        oxine->current_dialog_data.background_mrl =
            ho_strdup (background_mrl);
    else
        oxine->current_dialog_data.background_mrl = NULL;
    oxine->current_dialog_data.msg = ho_strdup (msg);

    hide_user_interface (oxine);
    oxine->current_menu = repaint_message_dialog;

    if (background_mrl)
        show_menu_background (background_mrl);
    else
        show_menu_background (OXINE_BACKGROUNDS "/mainmenu.png");
    create_new_window (false, true);

    /* First we count the lines we have to display. */
    int count = 0;
    char *cline = msg;
    while (cline) {
        cline = index (cline, '\n');
        if (cline)
            cline++;
        count++;
    }

    /* Display the border. */
    int h = count * 30 + 60;
    if (type != DIALOG_PLAIN)
        h += 70;
    int y = (600 - h) / 2;
    otk_border_new (oxine->otk, 100, y, 600, h);
    y += 60;

    /* Display the broken up message. */
    cline = msg;
    while (cline) {
        char *nline = index (cline, '\n');
        if (nline) {
            nline[0] = 0;
            nline++;
        }

        otk_label_new (oxine->otk, 400, y, 560,
                       OTK_ALIGN_CENTER | OTK_ALIGN_BOTTOM, cline);

        cline = nline;
        y += 30;
    }

    otk_widget_t *b;
    if (type == DIALOG_OK) {
        b = otk_text_button_new (oxine->otk, 320, y, 160, 40,
                                 _("OK"), ok_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focus (b);
    }

    else if (type == DIALOG_OK_CANCEL) {
        b = otk_text_button_new (oxine->otk, 230, y, 160, 40,
                                 _("OK"), ok_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focus (b);

        b = otk_text_button_new (oxine->otk, 410, y, 160, 40,
                                 _("Cancel"), cancel_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }

    else if (type == DIALOG_YES_NO) {
        b = otk_text_button_new (oxine->otk, 230, y, 160, 40,
                                 _("Yes"), ok_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focus (b);

        b = otk_text_button_new (oxine->otk, 410, y, 160, 40,
                                 _("No"), cancel_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }

    else if (type == DIALOG_RETRY_CANCEL) {
        b = otk_text_button_new (oxine->otk, 230, y, 160, 40,
                                 _("Retry"), ok_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focus (b);

        b = otk_text_button_new (oxine->otk, 410, y, 160, 40,
                                 _("Cancel"), cancel_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }

    show_user_interface (oxine);
}


void
playlist_rw_change_cb (playlist_t * playlist)
{
    assert (playlist == oxine->rw_playlist);
    playlist_save (playlist);
}


static void
playitem_play (oxine_t * oxine, playitem_t * playitem, otk_cb_t error_cb)
{
    if (!playitem) {
        playback_ended_menu_cb (oxine);
    } else {
        if ((strncasecmp (playitem->mrl, "http:/", 6) == 0)
            || (strncasecmp (playitem->mrl, "rtsp:/", 6) == 0)
            || (strncasecmp (playitem->mrl, "cdda:/", 6) == 0)
            || (strncasecmp (playitem->mrl, "dvd://", 6) == 0)
            || (strncasecmp (playitem->mrl, "vcd://", 6) == 0)
            || (strncasecmp (playitem->mrl, "dvb://", 6) == 0)
            || (strncasecmp (playitem->mrl, "v4l://", 6) == 0)) {
            /* This is only necessary when we start oxine and start playing a
             * video at once. If we call please_wait in this situation we
             * first load the background and a very short time after that
             * start playing the video. That would not look good! */
            if (odk_current_is_logo_mode (oxine->odk)
                || odk_current_is_playback_mode (oxine->odk)) {
                otk_cb_t current_menu = oxine->current_menu;
                please_wait ();
                oxine->current_menu = current_menu;
            }
        } else {
            otk_clear (oxine->otk);
        }

        if (!odk_play (oxine->odk, playitem->mrl,
                       playitem->subtitle_mrl, ODK_MODE_NORMAL)) {
            char *msg = NULL;
            otk_cb_t ok_cb = NULL;
            void *ok_cb_data = NULL;

            if (strncasecmp (playitem->mrl, "cdda:/", 6) == 0) {
                msg = ho_strdup_printf ("%s\n%s",
                                        _("Could not play Audio CD!"),
                                        _("Please insert an Audio CD!"));
                ok_cb = play_audiocd_cb;
            } else if (strncasecmp (playitem->mrl, "vcd://", 6) == 0) {
                msg = ho_strdup_printf ("%s\n%s",
                                        _("Could not play Video CD!"),
                                        _("Please insert a Video CD!"));
                ok_cb = play_vcd_cb;
            } else if (strncasecmp (playitem->mrl, "dvd://", 6) == 0) {
                msg = ho_strdup_printf ("%s\n%s",
                                        _("Could not play DVD!"),
                                        _("Please insert a DVD!"));
                ok_cb = play_dvd_cb;
            } else {
                msg = ho_strdup (_("Could not play title!"));
                ok_cb = error_cb;
                ok_cb_data = oxine;
            }

            playlist_remove (oxine->current_playlist, playitem);
            show_message_dialog (ok_cb, ok_cb_data,
                                 playback_ended_menu_cb, oxine,
                                 DIALOG_OK_CANCEL, NULL, msg);
            ho_free (msg);
        }
    }
}


void
playlist_play_first (oxine_t * oxine, otk_cb_t next_menu,
                     playlist_t * playlist)
{
    assert (playlist);
    oxine->backto_menu = next_menu;
    oxine->current_menu = next_menu;
    oxine->current_playlist = playlist;
    playitem_t *playitem = playlist_get_first (oxine->current_playlist);
    playitem_play (oxine, playitem, (otk_cb_t) playlist_play_next);
}


void
playlist_play_item (oxine_t * oxine, otk_cb_t next_menu,
                    playlist_t * playlist, playitem_t * playitem)
{
    assert (playlist);
    oxine->backto_menu = next_menu;
    oxine->current_menu = next_menu;
    oxine->current_playlist = playlist;
    playlist_set_current (oxine->current_playlist, playitem);
    playitem_play (oxine, playitem, (otk_cb_t) playlist_play_next);
}


void
playlist_play_next (oxine_t * oxine)
{
    playitem_t *playitem = playlist_get_next (oxine->current_playlist);
    playitem_play (oxine, playitem, (otk_cb_t) playlist_play_next);
}


void
playlist_play_prev (oxine_t * oxine)
{
    playitem_t *playitem = playlist_get_prev (oxine->current_playlist);
    playitem_play (oxine, playitem, (otk_cb_t) playlist_play_prev);
}
