// Tell emacs this is really -*- C++ -*-
#ifndef INCLUDE_FILTER_H
#define INCLUDE_FILTER_H

#include <stdlib.h>

class Connection;

#include "Buffer.h"

class Filter
{    
public:
    Filter(Connection * c, char * name);
    virtual ~Filter();
    
    // Consume data from this->input, appending to out. out may be non-empty.
    // Is not required to consume all of this->input; any remnants will be
    // re-presented to process() along with new input when new input arrives.
    // 
    // If the filter can detect easily that no changes are needed to the
    // data in 'input' (i.e. the filter will just act as a passthrough), then
    // an efficient way of doing the filtering (avoiding copying in most cases)
    // is to do:
    //
    //   input.transfer_to(out);
    //
    // Filters can safely manipulate the filter chain they belong to if:
    //  - all buffers are consistent before the manipulation, and
    //  - the filter immediately returns false after the manipulation (as its
    //    output buffer may have changed location)
    //
    // In all other cases, filters should return true.
    virtual bool process(Buffer &out) = 0;

    virtual char * query(char *);

    // Filters are ordered by this value. Ideally it should be unique across
    // a chain; if two filters have the same order value they will be ordered
    // arbitarily (based on heap location). Lower is earlier in the chain.
    virtual int order(void) const = 0;
    
    char *name;
    Buffer input; // Filled externally by FilterHandler & possibly on creation
    
protected:
    Connection * conn;
};

#endif // INCLUDE_FILTER_H
