/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     tifsave.c                                                      *
 *            Writing of TIFF files                                          *
 * Author:    Andreas Tille                                                  *
 * Date:      30.03.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <tiffio.h>
#include <stdlib.h>
#include "paul.h"

int GetTag(const char *key)
/* search number of the TAG which corresponds to key
 * --- Parameter: ---
 * char *key    : key of chunk to search
 * --- Return: ---
 * int  GetTag(): number of the TAG which corresponds to key
 */
{
  typedef struct { const char *key; int tag; } TAG_HASH;
  TAG_HASH table[] = { {ChunkNameSource,      TIFFTAG_DOCUMENTNAME},
                       {ChunkNameSoftware,    TIFFTAG_SOFTWARE},
                       {ChunkNameDescription, TIFFTAG_IMAGEDESCRIPTION},
                       {ChunkNameAuthor,      TIFFTAG_ARTIST},
                       {ChunkNameTyp,         TIFFTAG_MODEL},
                       {ChunkFilmType,        TIFFTAG_HOSTCOMPUTER},
                       {ChunkBrightness,      TIFFTAG_PAGENUMBER},
                       {ChunkContrast,        TIFFTAG_HALFTONEHINTS},
                       {ChunkReferenceFile,   TIFFTAG_MAKE},
                       {NULL,                 -1}
                     },
          *ap;

  for ( ap = table; ap->key != NULL; ap++ ) 
    if ( !strcmp(ap->key, key) ) return ap->tag;
  g_warning(_("No corresponding TAG for chunk %s defined."), key);
  return -1;
}


int WriteTIFFImage(PICTURE *bild, char *filename)
/* write TIFF files
 * --- Parameter: ---
 * PICTURE *bild        : image to write
 * char    *filename    : name of TIFF file (may differ from bild->file)
 * --- Return: ---
 * int WriteTIFFImage() : RET_ERR or RET_OK
 */
{
  TIFF       *tiff;
  register    unsigned char *ap, *sp;
  unsigned    char *scanline, *fip;
  int         scanlinesize, y;
  CHUNK      *cp;
  struct tm  *tmp;
  char       *buf;
  short       sval;
  const char *write_error = _("TIFFWriteScanline() - Error");
  
  g_return_val_if_fail ( (tiff = TIFFOpen(filename, "w")), RET_ERR ) ;

  TIFFSetField(tiff, TIFFTAG_IMAGELENGTH,     (uint32) bild->H);
  TIFFSetField(tiff, TIFFTAG_IMAGEWIDTH,      (uint32) bild->W);
  TIFFSetField(tiff, TIFFTAG_BITSPERSAMPLE,   8);

  /* COMPRESSION_CCITTRLE, COMPRESSION_CCITTFAX[34], COMPRESSION_CCITTRLEW
   *  --> Bits/sample must be 1 for Group 3/4 encoding/decoding.           */
  /* TIFFSetField(tiff, TIFFTAG_COMPRESSION,     COMPRESSION_PACKBITS);    *
   *  --> fast keine Kompression                                           */
  /* TIFFSetField(tiff, TIFFTAG_COMPRESSION,     COMPRESSION_LZW);         *
   *  --> nicht so doll                                                    */
  TIFFSetField(tiff, TIFFTAG_COMPRESSION,     COMPRESSION_DEFLATE);
  /* auch nicht berauschend, aber noch am besten von den getesteten */

  TIFFSetField(tiff, TIFFTAG_PLANARCONFIG,    PLANARCONFIG_CONTIG);
  TIFFSetField(tiff, TIFFTAG_SAMPLESPERPIXEL, bild->spp);
  TIFFSetField(tiff, TIFFTAG_ROWSPERSTRIP,    bild->H);

  if ( bild->spec ) 
    for ( cp = bild->spec; cp->key != NULL; cp++ ) {
      char *buf;
       
      if ( (buf = Chunk2Char(cp)) ) {
        y = GetTag(cp->key);
        if ( TIFFTAG_PAGENUMBER == y || TIFFTAG_HALFTONEHINTS == y) {
          if ( (1 == sscanf(buf, "%hi", &sval)) ) {
            TIFFSetField(tiff, y, sval);
            continue;
          }
        }
        if ( y >= 0 ) TIFFSetField(tiff, y, buf); 
      }
    }

  if ( !(bild->zeit) ) bild->zeit = time(NULL);
  tmp = gmtime(&(bild->zeit));
  buf = g_strdup_printf("%04i:%02i:%02i %02i:%02i:%02i",
                        1900+tmp->tm_year, 1+tmp->tm_mon, tmp->tm_mday, 
                        tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
  TIFFSetField(tiff, TIFFTAG_DATETIME, buf);
  FREE(buf);
  if ( bild->res ) {
    TIFFSetField(tiff, TIFFTAG_XRESOLUTION, (float)bild->res);
    TIFFSetField(tiff, TIFFTAG_YRESOLUTION, (float)bild->res);
    TIFFSetField(tiff, TIFFTAG_RESOLUTIONUNIT, RESUNIT_INCH);
  }
  if ( bild->x_offset != (unsigned long)-1 ) {
    TIFFSetField(tiff, TIFFTAG_XPOSITION, (float)bild->x_offset);
    TIFFSetField(tiff, TIFFTAG_YPOSITION, (float)bild->y_offset);
  }

  scanlinesize = TIFFScanlineSize(tiff);
  if ( bild->storepix == 3 ) {
    scanline = g_malloc(scanlinesize);
    if ( IsMonochrom(bild) ) {
      for ( ap = bild->DATA + 1, y = 0; y < bild->H; y++ ) {
        for ( fip = (sp = scanline) + scanlinesize; sp < fip; sp++, ap += 3)
          *sp = *ap;
        if ( TIFFWriteScanline(tiff, scanline, y, 0) < 0 ) {
          g_warning(write_error);
          break;
        }
      }
    } else {
      for ( fip = (ap = bild->DATA) + 3*bild->size, y = 0; ap < fip; 
            y++, ap += scanlinesize ) {
        if ( TIFFWriteScanline(tiff, ap, y, 0) < 0 ) break;
      }
    }
    FREE(scanline);
  } else {
    for ( ap = bild->DATA, y = 0; y < bild->H; y++, ap += scanlinesize ) 
      if ( TIFFWriteScanline(tiff, ap, y, 0) < 0 ) {
        g_warning(write_error);
        break;
      }
  }

  TIFFWriteDirectory(tiff);
  TIFFClose(tiff);
  return RET_OK;
}

