/*  Version 0.67  May, 2002
 *
 * pencam2-misc: This contains various routines common to both 
 * pencam and pensnap 
 *
 *  by Kevin Sisson  <kjsisson@bellsouth.net>
 *
 * Originally based on " pencam- a quick and dirty picture dumper for my 
 * pencam.       (C) Copyright 2001 by Bart Hartgers "
 *
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <term.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <math.h>
#include <usb.h>
#include <unistd.h>

#include "pencam2.h"
#include "unsharp.h"

#define pen_version 0.67


/****  stuff for kbhit DOS-like function  ****/

struct termios kborig, kbnew;
int peek = -1;

int kbhit (void)
{
	char ch;
	int nread;

	if (peek != -1) return 1;
	kbnew.c_cc[VMIN] = 0;
	tcsetattr (0, TCSANOW, &kbnew);
	nread = read (0,&ch,1);
	kbnew.c_cc[VMIN] = 1;
	tcsetattr (0, TCSANOW, &kbnew);

	if (nread == 1) {
		peek = ch;
		return 1;
	}
	return 0;
}

int readch (void)
{
	char ch;

	if (peek != -1) {
	        ch = peek;
		peek = -1;
		return ch;
	}
	read (0,&ch,1);
	return ch;
}

char get_key (void)
{
	char choice = ' ';

	fflush (stdout);		fflush (stdin);
	tcsetattr (0, TCSANOW, &kbnew);   /* turn on 'single character response' */

	for (;;) {
		if (kbhit()) break;
        }
	choice = (char) readch();   //	choice = tolower (choice); 
	tcsetattr (0,TCSANOW, &kborig);   /* turn off 'single character response' */	
	return choice;
}

/****  general purpose routines  ****/

void bad_value (void)
{
  
	fprintf (stderr,"\a\a Invalid entry. Press any key to try again.");
	get_key();
	return;
}


double get_val (float fmin, float fmax)
{
	double  r;
	int     i, j;
	char    ch, str[100];
	
	
	for (i=0; i<100; i++) {
loop1:
		j = 0;
		ch = get_key ();
		if (ch == '\n') {
			str[i] = ch;
			fprintf (stdout, "\n");
			goto cnvt;
		}
		if (ch == 0x7f) {
			fprintf (stdout, "\b \b");  /* backspace */
			i--;
			if (i < 0)
				i = 0;
			goto loop1;
		}
		if ((ch == '-') || (ch == '+') || (ch == '.'))
			j = 1;
		if (((ch >= '0') && (ch <= '9')) || (j == 1))
			str[i] = ch;
		else {
			fprintf (stderr, "\a\a");
			goto loop1;
		}
		fprintf (stdout, "%c", str[i]);
		if (i == 99) {
			fprintf (stderr, "\a\a\n Number too long! Press any key to try again ");
			return -11;
		}		
	}		

cnvt:
	if (str[0] == '\n')  {
		fprintf (stdout, "\n");
	     	return -1;
	} else 
		r = (double) atof (str);
	if ((r < fmin) || (r > fmax)) {
    		bad_value();	return -11;
	}
	else
		return r;
}


void clr_lines (int cnt)
{
	int i;
	
	for (i=0; i<cnt; i++) {
		fprintf (stdout, "\n");
	}
}

/********  pencam routines  ***********/

int pencam_get_num_pics (usb_dev_handle *pc, usb_stv *stv680)
{
	int i;
        unsigned char buffer[40];

	memset (buffer,0xff,40);
        if ((i = pencam_vendor(0, pc, 0x8d, 0x08, 0, buffer)) != 0x08) 
		return -1; 
	else
		return buffer[3];
}


int pencam_get_mode (usb_dev_handle *pc, usb_stv *stv680)
{
	unsigned char buffer[10];

        memset (buffer,0x0,10);

	/* get current mode */
	if (pencam_vendor (0, pc, 0x87, 0x08, 0, buffer) != 0x08) {
		fprintf (stderr, "\a\a Error getting camera mode in get_mode\n"); 
		return -1;
	}  else  {
		return buffer[0];
	}
}  /*  pencam_get_mode  */

int pencam_set_mode (usb_dev_handle *pc, usb_stv *stv680, int mode)
{
	unsigned char buffer[10];

	memset (buffer,0x0,10);

	buffer[0] = (unsigned char) mode;
	/*  set mode  */	
        if (pencam_vendor (1, pc, 0x07, 0x08, 0x0100, buffer) < 0) {
		pencam_vendor (0, pc, 0x80, 0x02, 0, buffer);
    		fprintf (stderr, "\a\a Set camera mode: Last error: %i,  command = 0x%x\n", 
			buffer[0], buffer[1]);
		return -1;
	}
        buffer[0] = 0xff;
	if (pencam_vendor (0, pc, 0x87, 0x08, 0, buffer) != 0x08) {
		fprintf (stderr, "\a\a Error getting camera mode in set_mode\n");
		return -1;
	} 
        if (mode != buffer[0]) {
		fprintf (stderr, "\a\a Error setting camera mode in set_mode\n");
		return -1;
	}
        return 0;
}  /*  pencam_set_mode  */

int pencam_write_config (usb_dev_handle *pc, usb_stv *stv680)
{
	FILE *of;
	char pathname[150];
	int i=0;
	
	strcpy (pathname, getenv("HOME"));
	strcat (pathname, "/.pencam2rc");

	if ((of = fopen(pathname,"wb")) == NULL) 
		return -1;

	/* write data in binary format; do entire structure at once */
	i = fwrite (stv680, sizeof(*stv680), 1, of);

	fclose (of);
	return i;	/* i = 1 -> OK, else error */
}  /* pencam_write_config  */

int pencam_read_config (usb_dev_handle *pc, usb_stv *stv680)
{
	FILE  *of;
	char  rc[150], ch1[10], ch2[10], choice = ' ';
	int   i=0, tmp1, tmp2, tmp3, tmp4, tmp5, tmp6, tmp7, tmp8, tmp9;
	usb_stv *stvtmp;
	
        if ((stvtmp = malloc (sizeof (*stvtmp))) == NULL) {
	        fprintf (stderr, "\a\a Couldn't malloc stvtmp struct\n");
    	        return -1;
	}
        if (sizeof (*stvtmp) != sizeof (*stv680)) {
	        fprintf (stderr, "\a\a stv struct sizes don't agree\n");
    	        return -1;
	}
	
	tmp1 = stv680->SupportedModes;		tmp2 = stv680->currentMode;
	tmp3 = stv680->origMode;		tmp4 = stv680->SensorID;
	tmp5 = stv680->SensorIDrev;		tmp6 = stv680->CIF;
	tmp7 = stv680->VGA;			tmp8 = stv680->QCIF;
	tmp9 = stv680->QVGA;
	strcpy (ch1, stv680->bus_dirname);	strcpy (ch2, stv680->dev_filename);
	
	strcpy (rc, getenv("HOME"));
	strcat (rc, "/.pencam2rc");

	if ((of = fopen (rc,"r")) == NULL) 
		return -1;
	i = fread (stvtmp, sizeof(*stvtmp), 1, of);
    	fclose (of);

	/* check if camera is different */
	if ((stvtmp->SensorID == stv680->SensorID) && 
	    (stvtmp->SensorIDrev == stv680->SensorIDrev))
		goto prc2;
	else {
prc1:	
		fflush (stdout);	fflush (stderr);
		fprintf (stderr, "\n\a\a SensorID %i.%i does not match original SensorID %i.%i \n",
			 stvtmp->SensorID, stv680->SensorIDrev, 
			 stv680->SensorID, stvtmp->SensorIDrev);
		fprintf (stderr, " Did you change cameras? Are you sure you want to proceed? (y/n) ");
		choice = get_key();
		if (choice == 'n') {
			fprintf (stdout, "\n");
			free (stvtmp);
			return 0;
		}
		if (choice == 'y') 
			goto prc2;
		fprintf (stderr, "\n\a\a Invalid entry. Press any key to try again ");
		get_key();  fprintf (stdout, "\n");
		goto prc1;
	}  /*  if  */

prc2:
	memcpy (stv680, stvtmp, sizeof (*stvtmp));
	stv680->SupportedModes = tmp1;		stv680->currentMode = tmp2;
	stv680->origMode = tmp3;		stv680->SensorID = tmp4;
	stv680->SensorIDrev = tmp5;		stv680->CIF = tmp6;
	stv680->VGA = tmp7;			stv680->QCIF = tmp8;
	stv680->QVGA = tmp9;
	strcpy (stv680->bus_dirname, ch1);	strcpy (stv680->dev_filename, ch2);
	free (stvtmp);

	stv680->num_pics = pencam_get_num_pics (pc, stv680);
	return i;	/* i = 1 -> OK, 0 = canceled, else error */

}  /* pencam_read_config  */


/*****   menu routines   *****/

void menu_header (usb_dev_handle *pc, usb_stv *stv680, int num_cams, 
		  camera_count *camera_addr)
{
	char str1[17], str2[17], str3[17], str0[17];
	int i;
	
	if ((i = pencam_get_num_pics (pc, stv680)) < 0)
		fprintf (stderr, "\a\a Error getting # of images in menu_header\n\n");
	else
		stv680->num_pics = i;

        strcpy(str1, " ");  strcpy(str2, " ");  strcpy(str3, " ");  strcpy(str0, " ");    
	if (stv680->VGA == 1)
    	    strcpy(str1, "VGA (640x480) ");
        if (stv680->QVGA == 1)
	    strcpy(str3, "QVGA (320x240) ");
        if (stv680->CIF == 1)
	    strcpy(str0, "CIF (352x288) ");
        if (stv680->QCIF == 1)
	    strcpy(str2, "QCIF (176x144) ");

        fprintf(stdout, "\n *  *  *  *  *   PENCAM2 ver %3.2f for STV0680 usb cameras   *  *  *  *  *\n", pen_version);

//	if (num_cams >= 2) {
		fprintf (stdout, " Cameras found: ");
		for (i=1; i<=num_cams; i++)
			fprintf (stdout, "%i @ %s:%s  **  ", i, camera_addr->bus_dirname[i],
				 camera_addr->dev_filename[i]);
		fprintf(stdout, "\n *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  * \n");	
//	} else
//		fprintf (stdout, "\n");
	fprintf(stdout, " Active Camera @     Camera Pictures     Firmware     ASIC rev     Sensor ID\n"); 
        fprintf(stdout, "     %s:%s             %3i              %4.2f         %4.2f         %i.%i\n", 
		    stv680->bus_dirname, stv680->dev_filename, stv680->num_pics, stv680->FW,
		    stv680->ASICrev, stv680->SensorID, stv680->SensorIDrev);
        fprintf(stdout, " *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  * \n");	
	fprintf(stdout, " Supported formats: %s %s %s %s\n", str0, str1, str2, str3); 
        fprintf(stdout, " *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  * \n");
	fprintf(stdout, "\n");
        	
}  /* menu_header */

char common_menu (usb_dev_handle *pc, usb_stv *stv680, char *cam_snap)
{
	char pathname[100], basename[50], choice = ' ';
        int light_type = 3, sharpen_percent = 40, sharp_toggle = 1, i;
	int saturate_toggle = 1, saturation_data = 1;
	char mode[4][19], lt[4][19];
	

	clr_lines (24);   /*  clear screen  */

	for (i=0; i<4; i++)
		strcpy (mode[i], " ");
		
	if (stv680->VGA == 1)
		strcpy (mode[1], "VGA (640x480) ");
    	if (stv680->QVGA == 1)
		strcpy (mode[3], "QVGA (320x240)");
	if (stv680->CIF == 1)
    		strcpy (mode[0], "CIF (352x288) ");
	if (stv680->QCIF == 1)
    		strcpy (mode[2], "QCIF (176x144)");

        strcpy (lt[0], " (Natural)     ");	strcpy (lt[1], " (Fluorescent) ");
        strcpy (lt[2], " (Incandescent)");	strcpy (lt[3], " (Auto Select) ");

	if ((i = pencam_get_num_pics (pc, stv680)) < 0)
		fprintf (stderr, "\a\a Error getting # of images in common_menu\n\n");
	else
		stv680->num_pics = i;

	if ((i = pencam_get_mode (pc, stv680)) < 0)
		fprintf (stderr, "\a\a Error getting mode in common_menu\n\n");
	else
		stv680->currentMode = i;

        if (strcmp(cam_snap, "PENCAM") == 0) {
		strcpy (pathname, stv680->pathname);
		strcpy (basename, stv680->basename); 
		light_type = stv680->light_type;
		sharp_toggle = stv680->sharp_toggle;
		sharpen_percent = stv680->sharpen_percent;
		saturate_toggle = stv680->saturate_toggle;
		saturation_data = stv680->saturation_data;
	} else if (strcmp(cam_snap, "PENSNAP") == 0) {	
		strcpy (pathname, stv680->pathname_ps);
		strcpy (basename, stv680->basename_ps); 
		light_type = stv680->light_type_ps;
		sharp_toggle = stv680->sharp_toggle_ps;
		sharpen_percent = stv680->sharpen_percent_ps;
		saturate_toggle = stv680->saturate_toggle_ps;
		saturation_data = stv680->saturation_data_ps;
	} 

        fprintf(stdout, "\n\n\n *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  * \n");	
    	fprintf(stdout, "    %s MENU:         YOUR CHOICES ARE:\n", cam_snap); 
	fprintf(stdout, " *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  * \n\n");	

	if (strcmp (cam_snap, "PENCAM") == 0)
	        fprintf (stdout, " Get Image:     i = All, Individual, or Range\n\n"); 

	fprintf (stdout, " Change name:   p = Path,   f = File   (%s%s)\n\n", 
			pathname, basename); 
    
        fprintf (stdout, " Filters:       l = Gamma Correction-White Balance "); 
    	if (light_type == -1)
        	fprintf (stdout, " (OFF)\n\n"); 
	else
		fprintf (stdout, "%15.15s\n\n", lt[light_type]);

        switch (sharp_toggle) {
		case -1:
			fprintf (stdout, "                t = Sharpness (OFF) \n\n"); 
			break;
		case 0:
			fprintf (stdout, "                t = Sharpness (unsharp mask)      h = Change values\n\n"); 
			break;
		case 1:
			fprintf (stdout, "                t = Sharpness (sharpen @ %3i%%)    g = Change value\n\n", 
				sharpen_percent);
			break;
		case 2:
			fprintf (stdout, "                t = Sharpness (both)       Change values separately\n\n"); 
			break;
	}

        if (saturate_toggle == -1)
		fprintf (stdout, "                s = Saturate (OFF)\n\n"); 
	if (saturate_toggle == 1)
		fprintf (stdout, "                s = Saturate (ON @ %3i%%)          m = Change value\n\n", 
			saturation_data); 

	fprintf (stdout, " Config File:   w = Write file                    e = Read file\n\n"); 

	if (strcmp(cam_snap, "PENSNAP") == 0) {	
	        fprintf (stdout, " Misc:          c = Picture mode: "); 
		fprintf (stdout, " %s (orig = %4.4s, ss = %4.4s)\n\n", 
			 mode[stv680->currentMode], mode[stv680->origMode], 
			 mode[stv680->ssMode]); 

	        if (stv680->beep_yes == 1)
		    	fprintf (stdout, "                u = Camera beep ( ON)");
		else
			fprintf (stdout, "                u = Camera beep (OFF)");	
	        if (stv680->overwrite == 1)
			fprintf (stdout, "         v = Overwrite image ( ON)\n\n");
		else
			fprintf (stdout, "         v = Overwrite image (OFF)\n\n");
	        if (stv680->annotate_ps == 1) {
			fprintf (stdout, "                o = Annotate pic ( ON)");
			fprintf (stdout, "        k = Message (%s)\n\n",
				 stv680->picmsg_ps);
		} else
			fprintf (stdout, "                o = Annotate pic (OFF)\n\n");

	}  /*  PENSNAP  */

	if (strcmp (cam_snap, "PENCAM") == 0)
		fprintf (stdout, "                b = Brightness (%4.2f)             j = Image count (%3i)\n\n", 
			 stv680->brightness, stv680->num_pics);

	if (strcmp (cam_snap, "PENSNAP") == 0) 
		fprintf (stdout, "                b = Brightness (%4.2f)         d = Time delay (%4i sec)\n\n", 
			 stv680->brightness_ps, stv680->time_delay);
	
	fprintf(stdout, "                z = Return to main menu");
	
	if (strcmp(cam_snap, "PENSNAP") == 0) 	
	        fprintf (stdout, "       x = Start/stop autosnap\n\n");

	if (strcmp(cam_snap, "PENCAM") == 0) 	
	        fprintf (stdout, "           r = Erase all pictures\n\n");
		
        fprintf (stdout, " Enter your choice: ");
	fflush (stdout);
        choice = get_key(); 
	return choice;

}  /*  common_menu  */


int common_image_process (usb_dev_handle *pc, usb_stv *stv680, unsigned char *raw,
			  unsigned char *pic, char *cam_snap)
{
	unsigned long int bufsize = stv680->rawbufsize;
        int s, sharp_toggle = 0, saturate_toggle = 0;
	unsigned char *mask;
        int  i, vw, vh;
	UnsharpMaskParams unp;

	if (strcmp(cam_snap, "PENCAM") == 0) {
		unp.radius = stv680->unsharpen_radius;
		unp.amount = stv680->unsharpen_amount;
		unp.threshold = stv680->unsharpen_threshold;
		saturate_toggle = stv680->saturate_toggle;
		sharp_toggle = stv680->sharp_toggle;
	}
	if (strcmp(cam_snap, "PENSNAP") == 0) {
		unp.radius = stv680->unsharpen_radius_ps;
		unp.amount = stv680->unsharpen_amount_ps;
		unp.threshold = stv680->unsharpen_threshold_ps;
		saturate_toggle = stv680->saturate_toggle_ps;
		sharp_toggle = stv680->sharp_toggle_ps;
	}

        /* picture processing  */
	bayer_unshuffle (raw, pic, stv680); 
	light_enhance (stv680, pic, cam_snap);
	bayer_demosaic (stv680, pic);

	/* alternate demosaic, with some sharpening */
//	demosaic_sharpen (stv680->vwidth, stv680->vheight, pic, pic, 2, 3);

	/* fix top and bottom row, left and right side */
	vw = stv680->vwidth;	i = vw * 3;	vh = stv680->vheight;
	memcpy (pic, (pic + i), i);
	memcpy ((pic + (vh * i)), (pic + ((vh - 1) * i)), i);
	for (s = 0; s < vh; s++) {
		i = s * vw * 3;
		memcpy ((pic + i), (pic + i + 3), 3);
		memcpy ((pic + i + (vw * 3)), (pic + i + (vw - 1) * 3), 3);
	}

	/* sharpen filters: -1 = disable, 0 = unsharp mask, 1 = sharpen, 2 = both  */
	if (sharp_toggle >= 0)
	{
		mask = malloc (3*bufsize);
		if (mask == NULL) {
			fprintf (stderr, "\a\a Not enough memory for sharp buffer!\n");
			free (raw);
			free (pic);
			return -1;
	        }
	
		if (sharp_toggle == 0) {
			unsharp_mask (pic, mask, vw, vh, 3, unp);
			memcpy (pic, mask, 3*bufsize);
		}
		if (sharp_toggle == 1) {
			sharpen (pic, mask, stv680, cam_snap);
			memcpy (pic, mask, 3*bufsize);
		}
		if (sharp_toggle == 2) {
			unsharp_mask (pic, mask, vw, vh, 3, unp);
			memcpy (pic, mask, 3*bufsize);
			sharpen (pic, mask, stv680, cam_snap);
			memcpy (pic, mask, 3*bufsize);
		}
		free (mask);
	}  /*  sharp toggle  */

	if (saturate_toggle == 1) {
		mask = malloc (3*bufsize);
		if (mask == NULL) {
			fprintf (stderr, "\a\a Not enough memory for saturate buffer!\n");
			free (raw);
			free (pic);
			return -1;
	        }

		hue_saturation (pic, mask, stv680, cam_snap);
	        memcpy (pic, mask, 3*bufsize);
	        free (mask);
	}

	if ((stv680->annotate_ps == 1)  && (strcmp(cam_snap, "PENSNAP") == 0))
		add_text (pic, stv680->vwidth, stv680->vheight, stv680->picmsg_ps);

	return 0;
}  /*  common_image_process  */


int common_save_image (usb_stv *stv680, int pic_no, unsigned char *pic, char *cam_snap)
{
	int   i;
        char  filename[160], pathandfilename[160];
	FILE  *of;
	
	
	/* done processing, write data to disk  */	
	if (strcmp (cam_snap, "PENCAM") == 0) {
    		strcpy (pathandfilename, stv680->pathname);
		strcat (pathandfilename, stv680->basename);
	} else if (strcmp (cam_snap, "PENSNAP") == 0) {
    		strcpy (pathandfilename, stv680->pathname_ps);
		strcat (pathandfilename, stv680->basename_ps);
	}

	/*  This creates numbered filenames  */
	sprintf (filename, "%s%03i%s", pathandfilename, pic_no, ".ppm");

	if ((of = fopen (filename,"wb")) == NULL) 
		return -1;

	fprintf (of,"P6\n# %s image\n%d %d\n255\n", cam_snap, stv680->vwidth, 
	         stv680->vheight);
	for (i=0; i<stv680->vheight; ++i) {
		fwrite (pic + (i*stv680->vwidth*3), 
		       (stv680->vwidth)*3, 1, of );
	}
	fclose (of);

	/*  routine to convert to jpeg for webcam.cgi, etc.  */
	if ((stv680->convert == 1)  && (strcmp (cam_snap, "PENSNAP") == 0)) {
		if (system (stv680->convert_path) != 0)
			fprintf (stderr, "\a\a Error with CONVERT routine!"); 
	}

	return 0;

}  /*  common_save_image  */


void common_do_function (usb_dev_handle *pc, usb_stv *stv680, char *cam_snap, char choice)
{
	int   i, light_type = 3, sharp_toggle = 1, saturate_toggle = 1;
	int   unsharpen_threshold = 4, sharpen_percent = 55, saturation_data = 40;
        char  pathname[100], basename[50], tempname[160];
        float   urmin, urmax, uamin, uamax, utmin, utmax, urd, utd;
        double  uad, r;
	double  unsharpen_radius = 7.0, unsharpen_amount = 0.5;
	float   brightness = 1.0;
	
        /*  set up min/max and defaults for sharpness.  
	    urmin = (u)nsharp mask.radius min; etc.,  */
        urmin = 1; urmax = 20; uamin = 0; uamax = 3; utmin = 0; utmax = 25;
	urd = 7.0; uad = 0.5; utd = 4;

	if (strcmp(cam_snap, "PENCAM") == 0) {
    		strcpy (pathname, stv680->pathname);
		strcpy (basename, stv680->basename);
		light_type = stv680->light_type;
		saturate_toggle = stv680->saturate_toggle;
		sharp_toggle = stv680->sharp_toggle;
		brightness = stv680->brightness;
		unsharpen_radius = stv680->unsharpen_radius;
		unsharpen_amount = stv680->unsharpen_amount;
		unsharpen_threshold = stv680->unsharpen_threshold;
		sharpen_percent = stv680->sharpen_percent;
		saturation_data = stv680->saturation_data;
	}
	if (strcmp(cam_snap, "PENSNAP") == 0) {
    		strcpy (pathname, stv680->pathname_ps);
		strcpy (basename, stv680->basename_ps);
		light_type = stv680->light_type_ps;
		saturate_toggle = stv680->saturate_toggle_ps;
		sharp_toggle = stv680->sharp_toggle_ps;
		brightness = stv680->brightness_ps;
		unsharpen_radius = stv680->unsharpen_radius_ps;
		unsharpen_amount = stv680->unsharpen_amount_ps;
		unsharpen_threshold = stv680->unsharpen_threshold_ps;
		sharpen_percent = stv680->sharpen_percent_ps;
		saturation_data = stv680->saturation_data_ps;
	}

	switch (choice) {

		case 'l':
			light_type++;
			/* -1=Disable, 0=Natural, 1=Fluor, 2=Incan, 3=Auto */
			if (light_type > 3)
		    		light_type = -1;
			break;

		case 'b':
			fprintf (stdout, " Enter Brightness factor (0.75 - 1.0): ");
			r = get_val (0.75, 1.0);
			if (r <= -1)  
				break;
			brightness = r;
			break;

		case 'p':
			fprintf (stdout, " Enter the new path name: ");
			scanf (" %s", tempname);
			if ((strlen(tempname) <= 1) || (strlen(tempname) > 98))
			{
				fprintf (stderr, "\a\a Name too short or too long. Press any key to try again");
				get_key(); 
				break;
			}
			else
				strcpy (pathname, tempname);

			if (pathname[strlen(pathname)-1] != '/')
				strcat (pathname, "/");

			break;		

    	        case 'f':
			fprintf (stdout, " Enter the new base file name: ");
			scanf (" %s", tempname);
			if ((strlen (tempname) <= 1) || (strlen (tempname) > 48))
			{
				fprintf (stderr, "\a\a Name too short or too long. Press any key to try again");
				get_key(); 
				break;
			}
			else
				strcpy (basename, tempname);
			break;
				
		case 'h':
	    		if ((sharp_toggle != 0))
				break;
	    		fprintf (stdout, "\n BE CAREFUL!! Image artifacts can be produced by changing these parameters.\n");
	    		fprintf (stdout, " If you increase 'amount', you should decrease 'radius'.\n");
			fprintf (stdout, " Enter radius (%3.2f <= r <= %3.2f, default: %3.2f,  current: %5.2lf) ", 
				 urmin, urmax, urd, unsharpen_radius);
			r = get_val (urmin, urmax);
			if (r <= -1)
				goto hnxt1;
			else   
				unsharpen_radius = r;
hnxt1:
			fprintf(stdout, " Enter amount (%3.2f <= a <= %3.2f, default: %4.2f,  current: %4.2lf) ", 
				uamin, uamax, uad, unsharpen_amount);
			r = get_val (uamin, uamax);
			if (r <= -1)
				goto hnxt2;
			else   
				unsharpen_amount = r;
hnxt2:
			fprintf (stdout, " Enter threshhold (%i <= t <= %i, default: %i,  current: %i) ", 
				(int) utmin, (int) utmax, (int) utd, unsharpen_threshold);
			r = get_val (utmin, utmax);
			if (r <= -1)
				break;
			else   
				unsharpen_threshold = (int) rint(r);
			break;

		case 't':
			sharp_toggle++;
			/* -1 = disable, 0 = unsharp_mask, 1 = sharpen, 2 = both */
			if (sharp_toggle > 2)
				sharp_toggle = -1;
			break;		

		case 'g':
	    		if (sharp_toggle != 1)  
				break;
	    		fprintf (stdout, " Enter percent to sharpen (0-100 current: %i) ", 
				 sharpen_percent);
			r = get_val (0, 100);
			if (r <= -1) 
				break;
			sharpen_percent = (int) r;
			break;

		case 's':
			saturate_toggle = -saturate_toggle;
			break;

		case 'm':
	    		if (saturate_toggle == -1) 
				break;
	    		fprintf (stdout, " Enter percent to saturate (-100:100 current: %i) ", 
				 saturation_data);
			r = get_val (-100, 100);
			if ((r == -11) || (r == -1)) 
				break;
			saturation_data = (int) r;
			break;

		case 'j':
			if ((i = pencam_get_num_pics (pc, stv680)) < 0) 
				fprintf (stderr, "\a\a Couldn't get # of images !\n");
			else
				stv680->num_pics = i;
			break;

		case 'w':
			fprintf (stdout, " Writing Config file... ");
			i = pencam_write_config (pc, stv680);
			if (i == 1) 
				fprintf (stdout, " Config file values written\n");
			else
				fprintf (stderr, "\a\a Error writing Config file!\n");
				fprintf (stdout, " Press any key to continue ");
				get_key();
			break;

	        case 'e':
			fprintf(stdout, " Reading Config file... ");
			i = pencam_read_config (pc, stv680);
			if (i == 1) 
				fprintf (stdout, " Config file read\n");
			else if (i == 0)
				fprintf (stdout, " Read canceled by user\n");
			else
				fprintf (stderr, "\a\a Error reading Config file!\n");
			fprintf (stdout, " Press any key to continue ");
    			get_key();
			return;
			break;

	        default:
		        break;
	} /* switch */

	if (strcmp(cam_snap, "PENCAM") == 0) {
    		strcpy (stv680->pathname, pathname);
		strcpy (stv680->basename, basename);
		stv680->light_type = light_type;
		stv680->saturate_toggle = saturate_toggle;
		stv680->sharp_toggle = sharp_toggle;
		stv680->brightness = brightness;
		stv680->unsharpen_radius = unsharpen_radius;
		stv680->unsharpen_amount = unsharpen_amount;
		stv680->unsharpen_threshold = unsharpen_threshold;
		stv680->sharpen_percent = sharpen_percent;
		stv680->saturation_data = saturation_data;
	}
	if (strcmp(cam_snap, "PENSNAP") == 0) {
    		strcpy (stv680->pathname_ps, pathname);
		strcpy (stv680->basename_ps, basename);
		stv680->light_type_ps = light_type;
		stv680->saturate_toggle_ps = saturate_toggle;
		stv680->sharp_toggle_ps = sharp_toggle;
		stv680->brightness_ps = brightness;
		stv680->unsharpen_radius_ps = unsharpen_radius;
		stv680->unsharpen_amount_ps = unsharpen_amount;
		stv680->unsharpen_threshold_ps = unsharpen_threshold;
		stv680->sharpen_percent_ps = sharpen_percent;
		stv680->saturation_data_ps = saturation_data;
	}

}  /*  common_do_function  */

