/* create and handle child processes */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>

#include <syslog.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "server.h"
#include "lk.h"

struct pkspxy_child
{
  pid_t pid;
  char *query;
  struct pkspxy_conn *conn;
  struct pkspxy_child *next;
};

static struct pkspxy_child *Children = NULL;

void pkspxy_collect_children (void)
{
  pid_t pid;
  int status;

  struct pkspxy_child *p, **q;

  if ((pid = waitpid (-1, &status, WNOHANG)) > 0 &&
      (WIFEXITED (status) || WIFSIGNALED (status)))
  {
    p = Children;
    q = &Children;

    if (Debug)
      fprintf (stderr, "%s: child %d died.\n", Progname, pid);

    while (p)
    {
      if (p->pid == pid)
      {
	if (WIFSIGNALED (status) || (WEXITSTATUS (status) != 0))
	  pkspxy_postponed_append (p->query);

	if (p->conn)
	{
	  pkspxy_notify_conn (p->conn);
	  
	  if (WIFSIGNALED (status))
	    syslog (LOG_WARNING, "child %d [fd = %d] died from signal %d.",
		    pid, p->conn->fd, WSTOPSIG (status));
	  else
	    syslog (LOG_INFO, "child %d [fd = %d] terminated with exit status %d.",
		    pid, p->conn->fd, WEXITSTATUS (status));
	}

	*q = p->next;
	free (p->query);
	free (p);
	break;
      }

      p = p->next;
    }
  }
}

void pkspxy_add_child (pid_t pid, const char *query, struct pkspxy_conn *conn)
{
  struct pkspxy_child *p = calloc (1, sizeof (struct pkspxy_child));

  if (!p) return;

  p->conn = conn;
  p->pid = pid;
  if ((p->query = strdup (query)) == NULL)
  {
    free (p);
    return;
  }
  
  p->next = Children;
  Children = p;
}

static void close_fds (int fd)
{
  int i;
  int s = getdtablesize ();

  for (i = 0; i < s; i++)
    if (i != fd)
      close (i);
}

int pkspxy_spawn_request (const char *query, unsigned int timeout,
			  struct pkspxy_conn *conn)
{
  pid_t childpid;
  char buff[2048];
  
  if (Debug)
  {
    fprintf (stderr, "%s: pkspxy_spawn_request (%s, %d, fd=%d)\n",
	     Progname, query, timeout, conn ? conn->fd : -1);
  }

  if ((childpid = fork ()) < 0)
  {
    syslog (LOG_ERR, "fork failed.");
    return -1;
  }

  else if (childpid == 0)
  {
    if (Debug > 1) sleep (10);
    closelog ();
    close_fds (-1);

    if (conn)
      snprintf (buff, sizeof (buff), "%s.request[fd=%d]", Progname, conn->fd);
    else
      snprintf (buff, sizeof (buff), "%s.request", Progname);

    openlog (buff, LOG_PID, LOG_DAEMON);
    alarm (timeout); /* may wish to die gracefully */
    _exit (pkspxy_update_cache (NULL, 0, query) > 0 ? 0 : 1);
  }
  else
  {
    if (Debug) fprintf (stderr, "%s: child pid = %d\n", Progname, childpid);
    pkspxy_add_child (childpid, query, conn);
  }

  return 0;
}

int pkspxy_spawn_reply (const char *query, time_t t, int fd)
{
  char spoolfile[_POSIX_PATH_MAX];
  char buff[2048];
  FILE *fp = NULL, *sfp = NULL;
  struct stat sb;
  int spoolfd = -1;
  
  const char *errstr = "internal error";

  pid_t childpid;

  if (Debug && query)
  {
    fprintf (stderr, "%s: pkspxy_spawn_reply (%s, %d, %d)\n",
	     Progname, query, t, fd);
  }

  if ((childpid = fork ()) < 0)
  {
    syslog (LOG_ERR, "fork failed");
    return -1;
  }
  else if (childpid > 0)
  {
    if (Debug) fprintf (stderr, "%s: child pid = %d\n", Progname, childpid);
    return 0;
  }

  if (Debug > 1) sleep (10);
  /* begin child code */

  closelog ();
  close_fds (fd);
  alarm (600); /* may wish to die gracefully */
  
  snprintf (buff, sizeof (buff), "%s.reply[fd=%d]", Progname, fd);
  openlog (buff, LOG_PID, LOG_DAEMON);

  if ((fp = fdopen (fd, "w+")) == NULL)
  {
    errstr = strerror (errno);
    goto internal;
  }

  if (!query)
  {
    errstr = "no query";
    goto internal;
  }
  
  pkspxy_spool_file (spoolfile, sizeof (spoolfile), query);
  
  /* XXX log */
  if ((spoolfd = open (spoolfile, O_RDONLY)) == -1)
  {
    errstr = strerror (errno);
    goto internal;
  }
  
  if (lk_lock (spoolfd, 0) == -1)
  {
    errstr = strerror (errno);
    goto internal;
  }
  
  if (fstat (spoolfd, &sb) == -1)
  {
    errstr = strerror (errno);
    goto internal;
  }

  if ((sfp = fdopen (spoolfd, "r")) == NULL)
  {
    errstr = strerror (errno);
    goto internal;
  }

  if (t >= sb.st_mtime || sb.st_size == 0)
  {
    syslog (LOG_INFO, "[fd = %d] sending \"no new data\"", fd);
    fputs ("HTTP/1.0 204 no new data\r\n\r\n", fp);
    fflush (fp);
    fclose (fp);
    _exit (0);
  }

  /* else */

  syslog (LOG_INFO, "[fd = %d] sending key material.", fd);
  fputs ("HTTP/1.0 200 key material follows\r\nContent-Type: text/html\r\n\r\n", fp);  

  while (fgets (buff, sizeof (buff), sfp))
    fputs (buff, fp);
  
  fclose (fp);
  lk_unlock (spoolfd);
  fclose (sfp);
  _exit (0);

  /* something went wrong */

  internal:

  if (fp)
    fprintf (fp, "HTTP/1.0 500 %s\r\n\r\n", errstr);

  syslog (LOG_ERR, "error: %s", errstr);
  
  if (spoolfd >= 0)
    lk_unlock (spoolfd);

  if (sfp)
    fclose (sfp);
  else if (spoolfd >= 0)
    close (spoolfd);

  if (fp)
    fclose (fp);
  else
    close (fd);

  _exit (1);

}

