/* the select-based event loop */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <syslog.h>

#include "server.h"
#include "hkp.h"

void pkspxy_select_loop (void)
{
  fd_set read_fds;
  int n, ls, as, i;

  struct timeval tv;
  time_t t;

  struct sockaddr_in sin;
  socklen_t slen;
  
  struct pkspxy_conn *cp, *cq;

  char buff[4096];
  ssize_t l;

  if (Debug)
    fprintf (stderr, "%s: Entering the main loop.\n", Progname);
  
  memset (&sin, 0, sizeof (sin));
  sin.sin_port = htons ((short) Port);
  sin.sin_family = AF_INET;

  if ((ls = socket (AF_INET, SOCK_STREAM, IPPROTO_IP)) < 0)
  {
    if (Debug)
      fprintf (stderr, "%s: socket failed (%s).\n", Progname, strerror (errno));
    
    return;

  }

  n = 1;
  setsockopt (ls, SOL_SOCKET, SO_REUSEADDR, &n, sizeof (n));

  
  if (bind (ls, (struct sockaddr *) &sin, sizeof (sin)) == -1)
  {
    if (Debug)
      fprintf (stderr, "%s: bind failed (%s).\n", Progname, strerror (errno));
    
    return;
  }
  
  if (listen (ls, 10) == -1)
  {
    if (Debug)
      fprintf (stderr, "%s: listen failed (%s).\n", Progname, strerror (errno));
    
    return;
  }

  while (!Signalled)
  {
    FD_ZERO (&read_fds);
    n = ls + 1;
    FD_SET (ls, &read_fds);
    for (cp = Connections; cp; cp = cp->next)
    {
      if (cp->state == CS_READREQ)
      {
	FD_SET (cp->fd, &read_fds);
	if (cp->fd + 1 > n)
	  n = cp->fd + 1;
      }
    }
    
    tv.tv_sec = 10;
    tv.tv_usec = 0;
    
    if (select (n, &read_fds, NULL, NULL, &tv) > 0)
    {
      if (FD_ISSET (ls, &read_fds))
      {
	slen = sizeof (sin);
	if ((as = accept (ls, &sin, &slen)) >= 0)
	{
	  if (Debug) printf ("%s: accept connection, fd = %d.\n", Progname, as);
	  syslog (LOG_INFO, "[fd = %d] connect from %s.", as, inet_ntoa (sin.sin_addr));
	  pkspxy_add_conn (as);
	}
	else if (Debug)
	  fprintf (stderr, "%s: accept () failed (%s).\n", Progname, strerror (errno));

      }
      
      for (cp = Connections; cp; cp = cp->next)
      {
	if (FD_ISSET (cp->fd, &read_fds))
	{
	  if ((l = read (cp->fd, buff, sizeof (buff))) > 0)
	  {
	    pkspxy_conn_add_data (cp, buff, l);
	    if (cp->state == CS_UPDATE)
	    {
	      
	      if (Debug) fprintf (stderr, "%s: HTTP query: %s\n", Progname, cp->buff);
	      
	      t = 0;
	      i = hkp_parse_query (cp->buff, buff, sizeof (buff), &t);

	      if (Debug) fprintf (stderr, "%s: search string: %s\n", Progname, buff);
	      syslog (LOG_INFO, "[fd = %d]: searching for `%s'", cp->fd, buff);

	      cp->query = strdup (buff);
	      cp->t = t;
	      if (!((i & (HKP_ERR | HKP_BADREQ)) || (i & (HKP_OP_GET|HKP_SEARCH)) != (HKP_OP_GET|HKP_SEARCH)))
	      {
		if (pkspxy_is_postponed (cp->query))
		{
		  syslog (LOG_DEBUG, "[fd = %d]: Query is on postponed list.", cp->fd);
		  pkspxy_spawn_reply (cp->query, cp->t, cp->fd);
		  cp->state = CS_DONE;
		}
		else if (!Online || pkspxy_spawn_request (cp->query, 60, cp) == -1)
		{
		  syslog (LOG_DEBUG, "[fd = %d]: Postponing query.", cp->fd);
		  pkspxy_postponed_append (cp->query);
		  pkspxy_spawn_reply (cp->query, cp->t, cp->fd);
		  cp->state = CS_DONE;
		}
	      }
	      else
	      {
		if (Debug) fprintf (stderr, "%s: Bad query.\n", Progname);
		syslog (LOG_INFO, "[fd = %d]: bad query.", cp->fd);
		pkspxy_spawn_reply (NULL, 0, cp->fd);
		cp->state = CS_DONE;
	      }
	    }
	  }
	  else
	    cp->state = CS_DONE;
	}
      }
    }

    pkspxy_collect_children ();

    for (cp = Connections; cp; cp = cq)
    {
      cq = cp->next;
      
      if (cp->state == CS_REPLY)
      {
	pkspxy_spawn_reply (cp->query, cp->t, cp->fd);
	cp->state = CS_DONE;
      }
      
      if (cp->state == CS_DONE)
      {
	close (cp->fd);
	pkspxy_delete_conn (cp->fd);
      }
    }

    if (Online)
    {
      pkspxy_handle_postponed_requests (ForcePostponed);
      ForcePostponed = 0;
    }
  }
}

