/**
 *
 * @file     callbacks.c
 * @brief    Playground callbacks
 * @author   Aleix Conchillo Flaque <aleix@member.fsf.org>
 * @date     Tue Aug 05, 2005 21:50
 *
 * Copyright (C) 2005, 2006 Aleix Conchillo Flaque
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <config.h>

#include "callbacks.h"

#include "interface.h"
#include "player.h"
#include "playground.h"
#include "preferences.h"

#include <gnome.h>

/* Static functions declaration (definition at end of file) */
static void timer_on (gpointer cb_data);
static void timer_off (void);
static gint cb_timer (gpointer cb_data);

/* Static variables definition */

static gint timeout_hnd_ = 0;    /* time out handler identifier */
static int const TIMEOUT_VALUE = 250;

/* General callbacks */

void
cb_realize (GtkWidget *w, playground_t *pg)
{
  gui_setup (playground_gui (pg), playground_player (pg));
  timer_on (pg);
}

void
cb_destroy (GtkWidget *w, playground_t *pg)
{
  timer_off ();
  playground_destroy (pg);
}

void
cb_change_size (GtkWidget *w, gint size, playground_t *pg)
{
  gui_set_size (playground_gui (pg), size);
  gui_setup (playground_gui (pg), playground_player (pg));
}

void
cb_change_orient (GtkWidget *w, PanelAppletOrient orient, playground_t *pg)
{
  gui_set_orient (playground_gui (pg), orient);
  gui_setup (playground_gui (pg), playground_player (pg));
}

void
cb_style_set (GtkWidget *w, GtkStyle *style, playground_t *pg)
{
  /* TODO */
  /* gui_theme_load (playground_gui (pg), style); */
}

/* General interface callbacks */

gboolean
cb_scrollwheel (GtkWidget *w, GdkEventScroll *event, playground_t *pg)
{
  switch (event->direction)
    {
    case GDK_SCROLL_UP:
      player_raise_volume (playground_player (pg));
      break;
    case GDK_SCROLL_DOWN:
      player_lower_volume (playground_player (pg));
      break;
    default:
      return FALSE;
    }

  return TRUE;
}

/* Progress bar callbacks */

gboolean
cb_seek (GtkWidget *w, GdkEventButton *event, playground_t *pg)
{
  return gui_seek (playground_gui (pg), playground_player (pg), event);
}

/* Buttons callbacks */

gboolean
cb_button_press (GtkWidget *w, GdkEventButton *event, playground_t *pg)
{
  return gui_propagate_event (playground_gui (pg),
                              playground_player (pg), event);
}

void
cb_prev_clicked (GtkWidget *w, playground_t *pg)
{
  player_prev (playground_player (pg));
}

void
cb_play_pause_clicked (GtkWidget *w, playground_t *pg)
{
  player_play_pause (playground_player (pg));
}

void
cb_stop_clicked (GtkWidget *w, playground_t *pg)
{
  player_stop (playground_player (pg));
}

void
cb_next_clicked (GtkWidget *w, playground_t *pg)
{
  player_next (playground_player (pg));
}

void
cb_eject_clicked (GtkWidget *w, playground_t *pg)
{
  player_eject (playground_player (pg));
}

/* Menu callbacks */

void
cb_repeat_menu (BonoboUIComponent *uic, playground_t *pg, gchar const *verb)
{
  /* empty (Bonobo UI component listener is used instead) */
}

void
cb_shuffle_menu (BonoboUIComponent *uic, playground_t *pg,
                 gchar const *verb)
{
  /* empty (Bonobo UI component listener is used instead) */
}

void
cb_toggle_repeat (BonoboUIComponent *uic, gchar const *path,
                  Bonobo_UIComponent_EventType type, gchar const *state,
                  playground_t *pg)
{
  player_repeat (playground_player (pg), strcmp(state, "0"));
}

void
cb_toggle_shuffle (BonoboUIComponent *uic, gchar const *path,
                   Bonobo_UIComponent_EventType type, gchar const *state,
                   playground_t *pg)
{
  player_shuffle (playground_player (pg), strcmp(state, "0"));
}

void
cb_main_win_menu (BonoboUIComponent *uic, playground_t *pg,
                  gchar const *verb)
{
  player_main_win (playground_player (pg),
                   !player_is_main_win (playground_player (pg)));
}

void
cb_playlist_win_menu (BonoboUIComponent *uic, playground_t *pg,
                      gchar const *verb)
{
  player_playlist_win (playground_player (pg),
                       !player_is_playlist_win (playground_player (pg)));
}

void
cb_show_hide_menu (BonoboUIComponent *uic, playground_t *pg,
                   gchar const *verb)
{
  player_show_hide (playground_player (pg));
}

void
cb_player_pref_menu (BonoboUIComponent *uic, playground_t *pg,
                     gchar const *verb)
{
  player_preferences (playground_player (pg));
}

void
cb_pref_menu (BonoboUIComponent *uic, playground_t *pg,
              gchar const *verb)
{
  preferences_show ();
}

void
cb_quit_menu (BonoboUIComponent *uic, playground_t *pg,
              gchar const *verb)
{
  player_quit (playground_player (pg));
}

void
cb_about_menu (BonoboUIComponent *uic, gpointer *ignore, gchar const *verb)
{
  static GtkWidget *about = NULL;
  static gchar const *authors[] = {
    "Aleix Conchillo Flaque <aleix@member.fsf.org>",
    NULL
  };

  if (about != NULL)
    {
      gdk_window_show (about->window);
      gdk_window_raise (about->window);
      return;
    }
  about =
    gnome_about_new (PACKAGE_NAME, PACKAGE_VERSION, "Copyright (C) 2005, 2006 Aleix Conchillo Flaque\n",
                     _("playground lets you control basic functions of audio players.\n"),
                     authors, NULL, NULL,
                     gdk_pixbuf_new_from_file (PG_PIXMAPSDIR"/playground.png", 0));

  g_signal_connect (G_OBJECT(about), "destroy",
                    G_CALLBACK (gtk_widget_destroyed), &about);
  gtk_widget_show (about);
}

/* Static functions definition */

static void
timer_on (gpointer cb_data)
{
  if (!timeout_hnd_)
    {
      timeout_hnd_ = gtk_timeout_add (TIMEOUT_VALUE, cb_timer, cb_data);
    }
}

static void
timer_off (void)
{
  if (timeout_hnd_)
    {
      gtk_timeout_remove (timeout_hnd_);
      timeout_hnd_ = 0;
    }
}

static gint
cb_timer (gpointer cb_data)
{
  static gboolean was_playing = FALSE;

  playground_t *pg = cb_data;
  pg_gui_t *gui = playground_gui (pg);
  pg_player_t *player = playground_player (pg);

  /* if player has been closed or if it was playing */
  if (was_playing && !player_is_running (player))
    {
      was_playing = FALSE;
    }
  if (player_is_playing (player))
    {
      was_playing = TRUE;
    }
  gui_update (gui, player);

  return TRUE;
}
