/**
 *
 * @file     interface.c
 * @brief    Playground GUI
 * @author   Aleix Conchillo Flaque <aleix@member.fsf.org>
 * @date     Tue Aug 05, 2003 21:59
 *
 * Copyright (C) 2005, 2006 Aleix Conchillo Flaque
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <config.h>

#include "interface.h"

#include "callbacks.h"
#include "player.h"

#include <string.h>

typedef struct
{
  GtkWidget *image;             /* button image */
  GtkWidget *button;            /* image container */
  gchar const *stock_id;        /* theme image stock identifier */
} pg_button_t;

struct pg_gui_t
{
  GtkWidget *applet;
  gint size;                    /* applet size */
  PanelAppletOrient orient;     /* applet orientation */
  GtkTooltips *tooltips;
  GtkIconSize icon_size;

  struct
  {
    GtkWidget *frame;
    GtkWidget *box;
  } main;
  struct
  {
    GtkWidget *bar;             /* progress bar */
    GtkWidget *box;             /* all widgets go inside this one */
  } progress;
  struct
  {
    pg_button_t prev;
    pg_button_t play_pause;
    pg_button_t stop;
    pg_button_t next;
    pg_button_t eject;
    GtkWidget *box;             /* all widgets go inside this one */
  } buttons;
};

/* Static functions declaration (definition at end of file) */
static void init_menu (pg_gui_t *gui, gpointer *cb_data);
static void init_stock_icons (void);
static void register_stock_icons (GtkIconFactory *factory);
static void reload_images (pg_button_t *b, GtkStyle *style,
                           GtkIconSize icon_size);
static void create_progress_box (pg_gui_t *gui, gpointer cb_data);
static void create_buttons_box (pg_gui_t *gui, gpointer cb_data);
static void create_button (pg_gui_t *gui, pg_button_t *b,
                           gchar const *stock_id, gchar const *tooltip,
                           GCallback callback, gpointer cb_data);
static void ref_and_remove (GtkWidget *w);
static void remove_button (pg_button_t *b);
static void remove_main_box (pg_gui_t *gui);
static void remove_progress_box (pg_gui_t *gui);
static void remove_buttons_box (pg_gui_t *gui);
static void destroy_widget (GtkWidget **w);
static void destroy_button (pg_button_t *b);
static void destroy_main_box (pg_gui_t *gui);
static void destroy_progress_box (pg_gui_t *gui);
static void destroy_buttons_box (pg_gui_t *gui);

/* Global stock icon identifiers */
static gchar const *PLAYGROUND_PREV       = "prev";
static gchar const *PLAYGROUND_PLAY       = "play";
static gchar const *PLAYGROUND_PAUSE      = "pause";
static gchar const *PLAYGROUND_STOP       = "stop";
static gchar const *PLAYGROUND_NEXT       = "next";
static gchar const *PLAYGROUND_EJECT      = "eject";

/* Image files extension */
static gchar const *PLAYGROUND_IMAGE_EXT = ".png";

pg_gui_t*
gui_create (void)
{
  return g_new0 (pg_gui_t, 1);
}

void
gui_destroy (pg_gui_t *gui)
{
  if (gui)
    {
      g_object_unref (gui->tooltips);
      destroy_main_box (gui);
      destroy_widget (&gui->main.frame);
      g_free(gui);
    }
}

void
gui_init (pg_gui_t *gui, GtkWidget *applet, gpointer *cb_data)
{
  gui->applet = applet;

  /* Get initial size, orientation and icon size */
  gui->size = panel_applet_get_size (PANEL_APPLET (applet));
  gui->orient = panel_applet_get_orient (PANEL_APPLET (applet));
  gui->icon_size = gtk_icon_size_register ("gdata_icon_size", 5, 5);

  /* Create main frame */
  gui->main.frame = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (gui->main.frame);

  /* Applet menu */
  init_menu (gui, cb_data);

  /* Create tooltips */
  gui->tooltips = gtk_tooltips_new ();

  /* Create progress song box */
  create_progress_box (gui, cb_data);

  /* Create applet buttons */
  init_stock_icons ();
  create_buttons_box (gui, cb_data);

  gtk_container_add (GTK_CONTAINER(applet), gui->main.frame);

  g_signal_connect_after (G_OBJECT (gui->main.frame), "realize",
                          G_CALLBACK (cb_realize), cb_data);

  gtk_widget_show (gui->applet);
}

void
gui_setup (pg_gui_t *gui, pg_player_t *player)
{
  gint spacing;
  GtkWidget *main_box;
  GtkWidget *progress_box;
  GtkWidget *buttons_box;
  GtkWidget *bar;

  if (gui->main.box)
    {
      remove_main_box (gui);
    }

  progress_box = gui->progress.box;
  bar = gui->progress.bar;

  if ((gui->orient == PANEL_APPLET_ORIENT_DOWN)
      || (gui->orient == PANEL_APPLET_ORIENT_UP))
    {
      main_box = gui->main.box = gtk_vbox_new (FALSE, 1);
      gtk_widget_set_size_request (gui->main.box, 80, -1);
      gtk_widget_show (main_box);

      buttons_box = gui->buttons.box = gtk_hbox_new (TRUE, 0);
      gtk_widget_set_size_request (buttons_box, -1, 15);
      gtk_widget_show (buttons_box);

      gtk_widget_set_size_request (progress_box, -1, 8);

      gtk_progress_bar_set_orientation (GTK_PROGRESS_BAR (bar),
                                        GTK_PROGRESS_LEFT_TO_RIGHT);

      spacing = 1;
    }
  else
    {
      main_box = gui->main.box = gtk_hbox_new (FALSE, 1);
      gtk_widget_set_size_request (gui->main.box, -1, 80);
      gtk_widget_show (main_box);

      buttons_box = gui->buttons.box = gtk_vbox_new (TRUE, 0);
      gtk_widget_set_size_request (buttons_box, 15, -1);
      gtk_widget_show (buttons_box);

      gtk_widget_set_size_request (progress_box, 8, -1);

      gtk_progress_bar_set_orientation (GTK_PROGRESS_BAR (bar),
                                        GTK_PROGRESS_TOP_TO_BOTTOM);

      spacing = 0;
    }

  gtk_box_pack_start (GTK_BOX(buttons_box), gui->buttons.prev.button,
                      TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX(buttons_box), gui->buttons.play_pause.button,
                      TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX(buttons_box), gui->buttons.stop.button,
                      TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX(buttons_box), gui->buttons.next.button,
                      TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX(buttons_box), gui->buttons.eject.button,
                      TRUE, TRUE, 0);

  if (gui->orient == PANEL_APPLET_ORIENT_RIGHT)
    {
      gtk_box_pack_start (GTK_BOX (main_box), buttons_box, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (main_box), progress_box, TRUE, TRUE, 0);
    }
  else
    {
      gtk_box_pack_start (GTK_BOX (main_box), progress_box, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (main_box), buttons_box, TRUE, TRUE, 0);
    }

  gtk_widget_show (main_box);

  gtk_box_pack_start (GTK_BOX (gui->main.frame), main_box, TRUE, TRUE, spacing);

  gtk_widget_show_all (gui->main.frame);

  gui_update (gui, player);
}

void
gui_update (pg_gui_t *gui, pg_player_t *player)
{
  gchar *track;
  gchar *track_album;
  gchar *track_artist;
  gint minutes;
  gint seconds;
  gint aux_time;
  gint cur_time;
  gint total_time;
  gdouble width;
  GtkWidget *bar;
  GtkWidget *play_btn_image;
  BonoboUIComponent *popup;
  enum { MAX_BUFFER = 1024 };
  gchar info[MAX_BUFFER] = "";

  strcpy (info, _("No track"));

  bar = gui->progress.bar;
  popup = panel_applet_get_popup_component (PANEL_APPLET (gui->applet));
  play_btn_image = gui->buttons.play_pause.image;
  if (!player_is_running (player))
    {
      gtk_image_set_from_stock (GTK_IMAGE (play_btn_image), PLAYGROUND_PLAY,
                                gui->icon_size);
      gtk_tooltips_set_tip (gui->tooltips, gui->progress.box, info, NULL);
      gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (bar), 0);
      bonobo_ui_component_set_prop(popup, "/commands/Repeat", "state",
                                   "0", NULL);
      bonobo_ui_component_set_prop(popup, "/commands/Shuffle", "state",
                                   "0", NULL);
      return;
    }

  track = player_track_name (player);
  track_album = player_track_album (player);
  track_artist = player_track_artist (player);
  total_time = player_track_total_time (player);
  cur_time = player_track_current_time (player);

  width = total_time ? ((gdouble) cur_time / (gdouble) total_time) : 0;

  gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (bar), width);

  aux_time = cur_time / 1000;
  seconds = aux_time % 60;
  minutes = aux_time / 60;
  if (track != NULL)
    {
      sprintf (info, "%s (%02d:%02d)", track, minutes, seconds);
    }
  gtk_tooltips_set_tip (gui->tooltips, gui->progress.box, info, NULL);

  /* Update play/pause button */
  gtk_image_set_from_stock
    (GTK_IMAGE (play_btn_image),
     player_is_playing (player) && !player_is_paused (player)
     ? PLAYGROUND_PAUSE : PLAYGROUND_PLAY,
     gui->icon_size);

  /* Update popup checkboxes */
  bonobo_ui_component_set_prop(popup, "/commands/Repeat", "state",
                               player_is_repeat (player) ? "1" : "0", NULL);
  bonobo_ui_component_set_prop(popup, "/commands/Shuffle", "state",
                               player_is_shuffle (player) ? "1" : "0", NULL);

  gtk_widget_show_all (gui->main.frame);
}

gboolean
gui_seek (pg_gui_t *gui, pg_player_t *player, GdkEventButton *event)
{
  gdouble t;
  GtkRequisition requisition;

  if (event->button == 3)
    {
      gtk_propagate_event (gui->applet, (GdkEvent *) event);

      return TRUE;
    }

  if (event->button == 2)
    {
      player_show_hide (player);
      return TRUE;
    }

  gtk_widget_size_request (gui->main.box, &requisition);

  if ((gui->orient == PANEL_APPLET_ORIENT_DOWN)
      || (gui->orient == PANEL_APPLET_ORIENT_UP))
    {
      t = event->x / (gdouble) (requisition.width - 1);
    }
  else
    {
      t = event->y / (gdouble) (requisition.height - 1);
    }
  player_seek (player, t);

  return FALSE;
}

gboolean
gui_propagate_event (pg_gui_t *gui, pg_player_t *player,
                     GdkEventButton *event)
{
  if (event->button == 3)
    {
      gtk_propagate_event (gui->applet, (GdkEvent *) event);

      return TRUE;
    }

  if (event->button == 2)
    {
      player_show_hide (player);
      return TRUE;
    }

  return FALSE;
}

void
gui_set_size (pg_gui_t *gui, gint size)
{
  gui->size = size;
}

void
gui_set_orient (pg_gui_t *gui, PanelAppletOrient orient)
{
  gui->orient = orient;
}

void
gui_theme_load (pg_gui_t *gui, GtkStyle *style)
{
  reload_images (&gui->buttons.prev, style, gui->icon_size);
  reload_images (&gui->buttons.play_pause, style, gui->icon_size);
  reload_images (&gui->buttons.stop, style, gui->icon_size);
  reload_images (&gui->buttons.next, style, gui->icon_size);
  reload_images (&gui->buttons.eject, style, gui->icon_size);
}

void
gui_show_error (gchar const *msg)
{
  GtkWidget *dialog;

  dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
                                   GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, msg);

  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
  gtk_window_set_screen (GTK_WINDOW (dialog), gdk_screen_get_default ());

  g_signal_connect (dialog, "response", G_CALLBACK (gtk_widget_destroy), NULL);
  g_signal_connect (dialog, "destroy",
                    G_CALLBACK (gtk_widget_destroyed), &dialog);

  gtk_widget_show_all (dialog);
}

/* Static functions declaration */

static void
init_menu (pg_gui_t *gui, gpointer *cb_data)
{
  static const char* menu_xml_file = "GNOME_PlaygroundApplet.xml";
  static const BonoboUIVerb menu_verbs[] =
    {
      BONOBO_UI_UNSAFE_VERB("Repeat", cb_repeat_menu),
      BONOBO_UI_UNSAFE_VERB("Shuffle", cb_shuffle_menu),
      BONOBO_UI_UNSAFE_VERB("MainWin", cb_main_win_menu),
      BONOBO_UI_UNSAFE_VERB("PlayListWin", cb_playlist_win_menu),
      BONOBO_UI_UNSAFE_VERB("ShowHide", cb_show_hide_menu),
      BONOBO_UI_UNSAFE_VERB("PlayerPref", cb_player_pref_menu),
      BONOBO_UI_UNSAFE_VERB("Pref", cb_pref_menu),
      BONOBO_UI_UNSAFE_VERB("Quit", cb_quit_menu),
      BONOBO_UI_UNSAFE_VERB("About", cb_about_menu),
      BONOBO_UI_VERB_END
    };
  BonoboUIComponent *popup = NULL;

  panel_applet_setup_menu_from_file (PANEL_APPLET (gui->applet),
                                     NULL,
                                     menu_xml_file,
                                     NULL,
                                     menu_verbs,
                                     cb_data);

  popup = panel_applet_get_popup_component (PANEL_APPLET (gui->applet));

  /* Set repeat and shuffle callbacks */
  bonobo_ui_component_add_listener(popup, "Repeat",
                                   (BonoboUIListenerFn) cb_toggle_repeat,
                                   cb_data);
  bonobo_ui_component_add_listener(popup, "Shuffle",
                                   (BonoboUIListenerFn) cb_toggle_shuffle,
                                   cb_data);
}

static void
init_stock_icons (void)
{
  GtkIconFactory *factory;

  factory = gtk_icon_factory_new ();
  gtk_icon_factory_add_default (factory);

  register_stock_icons (factory);

  g_object_unref (factory);
}

static void
register_stock_icons (GtkIconFactory *factory)
{
  enum { MAX_PATH = 4096 };
  gint i;
  static gchar file_name[MAX_PATH];
  gchar const *items[] =
    {
      PLAYGROUND_PREV, PLAYGROUND_PLAY,
      PLAYGROUND_PAUSE, PLAYGROUND_STOP,
      PLAYGROUND_NEXT, PLAYGROUND_EJECT
    };

  for (i = 0; i < sizeof(items) / sizeof(items[0]); ++i)
    {
      GdkPixbuf *pixbuf;
      GtkIconSet *icon_set;
      GError *error = NULL;

      sprintf (file_name, "%s/%s%s", PG_ICONSDIR, items[i],
               PLAYGROUND_IMAGE_EXT);
      pixbuf = gdk_pixbuf_new_from_file (file_name, &error);
      if (pixbuf == NULL)
        {
          gui_show_error (error->message);
        }

      icon_set = gtk_icon_set_new_from_pixbuf(pixbuf);
      gtk_icon_factory_add(factory, items[i], icon_set);

      gtk_icon_set_unref(icon_set);
      g_object_unref(G_OBJECT(pixbuf));
    }
}

static void
reload_images (pg_button_t *b, GtkStyle *style, GtkIconSize icon_size)
{
  GtkIconSet *icon_set;

  icon_set = gtk_style_lookup_icon_set (style, b->stock_id);
  gtk_image_set_from_icon_set (GTK_IMAGE (b->image), icon_set, icon_size);
}

static void
create_progress_box (pg_gui_t *gui, gpointer cb_data)
{
  GtkObject *adj;
  GtkWidget *bar;
  GtkWidget *progress_box;

  /* To catch progress bar events (real-time tooltips) */
  progress_box = gui->progress.box = gtk_event_box_new ();

  adj = gtk_adjustment_new (0.0, 0.0, 1.0, 0.1, 0.1, 0.0);
  bar = gui->progress.bar = gtk_progress_bar_new ();
  gtk_progress_set_adjustment (GTK_PROGRESS (bar), GTK_ADJUSTMENT (adj));
  gtk_widget_show (bar);

  gtk_container_add (GTK_CONTAINER (progress_box), bar);

  g_signal_connect (progress_box, "button_press_event",
                    G_CALLBACK (cb_seek), cb_data);

  gtk_widget_show (progress_box);
}

static void
create_buttons_box (pg_gui_t *gui, gpointer cb_data)
{
  create_button(gui, &gui->buttons.prev, PLAYGROUND_PREV, _("Previous track"),
                G_CALLBACK (cb_prev_clicked), cb_data);

  create_button(gui, &gui->buttons.play_pause, PLAYGROUND_PLAY,
                _("Play / Pause"),
                G_CALLBACK (cb_play_pause_clicked), cb_data);

  create_button(gui, &gui->buttons.stop, PLAYGROUND_STOP, _("Stop"),
                G_CALLBACK (cb_stop_clicked), cb_data);

  create_button(gui, &gui->buttons.next, PLAYGROUND_NEXT, _("Next track"),
                G_CALLBACK (cb_next_clicked), cb_data);

  create_button(gui, &gui->buttons.eject, PLAYGROUND_EJECT, _("Eject"),
                G_CALLBACK (cb_eject_clicked), cb_data);
}

static void
create_button (pg_gui_t *gui, pg_button_t *b, gchar const *stock_id,
               gchar const *tooltip, GCallback callback, gpointer cb_data)
{
  GtkWidget *button;
  GtkWidget *image;

  b->stock_id = stock_id;

  button = b->button = gtk_button_new ();
  gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_HALF);

  image = b->image = gtk_image_new ();
  gtk_image_set_from_stock (GTK_IMAGE (image), stock_id, gui->icon_size);
  gtk_widget_show (image);

  gtk_container_add (GTK_CONTAINER (button), image);

  gtk_tooltips_set_tip (gui->tooltips, button, tooltip, NULL);

  g_signal_connect (button, "clicked", callback, cb_data);
  g_signal_connect (button, "button_press_event",
                    G_CALLBACK (cb_button_press), cb_data);

  gtk_widget_show (button);
}

static void
ref_and_remove (GtkWidget *w)
{
  GtkWidget *parent = w->parent;

  if (parent)
    {
      g_object_ref (G_OBJECT (w));
      gtk_container_remove (GTK_CONTAINER (parent), w);
    }
}

static void
remove_button (pg_button_t *b)
{
  ref_and_remove (b->button);
}

static void
remove_main_box (pg_gui_t *gui)
{
  g_return_if_fail (GTK_IS_CONTAINER (gui->main.box));

  remove_progress_box (gui);
  remove_buttons_box (gui);

  destroy_widget (&gui->main.box);
}

static void
remove_progress_box (pg_gui_t *gui)
{
  g_return_if_fail (GTK_IS_CONTAINER (gui->progress.box));

  /* Do not need to remove progress bar from its container */
  ref_and_remove (gui->progress.box);
}

static void
remove_buttons_box (pg_gui_t *gui)
{
  g_return_if_fail (GTK_IS_CONTAINER (gui->buttons.box));

  remove_button (&gui->buttons.prev);
  remove_button (&gui->buttons.play_pause);
  remove_button (&gui->buttons.stop);
  remove_button (&gui->buttons.next);
  remove_button (&gui->buttons.eject);

  destroy_widget (&gui->buttons.box);
}

static void
destroy_widget (GtkWidget **w)
{
  if ((w == NULL) || (*w == NULL))
    {
      return;
    }

  gtk_widget_destroy (*w);
  *w = NULL;
}

static void
destroy_button (pg_button_t *b)
{
  if (b == NULL)
    {
      return;
    }

  destroy_widget (&b->image);
  destroy_widget (&b->button);
}

static void
destroy_main_box (pg_gui_t *gui)
{
  g_return_if_fail (GTK_IS_CONTAINER (gui->main.box));

  destroy_progress_box (gui);
  destroy_buttons_box (gui);

  destroy_widget (&gui->main.box);
}

static void
destroy_progress_box (pg_gui_t *gui)
{
  g_return_if_fail (GTK_IS_CONTAINER (gui->progress.box));

  destroy_widget (&gui->progress.bar);

  destroy_widget (&gui->progress.box);
}

static void
destroy_buttons_box (pg_gui_t *gui)
{
  g_return_if_fail (GTK_IS_CONTAINER (gui->buttons.box));

  destroy_button (&gui->buttons.prev);
  destroy_button (&gui->buttons.play_pause);
  destroy_button (&gui->buttons.stop);
  destroy_button (&gui->buttons.next);
  destroy_button (&gui->buttons.eject);

  destroy_widget (&gui->buttons.box);
}
