from patterns import *
from recog_domain import *

class DetectionMode:
    def __init__(self, name, descrip = ""):
        self.name = name
        self.description = descrip

Correct = DetectionMode('correct')
AlmostCorrect = DetectionMode('almost_correct', 'reduced confidence in precision and/or recall')
Violator = DetectionMode('violator', 'violates some topological constraint')
OutOfScope = DetectionMode('out__of__scope', 'constraining types that are defined externally')

class RecogMediator (SpecObj):
    """ fails if any potential colleagues are in sending_to of another"""
    pattern = Mediator
    detection = Correct
    inputs = ['colleagues = []']
    collections = [Recog('r2 in obj.out_to()',
                         [Recog('r3 in obj.in_from()',
                                [Recog('colleagues = \
                                 [x for x in r2.out_to() if x in r3.in_from()]')])])]
    stoppers = ["Exists([pair for pair in cross_product(colleagues) if pair[1] in pair[0].sending_to])",
                "len(colleagues) < 2"]
    focus_name = "mediator"

class RecogSingleProcessController (SpecObj):
    pattern = SingleProcessMediator
    detection = Correct
    collections = [Recog('r1 = \
                    [f for f in obj.out_to() if \
                    f in obj.in_from() and \
                    the_type(f) == "File" \
                    and len(f.in_edges) + len(f.out_edges) == 2]')]
    stoppers = ["len(r1) > 1"]
    focus_name = "mediator"

class RecogSingleProcessMediator (SpecObj):
    pattern = SingleProcessMediator
    detection = Correct
    collections = [Recog('r1 = \
                    [f for f in obj.out_to() if \
                    f in obj.in_from() and \
                    the_type(f) == "File" \
                    and len(f.in_edges) + len(f.out_edges) == 2]')]
    stoppers = ["len(r1) < 2"]
    focus_name = "mediator"

class RecogLogger1(SpecObj):
    pattern = Logger
    detection = Correct
    focus_name = "application"
    collections = [Recog('log_file = \
    [f for f in obj.out_to() if contains_str(f.name, "var/log")]')]


class RecogConfig1(SpecObj):
    pattern = Config
    detection = Correct
    focus_name = "application"
    collections = [Recog('config_file = \
    [f for f in obj.in_from() if config_namep(f.name)]')]

class RecogConfig2(SpecObj):
    pattern = Config
    detection =  OutOfScope
    focus_name = "application"
    collections = [Recog('config_file = \
    [f for f in obj.in_from() if singleton(f.out_edges) \
    and len(f.in_edges) == 0 and config_name_maybep(f.name)]')]

    
class RecogDaemon(SpecObj):
    pattern = Daemon
    detection = Correct
    focus_name = "parent"
    collections = [Recog('child = \
    [c for c in obj.processes.values() if "setsid" in c.special_calls]')]

class RecogCanCheckSelinux (SpecObj):
    pattern = CanCheckSelinux
    detection = Correct
    focus_name = "main"
    collections = [Recog('filesystems in \
    [f for f in obj.in_from() if contains_str(f.name,"/proc/filesystems")]',
                         [Recog('current = \
                         [f for f in obj.in_from() \
                         if contains_str(f.name, "/proc/self/attr/current")]')])]

class RecogInterpreter (SpecObj):
    pattern = Interpreter
    detection = Correct
    focus_name = "application"
    collections = [Recog('code_file in \
    [f for f in obj.in_from("exec") if contains_str(f.name, ".py")]',
                         [Recog('interpreter = [f for f in obj.in_from("exec") \
                         if f != code_file]')])]

class RecogPipe (SpecObj):
    pattern = Pipe
    detection = Correct
    focus_name = "parent"
    collections = [Recog('pipe in \
    [f for f in obj.out_to() if the_type(f) == "Fifo_file"]',
                         [Recog('child = [p for p in pipe.out_to() \
                         if obj == pipe.creator]')])]

class RecogClientOnly (SpecObj):
    pattern = Client
    detection = Correct
    focus_name = "client"
    collections = [Recog('socket in [s for s in obj.out_to() if \
    the_type(s) == "Unix_Stream_Socket" and not Exists(s.out_edges)]')]

class RecogClientServer (SpecObj):
    pattern = ClientServer
    detection = Correct
    focus_name = "client"
    collections = [Recog('socket in [f for f in obj.out_to() if \
    the_type(f) == "Unix_Stream_Socket"]',
                         [Recog('server = [f for f in socket.out_to() if \
                         f != obj and  Exists([a for a in f.with_accepts \
                         if a.listening_on() == socket.name])]')])]



class RecogS2S (SpecObj):
    pattern = ClientServer
    detection = Correct
    focus_name = "server"
    collections = [Recog('socket in \
    [f for f in obj.out_to() if the_type(f) == "Unix_Stream_Socket"]',
                         [Recog('client = \
    [f for f in socket.out_to() if  f == obj and \
    Exists([a for a in f.with_accepts if a.listening_on() == socket.name])]')])]

class RecogLibSelinuxInit (SpecObj):
    pattern = LibSelinuxInit
    detection = Correct
    focus_name = "application"
    collections = [Recog('mounts in \
    [f for f in obj.in_from() \
    if obj_name_contains_one_of(f, ["proc/mounts", "etc/selinux/config"])]',
                         [Recog('new_access = \
                                [t.not_obj(obj) for t in (obj.in_edges + obj.out_edges) if t.domain != mounts \
                                and obj_name_contains_one_of(t.not_obj(obj), ["etc/selinux/", "proc/mounts"])]')])]

class RecogSharedLibrary (SpecObj):
    pattern = SharedLibrary
    detection = Correct
    focus_name = "application"
    collections = [Recog('lib_read = \
    [f for f in obj.in_from() if contains_str(f.name,".so")]')]

class RecogTempUse (SpecObj):
    pattern = TempUse
    detection = Correct
    focus_name = "application"
    collections = [Recog('temp_file = \
    [f for f in obj.out_to() if contains_str(f.name,"tmp/")]')]


class RecogExecutable (SpecObj):
    #note: no direct connection between file and process
    pattern = Executable
    detection = Correct
    focus_name = "application"
    collections = [Recog('file in \
    [f for f in obj.out_to("exec") if \
                         not neither_local(obj, f)]',
                         [Recog('process in \
                         [y.range for y in flatten([x.out_edges for x in obj.out_to("activate")]) \
                         if similar(y.range.name, file.name)]')])]


class RecogExecutable2 (SpecObj):
    pattern = Executable
    detection = OutOfScope
    focus_name = "application"
    collections = [Recog('file in \
    [f for f in obj.out_to("exec") if \
                          neither_local(obj, f)]',
                         [Recog('process in \
                         [y.range for y in flatten([x.out_edges for x in obj.out_to("activate")])\
                         if similar(y.range.name, file.name)]')])]
    


class RecogPipeline(SpecObj):
    pattern = Pipeline
    detection = Correct
    collections = [Recog("r1 in [f for f in obj.out_to('write') if not config_namep(f.name)\
    and singleton(f.in_edges)]",
                         [Recog("middle in \
                         [m for m in r1.out_to('read') if m  != obj \
                         and m not in obj.receiving_from]", \
                                [Recog("r2 in \
                                [f for f in middle.out_to('write') if not config_namep(f.name) \
                                and f != r1 and singleton(f.in_edges)]", \
                                       [Recog("end = \
                                       [e for e in r2.out_to('read') if \
                                       e not in [obj, middle] + obj.receiving_from + middle.receiving_from]")])])])]
    focus_name = "begin"


class RecogFlawedIntegrityPipeline (SpecObj):
    pattern = Pipeline
    detection = Violator
    collections = [Recog("r1 in \
    obj.out_to('write')", \
                         [Recog("middle in \
                         [c.range for c in r1.out_edges if c.range  != obj and c.action == 'read' and c.domain not in obj.receiving_from]", \
                                [Recog("r2 in \
                                [f for f in middle.out_to('write') if not config_namep(f.name) \
                                and f != r1 and len(f.in_edges) > 1 or len(r1.in_edges) > 1]", \
                                       [Recog("end = \
                                       [c.range for c in  r2.out_edges if c.range not in obj.receiving_from \
                                       and c.range != obj and c.action == 'read' and c.range != middle and \
                                       c.range not in middle.receiving_from]")])])])]
    focus_name = "begin"

class RecogProxy(SpecObj):
    #note: no direct connection between r3 and client
    pattern = Proxy
    detection  = Correct
    search_set = "[x for x in resources_dict.values() if x.is_process() and Exists(x.in_edges) and Exists(x.out_edges)]"
    collections = [Recog('r2 in obj.out_to("activate", "False")',
                         [Recog('service in \
                                 [s for s in r2.out_to() if s != obj ]',
                                [Recog('r3 in [f for f in obj.in_from() if \
                                           f in service.out_to()]',
                                       [Recog('client in [x for x in obj.receiving_from \
                                          if x not in service.receiving_from and x not in [obj, service]]')])])])]
    focus_name = "proxy"


class RecogHubAndSpoke(SpecObj):
    #note: not necessarily direct connections between constituents
    pattern = HubAndSpoke
    detection  = Correct
    collections = [Recog('r2 in obj.out_to("clone")',
                         [Recog('colleague_generator in \
                                 r2.out_to()',
                                 [Recog('center in \
                                 [x for x in colleague_generator.executing if len(x.with_accepts) > 1]',                                    
                                       [Recog('colleague = \
                                       setdiff(execs(obj), [center, obj])'
                                             )])])])]
                                  
    focus_name = "initiator"

    


class RecogCGI_lexical (SpecObj):
    pattern = CGI
    detection = AlmostCorrect
    focus_name = "application"
    collections = [Recog('file in \
     [f for f in obj.out_to("exec") if  contains_str(f, "cgi")]',
                         [Recog('process in \
                         [y.range for y in flatten([x.out_edges for x in obj.out_to("activate")]) \
                         if Exists([r for r in reads_from(y.range) if contains_str(r.name,".so")])]')])]
    

class RecogCGI (SpecObj):
    pattern = CGI
    detection = Correct
    search_set = '[x for x in resources_dict.values() if x.is_process() and Exists(reads_from(x)) \
                    and Exists(writes_to(x)) and Exists(x.with_binds)]'
    collections = [Recog('r1 in \
                   [c.range for c in obj.out_edges if c.action in ["activate", "clone"]]', 
                         [Recog('shell in \
                             [f for f in r1.out_to() if f != obj]', 
                                [Recog('script in \
                                               [c for c in shell.executing if \
                                                     c not in [obj, shell] \
                                                     and Exists([f for f in \
                                                     c.in_from() if contains_str(f.name, "cgi")])]'
                                       )]
                                )]
                         )]                                               
    focus_name = "server"

   
class RecogActiveObject (SpecObj):
    pattern = ActiveObject
    detection = AlmostCorrect
    collections = [Recog("order in [f for f in obj.out_to() if singleton(f.in_edges)]",
                         [Recog("service in [s for s in order.out_to() if s != obj]", \
                                [Recog("filled_order in \
                                [f for f in service.out_to() if f != order and obj in f.out_to()]" \
                                       )])])]
    focus_name = "proxy"

# note: following list is order dependent
SPECIFIERS = [RecogHubAndSpoke, RecogExecutable, RecogExecutable2, RecogPipeline, RecogClientOnly, \
              RecogClientServer, RecogS2S, RecogPipe, RecogTempUse, RecogInterpreter, RecogCanCheckSelinux,\
              RecogSharedLibrary, RecogLibSelinuxInit, RecogLogger1, \
              RecogConfig1, RecogConfig2, RecogActiveObject, RecogFlawedIntegrityPipeline,  \
              RecogSingleProcessMediator, RecogDaemon, RecogSingleProcessController, RecogProxy, \
              RecogCGI, RecogMediator]
