#! /usr/bin/env python
# track-fd

# Date last modified:
# Time-stamp: <2004-08-30 10:21:58 olandgren>

# Build dictionaries keyed on the file descriptor for fields of
# interest from the output of 'strace -X' The raw output has already
# been filtered by the StraceFilter class

# When we read a close() record in the input, we fire print all of the
# relevant information in the following format:
#
# filename file-related.s.c. read-flag read-s.c. write-flag write-s.c.
import sys
import getopt
import re

class FileDescriptorTracker:
    def __init__(self):
        # Initialization of objects used during parsing
        self.container = []
        self.fieldCount = 0
        self.fileDescriptor = ""
        self.openDict = {}
        self.filenameDict = {}
        self.filenameSCDict = {}
        self.readDict = {}
        self.readSCDict = {}
        self.writeDict = {}
        self.writeSCDict = {}
        self.objectTypeDict = {}
        self.key = ""

    def writeRecord(self, fileDescriptor):
        output = ""
        if self.filenameDict.has_key(fileDescriptor):
            output = self.filenameDict[fileDescriptor] + "\t"
            if self.filenameSCDict.has_key(fileDescriptor):
                output = output + self.filenameSCDict[fileDescriptor] + "\t"
            if self.readDict.has_key(fileDescriptor):
                output  = output + self.readDict[fileDescriptor] + "\t"
            # This test is readDict.has_key() in the Perl, which seems
            # like a bug
            if self.readSCDict.has_key(fileDescriptor):
                output = output + self.readSCDict[fileDescriptor] + "\t"
            if self.writeDict.has_key(fileDescriptor):
                output = output + self.writeDict[fileDescriptor] + "\t"
            # Similar test condition where writeDict and not writeSCDict
            # was tested in the Perl
            if self.writeSCDict.has_key(fileDescriptor):
                output = output + self.writeSCDict[fileDescriptor] + "\t"
            # Prints out the object type information
            if self.objectTypeDict.has_key(fileDescriptor):
                output = output + self.objectTypeDict[fileDescriptor] + "\t"

        print output

    def parseType(self, str):
        # This function consumes a string of the form
        # O_<X> | O_<Y> | O_<Z>, where <X>, <Y>, and <Z> are
        # parsed. Currently, this function checks solely to determine if
        # an object is a file or directory, and returns the correct value as
        # the strings "FILE" or "DIRECTORY"
        isDir = re.compile(".*O_DIRECTORY.*")
        if isDir.match(str):
            return "DIRECTORY"
        else:
            return "FILE"

    def parseLine(self, line):
        # Clean off any newlines and split into the set of fields
        container = line.strip().split("~")
        fieldcount = len(container)
        if container[0] == "open" and fieldcount > 5:
            # this is an open() record
            if (fieldcount == 6):
                fileDescriptor = container[4]
            else:
                fileDescriptor = container[5]
            self.openDict[fileDescriptor] = "open"
            self.filenameDict[fileDescriptor] = container[1]
            self.filenameSCDict[fileDescriptor] = container[2]
            objtype = self.parseType(container[3])
            self.objectTypeDict[fileDescriptor] = objtype
        elif (container[0] == "read"):
            self.readDict[container[1]] = "R"
            self.readSCDict[container[1]] = container[2]
        elif (container[0] == "write" and (container[1] > 2)):
#            print "DEBUG-WRITE: " + container[1] + " : " + container[2]
            self.writeDict[container[1]] = "W"
            self.writeSCDict[container[1]] = container[2]
        elif (container[0] == "close"):
            fileDescriptor = container[1]
            if self.openDict.has_key(fileDescriptor):
                self.writeRecord(fileDescriptor)
            if self.openDict.has_key(fileDescriptor):
                del self.openDict[fileDescriptor]
            if self.filenameDict.has_key(fileDescriptor):
                del self.filenameDict[fileDescriptor]
            if self.filenameSCDict.has_key(fileDescriptor):
                del self.filenameSCDict[fileDescriptor]
            if self.readDict.has_key(fileDescriptor):
                del self.readDict[fileDescriptor]
            # This is a duplicate of readDict in the
            # original Perl
            if self.readSCDict.has_key(fileDescriptor):
                del self.readSCDict[fileDescriptor]
            if self.writeDict.has_key(fileDescriptor):
                del self.writeDict[fileDescriptor]
            if self.writeSCDict.has_key(fileDescriptor):
                del self.writeSCDict[fileDescriptor]
            # Clean out the object type information
            if self.objectTypeDict.has_key(fileDescriptor):
                del self.objectTypeDict[fileDescriptor]

        elif (container[0] == "fstat64"):
            # Extract file descriptor and source/destination security contexts
            fileDescriptor = container[1]
            sourceSC = container[2]
            destSC = container[7]
        elif (container[0] == "stat64"):
            # Extract filename, source/dest security context
            fileDescriptor = container[1]
            sourceSC = container[2]
            destSC = container[-1] # we want the last item in the array, so we use
                                   # reverse indexing
        elif (container[0] == "old_mmap"):
            # Extract R/W/E, file descriptor, and security context
            fileDescriptor = container[5]
            # There's some detailed parsing involved in getting this information
            RWE = getRWE(container[3])
            sourceSC = container[6]
        elif (container[0] == "mmap2"):
            # Extract R/W/E, fd. FD is almost always -1, which means "I want new memory"
            RWE = getRWE(container[3])
            destFD = container[5]
        elif (container[0] == "execve"):
            # Name of file being executed, security context from which it was executed
            name = container[1]
            fromContext = container[2]
            toContext = container[-1]

            # Here, we have this information and want to provide it to the user:

            print "Exec\t" + name + "\t" + fromContext + "\t" + toContext

        elif (container[0] == "socket"):
            # Extract file descriptor and security contexts
            pass
        elif (container[0] == "bind"):
            # Extract file desc, sec context, port number, IP addr
            fd = container[1]
            sourceSC = container[2]
            bindPort = container[5]
            destSC = container[-1]
        elif (container[0] == "listen"):
            # Extract file desc, sec context
            fd = container[1]
            sourceSC = container[2]
        elif (container[0] == "accept"):
            # Extract file desc, sec context, port number, IP, new fd, new sc
            fd = container[1]
            sourceSC = container[2]
            port = container[5]
            ip = container[7]
            newFD = checkNum(container[-2])
            destSC = container[-1]
        elif (container[0] == "recv"):
            # Extract file desc, sc
            fd = container[1]
            sc = container[2]
            pass
        elif (container[0] == "send"):
            # Extract fd, sc
            fd = container[1]
            sc = container[2]
            pass
        elif (container[0] == "unlink"):
            # Extract information about the file being deleted
            # unlink("/home/asegall/NewOrders/10794710348684" <<asegall:object_r:new_orders_dir_t>>) = 0 <<asegall:user_r:user_t>>
            # Filename, Security Context of deleted file, resultant security context
            filename = container[1]
            delSC = container[2]
            resSC = container[-1]
            print "Unlink\t" + filename + "\t" + delSC + "\t" + resSC
        else:
            # This means we hit a line where we don't recognize the input.
            # Print a warning since this should have been filtered out a while ago.
            pass

    def cleanup(self):
        for key in openDict.keys():
            self.writeRecord(key)

    def getRWE(str):
        # This changes the string so we'll end up with a
        # combination of RWE, which denote the read, write, and
        # execute permissions on the block of memory we're mmaping into
        str = string.replace(str, "PROT_READ", "R")
        str = string.replace(str, "PROT_WRITE", "W")
        str = string.replace(str, "PROT_EXEC", "E")
        str = string.replace(str, "|", "")
        return str

    def checkNum(str):
        try:
            val = int(str)
        except ValueError:
            return False
        return int(str)

    def run(self):
        for line in sys.stdin:
            self.parseLine(line)

class UnfilteredFileDescriptorTracker(FileDescriptorTracker):
    def __init__(self):
        # Initialization of objects used during parsing
        self.container = []
        self.fieldCount = 0
        self.fileDescriptor = ""
        self.openDict = {}
        self.filenameDict = {}
        self.filenameSCDict = {}
        self.readDict = {}
        self.writeDict = {}
        self.key = ""

    def writeRecord(self,key):
        output = ""
        if self.filenameDict.has_key(self.fileDescriptor):
            output = self.filenameDict[self.fileDescriptor] + "\t"
            if self.readDict.has_key(self.fileDescriptor):
                output  = output + self.readDict[self.fileDescriptor] + "\t"

            if self.writeDict.has_key(self.fileDescriptor):
                output = output + self.writeDict[self.fileDescriptor] + "\t"

            if self.objectTypeDict.has_key(fileDescriptor):
                output = output + self.objectTypeDict[fileDescriptor] + "\t"

        print output

    def parseLine(self, line):
        # Clean off any newlines and split into the set of fields
        container = line.strip().split("~")
        fieldcount = len(container)
        if container[0] == "open" and fieldcount > 2:
            # this is an open() record
            if (fieldcount == 3):
                self.fileDescriptor = container[2]
            else:
                self.fileDescriptor = container[3]
            self.openDict[self.fileDescriptor] = "open"
            self.filenameDict[self.fileDescriptor] = container[1]
            #            self.filenameSCDict[self.fileDescriptor] = container[2]
        elif (container[0] == "read"):
            self.readDict[container[1]] = "R"
        elif (container[0] == "write" and (container[1] > 2)):
            self.writeDict[container[1]] = "W"
        elif (container[0] == "close"):
            self.fileDescriptor = container[1]
            if self.openDict.has_key(self.fileDescriptor):
                self.writeRecord(self.fileDescriptor)
            if self.openDict.has_key(self.fileDescriptor):
                del self.openDict[self.fileDescriptor]
            if self.filenameDict.has_key(self.fileDescriptor):
                del self.filenameDict[self.fileDescriptor]
            if self.readDict.has_key(self.fileDescriptor):
                del self.readDict[self.fileDescriptor]
            if self.writeDict.has_key(self.fileDescriptor):
                del self.writeDict[self.fileDescriptor]
        elif (container[0] == "old_mmap"):
            pass
        elif (container[0] == "fstat64"):
            pass
        elif (container[0] == "stat64"):
            pass
        elif (container[0] == "mmap2"):
            pass
        elif (container[0] == "execve"):
            # We want to print out a message that says what's being executed and
            # what we want to do with it.
            pass
        elif (container[0] == "socket"):
            pass
        elif (container[0] == "bind"):
            pass
        elif (containter[0] == "listen"):
            pass
        elif (container[0] == "accept"):
            pass
        elif (container[0] == "recv"):
            pass
        elif (container[0] == "send"):
            pass

        else:
            # This means we hit a line where we don't recognize the input.
            pass

def main():
    try:
        opts, args = getopt.getopt(sys.argv[1:], "hu", ["help", "unfiltered"])
    except getopt.GetoptError:
        usage()
        sys.exit(2)
    f = None
    for o,a in opts:
        if o in ("-h", "--help"):
            usage()
            sys.exit()
        if o in ("-u, --unfiltered"):
            f = UnfilteredFileDescriptorTracker()

    if f == None:
        f = FileDescriptorTracker()

    f.run()

def usage():
    print """
    Usage:
    cat <strace output> | python filter.strace.py | python track.fd.py >
    <output file>

    Options:
    -h, --help:          Print this message
    -u, --unfiltered:    Utilize output from a non-security-enabled strace
    """

if __name__ == "__main__":
    main()
