/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pxmchem-ionizerule.h"
#include "pxmchem-formula.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmIonizerule * 
pxmchem_ionizerule_new (void)
{
  PxmIonizerule *ionizerule = NULL;

  ionizerule = g_malloc0 (sizeof (PxmIonizerule));

  return ionizerule;
}


PxmIonizerule * 
pxmchem_ionizerule_dup (const PxmIonizerule *ionizerule)
{
  PxmIonizerule * new_ionizerule = NULL;

  g_assert (ionizerule != NULL);

  new_ionizerule = pxmchem_ionizerule_new ();

  /* Beware that the ionizerule's actform may be null if no special
     chemistry is to be applied to the molecule for its ionization.
  */
  if (ionizerule->actform != NULL)
    new_ionizerule->actform = g_strdup (ionizerule->actform);
  
  new_ionizerule->charge = ionizerule->charge;

  new_ionizerule->level = ionizerule->level;

  return new_ionizerule;
}


gboolean
pxmchem_ionizerule_set_actform (PxmIonizerule *ionizerule, gchar *actform)
{
  g_assert (ionizerule != NULL && actform != NULL);
  
  /* The member data may be NULL, as this function can be called
     right after pxmchem_ionizerule_new () which leaves the members
     NULL (except the propGPA which is allocated).
  */
  if (ionizerule->actform != NULL)
    g_free (ionizerule->actform);
  
  ionizerule->actform = g_strdup (actform);  

  return TRUE;
}


gboolean
pxmchem_ionizerule_set_charge (PxmIonizerule *ionizerule, gint charge)
{
  g_assert (ionizerule != NULL);
  
  ionizerule->charge = charge;  

  return TRUE;
}


gboolean
pxmchem_ionizerule_set_level (PxmIonizerule *ionizerule, gint level)
{
  g_assert (ionizerule != NULL);
  
  ionizerule->level = level;

  return TRUE;
}




/* INTEGRITY CHECKING FUNCTIONS
 */
gboolean
pxmchem_ionizerule_validate (PxmIonizerule *ionizerule, GPtrArray *atom_refGPA,
			     gchar **valid)
{
  GString *gs = NULL;
  

  g_assert (ionizerule != NULL);
  g_assert (atom_refGPA != NULL);


  /* Note that for integrity reasons, *valid MUST BE NULL to ensure 
   * that it is empty.
   */
  g_assert (valid != NULL);
  g_assert (*valid == NULL);

  
  gs = g_string_new ("");
  

  /* The actform: it must be non-NULL and be chemically valid.
   */
  if (ionizerule->actform == NULL)
    {
      g_string_append_printf (gs, 
			      _("ionizerule has a NULL 'actform' member\n"));
    }
  else
    {
      /* We do not need to strip the string of its spaces, because
       * the actform checking function will do this itself.
       */
      if (FALSE == pxmchem_actform_check (ionizerule->actform, 
					  atom_refGPA))
	{
	  g_string_append_printf (gs, 
				  _("ionizerule has an invalid 'actform' member:"
				    " '%s'\n"), ionizerule->actform);
	}
    }
  
  /* The charge: well it is difficult to check for a reasonable value,
   * so let the user check it. We do not have any reasonable way
   * to check it!
   */

  /* The level: well it is difficult to check for a reasonable value,
   * so let the user check it. We do not have any reasonable way
   * to check it!
   */

  /* Finally the validation is finished.
   */
  if (strlen (gs->str) > 0)
    {
      /* String is not empty, which is there were errors.
       */
      *valid = gs->str;
      
      g_string_free (gs, FALSE);
      
      return FALSE;
    }

  g_string_free (gs, TRUE);
  
  return TRUE;
}




/* UTILITY FUNCTIONS
 */
gboolean
pxmchem_ionizerule_set_default (PxmIonizerule *ionizerule)
{
  g_assert (ionizerule != NULL);

  pxmchem_ionizerule_set_actform (ionizerule, "+H1");

  ionizerule->charge = +1;
    
  ionizerule->level = +1;
    
  return TRUE;
}



/* XML-format TRANSACTIONS
 */
gchar *
pxmchem_ionizerule_format_xml_string_ionizerule (PxmIonizerule *ionizerule, 
						 gchar *indent, gint offset)
{
  /* The pointer to the ionizerule will allow to get access to its
   * member data and to construct an xml string to describe these.
   */
  gint new_offset = 0;
  
  gchar *lead = NULL;
  gchar *help = NULL;
  
  GString *gs = NULL;
  
  g_assert (ionizerule != NULL && indent != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We are willing to create a <ionizerule> node that should 
   * look like this:
   *
   *    <ionizerule>
   *      <actform>+H1</actform>
   *      <charge>1</charge>
   *    </ionizerule>
   */

  /* Open the <ionizerule> element and insert the data.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);

  g_string_append_printf (gs, "%s<ionizerule>\n", lead);

  g_free (lead);

  new_offset = offset + 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  g_assert (ionizerule->actform != NULL 
	    && strlen (ionizerule->actform) > 0);
  
  g_string_append_printf (gs, "%s<actform>%s</actform>\n",
			  lead, ionizerule->actform);
  
  g_string_append_printf (gs, "%s<charge>%d</charge>\n",
			  lead, ionizerule->charge);

  g_string_append_printf (gs, "%s<level>%d</level>\n",
			  lead, ionizerule->level);

  g_free (lead);
  
  /* Finally close the ionizerule element.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);
  
  g_string_append_printf (gs, "%s</ionizerule>\n", lead);

  g_free (lead);
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}



PxmIonizerule *
pxmchem_ionizerule_render_xml_node_ionizerule (xmlDocPtr xml_doc,
					       xmlNodePtr xml_node,
					       gpointer user_data)
{
  /* The xml node we are in is structured this way:
   *
   *    <ionizerule>
   *      <actform>+H1</actform>
   *      <charge>1</charge>
   *    </ionizerule>
   *
   * And the xml_node parameter points to the 
   *
   * <ionizerule> element tag:
   *  ^
   *  |
   *  +----- here we are right now.
   * 
   * Which means that xml_node->name == "ionizerule" and that
   * we'll have to go one step down to the first child of the 
   * current node in order to get to the <actform> element.
   *
   */

  PxmIonizerule *ionizerule = NULL;

  gchar *help = NULL;
    
  
  /* Make sure we have parameters pointing bona fide to the right
   * xml element.
   */
  g_assert (xml_node != NULL);
  g_assert (0 == strcmp ((gchar *) xml_node->name, "ionizerule"));
  
  /* Now go to the first child of current node: <actform>.
   */
  xml_node = xml_node->children;

  /* From a rigorous XML parsing point of view, the blanks found in
   * the XML document are considered to be nodes, and we have to detect
   * these and take proper action: go next sibling (next blank) as long
   * as blanks are encountered.
   */
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;
  
  /* Check that we have effectively a <actform> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "actform"));

  /* Allocate the ionizerule instance.
   */
  ionizerule = pxmchem_ionizerule_new ();
  
  /* Since we have allocated the ionizerule instance at the line above,
     we know that its member data are NULL, so we can make direct
     assignements, without recoursing to the _set_xxx ().
  */
  ionizerule->actform = 
    (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
  g_assert (ionizerule->actform != NULL);
  
  /* Go to the next sibling of the <actform> which, by the DTD,
   * should be the <charge> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <charge> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "charge"));

  help = (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
 
  if (FALSE == libpolyxmass_globals_strtoi (help, &(ionizerule->charge), 10))
    {
      g_free (help);
      pxmchem_ionizerule_free (ionizerule);
      
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: failed to convert from '%s' to gint\n"),
	     __FILE__, __LINE__, xml_node->xmlChildrenNode->content);

      g_free (help);
      
      return NULL;
    }
  
  g_free (help);
  
  /* Go to the next sibling of the <actform> which, by the DTD,
   * should be the <level> element.
   */
  xml_node = xml_node->next;
  while (TRUE == xmlIsBlankNode (xml_node))
    xml_node = xml_node->next;

  /* Check that we have effectively a <level> element here.
   */
  g_assert (0 == strcmp ((gchar *) xml_node->name, "level"));

  help = (gchar *) xmlNodeListGetString (xml_doc, xml_node->xmlChildrenNode, 1);
 
  if (FALSE == libpolyxmass_globals_strtoi (help, &(ionizerule->level), 10))
    {
      g_free (help);
      pxmchem_ionizerule_free (ionizerule);
      
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: failed to convert from '%s' to gint\n"),
	     __FILE__, __LINE__, xml_node->xmlChildrenNode->content);
      
      g_free (help);
      
      return NULL;
    }
  
  g_free (help);
  

  /* At this point we have an ionizerule that is fully qualified for 
   * its actform and its charge and level.
   */
  
  /* Finished the rendering of the current <ionizerule> node.
   */
  return ionizerule;
}



/* FREE'ING FUNCTIONS
 */
gboolean
pxmchem_ionizerule_free (PxmIonizerule *ionizerule)
{
  g_assert (ionizerule != NULL);

  if (ionizerule->actform != NULL)
    g_free (ionizerule->actform);
    
  g_free (ionizerule);

  return TRUE;
}
