/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxmass-ui-code-completions.h"
#include "polyxmass-pixbuf-rendering.h"
#include "polyxmass-ui-seqed-widget-kbd.h"
#include "polyxmass-ui-seqed-widget.h"


enum
  {
    COMP_COLUMN_MNM_PIXBUF,
    COMP_COLUMN_MNM_CODE,
    COMP_COLUMN_MNM_NAME,
    COMP_COLUMN_MNM_COL_COUNT
  };


GtkWidget *
polyxmass_code_completions_mnm_code_show_list (PxmSeqedWidget *seqed_widget,
					       GPtrArray *GPA)
{
  GtkWidget *window = NULL;

 
  /* Each seqed_widget may elicit the opening of a completions list window with the
     keyboard hitting of the 'TAB' key or by choosing the corresponding contextual menu item.
  */
  g_assert (seqed_widget != NULL);
  
  g_assert (GPA != NULL);
  
  /* The window may be exising already with the seqed_widget!
   */
  if (seqed_widget->monomer_code_completions_wnd != NULL)
    {
      polyxmass_code_completions_wnd_empty_list_completions (seqed_widget);
      polyxmass_code_completions_wnd_update_list_completions (seqed_widget, GPA);
      
      return seqed_widget->monomer_code_completions_wnd;
    }
  
  /* Apparently the current editing context has no completions window
   * open, so we have to go through all the creation process.
   */
  window = polyxmass_code_completions_wnd_setup (seqed_widget, GPA);
  
  g_assert (window != NULL);
  
  /* Now that we have the window in which we want to list the
   * available completions, do it. However, remember that a window may
   * be pre-existing, and if it is so, it may contain already
   * previously needed completions. So, remove any existing
   * completions from the list_store.
   */
  seqed_widget->monomer_code_completions_wnd = window;
  
  /* Note that GPA is emptied after this function returns, so it is not
     possible to use the GPA pointer.
  */

  return seqed_widget->monomer_code_completions_wnd;
}



GtkWidget *
polyxmass_code_completions_wnd_setup (PxmSeqedWidget *seqed_widget,
				      GPtrArray *GPA)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;

  GtkWidget *completions_vbox = NULL;
  GtkWidget *scroll_wnd = NULL;
  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn* column = NULL;
  
  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;
  
  
  g_assert (seqed_widget != NULL);
  g_assert (GPA != NULL);
  

  gui_file = 
    g_strdup_printf ("%s/polyxedit-completions.glade", userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "completions_list_wnd", 
		       PACKAGE);
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "completions_list_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the completions window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Immediately set to the window a pointer to the seqed_widget:
   */
  g_object_set_data (G_OBJECT (window), "seqed_widget", seqed_widget);


  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window), "messages_entry",
		     widget);


  /* Set the polymer chemistry definition type to its correspondent
     GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polchemdef_type_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "polchemdef_type_entry", widget);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      seqed_widget->polchemdefctxt->polchemdef->type);
  

 /* Get from the glade definition file the vertical box into which
   * we are going to pack a number of widgets:
   */

  completions_vbox = glade_xml_get_widget (xml, "completions_vbox");
  g_assert (completions_vbox != NULL);


  /* Create the scrolled window that we'll pack into widget.
   */
  scroll_wnd = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll_wnd),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_wnd),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (completions_vbox), scroll_wnd, 
		      TRUE, TRUE, 0);  
  
  g_object_set_data (G_OBJECT (window), 
		     "scroll_wnd", scroll_wnd);
  


  /* Create the treeview model.
   */
  model = polyxmass_code_completions_wnd_create_treeview_model (seqed_widget,
								GPA);
  
  /* Set to the model a datum with a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "window", window);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible.
   */
  g_object_set_data (G_OBJECT (window), "TREEVIEW_MODEL", model);
    

  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later.
   */
  g_object_set_data (G_OBJECT (window), "TREEVIEW", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);

  /* Reacts when a double-click happens on an item.
   */
  g_signal_connect (G_OBJECT (treeview),
		    "row-activated",
		    G_CALLBACK 
		    (polyxmass_code_completions_treeview_row_activated_event), 
		    seqed_widget);
  
  /* We want to be able to trap the ENTER key.
   */
  gtk_widget_add_events (GTK_WIDGET (treeview), GDK_KEY_PRESS_MASK);
  
  g_signal_connect (G_OBJECT (treeview),
		    "key-release-event",
		    G_CALLBACK 
		    (polyxmass_code_completions_treeview_key_release_event),
		    seqed_widget);
  
  /* Pixbuf column.
   */
  renderer = gtk_cell_renderer_pixbuf_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Monicon"), renderer,
     "pixbuf", COMP_COLUMN_MNM_PIXBUF,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COMP_COLUMN_MNM_PIXBUF);


  /* Code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column",
		      (gint *) COMP_COLUMN_MNM_CODE);

  column = 
    gtk_tree_view_column_new_with_attributes (_("Code"),
					      
					      renderer,
					      
					      "text", 
					      COMP_COLUMN_MNM_CODE,
					      
					      NULL);

  gtk_tree_view_column_set_sort_column_id (column, 
					   COMP_COLUMN_MNM_CODE);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COMP_COLUMN_MNM_NAME);


  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column",
		      (gint *) COMP_COLUMN_MNM_CODE);

  column = 
    gtk_tree_view_column_new_with_attributes (_("Name"),
					      
					      renderer,
					      
					      "text", 
					      COMP_COLUMN_MNM_NAME,
					      
					      NULL);

  gtk_tree_view_column_set_sort_column_id (column, 
					   COMP_COLUMN_MNM_NAME);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);



  /* Add the treeview to the scroll_wnd.
   */
  gtk_container_add (GTK_CONTAINER (scroll_wnd), treeview);
  
  gtk_widget_show_all (window);
  
  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));


  /* Set the name of the polymer sequence to the title bar of the
   * window. Only if applicable, for example if the seqed_widget is
   * packed in a find/replace window, the polymer won't have a name !
   */
  if (seqed_widget->polymer->plminfo->name != NULL)
    {
      help = g_strdup_printf (_("Completions: %s"), 
			      seqed_widget->polymer->plminfo->name);
    }
  else
      help = g_strdup_printf (_("Completions: %s"), 
			      _("Not set"));
  
  gtk_window_set_title (GTK_WINDOW (window), help);
  
  g_free (help);
  

  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxmass_code_completions_wnd_delete_event), 
		    seqed_widget);
  
  return window;
}



gint
polyxmass_code_completions_wnd_empty_list_completions (PxmSeqedWidget *seqed_widget)
{
  gint count = 0;

  gboolean valid = FALSE;
  
  GtkTreeModel *model = NULL;
  GtkTreeIter treeiter ;
  GtkTreeView *treeview = NULL;
  GtkTreePath *path = NULL;
  
  GdkPixbuf *pixbuf = NULL;
    
  
  g_assert (seqed_widget != NULL);
  
  
  /* We get a pointer to a seqed_widget that contains a pointer to a
     window that may already exist and that might list a number of
     code completion items. We first have to remove them all.

     Of course, if the window does not exist, we have nothing to
     empty!
   */
  if (seqed_widget->monomer_code_completions_wnd == NULL)
    return 0;
  
  treeview = 
    (GtkTreeView *) g_object_get_data (G_OBJECT (seqed_widget->monomer_code_completions_wnd), 
				       "TREEVIEW");

  model = gtk_tree_view_get_model (treeview);

  path = gtk_tree_path_new_first ();  

  if (TRUE == gtk_tree_model_get_iter_first (model, &treeiter))
    {
      /* The tree is not empty. We should first empty it.
       */
      
      /* Get the first iter in the list.
       */
      valid = gtk_tree_model_get_iter_first (model, &treeiter);

      while (valid == TRUE)
	{
	  /* Walk through the list, reading each row.
	   */
	  /* Make sure to terminate calls to gtk_tree_model_get()
	   * with a '-1' value.
	   */
	  gtk_tree_model_get (model, &treeiter, 
			      COMP_COLUMN_MNM_PIXBUF, &pixbuf,
			      -1);
	  
	  /* Unref the pixbuf that was gotten from previous call.
	   */
	  g_object_unref (G_OBJECT (pixbuf));
	  
	  /* Do not do anything with the monomer's name and code, because
	   * they are pointed to in the model, and were not duplicated !
	   * This may need a fix later, with a duplication step when the
	   * items are appended to the list_store.
	   */

	  count++;
	  
	  valid = gtk_tree_model_iter_next (model, &treeiter);
	}
      
      /* Now that we have made the necessary with all the items, we 
       * can remove them from the list in a single step:
       */
      gtk_list_store_clear (GTK_LIST_STORE (model));
    }

  return count;
}


gint
polyxmass_code_completions_wnd_update_list_completions (PxmSeqedWidget *seqed_widget,
							GPtrArray *GPA)
{
  gint count = 0;
  gint iter = 0;

  GtkTreeModel *model = NULL;
  GtkTreeIter treeiter ;
  GtkTreeView *treeview = NULL;
  GtkTreePath *path = NULL;

  PxmPolchemdefCtxt *polchemdefctxt = NULL;
    
  GdkPixbuf *pixbuf = NULL;
    
  PxmMonomer *monomer;


  
  g_assert (seqed_widget != NULL);
  g_assert (GPA != NULL);

  polchemdefctxt = seqed_widget->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
    
  

  /* We get a pointer to a seqed_widget instance, where the user has
     hit the TAB kbd key in order to get the available monomer code
     completions for the currently edited multi-character code.

     That seqed_widget contains a pointer to the polchemdefctxt of the
     sequence being edited.

     That seqed_widget contains also a pointer to the window where the
     completions might have already been displayed ! If that's the
     case, we first have to remove them all and next fill them again *
     using the data in GPA.
   */
  if (seqed_widget->monomer_code_completions_wnd == NULL)
    return count;
  
  treeview = 
    (GtkTreeView *) g_object_get_data (G_OBJECT (seqed_widget->monomer_code_completions_wnd), 
				       "TREEVIEW");

  model = gtk_tree_view_get_model (treeview);

  path = gtk_tree_path_new_first ();  

  polyxmass_code_completions_wnd_empty_list_completions (seqed_widget);
    
  /* Now that we have emptied the store, we can append the items that 
   * are in the GPA.
   */
  if (GPA != NULL)
    {
      for (iter = 0 ; iter < GPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (GTK_LIST_STORE (model), &treeiter);
	  
	  monomer = g_ptr_array_index (GPA, iter);
	  g_assert (monomer != NULL);

	  /* We now have to get a pointer to the monomer's monicon's
	   * pixbuf.
	   */
	  pixbuf = 
	    polyxmass_pixbuf_rendering_get_from_data_list (monomer->code,
							   NULL,
							   &(seqed_widget->pixbufGData));
	  
	  if (pixbuf != NULL)
	    {
	      /* We should refcount this instance.
	       */
	      g_object_ref (G_OBJECT (pixbuf));
	    }
	  else	  
	    {
	      pixbuf = 
		polyxmass_pixbuf_rendering_render_no_modif (monomer->code,
							    seqed_widget->monicon_size,
							    &(seqed_widget->pixbufGData),
							    polchemdefctxt);
	      if (pixbuf != NULL)
		{
		  /* Let the polymer chemistry definition context know
		   * that we have rendered the pixbuf for the monomer
		   * 'code'.  Note that we just created a pixbuf from
		   * nothing and it was created by a reference count
		   * of 1, so we do not need to do anything.
		   */
		  polyxmass_pixbuf_rendering_set_to_data_list (pixbuf, 
							       monomer->code,
							       NULL, 
							       &(seqed_widget->pixbufGData));
		  /* We should refcount this instance.
		   */
		  g_object_ref (G_OBJECT (pixbuf));
		}
	      else
		{
		  g_error (_("%s@%d: failed to render pixbuf for code: '%s'\n"),
			 __FILE__, __LINE__, monomer->code);
		}
	    }
	  
	  gtk_list_store_set (GTK_LIST_STORE (model), &treeiter,
			      
			      COMP_COLUMN_MNM_PIXBUF,
			      pixbuf,
			      
			      COMP_COLUMN_MNM_CODE, 
			      monomer->code,
			      
			      COMP_COLUMN_MNM_NAME, 
			      monomer->name,
			      
			      -1);

	  count++;
	}
    }
    
  return count;
}


GtkTreeModel *
polyxmass_code_completions_wnd_create_treeview_model (PxmSeqedWidget *seqed_widget,
						      GPtrArray *GPA)
{
  gint iter = 0;
  
  GtkListStore *model = NULL;
  GtkTreeIter treeiter;

  PxmPolchemdefCtxt *polchemdefctxt = NULL;
    
  GdkPixbuf *pixbuf = NULL;
    
  PxmMonomer *monomer = NULL;
  


  g_assert (seqed_widget != NULL);
  
  polchemdefctxt = seqed_widget->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  
    
  /* Create the list store.
   */
  model = gtk_list_store_new (COMP_COLUMN_MNM_COL_COUNT, /*Numb. of columns*/
			      GDK_TYPE_PIXBUF, /* monicon's pixbuf */
			      G_TYPE_STRING, /* monomer code */
			      G_TYPE_STRING /* monomer name */);
  
  /* Add the items if GPA is non-null and there are items in it.
   */
  if (GPA != NULL)
    {
      for (iter = 0 ; iter < GPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (model, &treeiter);
	  
	  monomer = g_ptr_array_index (GPA, iter);
	  g_assert (monomer != NULL);
	  
	  /* We now have to get a pointer to the monomer's monicon's
	   * pixbuf. The function below refcounts the pixbuf if it is
	   * found existing already, so that it is not necessary to do
	   * it here.
	   */
	  pixbuf = polyxmass_pixbuf_rendering_render_no_modif (monomer->code,
							       seqed_widget->monicon_size,
							       &(seqed_widget->pixbufGData),
							       polchemdefctxt);
	  
	  gtk_list_store_set (model, &treeiter,
			      
			      COMP_COLUMN_MNM_PIXBUF,
			      pixbuf,
			      
			      COMP_COLUMN_MNM_CODE, 
			      monomer->code,
			      
			      COMP_COLUMN_MNM_NAME, 
			      monomer->name,
			      
			      -1);
	}
    }
  
  
  return GTK_TREE_MODEL (model);
}


/* Reacts when a double-click happens on an item.
 */
void
polyxmass_code_completions_treeview_row_activated_event (GtkTreeView *treeview,
						    GtkTreePath *path,
						    GtkTreeViewColumn *col,
						    gpointer data)
{
  PxmSeqedWidget *seqed_widget = data;
  
  GtkTreeIter iter;
  GtkTreeModel *model = NULL;
  GtkTreeSelection* selection = NULL; 
  
  PxmPolchemdefCtxt *polchemdefctxt = NULL;

  gchar* err = NULL;
  gchar* code = NULL;
  
  
  g_assert (treeview != NULL);
  
  g_assert (seqed_widget != NULL);

  polchemdefctxt = seqed_widget->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  

  /* The user has selected an item, which means that she wants the
     corresponding monomer to be inserted in the polymer sequence editor.
     Thus we need to get the name of the monomer item that's selected.
  */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);

  selection = gtk_tree_view_get_selection (treeview);

  if (FALSE == gtk_tree_selection_get_selected (selection,
						NULL,
						&iter))
    return; /*no selection, just return. Should never be the case. */
  

  /* Make sure you terminate calls to gtk_tree_model_get()
     with a '-1' value. We can do this call because we have the iter
     pointing to the selected node!
   */
  gtk_tree_model_get (model, &iter, 
		      COMP_COLUMN_MNM_CODE, &code,
		      -1);

  g_assert (code != NULL);
  
  /* The evaluation of the code is the process by which a given
   * monomer code is changed into a real monomer in the
   * array of monomers in the polymer sequence.
   */
  if (FALSE == polyxmass_seqed_widget_kbd_evaluate_code (code,
						  seqed_widget->last_point_1_idx,
						  GTK_WIDGET (seqed_widget)))
    {
      g_free (code);

      err = g_strdup_printf (_("Failed completion monomer code: '%s'"),
			     code);
      
      gtk_entry_set_text (GTK_ENTRY (seqed_widget->error_code_entry),
			  err);
      
      g_free (err);

      return;
    }

  g_free (code);

  /* If we are here, that means that the evaluation of the eval_code
     has been successful, so we have to do the following:
    
     1. emtpy the elab_code and the eval_code, so that we later can use
     them fresh.
     
     
     2. set to 0 the count of typed characters, since we have finished
     dealing witht he previously elaborating code.

     3. set to empty the text entry that hold the elaborating code.
  */
  memset (seqed_widget->eval_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);
  
  memset (seqed_widget->elab_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);

  seqed_widget->kb_typed_chars = 0;
  
  polyxmass_seqed_widget_kbd_echo_elab_code (GTK_WIDGET (seqed_widget));
	  
  return ;
}




/* Reacts when a key is released in the completions window.
 */
void
polyxmass_code_completions_treeview_key_release_event (GtkWidget *widget,
						  GdkEventKey *event,
						  gpointer data)
{
  PxmSeqedWidget *seqed_widget = data;

  GtkTreeView *treeview = GTK_TREE_VIEW (widget);
  GtkTreeSelection* selection = NULL; 
  GtkTreeIter iter;
  GtkTreeModel *model = NULL;
 
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  
 
  gchar* err = NULL;
  gchar* code = NULL;
  
  
  GdkEventKey *gdk_event_key = (GdkEventKey *) event;
  
  if (GDK_Return != gdk_event_key->keyval)
    return;
  
  g_assert (treeview != NULL);
  
  g_assert (seqed_widget != NULL);

  polchemdefctxt = seqed_widget->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  

  /* The user has selected an item, which means that she wants the
     corresponding monomer to be inserted in the polymer sequence editor.
     Thus we need to get the name of the monomer item that's selected.
  */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);

  selection = gtk_tree_view_get_selection (treeview);

  if (FALSE == gtk_tree_selection_get_selected (selection,
						NULL,
						&iter))
    return; /*no selection, just return. Should never be the case. */
  

  /* Make sure you terminate calls to gtk_tree_model_get()
     with a '-1' value. We can do this call because we have the iter
     pointing to the selected node!
   */
  gtk_tree_model_get (model, &iter, 
		      COMP_COLUMN_MNM_CODE, &code,
		      -1);

  g_assert (code != NULL);
  
  /* The evaluation of the code is the process by which a given
   * monomer code is changed into a real monomer in the
   * array of monomers in the polymer sequence.
   */
  if (FALSE == polyxmass_seqed_widget_kbd_evaluate_code (code,
						  seqed_widget->last_point_1_idx,
						  GTK_WIDGET (seqed_widget)))
    {
      g_free (code);

      err = g_strdup_printf (_("Failed completion monomer code: '%s'"),
			     code);
      
      gtk_entry_set_text (GTK_ENTRY (seqed_widget->error_code_entry),
			  err);
      
      g_free (err);

      return;
    }

  g_free (code);

  /* If we are here, that means that the evaluation of the eval_code
     has been successful, so we have to do the following:
    
    1. emtpy the elab_code and the eval_code, so that we later can use
       them fresh.
   
   
    2. set to 0 the count of typed characters, since we have finished
    dealing witht he previously elaborating code.

    3. set to empty the text entry that hold the elaborating code.
  */
  memset (seqed_widget->eval_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);
  
  memset (seqed_widget->elab_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);
  
  seqed_widget->kb_typed_chars = 0;
  
  polyxmass_seqed_widget_kbd_echo_elab_code (GTK_WIDGET (seqed_widget));
	  
  return ;
}


gint 
polyxmass_code_completions_wnd_delete_event (GtkWidget *window, 
					GdkEventAny *event, 
					gpointer data)
{
  PxmSeqedWidget *seqed_widget = data;

  
  g_assert (seqed_widget != NULL);

  /* We must empty the list of completions and set to NULL the 
   * member of the editctxt structure so that when completions are
   * required later a new window may be created ex nihilo.
   */
  polyxmass_code_completions_wnd_empty_list_completions (seqed_widget);
  
  seqed_widget->monomer_code_completions_wnd = NULL;
    
  return FALSE;
}

