/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef DEVICE_MANAGEMENT_H
#define DEVICE_MANAGEMENT_H

#include <string>
#include <map>
#include "device.h"


/** @brief class caring about all powermanagable devices */
class DeviceManagement {
public:
	/** @brief constructor */
	DeviceManagement();

	/** @brief init all devices */
	void init();
		
	/** @brief destructor resetting all devices to D0 */
	virtual ~DeviceManagement();

	/** @brief suspend devices
	 *
	 * @param device_class suspend only devices from this class
	 * @param device_num only suspend device device_num (default is
	 *        -1, means suspend all of this class)
	 * @param force set to true if devices should be suspended
	 *        although they are in use
	 *
	 * @return false if no devices where suspended
	 */
	bool suspend(DPM_DEVICE_CLASS device_class, int device_num = -1, bool force = false);

	/** @brief resume devices
	 *
	 * @param device_class resume only devices from this class
	 * @param device_num only resume device device_num (default is
	 *        -1, means resume all of this class)
	 *
	 * @return false if no devices where resumed
	 */
	bool resume(DPM_DEVICE_CLASS device_class, int device_num = -1);

	/** @brief updates the powerstate of all devices
	 *
	 * function suspends those devices whos @ref DPM_DEVICE_CLASS can
	 * be found in the given list rpm_classes. Devices not found are
	 * resumed
	 *
	 * @param rpm_classes list of @ref DPM_DEVICE_CLASS which should
	 *        get suspended
	 * @param force set to true if devices should be suspended
	 *        although they are in use
	 *
	 * @return false if no devices where suspended
	 */
	bool update(std::list< DPM_DEVICE_CLASS > rpm_classes, bool force = false);

	/** @brief get list of stored devices
	 *
	 * @param device_class get devices from this @ref DPM_DEVICE_CLASS
	 * @param udis stl list which get filled with udis
	 * @param power_states stl list which get filled with power states
	 *
	 * @return number of devices
	 */
	int getDevices(DPM_DEVICE_CLASS device_class,
		       std::list< std::string > &udis,
		       std::list< DPM_POWER_STATE > &power_states);

	/** @brief add a wlan device
	 *
	 * @param udi the device udi to add
	 */
	void addWlanDevice(const std::string &udi);

	/** @brief add a usb device
	 *
	 * @param udi the device udi to add
	 */
	void addUsbDevice(const std::string &udi);

	/** @brief add a sound device
	 *
	 * @param udi the device udi to add
	 */
	void addSoundDevice(const std::string &udi);

	/** @brief add a lan device
	 *
	 * @param udi the device udi to add
	 */
	void addLanDevice(const std::string &udi);
	
	/** @brief check if an added device can be added to out device list
	 *
	 * @param udi the device udi
	 *
	 * @return true if the the device was added
	 */
	bool checkDeviceAdded(char *udi);

	/** @brief check if a removed device was one of our stored devices
	 *
	 * @param udi the device udi
	 *
	 * @return true if the device got remoced
	 */
	bool checkDeviceRemoved(char *udi);

private:
	/** @brief list holding powermanagable devices */
	std::list< Device* > _devices;
};

#endif // DEVICE_MANAGEMENT_H
