/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *           Copyright (C) 2005 SUSE Linux Products GmbH                   *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef POWERSAVE_DBUS_H
#define POWERSAVE_DBUS_H
#define DBUS_API_SUBJECT_TO_CHANGE 1

#include <dbus-1.0/dbus/dbus.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/** @defgroup libpowersave_dbus Functions to communicate with the daemon via dbus
 * @ingroup libpower
 * @{ 
 */

#define PS_DBUS_SERVICE                "com.novell.powersave"
#define PS_DBUS_ACTION_INTERFACE       "com.novell.powersave.action"
#define PS_DBUS_ADMIN_INTERFACE        "com.novell.powersave.admin"
#define PS_DBUS_REQUEST_INTERFACE      "com.novell.powersave.request"
#define PS_DBUS_MANAGER_INTERFACE      "com.novell.powersave.manager"
#define PS_DBUS_SCRIPTS_INTERFACE      "com.novell.powersave.scripts"
#define PS_DBUS_PATH                   "/com/novell/powersave"

/** @defgroup client_capabilities Client capabilities: Capabilities
 * clients can have
 * @ingroup client_exchange Defines the different
 * capabilities clients can request from the daemon. Capabilities can be
 * added up. These are simple integer values.  @{
 */
/** Forward notifications the daemon throws */
#define CAPABILITY_NOTIFICATIONS 1
/** Forward screenlock requests */
#define CAPABILITY_SCREENLOCK 2
/** client is caring about brighness adjustments of the display */
#define CAPABILITY_BRIGHTNESS 4
	/** @} */// end of client_modes group
	
	/** @brief dbus error message ids. They can be mapped
	 *         to the DBus_Error_Array string array below
	 */
	enum DBUS_ERROR { REPLY_SUCCESS,
		REPLY_NO_CONNECTION,
		REPLY_NO_RIGHTS,
		REPLY_INVALID_PARAM,
		REPLY_INVALID_METHOD,
		REPLY_INVALID_INTERFACE,
		REPLY_HW_NOT_SUPPORTED,
		REPLY_DISABLED,
		REPLY_ALREADY_SET,
		REPLY_DBUS_ERROR,
		REPLY_GENERAL_ERROR,
		REPLY_INVALID_MESSAGE_TYPE,
		REPLY_ERROR_MAX
	};

	extern char *DBus_Error_Array[REPLY_ERROR_MAX];

	/** @brief Enum for the interface to use */
	enum POWERSAVE_MESSAGE_TYPE { MANAGER_MESSAGE,
				      REQUEST_MESSAGE,
				      ACTION_MESSAGE,
				      SCRIPTS_MESSAGE,
	                              ADMIN_MESSAGE };

	/** @brief frees and closes the connection to DBus
	 *
	 * Call this inside your application as soon as the dbus daemon
	 * brakes away
	 */
	void dbusFree(void);

	/** @brief get a string argument from a dbus messsage
	 *
	 * if sendMessageWithReply was successful (returned REPLY_SUCCESS)
	 * you may want to use this function to get the contents of the reply.
	 *
	 * @param msg The message where the contents should be read from
	 * @param s   The char double pointer that gets filled with the desired string
	 * @param no  The Xth string that was added to the message that 
	 *            s should be filled with (starting from 0)
	 *
	 * @return negative on failure, zero on success
	 * @retval DBUS_ERROR
	 */
	int dbusGetMessageString(DBusMessage *msg, char **s, int no);

	/** @brief get an integer argument from a dbus messsage
	 *
	 * if sendMessageWithReply was successful (returned REPLY_SUCCESS)
	 * you may want to use this function to get the contents of the reply.
	 *
	 * @param msg The message where the contents should be read from
	 * @param i pointer to an integer value that gets filled with the
	 *          desired value
	 * @param no The Xth integer that was added to the message that 
	 *           i should be filled with (starting from 0)
	 *
	 * @return negative on failure, zero on success
	 * @retval DBUS_ERROR
	 */
	int dbusGetMessageInteger(DBusMessage *msg, int *i, int no);

	/** @brief get the DBUS_ERROR id from a reply
	 *
	 * if sendMessageWithReply was unsuccessful (returned !REPLY_SUCCESS)
	 * you may want to use this function to get the contents of the reply.
	 *
	 * @param msg The message where the contents should be read from
	 * @param error_id pointer to an integer that gets filled with the
	 *                 error_id
	 *
	 * @return negative on failure, zero on success
	 * @retval DBUS_ERROR
	 */
	int dbusGetMessageErrorID(DBusMessage *msg, unsigned *error_id);

	/** @brief Send a message to the powersave daemon and receive the reply
	 *      
	 * You can only use the REQUEST and ACTION interface (msg_type)
	 * and you need to know what request/action you want to send and
	 * give the appropriate params for.
	 *
	 * Use the dbus type definitions and let the varargs end with
         * DBUS_TYPE_INVALID (also see the dbus doc/code for the function
	 * dbus_message_append_args(...)
	 *
	 * E.g.:
	 * @code
	 * DBusMessage *reply;
	 * int ret;
	 * char *scheme = "performance";
	 * ret = dbus_send_message(REQUEST, &reply, "set_scheme",
	 *                         DBUS_TYPE_STRING, &scheme, DBUS_TYPE_INVALID);
	 * if (ret == REPLY_SUCCESS){
	 *        // PROCESS MESSAGE;
	 *        dbus_message_unref(reply)
	 * }
	 * else{ // Handle error - no need to unref reply 
	 * } 
	 * @endcode
	 *
	 * Tries to set the current scheme ("set_scheme" -> method name) to
	 * "performance" (special argument for "set_scheme" messages)
	 *
	 * @param msg_type @ref POWERSAVE_MESSAGE_TYPE
	 * @param method the dbus method to invoke
	 *
	 * @return The error code (see @ref DBUS_ERROR . Use DBus_Error_Array[RET_VAL]
	 *         to get the corresponding error string).
	 */
	unsigned dbusSendSimpleMessage(int msg_type, const char *method);

	/** @brief send a message to the daemon and receive the reply
	 *
	 * @param msg_type @ref POWERSAVE_MESSAGE_TYPE
	 * @param reply double pointer in which the reply get stored
	 * @param method the dbus method to invoke
	 *
	 * @return @ref DBUS_ERROR
	 */
	unsigned dbusSendSimpleMessageWithReply(int msg_type, DBusMessage **reply, const char *method);

	/** @brief send message to the daemon with arguments
	 *
	 * @param msg_type @ref POWERSAVE_MESSAGE_TYPE
	 * @param method the dbus method to invoke
	 * @param first_arg_type the type of the first argument. End the list with DBUS_TYPE_INVALID
	 *
	 * @return @ref DBUS_ERROR
	 */
	unsigned dbusSendMessage(int msg_type, const char *method, int first_arg_type, ...);

	/** @brief send a message to the daemon with arguments and receive the reply
	 *
	 * @param msg_type @ref POWERSAVE_MESSAGE_TYPE
	 * @param reply double pointer in which the reply get stored
	 * @param method the dbus method to invoke
	 * @param first_arg_type the type of the first argument. End the list with DBUS_TYPE_INVALID
	 *
	 * @return @ref DBUS_ERROR
	 */
	unsigned dbusSendMessageWithReply(int msg_type,
					  DBusMessage **reply, const char *method, int first_arg_type, ...);

	/** @brief establishes a connection to the daemon 
	 *
	 * @param capabilities sum of integers from powersave_clientsocket.h
	 * @param connection the dbus connection from which the client's name is
	 *        taken
	 *
	 * @return DBUS_ERROR
	 */
	dbus_int32_t establishConnection(int capabilities, DBusConnection * connection);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /*  POWERSAVE_DBUS_H */
/** @} */// end of libpowersave_dbus
