# session-prime08.rb: Session library for the PRIME 0.8 protocol
# $Id: session-prime08.rb,v 1.2 2005/03/07 07:51:32 komatsu Exp $
#
# Copyright (C) 2005 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.

require 'prime/session'
require 'prime/prime-japanese'
require 'prime/prime-mixed'

require 'prime/engine/engine-basic'
require 'prime/engine/engine-english'
require 'prime/engine/engine-userdict2'
require 'prime/engine/engine-userdict2-en'
require 'prime/engine/engine-personaldict'
require 'prime/engine/engine-alphabet'
require 'prime/engine/engine-number'

class PrimeSessionPrime08 < PrimeSession
  include PrimeJapanese
  include PrimeMixed

  def initialize ()
    super("Prime0.8", PRIME_ENV["engines"])
    initialize_prime_japanese()
  end

  def close ()
    @engines.command(:close)
  end

  def update ()
    @engines.command(:update)
    refresh()
  end
  def refresh ()
    PrimeTypeConv::refresh()
    @engines.command(:refresh)
  end

  def get_label (pattern)
    return PrimeTypeConv::convert(pattern).join()
  end

  def preedit_convert_input (pattern)
    return PrimeTypeConv::convert(pattern).join("\t")
  end

  def set_context (context)
    @context = context
  end

  def learn_word (pron, literal, pos = "",
                  context = "", suffix = "", rest = "")
    pron    = (pron    or "").strip
    literal = (literal or "").strip
    pos     = (pos     or "").strip
    context = (context or "").strip
    suffix  = (suffix  or "").strip
    rest    = (rest    or "").strip

    @engines.command(:learn_word, pron, literal, pos, context, suffix, rest)
  end

  def lookup (string)
    if PRIME_ENV['typing_method'] == 'tcode' or 
        PRIME_ENV['typing_method'] == 'handwrite' then
      word_list = lookup_direct(string)
    else
      word_list = lookup_compact(string)
    end
    return word_list
  end
  def lookup_all (string)
    if PRIME_ENV['typing_method'] == 'tcode' or 
        PRIME_ENV['typing_method'] == 'handwrite' then
      word_list = lookup_direct_all(string)
    else
      word_list = lookup_compact_all(string)
    end
    return word_list
  end

  def lookup_hybrid (string)
    words  = search_hybrid(string, 10)
    result = PrimeWordList::merge_with_label(@context, words)
    return result
  end
  def lookup_hybrid_all (string)
    words  = search_hybrid(string)
    result = PrimeWordList::merge_with_label(@context, words)
    return result
  end

  def lookup_direct (string)
    results_expansion = lookup_expansion(string)
    results = PrimeWordList::merge_by_literal(@context, results_expansion)
    return results[0,3]
  end

  def lookup_direct_all (string)
    #     results_prefix    = lookup_prefix(string)
    results_expansion = lookup_expansion(string)
    results_mixed     = lookup_mixed(string)
    words_japanese  = search_japanese(string)
    words_overall   = search_overall(string)

    results = PrimeWordList::merge_by_literal(@context,
                                                # results_prefix[0, 3],
                                                results_expansion,
                                                results_mixed,
                                                words_japanese,
                                                words_overall)
    return PrimeWordList.new(results_expansion[0,3] | results)
  end

  def lookup_compact (string)
    words_compact = search_compact(string)
    words = PrimeWordList::merge_with_label(@context, words_compact)
    return words
  end
  def lookup_compact_all (string)
    words_compact  = search_compact(string)
    words_overall  = search_overall(string)
    words_japanese = search_japanese(string)
    results_compact    = PrimeWordList::merge_with_label(@context,
                                                         words_compact)
    results_conversion = PrimeWordList::merge_with_label(@context,
                                                         words_overall,
                                                         words_japanese)
    return PrimeWordList::concat(results_compact | results_conversion)
  end

  def lookup_prefix (string)
    words_prefix = search_prefix(string)
    return PrimeWordList::merge_with_label(@context, words_prefix)
  end

  def lookup_prefix_ex (string)
    words_prefix      = search_prefix(string)
    results_expansion = lookup_expansion(string)
    return PrimeWordList::merge_with_label(@context,
					   words_prefix,
					   results_expansion)
  end

  def lookup_exact (string)
    words_exact = search_exact(string)
    return PrimeWordList::merge_with_label(@context, words_exact)
  end

  def lookup_overall (string)
    words_overall = search_overall(string)
    return PrimeWordList::merge_with_label(@context, words_overall)
  end

  def lookup_expansion (string)
    # ͽͽ¬
    expands = PrimeTypeConv::expand(string)
    query = PrimeQuery.new(expands, nil, :literal_prefix, @context)
    results = lookup_internal(query)
    return results
  end

  def search (query)
    query.input.uniq!()
    words_list = @engines.command(:search, query)
    return PrimeWordList::merge(words_list)
  end

  def search_prefix (string)
    # ֤袪ͽ¬
    expands = PrimeTypeConv::expand(string)
    query = PrimeQuery.new(expands, nil, :prefix, @context)
    words = search(query)
    return words
  end

  def search_exact (string)
    # ֤褽ͽ¬
    converted = PrimeTypeConv::convert(string).join()
    query = PrimeQuery.new([converted, string], nil, :exact, @context)
    words = search(query)
    return words
  end
  def search_raw (string)
    ## FIXME: This method is an ad-hoc routine for search_japanese.
    ## FIXME: <komatsu@taiyaki.org> (2004-02-28)
    query = PrimeQuery.new([string], nil, :exact, @context)
    words = search(query)
    return words
  end

  def search_overall (string)
    # 1+1=2, aiueo
    query = PrimeQuery.new([string], nil, :overall)
    words_overall = search(query)
    return words_overall
  end

  def search_compact (string)
    words_prefix   = search_prefix(string)

    ## If the result of search_prefix is empty, this method stops the following
    ## search_japanese_uniclause for the quickness.
    if words_prefix.empty? then
      ## Ruby 1.6 does not keep the class PrimeWordList word[0,1] if the
      ## value of word is [], and the class of the result of word[0,1]
      ## becomes Array which is a super class of PrimewordList.
      return PrimeWordList.new()
    end

    words_japanese = search_japanese_uniclause(string)
    words_compact  = PrimeWordList::merge(words_prefix, words_japanese)[0,1]

    if words_compact.length > 0 then
      predict_with_multi_clauses!(words_compact)

      words_compact[1..-1].each {|word|
        word.score = words_compact[0].score
      }
    end
    return words_compact
  end

  def search_hybrid (string, max = nil)
    words_prefix   = search_prefix(string)
    words_japanese = search_japanese(string)
    words_overall  = search_overall(string)

    num_prefix = 3
    num_max_prefix = 10
    if (num_max_prefix - words_japanese.length) > num_prefix then
      num_prefix = (num_max_prefix - words_japanese.length)
    end
    words = PrimeWordList::merge_with_label(@context, 
                                            words_prefix[0,num_prefix],
                                            words_overall,
                                            words_japanese)
    if max then
      words = words[0,max]
    end
    predict_with_multi_clauses!(words)

    return words
  end


  private
  def lookup_internal (query)
    query.input.uniq!()
    words_list = @engines.command(:search, query)
    return PrimeWordList::merge_with_label(@context, words_list)
  end

  def predict_with_multi_clauses!(results)
    unless results.empty? then
      word1 = results.first
      context = (word1.literal + word1.conjugation)
      query = PrimeQuery.new([""], nil, :context, context)
      word2 = search(query).first
      if word2.non_nil? then
        results.push(_merge_words(word1, word2))
      end
    end
    return results
  end
end
