/*
    MakeBuilderDelegate.h

    Declaration of the MakeBuilderDelegate protocol for the ProjectManager app.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import <Foundation/NSObject.h>

@protocol MakeBuilderDelegate

/**
 * Asks the delegate to provide a list of build targets for the sending
 * MakeBuilder.
 *
 * @param sender The sending MakeBuilder.
 *
 * @return A list of target names.
 */
- (NSArray *) buildTargetsForMakeBuilder: (id) sender;

/**
 * Instructs a delegate to prepare itself for being built. The delegate
 * should regenerate it's makefiles and perform any additional tasks
 * necessary for a successful build.
 *
 * @param sender The builder which will execute the build.
 * @param target The target which will be built.
 *
 * @return YES if the builder may proceed with the build, NO if the
 *      build should be interrupted.
 */
- (BOOL) prepareForBuildByBuilder: (id) sender
                           target: (NSString *) target;

/**
 * Asks the delegate to provide arguments that should be passed to the
 * build task.
 *
 * @param sender The builder which will execute the build.
 * @param target The target which will be built.
 *
 * @return An array of string arguments which will be passed to the build
 *      task. Returning `nil' means `no extra arguments'.
 */
- (NSArray *) buildArgumentsForBuilder: (id) sender
                                target: (NSString *) target;

/**
 * Instructs a delegate to prepare itself for being cleaned. The delegate
 * should regenerate it's makefiles and perform any additional tasks
 * necessary for a successful clean.
 *
 * @param sender The builder which will execute the clean.
 * @param target The target which will be cleaned.
 *
 * @return YES if the builder may proceed with the clean, NO if the
 *      clean should be interrupted.
 */
- (BOOL) prepareForCleanByBuilder: (id) sender
                           target: (NSString *) target;

/**
 * Asks the delegate to provide arguments that should be passed to the
 * clean task. Please note that the builder <em>doesn't</em> automatically
 * append a `clean' argument when cleaning.
 *
 * @param sender The builder which will execute the clean.
 * @param target The target which will be cleaned.
 *
 * @return An array of string arguments which will be passed to the clean
 *      task. Returning `nil' means `no extra arguments'.
 */
- (NSArray *) cleanArgumentsForBuilder: (id) sender
                                target: (NSString *) target;

@end
