/*
 * pad.c - Main functions for plugin.
 * OmniJoy
 * by mike9010
 * Copyright (C) 2003 Michael Shoup
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <linux/joystick.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include "PSEmu_Plugin_Defs.h"
#include "pad.h"

// Static globals
static char    *lib_name      = "OmniJoy Beta 1"; // Library name
static int      version       = 1;
static int      revision      = MAJOR;
static int      build         = MINOR;
static long     pad_flags     = 0;
static char     pad_fd[2]     = {-1,-1}; // File descriptors for joydevs
static int      use_epsxe     = 0;
static int      use_keyboard  = 0;
static int      pad_opened[2] = {0,0};
static Display *disp          = NULL;

static unsigned short key_press   = 0;
static unsigned short key_release = 0;
static unsigned short pad_stat[2] = { 0xFFFF, 0xFFFF }; // Button status

static struct js_event js_ev;
static XEvent x_ev;

static struct cont_cfg cont[2]; // Controller configurations

// Local only prototypes
static void load_conf(); // Load Config
static int update_pad();
static void rezero_cfg();

char DECLSPEC *PSEgetLibName() // Return the library name
{
	return lib_name; // Libname
}

unsigned long DECLSPEC PSEgetLibType() // Return the library type
{
	return PSE_LT_PAD; // Standard Pad
}

unsigned long DECLSPEC PSEgetLibVersion() // Return the library version
{
	return version<<16|revision<<8|build; // plugin version
}

/* The Init and shutdown functions actually do just about nothing.
 * All the stuff needed is in PADopen() and PADclose()
 */

long DECLSPEC PADinit( long flags ) // Initialize the pad
{
	static int init = 0;
	if( !init )
	{
		printf( "OmniJoy Init: You are using v%d.%d.%d build %d\n", MAJOR, MINOR, PATCH, BUILD );
		pad_flags |= flags;
		init = 1;
	}
	return 0;
}

long DECLSPEC PADshutdown(void)
{
	return 0;
}

long DECLSPEC PADopen(void *Disp) // Open the pads
{
	char *buffer;

	static int opened = 0;

	if( !opened )
	{
		disp = (Display*)*(long*)Disp;
		load_conf(); // Load configuration

		if( use_epsxe ) pad_flags = 0x3;

		buffer = malloc( sizeof( char ) * 256 );

		if( pad_flags & 0x1 ) // If we are using controller 1, open it.
		{
			pad_fd[0] = open( cont[0].jsdev, O_RDONLY | O_NONBLOCK );
			if( pad_fd[0] < 0 )
			{
				perror( "Error OmniJoy Controller 1" );
				pad_fd[0] = -1;
				if( use_keyboard == 0 )
					return -1; // If an error occured, return such.
			}
			else
			{
				ioctl( pad_fd[0], JSIOCGNAME( 256 ), buffer );
				printf( "OmniJoy: Using Joystick %s for controller 1\n", buffer );
			}
			pad_opened[0] = 1;
		}
		if( pad_flags & 0x2 ) // If we are using controller 2, open it
		{
			pad_fd[1] = open( cont[1].jsdev, O_RDONLY | O_NONBLOCK );
			if( pad_fd[1] < 0 )
			{
				perror( "Error OmniJoy Controller 2" );
				if( use_keyboard == 0 )
				{
					printf( "OmniJoy: Disabling controller 2" );
					pad_fd[1] = -1; // If an error occured, complain and disable second pad
					pad_opened[1] = 0;
				}
			}
			if( pad_fd[1] > -1 )
			{
				ioctl( pad_fd[1], JSIOCGNAME( 256 ), buffer );
				printf( "OmniJoy: Using Joystick %s for controller 2\n", buffer );
			}
			pad_opened[1] = 1;
		}

		opened = 1;
	}

	XAutoRepeatOff( disp );
	return 0;
}

long DECLSPEC PADclose(void) // Close the pads here
{
	// If the file desc are open, close them
	if( pad_fd[0] >= 0 )
		close( pad_fd[0] );
	if( pad_fd[1] >= 0 )
		close( pad_fd[1] );

	// Reset the descriptors to show as closed (at least in OJ's terms)
	pad_fd[0] = -1;
	pad_fd[1] = -1;

	rezero_cfg(); // rezero everything

	XAutoRepeatOn( disp );

	return 0;
}

long DECLSPEC PADquery(void)
{
	return 3; // Both pads
}

long DECLSPEC PADconfigure(void)
{
	system( "cfg/cfgOmniJoy config" );
	return 0;
}


void DECLSPEC PADabout(void)
{
	system( "cfg/cfgOmniJoy" );
}

long DECLSPEC PADtest(void)
{

	/* Testing function is to be written at a later time */

	return 0;
}

long DECLSPEC PADreadPort1(PadDataS *pad) // Read the pad
{
	update_pad();

	pad->controllerType = 4;

	if( use_epsxe ) pad->buttonStatus = ( pad_stat[0] >> 8 | pad_stat[0] << 8 );
	else pad->buttonStatus   = pad_stat[0];

	return 0;
}

long DECLSPEC PADreadPort2(PadDataS *pad)
{
	update_pad();

	pad->controllerType = 4;

	if( use_epsxe ) pad->buttonStatus = ( pad_stat[1] >> 8 | pad_stat[1] << 8 );
	else pad->buttonStatus   = pad_stat[1];

	return 0;
}

long DECLSPEC PADkeypressed()
{
	if( use_keyboard == 0 ) return 0;
	update_pad();
	if( key_press != 0 ) return key_press;
	return 0;
}

/* The following two functions are more advanced, and shall be written
 * at a future time. For now, they are commented out.
 */

/*
unsigned char DECLSPEC PADstartPoll(int pad)
{
	return 0;
}

unsigned char DECLSPEC PADpoll(unsigned char value)
{
	return 0;
}

*/


static void rezero_cfg() // reset everything please!
{
	int i;
	strcpy( cont[0].jsdev, "not_configured" );
	strcpy( cont[1].jsdev, "not_configured" );
	for( i=0; i < 16; i++ )
	{
		cont[0].bits[i] = -1;
		cont[1].bits[i] = -1;
	}
	use_epsxe    = 0;
	use_keyboard = 0;
}

static void load_conf()
{
	FILE *fp;
	char buffer[256];
	rezero_cfg();
	fp = fopen( "cfg/omnijoy.cfg", "r" );
	if( fp == NULL )
		return;
	while( !feof(fp) )
	{
		int cont_no;
		if( fgets( buffer, 256, fp ) == NULL ) break;
		cont_no = atoi( buffer );
		buffer[(strlen(buffer))-1] = '\0';
		if( strncmp( &buffer[2], "DEVICE", 6 ) == 0 )
		{
			strcpy( cont[cont_no-1].jsdev, &buffer[11] );
		}

		else if( cont_no == 0 )
		{
			if( strncmp( &buffer[2], "USE_EPSXE", 9 ) == 0 )
				use_epsxe = atoi( &buffer[16] );
			else if( strncmp( &buffer[2], "USE_KEY", 7 ) == 0 )
				use_keyboard = atoi( &buffer[16] );
		}

		else if( strncmp( &buffer[2], "BTN_SEL", 7 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_SEL] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_L3", 6 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_L3] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_R3", 6 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_R3] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_START", 9 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_START] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_UP", 6 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_UP] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_RIGHT", 9 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_RIGHT] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_DOWN", 8 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_DOWN] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_LEFT", 8 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_LEFT] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_L2", 6 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_L2] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_R2", 6 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_R2] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_R1", 6 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_R1] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_L1", 6 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_L1] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_TRI", 7 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_TRI] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_CIRCLE", 10 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_CIRCLE] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_CROSS", 9 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_CROSS] = atoi( &buffer[16] );

		else if( strncmp( &buffer[2], "BTN_SQUARE", 10 ) == 0 )
			cont[cont_no-1].bits[OJ_BTN_SQUARE] = atoi( &buffer[16] );
	}
	fclose(fp);
}

static int update_pad()
{
	int i, j;


	// Read all events
	for( j=0; j < 2; j++ )
	{
		if( pad_fd[j] == -1 ) continue;
		while( read( pad_fd[j], &js_ev, sizeof( struct js_event ) ) >0 )
		{
			// Next check which type of even
			if( js_ev.type == JS_EVENT_BUTTON ) // If button...
			{
				// Check if pressed or released, and act accordingly
				if( js_ev.value == 1 )
				{
					for( i=0; i < 16; i++ )
					{
						if( cont[j].bits[i] == js_ev.number )
						{
							pad_stat[j] &= ~( 1 << i );
							break;
						}
					}
				}
				else if( js_ev.value == 0 )
				{
					for( i=0; i < 16; i++ )
					{
						if( cont[j].bits[i] == js_ev.number )
						{
							pad_stat[j] |= ( 1 << i );
							break;
						}
					}
				}
			}
			else if( js_ev.type == JS_EVENT_AXIS ) // If axis
			{
				// This is where things are a little bit more, shall we say, complicated.
				for( i=0; i < 16; i++ )
				{
					// Check to see if axis is moved and act accordingly
					if( js_ev.value > 4000 && cont[j].bits[i] > 127 &&
					    js_ev.number == (cont[j].bits[i]-128) )
					{
						pad_stat[j] &= ~( 1 << i );
						break;
					}
					else if( js_ev.value < -4000 && cont[j].bits[i] < -126 &&
					         js_ev.number == -(cont[j].bits[i]+128) )
					{
						pad_stat[j] &= ~( 1 << i );
						break;
					}
					else if( js_ev.value <= 4000 && js_ev.value >= -4000 &&
					         (js_ev.number == (cont[j].bits[i]-128) ||
						  js_ev.number == -(cont[j].bits[i]+128) ) )
					{
						pad_stat[j] |= ( 1 << i );
						break;
					}
				}
			}
		}
	}
	if( use_keyboard == 1 )
	{
		while( XPending( disp ) )
		{
			 XNextEvent( disp, &x_ev );
			 switch( x_ev.type )
			 {
			 	case FocusIn:
					XAutoRepeatOff(disp);
					break;
				case FocusOut:
					XAutoRepeatOn(disp);
					break;
				case KeyPress:
					key_press = XLookupKeysym((XKeyEvent *)&x_ev, 0);
					for( i=0; i < 16; i++ )
					{
						if( cont[0].bits[i] == key_press+256 )
						{
							pad_stat[0] &= ~( 1 << i );
							key_press = 0;
							break;
						}
						else if( cont[1].bits[i] == key_press+256 )
						{
							pad_stat[1] &= ~( 1 << i );
							key_press = 0;
							break;
						}
					}
					break;
				case KeyRelease:
					key_release = XLookupKeysym((XKeyEvent *)&x_ev, 0 );
					for( i=0; i < 16; i++ )
					{
						if( cont[0].bits[i] == key_release+256 )
						{
							pad_stat[0] |= ( 1 << i );
							key_release = 0;
							break;
						}
						else if( cont[1].bits[i] == key_release+256 )
						{
							pad_stat[1] |= ( 1 << i );
							key_release = 0;
							break;
						}
					}
					break;
			}
		}
	}
	return 0;
}
