from pydoctor import model, twisted

from nevow import rend, loaders, tags

import os, shutil, inspect, sys, urllib

try:
    from epydoc.markup import epytext
    EPYTEXT = True
except:
    print "no epytext found"
    EPYTEXT = False

def link(o):
    return urllib.quote(o.system.urlprefix+o.fullName()+'.html')

def srclink(o):
    system = o.system
    if not system.sourcebase:
        return None
    m = o
    while not isinstance(m, (model.Module, model.Package)):
        m = m.parent
        if m is None:
            return None
    sourceHref = '%s/%s'%(system.sourcebase, m.fullName().replace('.', '/'),)
    if isinstance(m, model.Module):
        sourceHref += '.py'
    if isinstance(o, model.Module):
        sourceHref += '#L1'
    elif hasattr(o, 'linenumber'):
        sourceHref += '#L'+str(o.linenumber)
    return sourceHref

def sibpath(path, sibling):
    return os.path.join(os.path.dirname(os.path.abspath(path)), sibling)

def boringDocstring(doc):
    """Generate an HTML representation of a docstring in a really boring
    way."""
    # inspect.getdoc requires an object with a __doc__ attribute, not
    # just a string :-(
    if doc is None or not doc.strip():
        return '<pre class="undocumented">Undocumented</pre>'
    def crappit(): pass
    crappit.__doc__ = doc
    return tags.pre[inspect.getdoc(crappit)]

class _EpydocLinker(object):
    def __init__(self, obj):
        self.obj = obj
    def translate_indexterm(self, something):
        # X{foobar} is meant to put foobar in an index page (like, a
        # proper end-of-the-book index). Should we support that? There
        # are like 2 uses in Twisted.
        return something.to_html(self)
    def translate_identifier_xref(self, fullID, prettyID):
        obj = self.obj.resolveDottedName(fullID, tryHarder=True)
        if obj is None:
            return '<code>%s</code>'%(prettyID,)
        else:
            if isinstance(obj, model.Function):
                linktext = link(obj.parent) + '#' + urllib.quote(obj.name)
            else:
                linktext = link(obj)
            return '<a href="%s"><code>%s</code></a>'%(linktext, prettyID)

class FieldDesc(object):
    def __init__(self):
        self.kind = None
        self.name = None
        self.type = None
        self.body = None
    def format(self):
        if self.body is None:
            body = ''
        else:
            body = self.body
        if self.type is not None:
            body = body, '(type: ', self.type, ')'
        return body

def format_desc_list(singular, descs, plural=None):
    if plural is None:
        plural = singular + 's'
    if not descs:
        return ''
    if len(descs) > 1:
        label = plural
    else:
        label = singular
    r = []
    first = True
    for d in descs:
        if first:
            row = tags.tr(class_="fieldStart")
            row[tags.td(class_="fieldName")[label]]
            first = False
        else:
            row = tags.tr()
            row[tags.td()]
        if d.name is None:
            row[tags.td(colspan=2)[d.format()]]
        else:
            row[tags.td(class_="fieldArg")[d.name], tags.td[d.format()]]
        r.append(row)
    return r

def format_field_list(obj, singular, fields, plural=None):
    if plural is None:
        plural = singular + 's'
    if not fields:
        return ''
    if len(fields) > 1:
        label = plural
    else:
        label = singular
    rows = []
    first = True
    for field in fields:
        if first:
            row = tags.tr(class_="fieldStart")
            row[tags.td(class_="fieldName")[label]]
            first=False
        else:
            row = tags.tr()
            row[tags.td()]
        row[tags.td(colspan=2)[field.body]]
        rows.append(row)
    return rows

class Field(object):
    """Like epydoc.markup.Field, but without the gross accessor
    methods and with a formatted body."""
    def __init__(self, field, obj):
        self.tag = field.tag()
        self.arg = field.arg()
        self.body = tags.raw(field.body().to_html(_EpydocLinker(obj)))

class FieldHandler(object):
    def __init__(self, obj):
        self.obj = obj

        self.parameter_descs = []
        self.ivar_descs = []
        self.cvar_descs = []
        self.var_descs = []
        self.return_desc = None
        self.raise_descs = []
        self.seealsos = []
        self.notes = []
        self.authors = []
        self.unknowns = []
        self.unattached_types = {}

    def handle_return(self, field):
        if not self.return_desc:
            self.return_desc = FieldDesc()
        if self.return_desc.body:
            print 'XXX'
        self.return_desc.body = field.body
    handle_returns = handle_return

    def handle_returntype(self, field):
        if not self.return_desc:
            self.return_desc = FieldDesc()
        if self.return_desc.type:
            print 'XXX'
        self.return_desc.type = field.body
    handle_rtype = handle_returntype

    def add_type_info(self, desc_list, field):
        if desc_list and desc_list[-1].name == field.arg:
            assert desc_list[-1].type is None
            desc_list[-1].type = field.body
        else:
            d = FieldDesc()
            d.kind = field.tag
            d.name = field.arg
            d.type = field.body
            desc_list.append(d)

    def add_info(self, desc_list, field):
        if desc_list and desc_list[-1].name == field.arg and desc_list[-1].body is None:
            desc_list[-1].body = field.body
        else:
            d = FieldDesc()
            d.kind = field.tag
            d.name = field.arg
            d.body = field.body
            desc_list.append(d)

    def handle_type(self, field):
        obj = self.obj
        if isinstance(obj, model.Function):
            self.add_type_info(self.parameter_descs, field)
        elif isinstance(obj, model.Class):
            ivars = self.ivar_descs
            cvars = self.cvar_descs
            if ivars and ivars[-1].name == field.arg:
                assert ivars[-1].type is None
                ivars[-1].type = field.body
            elif cvars and cvars[-1].name == field.arg:
                assert cvars[-1].type is None
                cvars[-1].type = field.body
            else:
                self.unattached_types[field.arg] = field.body
        else:
            self.add_type_info(self.var_descs, field)

    def handle_param(self, field):
        self.add_info(self.parameter_descs, field)
    handle_arg = handle_param

    def handle_ivar(self, field):
        self.add_info(self.ivar_descs, field)
        if field.arg in self.unattached_types:
            self.ivar_descs[-1].type = self.unattached_types[field.arg]
            del self.unattached_types[field.arg]

    def handle_cvar(self, field):
        self.add_info(self.cvar_descs, field)
        if field.arg in self.unattached_types:
            self.cvar_descs[-1].type = self.unattached_types[field.arg]
            del self.unattached_types[field.arg]

    def handle_var(self, field):
        self.add_info(self.var_descs, field)

    def handle_raises(self, field):
        self.add_info(self.raise_descs, field)
    handle_raise = handle_raises

    def handle_seealso(self, field):
        self.seealsos.append(field)
    handle_see = handle_seealso

    def handle_note(self, field):
        self.notes.append(field)

    def handle_author(self, field):
        self.authors.append(field)

    def handleUnknownField(self, field):
        print 'XXX', 'unknown field', field
        self.add_info(self.unknowns, field)

    def handle(self, field):
        m = getattr(self, 'handle_' + field.tag, self.handleUnknownField)
        m(field)

    def format(self):
        r = []
        for d, l in (('Parameters', self.parameter_descs),
                     ('Instance Variables', self.ivar_descs),
                     ('Class Variables', self.cvar_descs),
                     ('Variables', self.var_descs)):
            r.append(format_desc_list(d, l, d))
        if self.return_desc:
            r.append(tags.tr(class_="fieldStart")[tags.td(class_="fieldName")['Returns'],
                               tags.td(colspan="2")[self.return_desc.format()]])
        r.append(format_desc_list("Raises", self.raise_descs, "Raises"))
        for s, p, l in (('Author', 'Authors', self.authors),
                        ('See Also', 'See Also', self.seealsos),
                        ('Note', 'Notes', self.notes)):
            r.append(format_field_list(self.obj, s, l, p))
        unknowns = {}
        unknownsinorder = []
        for fieldinfo in self.unknowns:
            tag = fieldinfo.kind
            if tag in unknowns:
                unknowns[tag].append(fieldinfo)
            else:
                unknowns[tag] = [fieldinfo]
                unknownsinorder.append(unknowns[tag])
        for fieldlist in unknownsinorder:
            label = "Unknown Field: " + fieldlist[0].kind
            r.append(format_desc_list(label, fieldlist, label))

        return tags.table(class_='fieldTable')[r]

errcount = 0

def doc2html(obj, summary=False):
    """Generate an HTML representation of a docstring"""
    origobj = obj
    if isinstance(obj, model.Package):
        obj = obj.contents['__init__']
    doc = obj.docstring
    if doc is None or not doc.strip():
        text = "Undocumented"
        subdocstrings = {}
        subcounts = {}
        for subob in origobj.contents.itervalues():
            k = subob.kind.lower()
            if isinstance(subob, model.Function) and isinstance(origobj, model.Class):
                k = "method"
            subcounts[k] = subcounts.get(k, 0) + 1
            if subob.docstring is not None:
                subdocstrings[k] = subdocstrings.get(k, 0) + 1
        if subdocstrings:
            plurals = {'class':'classes'}
            text = "No %s docstring"%origobj.kind.lower()
            if summary:
                u = []
                for k in sorted(subcounts):
                    u.append("%s/%s %s"%(subdocstrings.get(k, 0), subcounts[k],
                                         plurals.get(k, k+'s')))
                text += '; ' + ', '.join(u) + " documented"
        if summary:
            return tags.span(class_="undocumented")[text]
        else:
            return tags.div(class_="undocumented")[text]
    if summary:
        for line in doc.split('\n'):
            if line.strip():
                doc = line
                break
    if not EPYTEXT:
        return boringDocstring(doc)
    errs = []
    pdoc = epytext.parse_docstring(doc, errs)
    if errs:
        errs = []
        def crappit(): pass
        crappit.__doc__ = doc
        doc = inspect.getdoc(crappit)
        pdoc = epytext.parse_docstring(doc, errs)
        if errs:
            if obj.system.options.verbosity > 0:
                print 'epytext error in', obj
            if obj.system.options.verbosity > 1:
                for i, l in enumerate(doc.splitlines()):
                    print "%4s"%(i+1), l
                for err in errs:
                    print err
            global errcount
            errcount += len(errs)
            return boringDocstring(doc)
    pdoc, fields = pdoc.split_fields()
    crap = pdoc.to_html(_EpydocLinker(getattr(obj, 'docsource', obj)))
    if not crap:
        return ()
    if summary:
        s = tags.span()[tags.raw(crap)]
    else:
        s = tags.div()[tags.raw(crap)]
        fh = FieldHandler(obj)
        for field in fields:
            fh.handle(Field(field, obj))
        s[fh.format()]
    return s

def getBetterThanArgspec(argspec):
    """Ok, maybe argspec's format isn't the best after all: This takes an
    argspec and returns (regularArguments, [(kwarg, kwval), (kwarg, kwval)])."""
    args = argspec[0]
    defaults = argspec[-1]
    if not defaults:
        return (args, [])
    backargs = args[:]
    backargs.reverse()
    defaults = list(defaults)
    defaults.reverse()
    kws = zip(backargs, defaults)
    kws.reverse()
    allargs = args[:-len(kws)] + kws
    return (args[:-len(kws)], kws)

def _strtup(tup):
    # Ugh
    if not isinstance(tup, (tuple, list)):
        return str(tup)
    return '(' + ', '.join(map(_strtup, tup)) + ')'

def signature(argspec):
    """Return a nicely-formatted source-like signature, formatted from an
    argspec.
    """
    regargs, kwargs = getBetterThanArgspec(argspec)
    varargname, varkwname = argspec[1:3]
    things = []
    for regarg in regargs:
        if isinstance(regarg, list):
            things.append(_strtup(regarg))
        else:
            things.append(regarg)
    if varargname:
        things.append('*%s' % varargname)
    things += ['%s=%s' % (t[0], t[1]) for t in kwargs]
    if varkwname:
        things.append('**%s' % varkwname)
    return ', '.join(things)

class NevowWriter:
    def __init__(self, filebase):
        self.base = filebase
        self.written_pages = 0

    def prepOutputDirectory(self):
        if not os.path.exists(self.base):
            os.mkdir(self.base)
        shutil.copyfile(sibpath(__file__, 'templates/apidocs.css'),
                        os.path.join(self.base, 'apidocs.css'))
        if self.system.options.htmlusesorttable:
            shutil.copyfile(sibpath(__file__, 'templates/sorttable.js'),
                            os.path.join(self.base, 'sorttable.js'))

    def writeIndividualFiles(self, obs, functionpages=False):
        for ob in obs:
            self.writeDocsFor(ob, functionpages=functionpages)

    def writeModuleIndex(self, system):
        for pclass in summarypages:
            page = pclass(self, system)
            f = open(os.path.join(self.base, pclass.filename), 'w')
            f.write(page.renderSynchronously())
            f.close()

    def writeDocsFor(self, ob, functionpages):
        isfunc = isinstance(ob, (model.Function, twisted.Attribute)) # cough
        if (isfunc and functionpages) or not isfunc:
            f = open(os.path.join(self.base, link(ob)), 'w')
            self.writeDocsForOne(ob, f)
            f.close()
        for o in ob.orderedcontents:
            self.writeDocsFor(o, functionpages)

    def writeDocsForOne(self, ob, fobj):
        # brrrrrrrr!
        for c in ob.__class__.__mro__:
            n = c.__name__ + 'Page'
            if n in globals():
                pclass = globals()[n]
                break
        else:
            pclass = CommonPage
        page = pclass(self, ob)
        self.written_pages += 1
        print '\rwritten', self.written_pages, 'pages',
        sys.stdout.flush()
        fobj.write(page.renderSynchronously())

def mediumName(obj):
    fn = obj.fullName()
    if '.' not in fn:
        return fn
    path, name = fn.rsplit('.', 1)
    def process(part):
        return obj.system.abbrevmapping.get(part, part[0])
    return '.'.join([process(p) for p in path.split('.')]) + '.' + name

def moduleSummary(modorpack):
    r = tags.li[taglink(modorpack), ' - ', doc2html(modorpack, summary=True)]
    if isinstance(modorpack, model.Package) and len(modorpack.orderedcontents) > 1:
        ul = tags.ul()
        for m in sorted(modorpack.orderedcontents,
                        key=lambda m:m.fullName()):
            if m.name != '__init__':
                ul[moduleSummary(m)]
        r[ul]
    return r

class ModuleIndexPage(rend.Page):
    filename = 'moduleIndex.html'
    docFactory = loaders.xmlfile(sibpath(__file__, 'templates/summary.html'))
    def __init__(self, writer, system):
        self.writer = writer
        self.system = system
    def render_title(self, context, data):
        return context.tag.clear()["Module Index"]
    def render_stuff(self, context, data):
        r = []
        for o in self.system.rootobjects:
            r.append(moduleSummary(o))
        return context.tag.clear()[r]
    def render_heading(self, context, data):
        return context.tag().clear()["Module Index"]

def findRootClasses(system):
    roots = {}
    for cls in system.objectsOfType(model.Class):
        if ' ' in cls.name:
            continue
        if cls.bases:
            for n, b in zip(cls.bases, cls.baseobjects):
                if b is None:
                    roots.setdefault(n, []).append(cls)
                elif b.system is not system:
                    roots[b.fullName()] = b
        else:
            roots[cls.fullName()] = cls
    return sorted(roots.items())

def subclassesFrom(hostsystem, cls, anchors):
    r = tags.li()
    name = cls.fullName()
    if name not in anchors:
        r[tags.a(name=name)]
        anchors.add(name)
    r[taglink(cls), ' - ', doc2html(cls, summary=True)]
    scs = [sc for sc in cls.subclasses if sc.system is hostsystem and ' ' not in sc.fullName()]
    if len(scs) > 0:
        ul = tags.ul()
        for sc in sorted(scs, key=lambda sc2:sc2.fullName()):
            ul[subclassesFrom(hostsystem, sc, anchors)]
        r[ul]
    return r

class ClassIndexPage(rend.Page):
    filename = 'classIndex.html'
    docFactory = loaders.xmlfile(sibpath(__file__, 'templates/summary.html'))
    def __init__(self, writer, system):
        self.writer = writer
        self.system = system
    def render_title(self, context, data):
        return context.tag.clear()["Class Hierarchy"]
    def render_stuff(self, context, data):
        t = context.tag
        anchors = set()
        for b, o in findRootClasses(self.system):
            if isinstance(o, model.Class):
                t[subclassesFrom(self.system, o, anchors)]
            else:
                item = tags.li[b]
                if o:
                    ul = tags.ul()
                    for sc in sorted(o, key=lambda sc2:sc2.fullName()):
                        ul[subclassesFrom(self.system, sc, anchors)]
                    item[ul]
                t[item]
        return t
    def render_heading(self, context, data):
        return context.tag.clear()["Class Hierarchy"]


class NameIndexPage(rend.Page):
    filename = 'nameIndex.html'
    docFactory = loaders.xmlfile(sibpath(__file__, 'templates/nameIndex.html'))
    def __init__(self, writer, system):
        self.writer = writer
        self.system = system
        used_initials = {}
        for ob in self.system.orderedallobjects:
            used_initials[ob.name[0].upper()] = True
        self.used_initials = sorted(used_initials)

    def render_title(self, context, data):
        return context.tag.clear()["Index Of Names"]
    def render_heading(self, context, data):
        return context.tag.clear()["Index Of Names"]
    def data_letters(self, context, data):
        # this is confusing
        # we return a list of lists of lists
        # if r is the return value, r[i][j] is a list of things with the same .name
        # all the things in the lists contained in r[i] share the same initial.
        initials = {}
        for ob in self.system.orderedallobjects:
            initials.setdefault(ob.name[0].upper(), {}).setdefault(ob.name, []).append(ob)
        r = []
        for k in sorted(initials):
            r.append(sorted(
                sorted(initials[k].values(), key=lambda v:v[0].name),
                key=lambda v:v[0].name.upper()))
        return r
    def render_letter(self, context, data):
        return context.tag.clear()[data[0][0].name[0].upper()]
    def render_letters(self, context, data):
        cur = data[0][0].name[0].upper()
        r = []
        for i in self.used_initials:
            if i != cur:
                r.append(tags.a(href='#' + i)[i])
            else:
                r.append(i)
            r.append(' - ')
        if r:
            del r[-1]
        return context.tag.clear()[r]
    def render_linkToThing(self, context, data):
        tag = context.tag.clear()
        if len(data) == 1:
            ob, = data
            return tag[ob.name, ' - ', taglink(ob)]
        else:
            ul = tags.ul()
            for d in sorted(data, key=lambda o:o.fullName()):
                ul[tags.li[taglink(d)]]
            return tag[data[0].name, ul]

class IndexPage(rend.Page):
    filename = 'index.html'
    docFactory = loaders.xmlfile(sibpath(__file__, 'templates/index.html'))
    def __init__(self, writer, system):
        self.system = system
    def render_project_link(self, context, data):
        if self.system.options.projecturl:
            return tags.a(href=self.system.options.projecturl)[self.system.options.projectname]
        else:
            return self.system.options.projectname
    def render_project(self, context, data):
        return self.system.options.projectname
    def render_onlyIfOneRoot(self, context, data):
        if len(self.system.rootobjects) != 1:
            return []
        else:
            root, = self.system.rootobjects
            return context.tag.clear()[
                "Start at ", taglink(root),
                ", the root ", root.kind.lower(), "."]
    def render_onlyIfMultipleRoots(self, context, data):
        if len(self.system.rootobjects) == 1:
            return []
        else:
            return context.tag.clear()

summarypages = [ModuleIndexPage, ClassIndexPage, IndexPage, NameIndexPage]


class CommonPage(rend.Page):
    docFactory = loaders.xmlfile(sibpath(__file__, 'templates/common.html'))
    def __init__(self, writer, ob):
        self.writer = writer
        self.ob = ob
    def render_title(self, context, data):
        return self.ob.fullName()
    def render_heading(self, context, data):
        tag = context.tag()
        tag.clear()
        kind = self.ob.kind
        return tag(class_=kind.lower())[kind + " " + mediumName(self.ob)]
    def render_part(self, context, data):
        tag = context.tag()
        tag.clear()
        if self.ob.parent:
            parent = self.ob.parent
            parts = []
            while parent.parent:
                parts.append(tags.a(href=link(parent))[parent.name])
                parts.append('.')
                parent = parent.parent
            parts.append(tags.a(href=link(parent))[parent.name])
            parts.reverse()
            return tag['Part of ', parts]
        else:
            return tag

    def render_project(self, context, data):
        if self.ob.system.options.projecturl:
            return tags.a(href=self.ob.system.options.projecturl)[self.ob.system.options.projectname]
        else:
            return self.ob.system.options.projectname

    def render_source(self, context, data):
        sourceHref = srclink(self.ob)
        if not sourceHref:
            return ()
        return context.tag(href=sourceHref)

    def render_inhierarchy(self, context, data):
        return ()

    def render_extras(self, context, data):
        return ()

    def render_docstring(self, context, data):
        return doc2html(self.ob)

    def render_maybechildren(self, context, data):
        tag = context.tag()
        if not self.ob.orderedcontents:
            tag.clear()
        return tag

    def has_lineno_col(self):
        if not self.writer.system.options.htmlusesorttable:
            return False
        return isinstance(self.ob, (model.Class, model.Module))

    def render_maybeheadings(self, context, data):
        if not self.writer.system.options.htmlusesorttable:
            return ()
        return context.tag()

    def render_maybelinenohead(self, context, data):
        if self.has_lineno_col():
            return context.tag()
        else:
            return ()

    def render_sequence3(self, context, data):
        return context.tag().clear()

    def data_bases(self, context, data):
        return []

    def data_children(self, context, data):
        return self.ob.orderedcontents

    def render_childlist(self, context, data):
        return ()

    def render_childclass(self, context, data):
        return data.kind.lower()

    def render_childline(self, context, data):
        tag = context.tag()
        tag.clear()
        if not self.has_lineno_col():
            return ()
        if hasattr(data, 'linenumber'):
            sourceHref = srclink(data)
            if not sourceHref:
                return tag[data.linenumber]
            return tag[tags.a(href=sourceHref)[data.linenumber]]
        else:
            return tag

    def render_childkind(self, context, data):
        tag = context.tag()
        tag.clear()
        return tag[data.kind]

    def render_childname(self, context, data):
        tag = context.tag()
        tag.clear()
        return tag[tags.a(href=link(data))[data.name]]

    def render_childsummaryDoc(self, context, data):
        tag = context.tag()
        tag.clear()
        return tag[doc2html(data, summary=True)]

    def data_methods(self, context, data):
        return []


class PackagePage(CommonPage):
    def data_children(self, context, data):
        return sorted([o for o in self.ob.orderedcontents
                       if o.name != '__init__'],
                      key=lambda o2:o2.fullName())

class FunctionParentMixin(object):
    def render_childlist(self, context, data):
        function = context.tag.patternGenerator('function')
        attribute = context.tag.patternGenerator('attribute')
        tag = context.tag().clear()
        for d in data:
            if isinstance(d, model.Function):
                tag[function(data=d)]
            else:
                tag[attribute(data=d)]
        return tag
        
    def render_functionName(self, context, data):
        tag = context.tag()
        tag.clear()
        return tag[data.name, '(', signature(data.argspec), '):']

    def render_attribute(self, context, data):
        tag = context.tag()
        tag.clear()
        return tag[data.name]

    def render_childname(self, context, data):
        if not isinstance(data, model.Function):
            sup = super(FunctionParentMixin, self)
            return sup.render_childname(context, data)
        tag = context.tag()
        tag.clear()
        return tag[tags.a(href='#' + urllib.quote(data.name))[data.name]]

    def render_functionAnchor(self, context, data):
        return data.fullName()

    def render_shortFunctionAnchor(self, context, data):
        return data.name

    def render_functionBody(self, context, data):
        tag = context.tag()
        tag.clear()
        return tag[doc2html(data)]

    def render_functionSourceLink(self, context, data):
        sourceHref = srclink(data)
        if not sourceHref:
            return ()
        return context.tag(href=sourceHref)

class ModulePage(FunctionParentMixin, CommonPage):
    def data_methods(self, context, data):
        return [o for o in self.ob.orderedcontents
                if isinstance(o, model.Function)]


def taglink(o, label=None):
    if label is None:
        label = o.fullName()
    if isinstance(o, model.Function):
        linktext = link(o.parent) + '#' + urllib.quote(o.name)
    else:
        linktext = link(o)
    return tags.a(href=linktext)[label]

class ClassPage(FunctionParentMixin, CommonPage):
    def render_extras(self, context, data):
        r = super(ClassPage, self).render_extras(context, data)
        if self.ob.subclasses:
            if r == ():
                r = context.tag.clear()
            sc = self.ob.subclasses[0]
            p = tags.p()
            p["Known subclasses: ", taglink(sc)]
            for sc in self.ob.subclasses[1:]:
                p[', ', taglink(sc)]
            r[p]
        return r

    def render_childkind(self, context, data):
        tag = context.tag()
        tag.clear()
        if isinstance(data, model.Function):
            kind = "Method"
        else:
            kind = data.kind
        return tag[kind]

    def render_heading(self, context, data):
        tag = super(ClassPage, self).render_heading(context, data)
        zipped = zip(self.ob.rawbases, self.ob.baseobjects)
        if zipped:
            tag['(']
            for i, (n, o) in enumerate(zipped):
                if o is None:
                    tag[n]
                else:
                    tag[tags.a(href=link(o))[n]]
                if i != len(zipped)-1:
                    tag[', ']
            tag[')']
        tag[':']
        return tag

    def render_inhierarchy(self, context, data):
        return context.tag(href="classIndex.html#"+self.ob.fullName())

    def data_methods(self, context, data):
        return [o for o in self.ob.orderedcontents
                if isinstance(o, model.Function)]

    def nested_bases(self, b):
        r = [(b,)]
        for b2 in b.baseobjects:
            if b2 is None:
                continue
            for n in self.nested_bases(b2):
                r.append(n + (b,))
        return r

    def data_bases(self, context, data):
        r = []
        for b in self.ob.baseobjects:
            if b is None:
                continue
            r.extend(self.nested_bases(b))
        return r

    def render_base_sequence(self, context, data):
        header = context.tag.onePattern('header2')
        p = context.tag.patternGenerator('item2')
        tag = context.tag.clear()
        tag[header]
        for d in data[0].orderedcontents:
            for b in (self.ob,) + data[1:]:
                if d.name in b.contents:
                    break
            else:
                tag[p(data=d)]
        return tag

    def render_base_name(self, context, data):
        tag = context.tag.clear()
        tag[tags.a(href=link(data[0]))[data[0].name]]
        if data[1:]:
            tail = []
            for b in data[1:][::-1]:
                tail.append(tags.a(href=link(b))[b.name])
                tail.append(', ')
            del tail[-1]
            tag[' (via ', tail, ')']
        return tag

    def render_maybe_base(self, context, data):
        for d in data[0].orderedcontents:
            for b in (self.ob,) + data[1:]:
                if d.name in b.contents:
                    break
            else:
                return context.tag()
        else:
            return context.tag.clear()

    def render_basechildclass(self, context, data):
        return 'base' + data.kind.lower()

    baseindex = 0
    def render_basechildtableid(self, context, data):
        self.baseindex += 1
        return 'basetable' + str(self.baseindex)

    def render_basechildname(self, context, data):
        tag = context.tag()
        tag.clear()
        return tag[tags.a(href=link(data.parent)+'#'+data.name)[data.name]]

class TwistedClassPage(ClassPage):
    def data_methods(self, context, data):
        return [o for o in self.ob.orderedcontents
                if isinstance(o, model.Function) or isinstance(o, twisted.Attribute)]

    def render_extras(self, context, data):
        r = super(TwistedClassPage, self).render_extras(context, data)
        system = self.ob.system
        def tl(s):
            if s in system.allobjects:
                return taglink(system.allobjects[s])
            else:
                return s
        if self.ob.isinterface:
            namelist = self.ob.implementedby_directly
            label = 'Known implementations: '
        else:
            namelist = self.ob.implements_directly
            label = 'Implements interfaces: '
        if namelist:
            if r == ():
                r = context.tag.clear()
            tag = tags.p()[label, tl(namelist[0])]
            for impl in namelist[1:]:
                tag[', ', tl(impl)]
            r[tag]
        return r

    def interfaceMeth(self, methname):
        system = self.ob.system
        for interface in self.ob.implements_directly + self.ob.implements_indirectly:
            if interface in system.allobjects:
                io = system.allobjects[interface]
                if methname in io.contents:
                    return io.contents[methname]
        return None

    def render_functionBody(self, context, data):
        imeth = self.interfaceMeth(data.name)
        tag = context.tag()
        tag.clear()
        if imeth:
            tag[tags.div(class_="interfaceinfo")
                ['from ', tags.a(href=link(imeth.parent) + '#' + imeth.name)
                 [imeth.parent.fullName()]]]
        for b in self.ob.allbases():
            if data.name not in b.contents:
                continue
            overridden = b.contents[data.name]
            tag[tags.div(class_="interfaceinfo")
                ['overrides ',
                 tags.a(href=link(overridden.parent) + '#' + overridden.name)
                 [overridden.fullName()]]]
            break
        ocs = list(overriding_subclasses(self.ob, data.name))
        if ocs:
            def one(sc):
                return tags.a(
                    href=link(sc) + '#' + sc.contents[data.name].name
                    )[sc.fullName()]
            t = tags.div(class_="interfaceinfo")['overridden in ']
            t[one(ocs[0])]
            for sc in ocs[1:]:
                t[', ', one(sc)]
            tag[t]
        tag[doc2html(data)]
        return tag

def overriding_subclasses(c, name, firstcall=True):
    if not firstcall and name in c.contents:
        yield c
    else:
        for sc in c.subclasses:
            for sc2 in overriding_subclasses(sc, name, False):
                yield sc2

class FunctionPage(CommonPage):
    def render_heading(self, context, data):
        tag = super(FunctionPage, self).render_heading(context, data)
        return tag['(', signature(self.ob.argspec), '):']
