__license__ = """
Pygmy, a GTK+ client for the Music Player Daemon
Copyright 2005 Andrew Conkling <andrewski@fr.st>

This file is part of Pygmy.

Pygmy is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

Pygmy is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pygmy; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
"""

import sys
import os
import gobject
import ConfigParser

try:
	import cPickle as pickle
except ImportError:
	import pickle

try:
	import gtk
	import gtk.glade
	import pango
	import mpdclient2
	import em
except ImportError, (strerror):
	print >>sys.stderr, "%s.  Please make sure you have this library installed into a directory in Python's path or in the same directory as Pygmy.\n" % strerror
	sys.exit(1)

try:
	# We'll make the system tray functionality optional...
	import egg.trayicon
except ImportError:
	# so we'll pass on any errors in loading it
	pass

# Test pygtk version
if gtk.pygtk_version < (2, 4, 0):
	sys.stderr.write("Pygmy requires PyGTK 2.4.0 or newer.\n")
	sys.exit(1)


class WindowProps:
	def __init__(self, x, y, w, h, we=0, he=0, expanded=False, visible=True, withdrawn=False, screen=0):
		self.x = x
		self.y = y
		self.w = w
		self.h = h
		self.we = we
		self.he = he
		self.expanded = expanded
		self.visible = visible
		self.withdrawn = withdrawn
		self.screen = screen
	def get_size(self):
		if self.expanded: return self.we, self.he
		else: return self.w, self.h
	def set_size(self, size):
		if self.expanded: self.we, self.he = size
		else: self.w, self.h = size


class GladeWidgetTree:
	def __init__(self, tree):
		if tree:
			self.tree = tree
		else:
			if os.path.exists(os.path.join(os.path.split(__file__)[0], 'pygmy.glade')):
				self.tree = gtk.glade.XML(os.path.join(os.path.split(__file__)[0], 'pygmy.glade'))
			elif os.path.exists(os.path.join(sys.prefix, 'share', 'pygmy', 'pygmy.glade')):
				self.tree = gtk.glade.XML(os.path.join(sys.prefix, 'share', 'pygmy', 'pygmy.glade'))
			else:
				print "Could not load Glade UI."
				sys.exit(1)
	def __getattr__(self, widget):
		return self.tree.get_widget(widget)


class Config:
	"""Our persistent configuration values. Only one of these should be opened per proccess; different Clients may share them."""
	def __init__(self):
		# defaults
		self.host = 'localhost'
		self.port = 6600
		self.password = ''
		
		self.flat_buttons = False
		
		self.player_winprops = WindowProps(0, 0, 250, 72, 250, 200, expanded=True, withdrawn=False, screen=0)
		self.browser_winprops = WindowProps(0, 0, 375, 250, visible=False)
		
		self.cursong_fmt = '@[if playing]<big>@[if artist]@artist:@[end if]</big>\n<small>@[if title]@title@[else]@file@[end if]</small>@[else]Playlist@[end if]'
		self.wintitle_fmt = '@[if playing]@[if artist and title]@artist: @title@[else]@file@[end if] - @[end if]Pygmy'
		self.playlist_fmt = '@[if artist and title]@artist: @title@[else]@file@[end if]'
		self.progress_fmt = '@[if connected]@[if playing]@at@[if time] / @time@[end if]@[else]Stopped@[end if]@[else]Not Connected@[end if]'

		self.show_ppbutton = True
		self.show_stopbutton = True
		self.show_prevbutton = True
		self.show_nextbutton = True
		self.show_volumebutton = True
		self.show_progressbar = True
		self.show_toolbar = True
		self.show_systrayicon = True
		
		# now load from disk
		conf = ConfigParser.ConfigParser()
		conf.read(os.path.expanduser("~/.pygmyrc"))

		try:
			self.host = conf.get('connection', 'host')
			self.port = int(conf.get('connection', 'port'))
			self.password = conf.get('connection', 'password')

			self.player_winprops.x = conf.getint('player', 'x')
			self.player_winprops.y = conf.getint('player', 'y')
			self.player_winprops.w = conf.getint('player', 'w')
			self.player_winprops.h = conf.getint('player', 'h')
			self.player_winprops.we = conf.getint('player', 'we')
			self.player_winprops.he = conf.getint('player', 'he')
			self.player_winprops.expanded = conf.getboolean('player', 'expanded')
			self.player_winprops.withdrawn = conf.getboolean('player', 'withdrawn')
			self.player_winprops.screen = conf.getint('player', 'screen')
			
			self.browser_winprops.x = conf.getint('browser', 'x')
			self.browser_winprops.y = conf.getint('browser', 'y')
			self.browser_winprops.w = conf.getint('browser', 'w')
			self.browser_winprops.h = conf.getint('browser', 'h')
			self.browser_winprops.visible = conf.getboolean('browser', 'visible')
			
			self.flat_buttons = conf.getboolean('ui', 'flat_buttons')
			self.show_ppbutton = conf.getboolean('ui', 'ppbutton')
			self.show_stopbutton = conf.getboolean('ui', 'stopbutton')
			self.show_prevbutton = conf.getboolean('ui', 'prevbutton')
			self.show_nextbutton = conf.getboolean('ui', 'nextbutton')
			self.show_volumebutton = conf.getboolean('ui', 'volumebutton')
			self.show_progressbar = conf.getboolean('ui', 'progressbar')
			self.show_toolbar = conf.getboolean('ui', 'toolbar')
			self.show_systrayicon = conf.getboolean('ui', 'systrayicon')

			self.cursong_fmt = conf.get('formats', 'cursong')
			self.wintitle_fmt = conf.get('formats', 'wintitle')
			self.playlist_fmt = conf.get('formats', 'playlist')
			self.progress_fmt = conf.get('formats', 'progress')

		except (ConfigParser.NoOptionError, ConfigParser.NoSectionError):
			pass
			
	def save(self):
		conf = ConfigParser.ConfigParser()

		conf.add_section('connection')
		conf.set('connection', 'host', self.host)
		conf.set('connection', 'port', self.port)
		conf.set('connection', 'password', self.password)

		conf.add_section('player')
		conf.set('player', 'w', self.player_winprops.w)
		conf.set('player', 'h', self.player_winprops.h)
		conf.set('player', 'x', self.player_winprops.x)
		conf.set('player', 'y', self.player_winprops.y)
		conf.set('player', 'we', self.player_winprops.we)
		conf.set('player', 'he', self.player_winprops.he)
		conf.set('player', 'expanded', self.player_winprops.expanded)
		conf.set('player', 'withdrawn', self.player_winprops.withdrawn)
		conf.set('player', 'screen', self.player_winprops.screen)

		conf.add_section('browser')
		conf.set('browser', 'w', self.browser_winprops.w)
		conf.set('browser', 'h', self.browser_winprops.h)
		conf.set('browser', 'x', self.browser_winprops.x)
		conf.set('browser', 'y', self.browser_winprops.y)
		conf.set('browser', 'visible', self.browser_winprops.visible)
		
		conf.add_section('ui')
		conf.set('ui', 'flat_buttons', self.flat_buttons)
		conf.set('ui', 'ppbutton', self.show_ppbutton)
		conf.set('ui', 'stopbutton', self.show_stopbutton)
		conf.set('ui', 'prevbutton', self.show_prevbutton)
		conf.set('ui', 'nextbutton', self.show_nextbutton)
		conf.set('ui', 'volumebutton', self.show_volumebutton)
		conf.set('ui', 'progressbar', self.show_progressbar)
		conf.set('ui', 'toolbar', self.show_toolbar)
		conf.set('ui', 'systrayicon', self.show_systrayicon)
		
		conf.add_section('formats')
		conf.set('formats', 'cursong', self.cursong_fmt)
		conf.set('formats', 'wintitle', self.wintitle_fmt)
		conf.set('formats', 'playlist', self.playlist_fmt)
		conf.set('formats', 'progress', self.progress_fmt)

		conf.write(file(os.path.expanduser('~/.pygmyrc'), 'w'))


class Connection(mpdclient2.mpd_connection):
	"""A connection to the daemon. Will use MPD_HOST/MPD_PORT in preference to the supplied config if available."""

	def __init__(self, conf):
		"""Open a connection using the host/port values from the provided config. If conf is None, an empty object will be returned, suitable for comparing != to any other connection."""
		host = conf.host
		port = conf.port
		password = conf.password

		if os.environ.has_key('MPD_HOST'):
			if '@' in os.environ['MPD_HOST']:
				password, host = os.environ['MPD_HOST'].split('@')
			else:
				host = os.environ['MPD_HOST']
		if os.environ.has_key('MPD_PORT'):
			port = int(os.environ['MPD_PORT'])

		mpdclient2.mpd_connection.__init__(self, host, port)
		if password: self.password(password)
		
	def __repr__(self, host, port, password):
		if password:
			return "<Connection to %s:%s, using password>" % (host, port)
		else:
			return "<Connection to %s:%s>" % (host, port)


class Client:
	"""Holds open a Connection. This is the base for all user-facing classes."""
	def __init__(self, conf=None, conn=None):
		if conf:
			self.conf = conf
		else:
			self.conf = Config()
		if conn:
			self.conn = conn
		else:
			self.conn = self.connect()

	def connect(self):
		try:
			return Connection(self.conf)
		except (mpdclient2.socket.error, EOFError):
			return None


class Player(Client):
	"""Keeps track of what is playing and handles changes in state as necessary. Derived classes should override handle_change_* to do things such as redrawing their UI."""
	def __init__(self, conf=None, conn=None):
		self.prevconn = []
		self.prevstatus = None
		self.prevsonginfo = None
		self.stopaftercurrent = False
		Client.__init__(self, conf, conn)

	def update_status(self):
		try:
			if not self.conn:
				self.conn = self.connect()
			if self.conn:
				self.status = self.conn.do.status()
				self.songinfo = self.conn.do.currentsong()
			else:
				self.status = None
				self.songinfo = None
		except (mpdclient2.socket.error, EOFError):
			self.prevconn = self.conn
			self.prevstatus = self.status
			self.prevsonginfo = self.songinfo
			self.conn = None
			self.status = None
			self.songinfo = None

	def iterate(self):
		self.update_status()
		
		if self.conn != self.prevconn:
			self.handle_change_conn()
		if self.status != self.prevstatus:
			self.handle_change_status()
		if self.songinfo != self.prevsonginfo:
			self.handle_change_song()

		self.prevconn = self.conn
		self.prevstatus = self.status
		self.prevsonginfo = self.songinfo

	def handle_change_conn(self):
		pass
	def handle_change_status(self):
		pass
	def handle_change_song(self):
		pass


class UI:
	"""Something with a glade tree that can be main'd and that the user can interact with."""
	def __init__(self, tree):
		self.w = GladeWidgetTree(tree)
		self.interpreter = em.Interpreter()

	def __del__(self):
		self.interpreter.shutdown()

	def main(self):
		gtk.main()

	def show(self):
		self.window.move(self.winprops.x, self.winprops.y)
		self.window.resize(*self.winprops.get_size())
		self.window.show()

	def save_size(self):
		self.winprops.set_size(self.window.get_size())

	def save_position(self):
		self.winprops.x, self.winprops.y = self.window.get_position()

	def expand_fmt(self, fmt, songinfo, status=None, escape=True):
		vars = {
			'connected': self.conn != None,
			'playing': False,
			}
		if status and status.state in ('play', 'pause'):
			at, len = [int(c) for c in status.time.split(':')]
			vars['at'] = convert_time(at)
			vars['playing'] = True
		try:
			vars['time'] = convert_time(int(songinfo.time))
		except:
			vars['time'] = None
		for a in ('artist', 'title', 'album', 'genre', 'date', 'track', 'composer', 'performer', 'comment', 'file'):
			v = getattr(songinfo, a, None)
			if v and escape:
				vars[a] = escape_html(v)
			else:
				vars[a] = v
		try:
			return self.interpreter.expand(fmt, locals=vars)
		except Exception, e:
			return '(error processing format: %s)' % e


class PlayerUI(Player, UI):
	def __init__(self, conf=None, conn=None, tree=None):
		Player.__init__(self, conf, conn)
		UI.__init__(self, tree)

		self.window = self.w.playerwindow
		self.winprops = self.conf.player_winprops

		###############################################
		# Initialize widgets... in a non-Glade manner #
		###############################################
		self.tooltips = gtk.Tooltips()
		self.w.cursonglabel.connect('notify::label', self.labelnotify)
		self.w.progressbar.connect('notify::fraction', self.progressbarnotify_fraction)
		self.w.progressbar.connect('notify::text', self.progressbarnotify_text)
		if self.conf.show_systrayicon:
			self.traytips = TrayIconTips()
			self.traytips.add_widget(self.w.tipbox)
			self.w.tipbox.show_all()
		
		# Make window icon and make it themeable.
		self.iconfactory = gtk.IconFactory()
		self.pygmyset = gtk.IconSet()
		self.pygmyplaylistset = gtk.IconSet()
		
		# Load the SVG icon, if supported
		for item in gtk.gdk.pixbuf_get_formats():
			if item['name'] == 'svg':
				pygmyicon = 'pygmy.svg'
				pygmyplaylisticon = 'pygmyplaylist.svg'
			else:
				pygmyicon = 'pygmy.png'
				pygmyplaylisticon = 'pygmyplaylist.png'
		
		if os.path.exists(os.path.join(sys.prefix, 'share', 'pixmaps', pygmyicon)):
			filename1 = [os.path.join(sys.prefix, 'share', 'pixmaps', pygmyicon)]
			filename2 = [os.path.join(sys.prefix, 'share', 'pixmaps', pygmyplaylisticon)]
		elif os.path.exists(os.path.join(os.path.split(__file__)[0], pygmyicon)):
			filename1 = [os.path.join(os.path.split(__file__)[0], pygmyicon)]
			filename2 = [os.path.join(os.path.split(__file__)[0], pygmyplaylisticon)]
		self.icons1 = [gtk.IconSource() for i in filename1]
		self.icons2 = [gtk.IconSource() for i in filename2]
		for i, iconsource in enumerate(self.icons1):
			iconsource.set_filename(filename1[i])
			self.pygmyset.add_source(iconsource)
		for i, iconsource in enumerate(self.icons2):
			iconsource.set_filename(filename2[i])
			self.pygmyplaylistset.add_source(iconsource)
		self.iconfactory.add('pygmy', self.pygmyset)
		self.iconfactory.add('pygmyplaylist', self.pygmyplaylistset)
		self.iconfactory.add_default()
		
		self.w.expander.set_expanded(self.winprops.expanded)
		self.w.ppcheckbutton.set_active(self.conf.show_ppbutton)
		self.w.stopcheckbutton.set_active(self.conf.show_stopbutton)
		self.w.nextcheckbutton.set_active(self.conf.show_nextbutton)
		self.w.prevcheckbutton.set_active(self.conf.show_prevbutton)
		self.w.volumecheckbutton.set_active(self.conf.show_volumebutton)
		self.w.pbcheckbutton.set_active(self.conf.show_progressbar)
		self.w.tbcheckbutton.set_active(self.conf.show_toolbar)
		self.w.flatcheckbutton.set_active(self.conf.flat_buttons)
		self.w.systraycheckbutton.set_active(self.conf.show_systrayicon)

		self.flatcheckbutton_clicked(self.w.flatcheckbutton)
		self.tbcheckbutton_clicked(self.w.tbcheckbutton)

		self.w.ppbutton.set_property('visible', self.conf.show_ppbutton)
		self.w.stopbutton.set_property('visible', self.conf.show_stopbutton)
		self.w.prevbutton.set_property('visible', self.conf.show_prevbutton)
		self.w.nextbutton.set_property('visible', self.conf.show_nextbutton)
		self.w.volumebutton.set_property('visible', self.conf.show_volumebutton)
		self.w.progressbar.set_property('visible', self.conf.show_progressbar)
		self.w.toolbar.set_property('visible', self.conf.show_toolbar)
		
		# Initialize playlist data and widget
		self.playlistdata = gtk.ListStore(int, str)
		self.w.playlist.set_model(self.playlistdata)
		self.w.playlist.connect('drag-data-get',  self.playlist_data_get)
		self.playlistdata.connect('row-changed',  self.playlist_changed)
		self.playlistcell = gtk.CellRendererText()
		self.playlistcolumn = gtk.TreeViewColumn('Pango Markup', self.playlistcell, markup=1)
		self.w.playlist.append_column(self.playlistcolumn)
		self.selection = self.w.playlist.get_selection()
		self.selection.set_mode(gtk.SELECTION_MULTIPLE)
		targets = [('STRING', 0, 0)]
		self.w.playlist.enable_model_drag_source(gtk.gdk.BUTTON1_MASK, targets, gtk.gdk.ACTION_MOVE)
		self.w.playlist.enable_model_drag_dest(targets, gtk.gdk.ACTION_MOVE)

		# Init preference window text views
		self.cursong_textbuf = gtk.TextBuffer()
		self.cursong_textbuf.set_text(self.conf.cursong_fmt)
		self.w.cursong_textview.set_buffer(self.cursong_textbuf)
		self.wintitle_textbuf = gtk.TextBuffer()
		self.wintitle_textbuf.set_text(self.conf.wintitle_fmt)
		self.w.wintitle_textview.set_buffer(self.wintitle_textbuf)
		self.playlist_textbuf = gtk.TextBuffer()
		self.playlist_textbuf.set_text(self.conf.playlist_fmt)
		self.w.playlist_textview.set_buffer(self.playlist_textbuf)
		self.progress_textbuf = gtk.TextBuffer()
		self.progress_textbuf.set_text(self.conf.progress_fmt)
		self.w.progress_textview.set_buffer(self.progress_textbuf)
		
		icon = self.window.render_icon('pygmy', gtk.ICON_SIZE_DIALOG)
		self.window.set_icon(icon)

		self.show()

		# Only withdraw if the system tray icon is showing
		if self.winprops.withdrawn and self.conf.show_systrayicon:
			self.window.window.withdraw()

		# self.configure accelerators
		self.accelerators = gtk.AccelGroup()
		self.window.add_accel_group(self.accelerators)
		self.accelerators.connect_group(gtk.keysyms.Delete, (), gtk.ACCEL_LOCKED, self.accelerator_activated)
		self.accelerators.connect_group(ord('i'), gtk.gdk.CONTROL_MASK, gtk.ACCEL_LOCKED, self.accelerator_activated)
		self.accelerators.connect_group(ord('q'), gtk.gdk.CONTROL_MASK, gtk.ACCEL_LOCKED, self.accelerator_activated)

		# Connect to callbacks		
		self.w.tree.signal_autoconnect(self) # Glade callbacks
		self.shuffleid = self.w.shufflebutton.connect('toggled',  self.shuffle)
		self.repeatid = self.w.repeatbutton.connect('toggled', self.repeat)
		
		if self.conf.show_systrayicon:
			self.initialize_systrayicon()
		
		# Call self.iterate every 250ms to keep current info displayed
		gobject.timeout_add(250, self.iterate)
		
		# XXX: get from conn not conf?
		self.w.hostentry.set_text(self.conf.host)
		self.w.portentry.set_text(str(self.conf.port))
		self.w.passwordentry.set_text(self.conf.password)

		# Finally, initialize the browser so that it loads quickly when called upon
		self.browser = BrowserUI(self.conf, self.conn, self.w.tree, slave=True)

	def expand_fmt(self, fmt, songinfo=None, status=None, escape=True):
		if not songinfo: songinfo = self.songinfo
		if not status: status = self.status
		return UI.expand_fmt(self, fmt, songinfo, status, escape)

	#################
	# Main GTK Loop #
	#################

	def iterate(self):
		Player.iterate(self)
		return True

	def handle_change_conn(self):
		if self.conn is None:
			self.browser.window.hide()
			self.w.ppbutton.set_property('sensitive', False)
			self.w.stopbutton.set_property('sensitive', False)
			self.w.prevbutton.set_property('sensitive', False)
			self.w.nextbutton.set_property('sensitive', False)
			self.w.volumebutton.set_property('sensitive', False)
			self.w.playlistbutton.set_property('sensitive', False)
			self.w.rmbutton.set_property('sensitive', False)
			self.w.clbutton.set_property('sensitive', False)
			self.w.repeatbutton.set_property('sensitive', False)
			self.w.shufflebutton.set_property('sensitive', False)
			self.w.statusimage.set_from_stock(gtk.STOCK_NO, gtk.ICON_SIZE_SMALL_TOOLBAR)
			if self.conf.show_systrayicon:
				self.trayimage.set_from_stock('pygmy',  gtk.ICON_SIZE_BUTTON)
			self.playlistdata.clear()
			self.w.playlistlength.set_text('')
			self.tooltips.set_tip(self.w.statusbox, "Not connected")
			self.w.prefsbutton.grab_focus()
		else:
			if self.browser.winprops.visible:
				self.browser.window.show()
			self.w.ppbutton.set_property('sensitive', True)
			self.w.stopbutton.set_property('sensitive', True)
			self.w.prevbutton.set_property('sensitive', True)
			self.w.nextbutton.set_property('sensitive', True)
			self.w.volumebutton.set_property('sensitive', True)
			self.w.playlistbutton.set_property('sensitive', True)
			self.w.rmbutton.set_property('sensitive', True)
			self.w.clbutton.set_property('sensitive', True)
			self.w.repeatbutton.set_property('sensitive', True)
			self.w.shufflebutton.set_property('sensitive', True)
			self.w.statusimage.set_from_stock(gtk.STOCK_YES, gtk.ICON_SIZE_SMALL_TOOLBAR)
			#self.w.updateimage.set_from_stock(gtk.STOCK_YES, gtk.ICON_SIZE_MENU)
			self.tooltips.set_tip(self.w.statusbox, "Connected")
			self.w.ppbutton.grab_focus()

	def handle_change_status(self):
		if self.status == None:
			# clean up and bail out
			self.update_progressbar()
			self.update_cursong()
			self.update_wintitle()
			self.update_trayicon()
			return
		
		# Update progress frequently if we're playing
		if self.status.state in ['play', 'pause']:
			self.update_progressbar()

		# Display current playlist
		if self.prevstatus == None or self.prevstatus.playlist != self.status.playlist:
			self.update_playlist()

		# If state changes
		if self.prevstatus == None or self.prevstatus.state != self.status.state:

			# Update progressbar if the state changes too
			self.update_progressbar()
			self.update_cursong()
			self.update_wintitle()
			self.update_trayicon()
			if self.status.state == 'stop':
				self.w.ppimage.set_from_stock(gtk.STOCK_MEDIA_PLAY, gtk.ICON_SIZE_BUTTON)
				self.tooltips.set_tip(self.w.ppbutton, "Play")
				# Unbold playing song (if we were playing)
				if self.prevstatus and self.prevstatus.state == 'play':
					oldrow = int(self.prevsonginfo.pos)
					try:
						self.playlistdata[oldrow][1] = make_unbold(self.playlistdata[oldrow][1])
					except IndexError: # it's gone, playlist was probably cleared
						pass
			elif self.status.state == 'pause':
				self.w.ppimage.set_from_stock(gtk.STOCK_MEDIA_PLAY, gtk.ICON_SIZE_BUTTON)
				self.tooltips.set_tip(self.w.ppbutton, "Play")
			elif self.status.state == 'play':
				self.w.ppimage.set_from_stock(gtk.STOCK_MEDIA_PAUSE, gtk.ICON_SIZE_BUTTON)
				self.tooltips.set_tip(self.w.ppbutton, "Pause")
			if self.status.state in ['play', 'pause']:
				row = int(self.songinfo.pos)
				self.playlistdata[row][1] = make_bold(self.playlistdata[row][1])

		# If shuffle/repeat have changed, update the button state
		if self.prevstatus == None or self.status.repeat != self.prevstatus.repeat:
			self.w.repeatbutton.handler_block(self.repeatid)
			self.w.repeatbutton.set_active('1' == self.status.repeat)
			self.w.repeatbutton.handler_unblock(self.repeatid)
		if self.prevstatus == None or self.status.random != self.prevstatus.random:
			self.w.shufflebutton.handler_block(self.shuffleid)
			self.w.shufflebutton.set_active('1' == self.status.random)
			self.w.shufflebutton.handler_unblock(self.shuffleid)

		# Check if MPD is updating the library
		if self.prevstatus == None or self.prevstatus.get('updating_db', 0) != self.status.get('updating_db', 0):
			if self.status.get('updating_db', 0):
				self.w.statusimage.set_from_stock(gtk.STOCK_REFRESH, gtk.ICON_SIZE_SMALL_TOOLBAR)
				self.tooltips.set_tip(self.w.statusbox, "Updating...")
			else:
				self.w.statusimage.set_from_stock(gtk.STOCK_YES, gtk.ICON_SIZE_SMALL_TOOLBAR)
				self.tooltips.set_tip(self.w.statusbox, "Connected")
				
		if self.prevstatus is None or self.status.volume != self.prevstatus.volume:
			self.w.volumescale.get_adjustment().set_value(int(self.status.volume))

	def handle_change_song(self):
		if self.conn and self.stopaftercurrent == True:
			self.conn.do.stop()
			self.stopaftercurrent = False

		for song in self.playlistdata:
			song[1] = make_unbold(song[1])
		
		if self.status and self.status.state in ['play', 'pause']:
			row = int(self.songinfo.pos)
			self.playlistdata[row][1] = make_bold(self.playlistdata[row][1])

		self.update_cursong()
		self.update_wintitle()
		self.update_trayicon()

	def update_trayicon(self):
		if not self.conf.show_systrayicon:
			return
		
		if self.status == None:
			self.trayimage.set_from_stock('pygmy', gtk.ICON_SIZE_BUTTON)
		elif self.prevstatus == None or self.prevstatus.state != self.status.state:
			if self.status.state == 'play':
				self.trayimage.set_from_stock(gtk.STOCK_MEDIA_PLAY, gtk.ICON_SIZE_BUTTON)
			elif self.status.state == 'pause':
				self.trayimage.set_from_stock(gtk.STOCK_MEDIA_PAUSE, gtk.ICON_SIZE_BUTTON)
			elif self.status.state == 'stop':
				self.trayimage.set_from_stock('pygmy', gtk.ICON_SIZE_BUTTON)

	def update_progressbar(self):
		if self.conn and self.status and self.status.state in ['play', 'pause']:
			at, len = [float(c) for c in self.status.time.split(':')]
			try:
				self.w.progressbar.set_fraction(at/len)
			except ZeroDivisionError:
				self.w.progressbar.set_fraction(0)
		else:
			self.w.progressbar.set_fraction(0)
		self.w.progressbar.set_text(self.expand_fmt(self.conf.progress_fmt, escape=False))

	def update_cursong(self):
		self.w.cursonglabel.set_markup(self.expand_fmt(self.conf.cursong_fmt, escape=True))
		# Hide traytip's progressbar when stopped
		if self.status and self.status.state == 'stop':
			self.w.traycursonglabel.set_label('Stopped')
			self.w.trayprogressbar.hide()
		else:
			self.w.trayprogressbar.show()
	
	def update_wintitle(self):
		self.window.set_property('title', self.expand_fmt(self.conf.wintitle_fmt, escape=False))
		
	def update_playlist(self):
		if self.conn:
			self.songs = self.conn.do.playlistinfo()
			self.playlistdata.clear()
			total_time = 0
			for track in self.songs:
				self.playlistdata.append([int(track.id), self.expand_fmt(self.conf.playlist_fmt, track)])
				try:
					total_time = total_time + int(track.time)
				except AttributeError:
					pass
			if total_time == 0:
				self.w.playlistlength.set_text('')
			else:
				self.w.playlistlength.set_text(convert_time(total_time))

			if self.status.state in ['play', 'pause']:
				row = int(self.songinfo.pos)
				self.playlistdata[row][1] = make_bold(self.playlistdata[row][1])
	
	def update_browser(self):
		buttons = self.w.buttonbox.get_children()
		if buttons and buttons[0].get_label() == 'Search results':
			self.browser.findbutton_clicked(None)
		else:
			self.browser.browse(None, self.browser.root)
	
	def labelnotify(self, *args):
		self.w.traycursonglabel.set_label(self.w.cursonglabel.get_label())
		if self.conf.show_systrayicon and self.traytips.get_property('visible'):
			self.traytips._real_display(self.trayeventbox)
		
	def progressbarnotify_fraction(self, *args):
		self.w.trayprogressbar.set_fraction(self.w.progressbar.get_fraction())
	
	def progressbarnotify_text(self, *args):
		self.w.trayprogressbar.set_text(self.w.progressbar.get_text())

	#################
	# Gui Callbacks #
	#################

	# This one makes sure the program exits when the window is closed
	def delete_event(self, widget, data=None):
		self.conf.save()
		gtk.main_quit()
		return False
		
	def prefs_delete_event(self, *args):
		self.w.prefswindow.hide()
		return True

	def on_window_state_change(self, widget, event):
		self.volume_hide()

	def on_window_configure(self, widget, event):
		self.save_position()
		self.save_size()
		self.volume_hide()

	# keep the window sized correctly when the expander changes... at this
	# point it hasn't changed yet.
	def expander_activate(self, expander):
		if expander.get_expanded():
			expander.child.hide_all()
		else:
			expander.child.show_all()
		self.winprops.set_size(self.window.get_size())
		return

	def after_expander_activate(self, expander):
		self.winprops.expanded = expander.get_expanded()
		if self.winprops.expanded:
			if self.winprops.we != 0 and self.winprops.he != 0:
				if self.winprops.we < self.winprops.w:
					self.winprops.we = self.winprops.w
				self.window.resize(self.winprops.we, self.winprops.he)
			#self.window.set_geometry_hints(None)
		else:
			w, h = self.window.get_size()
			self.window.resize(w, 1)
			if self.winprops.w !=0 and self.winprops.h != 0:
				self.window.resize(self.winprops.w, 1)
			#self.window.set_geometry_hints(None, max_height=72)
		return

	# This callback allows the user to seek to a specific portion of the song
	def progressbar_button_press_event(self, widget, event):
		if self.status.state in ['play', 'pause']:
			at, len = [int(c) for c in self.status.time.split(':')]
			progressbarsize = self.w.progressbar.allocation
			seektime = int((event.x/progressbarsize.width) * len)
			self.seek(int(self.status.song), seektime)
		return True

	def on_drag_drop(self, treeview, drag_context, x, y, selection, info, timestamp):
		model = treeview.get_model()
		foobar, self._selected = self.selection.get_selected_rows()
		data = pickle.loads(selection.data)
		drop_info = treeview.get_dest_row_at_pos(x, y)

		# calculate all this now before we start moving stuff
		drag_sources = []
		#self.selection_ids = []
		for path in data:
			index = path[0]
			iter = model.get_iter(path)
			id = model.get_value(iter, 0)
			text = model.get_value(iter, 1)
			drag_sources.append([index, iter, id, text])
			#self.selection_ids.append(id)

		offset = 0
		for source in drag_sources:
			index, iter, id, text = source
			if drop_info:
				destpath, position = drop_info
				if destpath[0] > index:
					# if moving ahead, all the subsequent indexes decrease by 1
					dest = destpath[0] + offset - 1
				else:
					# if moving back, next one will need to go after it
					dest = destpath[0] + offset
					offset += 1
				if position in (gtk.TREE_VIEW_DROP_BEFORE, gtk.TREE_VIEW_DROP_INTO_OR_BEFORE):
					model.insert_before(iter, [index, text])
					self.conn.do.moveid(id, dest)
				else:
					model.insert_after(iter, [index, text])
					self.conn.do.moveid(id, dest + 1)
			else:
				dest = len(self.conn.do.playlistinfo()) - 1
				self.conn.do.moveid(id, dest)
				model.append([0, text])
			# now fixup
			for source in drag_sources:
				if dest < index:
					# we moved it back, so all indexes inbetween increased by 1
					if dest < source[0] < index:
						source[0] += 1
				else:
					# we moved it ahead, so all indexes inbetween decreased by 1
					if index < source[0] < dest:
						source[0] -= 1
			model.remove(iter)

		if drag_context.action == gtk.gdk.ACTION_MOVE:
			drag_context.finish(True, True, timestamp)
		
	def after_drag_drop(self, treeview, drag_context):
		model = treeview.get_model()
		sel = treeview.get_selection()
		for path in self._selected:
			sel.select_path(path)
	
	def playlist_changed(self, treemodel, path, iter):
		pass
	
	def playlist_data_get(self, widget, drag_context, selection, info, timestamp):
		model, selected = self.selection.get_selected_rows()
		selection.set(selection.target, 8, pickle.dumps(selected))
		return

	def playlist_button_press_event(self, widget, event):
		if event.button == 3:
			self.w.playlistmenu.popup(None, None, None, event.button, event.time)

	def playlist_popup_menu(self, widget):
		self.w.playlistmenu.popup(None, None, None, 3, 0)

	def update_activate(self, widget, event):
		if event.button == 3:
			self.w.updatemenu.popup(None, None, None, event.button, event.time)
		return False

	def updatedb(self, widget):
		self.conn.do.update('/')

	def stop_activate(self, widget, event):
		if event.button == 3:
			self.w.stopmenu.popup(None, None, None, event.button, event.time)
		return False

	def stop_after_current(self, widget):
		self.stopaftercurrent = True

	# What happens when you click on the system tray icon?
	def trayaction(self, widget, event):
		if event.button == 1: # Left button shows/hides window(s)
			if self.window.window.get_state() & gtk.gdk.WINDOW_STATE_WITHDRAWN: # window is hidden
				self.show()
				self.winprops.withdrawn = False
				if self.browser.winprops.visible:
					self.browser.show()
			elif not (self.window.window.get_state() & gtk.gdk.WINDOW_STATE_WITHDRAWN): # window is showing
				self.window.hide()
				self.winprops.withdrawn = True
				if self.browser.window.get_property('visible'):
					self.browser.window.hide()
		elif event.button == 2: # Middle button will play/pause
			self.pp(self.trayeventbox)
		elif event.button == 3: # Right button pops up menu
			self.w.traymenu.popup(None, None, None, event.button, event.time)
		return False

	# Accelerator callback; works globally
	def accelerator_activated(self, accelgroup, widget, key, mods):
		if key == gtk.keysyms.Delete:
			self.remove(widget)
		elif key == ord('I') or key == ord('i'):
			self.invert_selection(self.w.playlist)
		return True

	# Tray menu callbacks, because I can't reuse all of them.
	def quit_activate(self, widget):
		self.window.destroy()

	# TODO: combine with pp()?
	def playlist_click(self, treeview, path, column):
		iter = self.playlistdata.get_iter(path)
		self.conn.do.playid(self.playlistdata.get_value(iter, 0))
		
	def invert_selection(self, widget):
		for row in self.playlistdata:
			if row.path in self.selection.get_selected_rows()[1]:
				self.selection.unselect_path(row.path)
			else:
				self.selection.select_path(row.path)
	
	# Volume control
	def on_volumebutton_clicked(self, widget):
		if not self.w.volumewindow.get_property('visible'):
			x_win, y_win = self.w.volumebutton.window.get_origin()
			button_rect = self.w.volumebutton.get_allocation()
			x_coord, y_coord = button_rect.x + x_win, button_rect.y+y_win
			width, height = button_rect.width, button_rect.height
			self.w.volumewindow.set_size_request(width, -1)
			self.w.volumewindow.move(x_coord, y_coord+height)
			self.w.volumewindow.present()
		else:
			self.volume_hide()
		return

	def on_volumescale_change(self, obj, value, data):
		new_volume = int(obj.get_adjustment().get_value())
		self.conn.do.setvol(new_volume)
		return

	def on_volumewindow_unfocus(self, obj, data):
		self.volume_hide()
		return True

	def volume_hide(self):
		self.w.volumebutton.set_active(False)
		self.w.volumewindow.hide()
	
	# Control callbacks
	def pp(self, widget):
		if self.status.state in ('stop', 'pause'):
			self.conn.do.play()
		elif self.status.state == 'play':
			self.conn.do.pause(1)
		return

	def stop(self, widget):
		self.conn.do.stop()
		return

	def prev(self, widget):
		self.conn.do.previous()
		return

	def next(self, widget):
		self.conn.do.next()
		return

	def show_browser(self, widget):
		self.browser.show()
		return

	def remove(self, widget):
		model, selected = self.selection.get_selected_rows()
		iters = [model.get_iter(path) for path in selected]
		for iter in iters:
			self.conn.do.deleteid(self.playlistdata.get_value(iter, 0))
		
	def randomize(self, widget):
		# Ironically enough, the command to turn shuffle on/off is called
		# random, and the command to randomize the playlist is called shuffle.
		self.conn.do.shuffle()
		return

	def clear(self, widget):
		self.conn.do.clear()
		return

	def repeat(self, widget):
		if self.status.repeat == '0':
			self.w.repeatbutton.set_active(True)
			self.w.repeatmenu.set_active(True)
			self.conn.do.repeat(1)
		elif self.status.repeat == '1':
			self.w.repeatbutton.set_active(False)
			self.w.repeatmenu.set_active(False)
			self.conn.do.repeat(0)

	def repeat_toggled(self, widget):
		return True

	def shuffle(self, widget):
		if self.status.random == '0':
			self.conn.do.random(1)
			self.w.shufflebutton.set_active(True)
			self.w.shufflemenu.set_active(True)
		elif self.status.random == '1':
			self.conn.do.random(0)
			self.w.shufflebutton.set_active(False)
			self.w.shufflemenu.set_active(False)

	def shuffle_toggled(self, widget):
		return True
	
	def prefs(self, widget):
		self.w.prefswindow.present()

	def seek(self, song, seektime):
		self.conn.do.seek(song, seektime)
		return
		
	# Preferences callbacks
	def applybutton_clicked(self, widget):
		old_host = self.conf.host
		old_port = self.conf.port
		old_password = self.conf.password
		self.conf.host = self.w.hostentry.get_text()
		self.conf.port = int(self.w.portentry.get_text())
		self.conf.password = self.w.passwordentry.get_text()
		if self.conf.host != old_host or self.conf.port != old_port or self.conf.password != old_password:
			self.conn = self.connect()
			self.browser.reconnect(self.conn)
	
	def ppcheckbutton_clicked(self, widget):
		self.conf.show_ppbutton = widget.get_active()
		self.w.ppbutton.set_property('visible', self.conf.show_ppbutton)
	
	def stopcheckbutton_clicked(self, widget):
		self.conf.show_stopbutton = widget.get_active()
		self.w.stopbutton.set_property('visible', self.conf.show_stopbutton)

	def prevcheckbutton_clicked(self, widget):
		self.conf.show_prevbutton = widget.get_active()
		self.w.prevbutton.set_property('visible', self.conf.show_prevbutton)
		
	def nextcheckbutton_clicked(self, widget):
		self.conf.show_nextbutton = widget.get_active()
		self.w.nextbutton.set_property('visible', self.conf.show_nextbutton)
		
	def volumecheckbutton_clicked(self, widget):
		self.conf.show_volumebutton = widget.get_active()
		self.w.volumebutton.set_property('visible', self.conf.show_volumebutton)
		
	def pbcheckbutton_clicked(self, widget):
		self.conf.show_progressbar = widget.get_active()
		self.w.progressbar.set_property('visible', self.conf.show_progressbar)
		
	def flatcheckbutton_clicked(self, widget):
		self.conf.flat_buttons = widget.get_active()
		if self.conf.flat_buttons:
			set_buttons(self.window, gtk.RELIEF_NONE)
		else:
			set_buttons(self.window, gtk.RELIEF_NORMAL)
		
	def tbcheckbutton_clicked(self, widget):
		self.conf.show_toolbar = widget.get_active()
		if self.conf.show_toolbar:
			self.w.toolbar.show()
			self.w.toolbar.set_no_show_all(False)
		elif not widget.get_active():
			self.w.toolbar.hide()
			self.w.toolbar.set_no_show_all(True)
	
	def systraycheckbutton_clicked(self, widget):
		self.conf.show_systrayicon = widget.get_active()
		if self.conf.show_systrayicon:
			self.initialize_systrayicon()
			self.prevstatus = None
		else:
			self.trayicon.destroy()
	
	def textview_focus_out(self, widget, event):
		if widget == self.w.cursong_textview:
			self.conf.cursong_fmt = self.cursong_textbuf.get_text(self.cursong_textbuf.get_start_iter(), self.cursong_textbuf.get_end_iter(), True)
			self.update_cursong()
		elif widget == self.w.wintitle_textview:
			self.conf.wintitle_fmt = self.wintitle_textbuf.get_text(self.wintitle_textbuf.get_start_iter(), self.wintitle_textbuf.get_end_iter(), True)
			self.update_wintitle()
		elif widget == self.w.playlist_textview:
			self.conf.playlist_fmt = self.playlist_textbuf.get_text(self.playlist_textbuf.get_start_iter(), self.playlist_textbuf.get_end_iter(), True)
 			self.update_playlist()
 			self.update_browser()
		elif widget == self.w.progress_textview:
			self.conf.progress_fmt = self.progress_textbuf.get_text(self.progress_textbuf.get_start_iter(), self.progress_textbuf.get_end_iter(), True)
 			self.update_progressbar()
	
	def initialize_systrayicon(self):
		# Make system tray 'icon' to sit in the system tray
		self.trayeventbox = gtk.EventBox()
		self.trayeventbox.connect('button_press_event', self.trayaction)
		self.traytips.set_tip(self.trayeventbox)
		self.trayimage = gtk.Image()
		self.trayimage.set_from_stock('pygmy', gtk.ICON_SIZE_BUTTON)
		self.trayeventbox.add(self.trayimage)
		try:
			self.trayicon = egg.trayicon.TrayIcon("TrayIcon")
			self.trayicon.add(self.trayeventbox)
			self.trayicon.show_all()
		except NameError:
			self.w.systraycheckbutton.set_property('sensitive', False)


class Browser(Player):
	def __init__(self, conf=None, conn=None):
		Player.__init__(self, conf, conn)


class BrowserUI(Browser, UI):
	def __init__(self, conf=None, conn=None, tree=None, slave=False):
		Browser.__init__(self, conf, conn)
		UI.__init__(self, tree)
		self.slave = slave

		self.window = self.w.browserwindow
		self.winprops = self.conf.browser_winprops

		# Connect to glade callbacks
		self.w.tree.signal_autoconnect(self)

#		self.w.librarybutton.connect_object('button-press-event', self.browse, '/')

		self.browserposition = {}
		self.root = '/'
		self.w.browser.wd = '/'
		self.w.metadatacombobox.set_active(0)
		self.prevstatus = None
		self.tooltips = gtk.Tooltips()

		# Initialize browser data and widget
		self.browserdata = gtk.ListStore(str, str)
		self.w.browser.set_model(self.browserdata)
		self.browsercell = gtk.CellRendererText()
		self.browsercolumn = gtk.TreeViewColumn('Pango Markup', self.browsercell, markup=1)
		self.browsercolumn.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
		self.w.browser.append_column(self.browsercolumn)
		self.selection = self.w.browser.get_selection()
		self.selection.set_mode(gtk.SELECTION_MULTIPLE)
		self.hadjustment = self.w.browser.get_hadjustment()
		self.vadjustment = self.w.browser.get_vadjustment()
		
		# self.configure accelerators
		self.accelerators = gtk.AccelGroup()
		self.window.add_accel_group(self.accelerators)
		self.accelerators.connect_group(ord('f'), gtk.gdk.CONTROL_MASK, gtk.ACCEL_LOCKED, self.accelerator_activated)

		icon = self.window.render_icon('pygmyplaylist', gtk.ICON_SIZE_DIALOG)
		self.window.set_icon(icon)

		if self.winprops.visible or self.slave == False:
			self.show()
			
		# Only show updating icon if Pygmy is not running
		self.w.updateimage.set_property('visible', (False == self.slave))
			
		# Call self.iterate every 500ms to make sure current info is displayed.
		gobject.timeout_add(500, self.iterate)
	
	#################
	# Gui Callbacks #
	#################

	def browserwindow_after_show(self, *args):
		# Browse to cwd when opening window
		self.browse(None, self.w.browser.wd)
		if self.slave:
			self.winprops.visible = True
			self.window.set_property('visible', self.winprops.visible)
		return True
	
	def browserwindow_delete_event(self, *args):
		if self.slave:
			self.winprops.visible = False
			self.window.set_property('visible', self.winprops.visible)
			self.window.hide()
		else:
			self.conf.save()
			gtk.main_quit()
		return True

	def browserwindow_on_window_configure(self, widget, event):
		if not (self.window.window.get_state() & gtk.gdk.WINDOW_STATE_WITHDRAWN):
			self.save_position()
			self.save_size()

	def browse(self, widget=None, root='/'):
		if not self.conn:
			return
		
		# Handle special cases
		while self.conn.do.lsinfo(root) == []:
			if self.conn.do.listallinfo(root):
				# Info exists if we try to browse to a song
				self.addbutton_clicked(self.w.browser)
				return
			elif root == '/':
				# Nothing in the library at all
				return
			else:
				# Back up and try the parent
				root = '/'.join(root.split('/')[:-1]) or '/'
		
		self.root = root
		# Save coordinates for where we just were
		self.browserposition[self.w.browser.wd] = (self.hadjustment.value, self.vadjustment.value)		

		self.w.browser.wd = root
		self.browserdata.clear()
		for item in self.conn.do.lsinfo(root):
			if item.type == 'directory':
				name = item.directory.split('/')[-1]
				self.browserdata.append([item.directory, escape_html(name)])
			elif item.type == 'file':
				name = item.file.split('/')
				self.browserdata.append([item.file, self.expand_fmt(self.conf.playlist_fmt, item)])
		# Set coordinates for current dir (i.e. root)
		try:
			self.hadjustment.value, self.vadjustment.value = self.browserposition[root]
		except KeyError:
			self.hadjustment.value, self.vadjustment.value = (0, 0)
		
		self.w.buttonbox.foreach(self.w.buttonbox.remove)
		if root != '/':
			p = []
			for dir in root.split('/'):
				p.append(dir)
				button = gtk.ToggleButton()
				button.set_label(dir)
				self.w.buttonbox.pack_start(button, False, False, 1)
				button.show()
				button.connect_object('clicked', self.browse, button, '/'.join(p))
				button.connect('toggled', self.toggled)
				
	def toggled(self, button):
		if button.get_active():
			button.set_label(make_bold(button.get_label()))
		else:
			button.set_label(make_unbold(button.get_label()))

	def browserow(self, widget, path, column=0):
		self.browse(None, self.browserdata.get_value(self.browserdata.get_iter(path), 0))
		
	# Accelerator callback; works globally for the window
	def accelerator_activated(self, accelgroup, widget, key, mods):
		if key == ord('f') and mods == gtk.gdk.CONTROL_MASK:
			self.w.findtogglebutton.set_active(False == self.w.findtogglebutton.get_active())
		return True
	
	def toggle_findbox(self, widget=None):
		if not self.w.findbox.get_property('visible'):
			self.show_findbox()
		elif self.w.findbox.get_property('visible'):
			self.hide_findbox()
		return False
	
	def show_findbox(self):
			self.w.findentry.grab_focus()
			self.w.findbutton.grab_default()
			self.w.findbox.show()
	
	def hide_findbox(self):
			self.w.findbox.hide()
			self.browse(root=self.w.browser.wd)
			self.w.findentry.set_text('')
			self.w.closebutton.grab_focus()
			self.w.closebutton.grab_default()
		
	def findbutton_clicked(self, widget):
		self.list = self.conn.do.search(self.w.metadatacombobox.get_active_text().lower(), self.w.findentry.get_text())
		self.browserdata.clear()
		for item in self.list:
			if item.type == 'directory':
				name = item.directory.split('/')
				self.browserdata.append([item.directory, name[len(name)-1]])
			elif item.type == 'file':
				name = item.file.split('/')
				self.browserdata.append([item.file, self.expand_fmt(self.conf.playlist_fmt, item)])
		
		self.w.buttonbox.foreach(self.w.buttonbox.remove)
		self.w.resultsbutton = gtk.Button()
		self.w.resultsbutton.set_label('Search results')
		self.w.buttonbox.pack_start(self.w.resultsbutton, False, False, 1)
		self.w.resultsbutton.show()

	def addbutton_clicked(self, widget):
		model, selected = self.selection.get_selected_rows()
		iters = [model.get_iter(path) for path in selected]
		for iter in iters:
			self.conn.do.add(model.get_value(iter, 0))

	def reconnect(self, conn):
		self.conn = conn
	
	def iterate(self):
		Player.iterate(self)
		return True
	
	def handle_change_conn(self):
		if self.conn:
			self.browse(root='/')

	def handle_change_status(self):
		if self.conn:
			if self.prevstatus == None or self.prevstatus.get('updating_db', 0) != self.status.get('updating_db', 0):
				if self.status and self.status.get('updating_db', 0):
					self.w.updateimage.set_from_stock(gtk.STOCK_REFRESH, gtk.ICON_SIZE_SMALL_TOOLBAR)
					self.tooltips.set_tip(self.w.updateimage, "Updating...")
				else:
					self.browse(root=self.root)
					#self.w.statusimage.clear() # Only works for GTK+ 2.8+
					self.w.updateimage.set_from_image(None, None)
					self.tooltips.set_tip(self.w.updateimage, "")
		return True


class TrayIconTips(gtk.Window):
	"""Custom tooltips derived from gtk.Window() that allow for markup text and multiple widgets, e.g. a progress bar. ;)"""
	# For placing the tooltip win only; internal spacing is done in glade
	MARGIN = 4
	
	def __init__(self, widget=None):
		gtk.Window.__init__(self, gtk.WINDOW_POPUP)
		# from gtktooltips.c:gtk_tooltips_force_window
		self.set_app_paintable(True)
		self.set_resizable(False)
		self.set_name("gtk-tooltips")
		self.connect('expose-event', self._on__expose_event)

		if widget != None:
			self._label = gtk.Label()
			self.add(self._label)
		
		self._show_timeout_id = -1
		self.timer_tag = None
		
	# from gtktooltips.c:gtk_tooltips_draw_tips
	def _calculate_pos(self, widget):
		screen = widget.get_screen()
		x, y = widget.window.get_origin()
		w, h = self.size_request()

		if widget.flags() & gtk.NO_WINDOW:
			x += widget.allocation.x
			y += widget.allocation.y
			
		pointer_screen, px, py, _ = screen.get_display().get_pointer()
		if pointer_screen != screen:
			px = x
			py = y

		monitor_num = screen.get_monitor_at_point(px, py)
		monitor = screen.get_monitor_geometry(monitor_num)
		
		# If the tooltip goes off the screen horizontally, realign it so that
		# it all displays.
		if (x + w) > monitor.width:
			x = monitor.width - w

		# If the tooltip goes off the screen vertically (i.e. the system tray
		# icon is on the bottom of the screen), realign the icon so that it
		# shows above the icon.
		if ((y + h + widget.allocation.height + self.MARGIN) >
			monitor.y + monitor.height):
			y = y - h - self.MARGIN
		else:
			y = y + widget.allocation.height + self.MARGIN

		return x, y

	def _event_handler (self, widget):
		widget.connect_after("event-after", self._motion_cb)

	def _motion_cb (self, widget, event):
		if event.type == gtk.gdk.LEAVE_NOTIFY:
			self._remove_timer()
		if event.type == gtk.gdk.ENTER_NOTIFY: 
			self._start_delay(widget)

	def _start_delay (self, widget):
		self.timer_tag = gobject.timeout_add(500, self._tips_timeout, widget)

	def _tips_timeout (self, widget):
		gtk.gdk.threads_enter()
		self._real_display(widget)
		gtk.gdk.threads_leave()

	def _remove_timer(self):
		self.hide()
		gobject.source_remove(self.timer_tag)
		self.timer_tag = None

	# from gtktooltips.c:gtk_tooltips_paint_window
	def _on__expose_event(self, window, event):
		w, h = window.size_request()
		window.style.paint_flat_box(window.window,
									gtk.STATE_NORMAL, gtk.SHADOW_OUT,
									None, window, "tooltip",
									0, 0, w, h)
		return False
	
	def _real_display(self, widget):
		x, y = self._calculate_pos(widget)
		w, h = self.size_request()
		self.move(x, y)
		self.resize(w, h)
		self.show()

	# Public API
	
	def set_text(self, text):
		self._label.set_text(text)

	def hide(self):
		gtk.Window.hide(self)
		gobject.source_remove(self._show_timeout_id)
		self._show_timeout_id = -1

	def display(self, widget):
		if not self._label.get_text():
			return
		
		if self._show_timeout_id != -1:
			return
		
		self._show_timeout_id = gobject.timeout_add(500, self._real_display, widget)

	def set_tip (self, widget):
		self.widget = widget
		self._event_handler (self.widget)

	def add_widget (self, widget_to_add):
		self.widget_to_add = widget_to_add
		self.add(self.widget_to_add)


def convert_time(raw):
	# Converts raw time to 'hh:mm:ss' with leading zeros as appropriate
	h, m, s = ['%02d' % c for c in (raw/3600, (raw%3600)/60, raw%60)]
	if h == '00':
		return m + ':' + s
	else:
		return h + ':' + m + ':' + s

def make_bold(s):
	if not (s.startswith('<b>') and s.endswith('</b>')):
		return '<b>%s</b>' % s
	else:
		return s

def make_unbold(s):
	if s.startswith('<b>') and s.endswith('</b>'):
		return s[3:-4]
	else:
		return s

def escape_html(s):
	s = s.replace('&', '&amp;')
	s = s.replace('<', '&lt;')
	s = s.replace('>', '&gt;')
	return s

def set_buttons(parent, relief):
	try:
		for widget in parent.get_children():
			try:
				widget.set_relief(relief)
			except AttributeError:
				pass
			set_buttons(widget, relief)
	except AttributeError:
		pass
