# btsutils -- Python module to interact with debbugs servers.
# Copyright (C) 2007  Gustavo R. Montesino
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import re

import BugExceptions
import BugStatus
import BugSeverity
import BugTags

class Bug:
    """A single debbugs bug report"""

    def __init__(self, bug=None, package=None, summary=None, 
        severity=BugSeverity.DEFAULT, submitter=None, tags=None,
        usertags=None, status=BugStatus.DEFAULT, forwarded=None,
        url=''):
        """Creates a bug instance

        Creates a bug instance with the information specified as
        keyword arguments."""

        self.setBug(bug)
        self.setPackage(package)
        self.setSummary(summary)
        self.setSeverity(severity)
        self.setSubmitter(submitter)
        self.setTags(tags)
        self.setUserTags(usertags)
        self.setStatus(status)
        self.setForwarded(forwarded)
        self.setURL(url)

    def __str__(self):
        return """Bug: %s
Package: %s
Summary: %s
Status: %s
Severity: %s
Submitter: %s
Tags: %s
Usertags: %s
Forwarded: %s""" % (self.getBug().encode("utf-8"),
        self.getPackage().encode("utf-8"),
        self.getSummary().encode("utf-8"),
        self.getStatus().encode("utf-8"),
        self.getSeverity().encode("utf-8"),
        self.getSubmitter().encode("utf-8"),
        ", ".join(self.getTags()).encode("utf-8"),
        ", ".join(self.getUserTags()).encode("utf-8"),
        self.getForwarded().encode("utf-8"))
    
    def getBug(self):
        return self.bug
    
    def setBug(self, bug):
        if bug == None or re.match("^\d+$", str(bug)):
            self.bug = unicode(bug)
        else:
            raise BugExceptions.InvalidBugIdError("Invalid Bug ID: %s" % bug)
    
    def getPackage(self):
        return self.package

    def setPackage(self, package):
        self.package = unicode(package)
    
    def getSummary(self):
        return self.summary;
    
    def setSummary(self, summary):
        self.summary = unicode(summary)
    
    def getSeverity(self):
        return self.severity
    
    def setSeverity(self, severity):
        if severity.lower() in BugSeverity.VALID:
            self.severity = unicode(severity.lower())
        else:
            raise BugExceptions.InvalidBugSeverityError, "Invalid severity: %s" % severity
    
    def getSubmitter(self):
        return self.submitter
    
    def setSubmitter(self, submitter):
        self.submitter = unicode(submitter)
    
    def getTags(self):
        return self.tags
    
    def setTags(self, tags):
        self.tags = []
        if tags:
            for tag in tags:
                self.addTag(tag)

    def addTag(self, tag):
        """Add a tag for the list of tags or usertags of this Bug"""
        if tag.lower() in BugTags.VALID:
            self.tags.append(tag)
        else:
            raise BugExceptions.InvalidBugTagError("Invalid tag: %s" % tag)

    def getUserTags(self):
        return self.usertags
    
    def setUserTags(self, usertags):
        if usertags:
            self.usertags = usertags
        else:
            self.usertags = []

    def addUserTag(self, usertag):
        self.usertags.append(usertag)

    def getStatus(self):
        return self.status
    
    def setStatus(self, status):
        if status.lower() in BugStatus.VALID:
            self.status = unicode(status.lower())
        else:
            raise BugExceptions.InvalidBugStatusError, "Invalid status: %s" % status

    def getForwarded(self):
        return self.forwarded

    def setForwarded(self, forwarded):
        self.forwarded = unicode(forwarded)

    def getURL(self):
        return self.url

    def setURL(self, url):
        self.url = url


# vim: tabstop=4 expandtab shiftwidth=4
