#import <BTUtil/BTUtil.h>
#import "MyFileSender.h"

@implementation MyFileSender

- (void)awakeFromNib
{
	[BBOBEXClientSession setDebug:YES];
	[self resetUI];
	
}

- (IBAction)connectOrDisconnect:(id)sender
{
	if ([[connectOrDisconnectButton title] isEqual:@"Connect"]) {
		[self connect];
	} else {
		[self disconnect];		
	}
}

- (void)connect
{
	BBOBEXClientSession *newOBEXClient = [[BBOBEXClientSession alloc] initWithDelegate:self];
	[myOBEXClient release];
	myOBEXClient = newOBEXClient;
	
    // start the connection process
	OBEXError status = [myOBEXClient connectToDeviceWithAddress:[addressField stringValue]
												withChannelID:[portField intValue]
													 toTarget:nil];
	if (status == kOBEXSuccess) {
		[self log:@"Connecting..."];	
		[connectOrDisconnectButton setEnabled:NO];
	} else {
		[self log:[NSString stringWithFormat:@"Error connecting: %@ (OBEXError=%d)", [myOBEXClient lastErrorDescription], status]];
	}
}

- (void)disconnect
{
	// start the disconnection process
	OBEXError status = [myOBEXClient disconnect];
	
	if (status == kOBEXSuccess) {
		[self log:@"Disconnecting..."];	
		[connectOrDisconnectButton setEnabled:NO];
	} else {
		[self log:[NSString stringWithFormat:@"Error disconnecting (OBEXError=%d)", status]];
	}	
}

- (IBAction)sendFile:(id)sender
{
	// reset the progress bar 
	int fileSize = [self getFileSize:[filePathField stringValue]];
	[sendFileProgressBar setMaxValue:fileSize];
	[sendFileProgressBar setDoubleValue:0.0];
	
	// start the file sending process
	OBEXError status = [myOBEXClient putFileAtPath:[filePathField stringValue]];
	
	if (status == kOBEXSuccess) {
		[self log:@"Sending file..."];	
		[sendFileButton	setEnabled:NO];
		[abortButton setEnabled:YES];
	} else {
		[self log:[NSString stringWithFormat:@"Error sending file (OBEXError=%d)", status]];
	}		
}

- (IBAction)abort:(id)sender
{
	// start the abort process
	OBEXError status = [myOBEXClient abort];
	
	if (status == kOBEXSuccess) {
		[self log:@"Aborting..."];	
		[abortButton setEnabled:NO];
	} else {
		[self log:[NSString stringWithFormat:@"Error aborting (OBEXError=%d)", status]];
	}	
	
	[sendFileProgressBar setDoubleValue:0.0];
}

- (IBAction)deviceDisconnect:(id)sender
{	
	BluetoothDeviceAddress deviceAddress;
	IOBluetoothNSStringToDeviceAddress([addressField stringValue], &deviceAddress);
	IOBluetoothDevice *remoteDevice = [IOBluetoothDevice withAddress:&deviceAddress];
	
	OBEXError status = [remoteDevice closeConnection];
	[self log:[NSString stringWithFormat:@"Device disconnection result: %d", status]];
	
	if (status == kOBEXSuccess) {
		[remoteDevice release];
		remoteDevice = nil;
	}
	
	// reset UI
	[self resetUI];
}


#pragma mark -
#pragma mark Delegate methods

// The following delegate methods allow the BBOBEXClientSession to notify this 
// delegate when an OBEX event has occurred. You must wait for a notification
// that a command has finished before performing the next command -- e.g. when
// this client connects, it cannot send a file until the connection process is 
// complete (i.e. until the obexClientConnectComplete:error: method below is 
// called).


// Called when the connection process is complete.
- (void)obexClientSessionConnectComplete:(BBOBEXClientSession *)session 
								   error:(OBEXError)error
								 fromWho:(NSData *)who
							connectionID:(long)connectionID
{	
	// Log the result
	if (error == kOBEXSuccess) {
		[self log:@"Connection success!"];
		[connectOrDisconnectButton setTitle:@"Disconnect"];
		
	} else {
		[self log:[NSString stringWithFormat:@"Connection failed (error=%d): %@", 
            error, [myOBEXClient lastErrorDescription]]];
	}
	
	[connectOrDisconnectButton setEnabled:YES];
}

// Called when the disconnection process is complete.
- (void)obexClientSessionDisconnectComplete:(BBOBEXClientSession *)session 
									  error:(OBEXError)error
{
	// Log the result	
	if (error == kOBEXSuccess) {
		[self log:@"Disconnection success!"];
		[connectOrDisconnectButton setTitle:@"Connect"];
		
		// should disconnect the transport connection now that the OBEX session
		// is disconnected.
		OBEXError status = [myOBEXClient close];
		[self log:[NSString stringWithFormat:@"Transport disconnection result: %d", status]];
		
	} else {
		[self log:[NSString stringWithFormat:@"Disconnection failed (error=%d): %@", 
            error, [myOBEXClient lastErrorDescription]]];
	}	
	
	[connectOrDisconnectButton setEnabled:YES];
}

- (void)obexClientSessionGetProgress:(BBOBEXClientSession *)session 
							byAmount:(int)bytesSent
{
	[self log:[NSString stringWithFormat:@"GET received data: %d", bytesSent]];
}

- (void)obexClientSessionGetComplete:(BBOBEXClientSession *)session 
							   error:(OBEXError)error
{
	if (error == kOBEXSuccess) {
		[self log:@"GET success!"];
	} else {
		[self log:@"GET error!"];		
	}
}



// Called each time data is sent for a PUT operation.
- (void)obexClientSessionPutProgress:(BBOBEXClientSession *)session 
							byAmount:(int)bytesSent
{
	// Log the result
	[self log:[NSString stringWithFormat:@"Sent another %d bytes", bytesSent]];
	
	// update the progress bar
	[sendFileProgressBar incrementBy:bytesSent];
}

// Called when a PUT process is complete.
- (void)obexClientSessionPutComplete:(BBOBEXClientSession *)session 
							   error:(OBEXError)error
{
	// Log the result
	if (error == kOBEXSuccess) {
		[self log:@"PUT success!"];
		[sendFileProgressBar setDoubleValue:[sendFileProgressBar maxValue]];
	} else {
		[self log:[NSString stringWithFormat:@"PUT failed (error=%d): %@", 
            error, [myOBEXClient lastErrorDescription]]];
	}	

	[sendFileButton setEnabled:YES];
	[abortButton setEnabled:NO];
}

- (void)obexClientSessionAbortComplete:(BBOBEXClientSession *)session 
								 error:(OBEXError)error
{
	// Log the result
	if (error == kOBEXSuccess) {
		[self log:@"ABORT success!"];
		[sendFileButton setEnabled:YES];
	} else {
		[self log:[NSString stringWithFormat:@"ABORT failed (error=%d): %@", 
            error, [myOBEXClient lastErrorDescription]]];
		[abortButton setEnabled:YES];
	}
}


#pragma mark -
#pragma mark Less important methods

- (IBAction)chooseFile:(id)sender
{
	NSOpenPanel *openPanel = [NSOpenPanel openPanel];
	if ([openPanel runModalForTypes:nil] == NSOKButton) {
		[filePathField setStringValue:[[openPanel filenames] objectAtIndex:0]];
		[sendFileButton setEnabled:YES];
	}
}

- (int)getFileSize:(NSString *)filePath
{
	NSNumber *fileSize;
	NSDictionary *attrs;
	if (attrs = [[NSFileManager defaultManager] fileAttributesAtPath:filePath
														traverseLink:YES]) {
		if (fileSize = [attrs objectForKey:NSFileSize])
			return [fileSize intValue];
	}	
	return -1;
}

- (void)resetUI
{
	[connectOrDisconnectButton setEnabled:YES];	
	[connectOrDisconnectButton setTitle:@"Connect"];
	[sendFileButton setEnabled:NO];
	[abortButton setEnabled:NO];
	[sendFileProgressBar setDoubleValue:0.0];
}

- (void)controlTextDidChange:(NSNotification *)aNotification
{
	[sendFileButton setEnabled:![[filePathField stringValue] isEqual:@""]];
}

- (void)windowWillClose:(NSNotification *)aNotification
{
	// Make sure everything is disconnected before shutting down this app.
	[myOBEXClient close];
	[self deviceDisconnect:nil];
}

- (void)log:(NSString *)logText
{
	[logTextView insertText:logText];
	[logTextView insertText:@"\n"];	
}


#pragma mark -

- (void)dealloc
{
	[myOBEXClient release];
	
	[super dealloc];
}

@end

/*
	// To connect for a File Transfer session instead of an Object-Push type session, you can 
	// do something like this instead when connecting:
	
	unsigned char rawUUID[] =	// F9EC7BC4-953C-11D2-984E-525400DC9E09 (File Transfer Profile target UUID)
    {
        0xf9, 0xec, 0x7b, 0xc4, 0x95, 0x3c, 0x11, 0xd2,
        0x98, 0x4e, 0x52, 0x54, 0x00, 0xdc, 0x9e, 0x09
    };
	
	IOBluetoothSDPUUID *uuid = [IOBluetoothSDPUUID uuidWithBytes:rawUUID length:16];
	NSData *d = [NSData dataWithBytes:[uuid bytes] length:[uuid length]];
	
	// start the connection process
	OBEXError status = [myOBEXClient connectToDeviceWithAddress:[addressField stringValue]
												withChannelID:[portField intValue]
													 toTarget:d];
    
    // (You can also use the setWhoHeaderValue method in BBOBEXClientSession
    // to set the value of the "Who" header for the connect request.)
 
 
 
    // Then once you have a connection set up (e.g. in the 
    // obexClientSessionConnectComplete callback), you can do something like 
    // this to get a file listing from the other device:
    [[NSFileManager defaultManager] createFileAtPath:@"fileListing.txt"
                                            contents:nil 
                                          attributes:nil];		
    NSFileHandle *file = [NSFileHandle fileHandleForWritingAtPath:@"fileListing.txt"];
    if (file) {
        NSLog(@"sending GET");
        [myOBEXClient getWithName:nil type:@"x-obex/folder-listing" writeToFile:file];
    } 
 
 
    // See the File Transfer Profile and Object Push Profile specs at
    // www.bluetooth.org for more information.
*/
