/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/
#include "pixlabel.h"
#include "globals.h"

#include <assert.h>

#include <QFileDialog>
#include <QImageReader>
#include <QBuffer>

#include "picturescrollarea.h"

PixLabel::PixLabel(QString label, QWidget* parent)
 : QLabel(label, parent),
  m_filename(""),
  m_lastdir ("/"),
  m_label   (label),
  m_readOnly(false)
{
  m_pixmap      = new QPixmap();
  m_menu        = new QMenu  (this);
  m_cancelAct   = new QAction(this);
  m_fullSizeAct = new QAction(this);
  m_fullPicture = new QLabel(0);
  m_scrollPic   = new PictureScrollArea;

  m_menu->addAction(m_cancelAct);
  m_menu->addAction(m_fullSizeAct);

  connect(this         , SIGNAL(clicked  (    )), this, SLOT(slot_choosePictureDlg()));
  connect(m_cancelAct  , SIGNAL(triggered(bool)), this, SLOT(slot_cancelPicture   ()));
  connect(m_fullSizeAct, SIGNAL(triggered(bool)), this, SLOT(slot_showFullSize    ()));

  setScaledContents (true);
  setAlignment(Qt::AlignCenter);
  setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);

  retranslate();
}

void PixLabel::retranslate()
{
  m_cancelAct  ->setText(tr("Cancel Picture"));
  m_fullSizeAct->setText(tr("Full Size"));
}

//\*****************************************************************************
void PixLabel::slot_choosePictureDlg()
{
  if(m_readOnly)
    return;

  qDebug("[PixLabel::slot_choosePictureDlg]: Old_Filename = %s", m_filename.toLocal8Bit().data());

  QString new_filename;
  QString image_formats = "";
  QList<QByteArray> list = QImageReader::supportedImageFormats ();

  for(int i = 0; i < list.size(); i++)
    image_formats += QString("*.") + QString().fromUtf8(list[i].data()) + " ";

  new_filename = QFileDialog::getOpenFileName(this, tr("Choose Picture"), m_lastdir, image_formats, 0, QFileDialog::DontResolveSymlinks);

  if(!new_filename.isEmpty())
  {
    if(!m_pixmap)
    {
      m_pixmap = new QPixmap();
      assert(NULL != m_pixmap);
    }

    if(m_pixmap->load(new_filename))
    {
      setScaledContents(false);
      setPixmap(m_pixmap->scaled(width(), height(), Qt::KeepAspectRatio));
      m_filename = new_filename;
      emit changed();
    }
  }

  if(!new_filename.isEmpty())
    m_lastdir = new_filename;

  qDebug("[PixLabel::slot_choosePictureDlg]: New_Filename = %s", m_filename.toLocal8Bit().data());
}
//\*****************************************************************************
void PixLabel::freePixmap()
{
  setText(m_label);
  *m_pixmap = QPixmap();
}
//\*****************************************************************************
void PixLabel::slot_cancelPicture   ()
{
  if(!m_readOnly)
  {
    m_filename = "";
    freePixmap();
    emit changed();
  }
}
//\*****************************************************************************
void PixLabel::slot_setPixmap(const QPixmap & pix)
{
  if(!m_pixmap)
  {
    m_pixmap = new QPixmap(pix);
    assert(NULL != m_pixmap);
  }
  else
    *m_pixmap = pix;

  if(m_pixmap->isNull())
  {
    qDebug("[PixLabel::slot_setPixmap]: pixmap is null");
    setText(m_label);
  }
  else
  {
    setScaledContents(false);
    setPixmap(m_pixmap->scaled(width(), height(), Qt::KeepAspectRatio));
    qDebug("[PixLabel::slot_setPixmap]: pixmap is NOT null");
  }
}
//\*****************************************************************************
void PixLabel::slot_setPixmap(const QPixmap* pix)
{
  if(pix)
  {
    if(!m_pixmap)
    {
      m_pixmap = new QPixmap(*pix);
      assert(NULL != m_pixmap);
    }
    else
      *m_pixmap = *pix;

    if(m_pixmap->isNull())
      setText(m_label);
    else
    {
      setScaledContents(false);
      setPixmap(m_pixmap->scaled(width(), height(), Qt::KeepAspectRatio));
    }
  }
}
//\*****************************************************************************
void PixLabel::slot_showFullSize    ()
{
  if(!m_pixmap->isNull())
  {
    m_fullPicture->setWindowTitle(tr("Full Size of Picture"));

    m_fullPicture->setPixmap(*m_pixmap);
    m_fullPicture->setScaledContents(true);

    m_scrollPic->setWidget((QWidget*)m_fullPicture);
    m_scrollPic->setWidgetResizable(true);

    // FIXME nado chto-to bolee adekvatnoe, chem +4 :)
    m_scrollPic->setMaximumWidth (m_pixmap->width()  + 4);
    m_scrollPic->setMaximumHeight(m_pixmap->height() + 4);

    m_scrollPic->resize(800, 600);
    m_scrollPic->move(x(), y());

    m_scrollPic->show();
  }
}
//\*****************************************************************************
