/***************************************************************************
 *   Copyright (C) 2007 by Κωνσταντίνος Σμάνης   *
 *   kon.smanis@gmail.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation                                              *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301 USA                                             *
 ***************************************************************************/

#include "filetransactions.h"

bool FileTransactions::definePermissions( const QString menulst, const QString devicemap, const QString mtab )
{
	QFileInfo menulst_check( menulst );
	QFileInfo devicemap_check( devicemap );
	QFileInfo mtab_check( mtab );

	if ( ( menulst_check.isReadable() && menulst_check.isWritable() ) && ( devicemap_check.isReadable() && devicemap_check.isWritable() ) && ( mtab_check.isReadable() && mtab_check.isWritable() ) )
		return true;
	else
		return false;
}

void FileTransactions::menuInput( const QString path, GRUBSettings *settings, QVector<GRUBItem> *items )
{
	GRUBItem tmp_item;
	settings->clear();
	items->clear();

	QRegExp space( "\\s+" );
	QString line;
	QFile file( path );

	if ( file.open( QIODevice::ReadOnly ) )
	{
		QTextStream stream( &file );
		while ( !stream.atEnd() )
		{
			line=stream.readLine().trimmed();
			//GRUBSettings identification
			if ( line.startsWith( "splashimage", Qt::CaseInsensitive ) )
			{
				SPLASHIMAGE:
				settings->setSplashImage( line.section( "=", 1 ) );
				continue;
			}
			if (line.startsWith( "default", Qt::CaseInsensitive ) )
			{
				DEFAULT:
				( line.section( space, 1 ) == "saved" ? settings->setDefault( -2 ) : settings->setDefault( line.section( space, 1 ).toInt() ) );
				continue;
			}
			if (line.startsWith( "fallback", Qt::CaseInsensitive ) )
			{
				FALLBACK:
				settings->setFallback( line.section( space, 1 ).toInt() );
				continue;
			}
			if ( line.startsWith( "timeout", Qt::CaseInsensitive ) )
			{
				TIMEOUT:
				settings->setTimeout( line.section( space, 1 ).toInt() );
				continue;
			}
			if ( line.startsWith( "hiddenmenu", Qt::CaseInsensitive ) )
			{
				HIDDENMENU:
				settings->setHiddenMenu( true );
				continue;
			}
			if ( line.startsWith( "color", Qt::CaseInsensitive ) )
			{
				COLOR:
				settings->setColor( line.section( space, 1 ) );
				continue;
			}
			if ( line.startsWith( "password", Qt::CaseInsensitive ) )
			{
				PASSWORD:
				settings->setPassword( line.section( space, 1 ) );
				continue;
			}
			//GRUBItem identified
			if ( line.startsWith( "title", Qt::CaseInsensitive ) )
			{
				TITLE:
				tmp_item.setTitle( line.section( space, 1 ) );

				while ( !stream.atEnd() )
				{
					line=stream.readLine().trimmed();
					if ( line.startsWith( "title", Qt::CaseInsensitive ) )
						break;
					if ( line.startsWith( "splashimage", Qt::CaseInsensitive ) )
						break;
					if (line.startsWith( "default", Qt::CaseInsensitive ) )
						break;
					if ( line.startsWith( "fallback", Qt::CaseInsensitive ) )
						break;
					if ( line.startsWith( "timeout", Qt::CaseInsensitive ) )
						break;
					if ( line.startsWith( "hiddenmenu", Qt::CaseInsensitive ) )
						break;
					if ( line.startsWith( "color", Qt::CaseInsensitive ) )
						break;
					if ( line.startsWith( "password", Qt::CaseInsensitive ) )
						break;

					if ( line.startsWith( "root", Qt::CaseInsensitive ) )
					{
						tmp_item.setRoot( line.section( space, 1 ) );
						continue;
					}

					if ( line.startsWith( "kernel", Qt::CaseInsensitive ) )
					{
						tmp_item.setKernel( line.section( space, 1 ) );
						continue;
					}

					if ( line.startsWith( "initrd", Qt::CaseInsensitive ) )
					{
						tmp_item.setInitrd( line.section( space, 1 ) );
						continue;
					}

					if ( line.startsWith( "chainloader", Qt::CaseInsensitive ) )
					{
						tmp_item.setChainLoader( line.section( space, 1 ) );
						continue;
					}

					if ( line.startsWith( "quiet", Qt::CaseInsensitive ) )
					{
						tmp_item.setQuiet( true );
						continue;
					}

					if ( line.startsWith( "savedefault", Qt::CaseInsensitive ) )
					{
						tmp_item.setSaveDefault( true );
						continue;
					}

					if ( line.startsWith( "makeactive", Qt::CaseInsensitive ) )
					{
						tmp_item.setMakeActive( true );
						continue;
					}
	
					if ( line.startsWith( "lock", Qt::CaseInsensitive ) )
					{
						tmp_item.setLock( true );
						continue;
					}
				}

				items->append( tmp_item );
				tmp_item.clear();

				if ( line.startsWith( "title", Qt::CaseInsensitive ) )
					goto TITLE;
				if ( line.startsWith( "splashimage", Qt::CaseInsensitive ) )
					goto SPLASHIMAGE;
				if (line.startsWith( "default", Qt::CaseInsensitive ) )
					goto DEFAULT;
				if (line.startsWith( "fallback", Qt::CaseInsensitive ) )
					goto FALLBACK;
				if ( line.startsWith( "timeout", Qt::CaseInsensitive ) )
					goto TIMEOUT;
				if ( line.startsWith( "hiddenmenu", Qt::CaseInsensitive ) )
					goto HIDDENMENU;
				if ( line.startsWith( "color", Qt::CaseInsensitive ) )
					goto COLOR;
				if ( line.startsWith( "password", Qt::CaseInsensitive ) )
					goto PASSWORD;
			}
		}
		file.close();
	}
}
void FileTransactions::menuOutput( const QString path, const GRUBSettings *settings, const QVector<GRUBItem> *items )
{
	QFile file( path );
	if ( file.open( QIODevice::WriteOnly ) )
	{
		QTextStream stream( &file );
	
		/* GRUBSETTINGS */
		if ( !settings->splashImage().isEmpty() )
			stream << "splashimage=" << settings->splashImage() << endl;
		if ( settings->default_entry() != -1 )
			stream << "default " << ( settings->default_entry() != -2 ? QString().setNum( settings->default_entry() ) : "saved" ) << endl;
		if ( settings->fallback() != -1 )
			stream << "fallback " << settings->fallback() << endl;
		if ( settings->timeout() != -1 )
			stream << "timeout " << settings->timeout() << endl;
		if ( settings->hiddenMenu() )
			stream << "hiddenmenu" << endl;
		if ( !settings->color().isEmpty() )
			stream << "color " << settings->color() << endl;
		if ( !settings->password().isEmpty() )
			stream << "password " << settings->password() << endl;

		/* GRUBITEMS */
		for ( int i = 0; i < items->size(); i++ )
		{
				stream << endl;

				stream << "title " << items->at( i ).title() << endl;

			if ( items->at( i ).lock() )
				stream << "lock" << endl;

				stream << "root " << items->at( i ).root() << endl;

			if ( !items->at( i ).kernel().isEmpty() )
				stream << "kernel " << items->at( i ).kernel() << endl;

			if ( !items->at( i ).initrd().isEmpty() )
				stream << "initrd " << items->at( i ).initrd() << endl;

			if (!items->at( i ).chainLoader().isEmpty())
				stream << "chainloader " << items->at( i ).chainLoader() << endl;

			if ( items->at( i ).quiet() )
				stream << "quiet" << endl;

			if ( items->at( i ).saveDefault() )
				stream << "savedefault" << endl;

			if ( items->at( i ).makeActive() )
				stream << "makeactive" << endl;
		}
		file.close();
	}
}

void FileTransactions::mtabInput( const QString path, QVector<Device> *devices )
{
	devices->clear();

	QRegExp space( "\\s+" );
	QString line;
	QFile file( path );

	if ( file.open( QIODevice::ReadOnly ) )
	{
		QTextStream stream( &file );
		while ( !stream.atEnd() )
		{
			line = stream.readLine();
			if ( line.startsWith( "/dev" ) )
			{
				devices->append( Device( line.section( space, 0, 0 ), line.section( space, 0, 0 ).remove( QRegExp( "\\d" ) ), line.section( space, 1, 1 ) ) );
			}
		}
		file.close();
	}
}
void FileTransactions::deviceInput( const QString path, QVector<Device> *devices )
{
	QRegExp space( "\\s+" );
	QString line;
	QFile file( path );

	if ( file.open( QIODevice::ReadOnly ) )
	{
		QTextStream stream( &file );
		while ( !stream.atEnd() )
		{
			line = stream.readLine();
			for ( int i = 0 ; i < devices->size() ; i++ )
			{
				if ( devices->at( i ).device() == line.section( space, 1, 1 ) )
				{
					(*devices)[i].setGRUBDevice( line.section( space, 0, 0 ) );
					(*devices)[i].setGRUBPartition( (*devices)[i].grubDevice().replace( QString( ")" ), QString( "," ) + QString().setNum( (*devices)[i].partition().right( 1 ).toInt() - 1 ) + QString( ")" ) ) );
				}
			}
		}
		file.close();
	}
//
	QProcess uuid;
	QString output;

	uuid.start( "ls" , QStringList() << "/dev/disk/by-uuid/" << "-l" );
	uuid.waitForFinished();

	output = uuid.readAllStandardOutput();
	for ( int j = 1 ; j < output.count( "\n" ) ; j++ )
	{
		for ( int i = 0 ; i < devices->size() ; i++ )
		{
			if ( devices->at( i ).partition() == output.section( "\n", j, j ).section( space, 9, 9 ).remove( "../.." ).prepend( "/dev" ) )
			{
				(*devices)[i].setUuid( output.section( "\n", j, j ).section( space, 7, 7 ) );
			}
		}
	}
}
QString FileTransactions::view( const QString path )
{
	QString data;
	QFile file( path );
	if ( file.open( QIODevice::ReadOnly ) )
	{
		QTextStream stream( &file );
		data = stream.readAll();

		file.close();
	}
	return data;
}

void FileTransactions::backup( const QString source, const QString target )
{
	if ( QFile::exists( target ) )
	{
		QMessageBox msgBox( QMessageBox::Question, QObject::tr( "File already exists!" ), QString( QObject::tr( "%1 already exists! Should I overwrite it?" ) ).arg( target ), QMessageBox::Yes | QMessageBox::No, 0, Qt::Dialog );
		switch ( msgBox.exec() )
		{
			case QMessageBox::Yes:
				if ( QFile::remove( target  ) )
				{
					if ( QFile::copy( source, target ) )
					{
						QMessageBox::information( 0, QObject::tr( "Success!" ), QString( QObject::tr( "Operation was successfully completed! %1 has been created." ) ).arg( target ), QMessageBox::Ok );
					}
					else
					{
						QMessageBox::warning( 0, QObject::tr( "Failure!" ), QObject::tr( "An error occured." ), QMessageBox::Ok );
					}
				}
				else
				{
					QMessageBox::warning( 0, QObject::tr( "Failure!" ), QObject::tr( "An error occured." ), QMessageBox::Ok );
				}
			break;
			case QMessageBox::No:
				return;
			break;
		}
	}
	else
	{
		if ( QFile::copy( source, target ) )
		{
			QMessageBox::information( 0, QObject::tr( "Success!" ), QString( QObject::tr( "Operation was successfully completed! %1 has been created." ) ).arg( target ), QMessageBox::Ok );
		}
		else
		{
			QMessageBox::warning( 0, QObject::tr( "Failure!" ), QObject::tr( "An error occured." ), QMessageBox::Ok );
		}
	}
}
void FileTransactions::backup_original( const QString source )
{
	QString target = source + "_original";
	if ( QFile::exists( target ) )
	{
		if ( QFile::remove( target  ) )
		{
			QFile::copy( source, target );
		}
	}
	else
	{
		QFile::copy( source, target );
	}
}
void FileTransactions::restore( const QString source, const QString target)
{
	if ( QFile::remove( target ) )
	{
		if ( QFile::copy( source, target ) )
		{
			QMessageBox::information( 0, QObject::tr( "Success!" ), QString( QObject::tr( "Operation was successfully completed! %1 has been restored to %2." ) ).arg( source ).arg( target ), QMessageBox::Ok );
		}
		else
		{
			QMessageBox::warning( 0, QObject::tr( "Failure!" ), QObject::tr( "An error occured." ), QMessageBox::Ok );
		}
	}
	else
	{
		QMessageBox::warning( 0, QObject::tr( "Failure!" ), QObject::tr( "An error occured." ), QMessageBox::Ok );
	}
}
void FileTransactions::deleteBackup( const QString backup )
{
	if ( QFile::remove( backup ) )
	{
		QMessageBox::information( 0, QObject::tr( "Success!" ), QString( QObject::tr( "Operation was successfully completed! %1 has been removed." ) ).arg( backup ), QMessageBox::Ok );
	}
	else
	{
		QMessageBox::warning( 0, QObject::tr( "Failure!" ), QObject::tr( "An error occured." ), QMessageBox::Ok );
	}
}

void FileTransactions::addEntry( const QString path, const GRUBItem grubitem, const GRUBSettings *settings, QVector<GRUBItem> *items )
{
	items->append( grubitem );
	menuOutput( path, settings, items );
}
void FileTransactions::editEntry( const QString path, const int index, const GRUBItem grubitem, const GRUBSettings *settings, QVector<GRUBItem> *items )
{
	items->replace( index, grubitem );
	menuOutput( path, settings, items );
}
void FileTransactions::copyEntry( const QString path, const int index, const GRUBSettings *settings, QVector<GRUBItem> *items )
{
	items->append( (*items)[index] );
	menuOutput( path, settings, items );
}
void FileTransactions::removeEntry( const QString path, const int index, const GRUBSettings *settings, QVector<GRUBItem> *items )
{
	items->remove( index );
	menuOutput( path, settings, items );
}

void FileTransactions::move( const QString path, const int source, const int target, const GRUBSettings *settings, QVector<GRUBItem> *items )
{
	if ( source != target )
	{
		GRUBItem tmp_item = (*items)[source];
		items->remove( source );
		items->insert( target, tmp_item );
		menuOutput( path, settings, items );
	}
}
