/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/
#include "CToolViewWpt.h"
#include "CGarminDBWpt.h"
#include "CCentralResources.h"
#include "icons.h"
#include "CDlgDeleteWpt.h"

#include <QtGui>

CToolViewWpt::CToolViewWpt(CGarminDBWpt* db, QWidget * parent)
: QWidget(parent)
{
    setupUi(this);
    setObjectName("Waypoints");
    connect(listWaypoints,SIGNAL(itemDoubleClicked(QTreeWidgetItem*,int) ),this,SLOT(slotItemDoubleClicked(QTreeWidgetItem*)));
    connect(listWaypoints,SIGNAL(customContextMenuRequested(const QPoint&)),this,SLOT(slotContextMenu(const QPoint&)));

    contextMenu = new QMenu(this);
    contextMenu->addAction(QPixmap(iconEditWpt16x16),tr("Edit..."),this,SLOT(slotEdit()));
    contextMenu->addAction(QPixmap(":/icons/iconProximity16x16"),tr("Proximity ..."),this,SLOT(slotGrpEditProxy()));
    contextMenu->addAction(QPixmap(iconTrash16x16),tr("Delete"),this,SLOT(slotDelete()));
    contextMenu->addAction(QPixmap(iconTrash16x16),tr("Delete by ..."),this,SLOT(slotDeleteBy()));

    connect(db,SIGNAL(sigWaypointListChanged()), this, SLOT(slotUpdateWptList()));
}


CToolViewWpt::~CToolViewWpt()
{
}


void CToolViewWpt::slotUpdateWptList()
{
    listWaypoints->clear();

    const QMap<QString,CGarminWpt*>& waypoints = gpResources->wptdb().getWaypoints();
    QMap<QString,CGarminWpt*>::const_iterator wpt = waypoints.begin();

    QTreeWidgetItem * selected = 0;

    while(wpt != waypoints.end()) {
        QTreeWidgetItem * entry = new QTreeWidgetItem(listWaypoints);
        entry->setText(0,(*wpt)->ident);
        entry->setData(0,Qt::UserRole,wpt.key());
        entry->setIcon(0,(*wpt)->icon);
        entry->setText(1,wpt.key().section('|',0,0));
        entry->setData(1,Qt::UserRole,(*wpt)->smbl);
        entry->setText(2,(*wpt)->comment);
        if((*wpt)->isSelected()) {
            selected = entry;
            entry->setSelected((*wpt)->isSelected());
        }
        ++wpt;
    }

    listWaypoints->header()->setResizeMode(0,QHeaderView::Interactive);
    listWaypoints->resizeColumnToContents(0);
    listWaypoints->header()->setResizeMode(1,QHeaderView::Interactive);
    listWaypoints->resizeColumnToContents(1);

    if(selected) {
        listWaypoints->scrollToItem(selected);
    }
}


void CToolViewWpt::slotItemDoubleClicked(QTreeWidgetItem * item)
{
    QString key = item->data(0,Qt::UserRole).toString();
    const CGarminWpt * wpt = gpResources->wptdb().getWaypoints()[key];
    gpResources->canvas().move(wpt->lon,wpt->lat,"");
}


void CToolViewWpt::keyPressEvent(QKeyEvent * e)
{
    if(e->key() == Qt::Key_Delete) {
        QStringList keys;
        QString key;
        QTreeWidgetItem * item;
        const QList<QTreeWidgetItem*>& items = listWaypoints->selectedItems();
        foreach(item,items) {
            keys << item->data(0,Qt::UserRole).toString();
        }
        gpResources->wptdb().removeWaypoint(keys);

    }
    else {
        QWidget::keyPressEvent(e);
    }
}


void CToolViewWpt::slotContextMenu(const QPoint& pos)
{
    if(listWaypoints->currentItem()) {
        QPoint p = listWaypoints->mapToGlobal(pos);
        p.setY(p.y() + listWaypoints->header()->height());
        contextMenu->exec(p);
    }
}


void CToolViewWpt::slotEdit()
{
    gpResources->wptdb().editWaypoint(listWaypoints->currentItem()->data(0,Qt::UserRole).toString());
}


void CToolViewWpt::slotDelete()
{
    QStringList keys;
    QTreeWidgetItem * item;
    const QList<QTreeWidgetItem*>& items = listWaypoints->selectedItems();
    foreach(item,items) {
        keys << item->data(0,Qt::UserRole).toString();
        delete item;
    }
    gpResources->wptdb().removeWaypoint(keys);
}


void CToolViewWpt::slotDeleteBy()
{
    CDlgDeleteWpt dlg(*this);
    dlg.exec();
}


void CToolViewWpt::slotGrpEditProxy()
{
    bool ok = false;
    QString unit = gpResources->doMetric() ? tr("Distance [m]") : tr("Distance [ft]");
    double dist = QInputDialog::getDouble(0,tr("Proximity distance ..."),unit,0,0,2147483647,0,&ok);
    if(ok) {
        if(!gpResources->doMetric()) {
            dist = dist * 0.3048;
        }

        QStringList keys;
        QTreeWidgetItem * item;
        const QList<QTreeWidgetItem*>& items = listWaypoints->selectedItems();
        foreach(item,items) {
            keys << item->data(0,Qt::UserRole).toString();
        }
        gpResources->wptdb().setProxyDistance(keys,(dist == 0 ? 1e25f : dist));
    }
}
