#include "general_menu.h"
#include <stdlib.h>
#include <QFileDialog>
#include <QFontDialog>
#include <QMessageBox>
#include <QColorDialog>
#include <QMap>

inline QString CHECK_TO_STRING(Qt::CheckState s)
{
  return s?"true":"false";
}

inline Qt::CheckState CHECK_FROM_STRING(const QString &s)
{
  if( s.isEmpty() || s == "false" )
    return Qt::Unchecked;
  else
    return Qt::Checked;
}

General_Menu::General_Menu(QWidget *parent):QDialog(parent)
{
	ui.setupUi(this);
	
	ui.use_external_editor->setCheckState(CHECK_FROM_STRING(get_config("external_editor")));
	ui.editor_lineEdit->setText(get_config("editor"));
	
	ui.help_path_lineEdit->setText(get_config("help_path"));
	
	if(!get_config("terminal_font").isEmpty()) config_font.fromString(get_config("terminal_font"));
	
	set_font_label();
	
	foreground_color.setNamedColor(get_config("terminal_foreground_color"));
	background_color.setNamedColor(get_config("terminal_background_color"));
	error_color.setNamedColor(get_config("terminal_error_color"));
	
	set_color_label();
	
	ui.max_line_num_lineEdit->setText(get_config("lines_in_terminal"));
	ui.max_col_in_terminal_lineEdit->setText(get_config("cols_in_terminal"));
	
	connect(ui.editor_select_button,SIGNAL(clicked()),this,SLOT(editor_select_button_callback()));
	connect(ui.help_path_select_pushButton,SIGNAL(clicked()),this,SLOT(help_path_select_pushButton_callback()));
	connect(ui.font_button,SIGNAL(clicked()),this,SLOT(font_button_callback()));
	connect(ui.foreground_button,SIGNAL(clicked()),this,SLOT(foreground_button_callback()));
	connect(ui.background_button,SIGNAL(clicked()),this,SLOT(background_button_callback()));
	connect(ui.error_button,SIGNAL(clicked()),this,SLOT(error_button_callback()));
}

void General_Menu::background_button_callback()
{
	QColor color=QColorDialog::getColor(foreground_color,this);
	
	if(color.isValid())
	{
		background_color=color;
		set_color_label();
	}
}

void General_Menu::error_button_callback()
{
	QColor color=QColorDialog::getColor(error_color,this);
	
	if(color.isValid())
	{
		error_color=color;
		set_color_label();
	}
}

void General_Menu::foreground_button_callback()
{
	QColor color=QColorDialog::getColor(foreground_color,this);
	
	if(color.isValid())
	{
		foreground_color=color;
		set_color_label();
	}
}

void General_Menu::set_color_label()
{
	QPalette p(ui.color_label->palette());
	p.setColor(QPalette::ButtonText,foreground_color);
	p.setColor(QPalette::Button,background_color);
	ui.color_label->setPalette(p);
	
	p=(ui.error_example_label->palette());
	p.setColor(QPalette::ButtonText,error_color);
	p.setColor(QPalette::Button,background_color);
	ui.error_example_label->setPalette(p);
}

void General_Menu::set_font_label()
{
	ui.font_label->setFont(config_font);
	ui.color_label->setFont(config_font);
	ui.error_example_label->setFont(config_font);
	
	QString font;
	
	font+=config_font.family()+" "+QString().setNum(config_font.pointSize())+" ";
	int weight=config_font.weight();
	switch(weight)
	{
		case QFont::Light:
			font+="Light"; break;
		case QFont::Normal:
			font+="Normal"; break;
		case QFont::DemiBold:
			font+="DemiBold"; break;
		case QFont::Bold:
			font+="Bold"; break;
		case QFont::Black:
			font+="Black"; break;
	};
	
	if(config_font.italic())
		font+=" Italic";
	
	ui.font_label->setText(font);
	
}

void General_Menu::font_button_callback()
{
	bool ok;
	QFont font = QFontDialog::getFont(&ok, config_font, this);
	if (ok)
	{
		config_font=font;
		set_font_label();
	}
}

void General_Menu::editor_select_button_callback()
{
	QString file = QFileDialog::getOpenFileName(this,"Choose a file", ".","*");
	
	if(file.isEmpty()) return;
	
	file.replace(" ","\\ ");
	
	ui.editor_lineEdit->setText(file);
}

void General_Menu::help_path_select_pushButton_callback()
{
	QString file = QFileDialog::getOpenFileName(this,"Choose a file", ".","*");
	
	if(file.isEmpty()) return;
	
	file.replace(" ","\\ ");
	
	ui.help_path_lineEdit->setText(file);
}


void General_Menu::apply_config()
{
	QMap<QString,QString> config;
	config["external_editor"] = CHECK_TO_STRING(ui.use_external_editor->checkState());
	config["editor"]=ui.editor_lineEdit->text();
	config["help_path"]=ui.help_path_lineEdit->text();
	config["terminal_font"]=config_font.toString();
	config["terminal_foreground_color"]=foreground_color.name();
	config["terminal_background_color"]=background_color.name();
	config["terminal_error_color"]=error_color.name();
	config["lines_in_terminal"]=ui.max_line_num_lineEdit->text();
	config["cols_in_terminal"]=ui.max_col_in_terminal_lineEdit->text();
	
	
	set_config(config);

	QMessageBox::information(this, "Note","Reload QtOctave to apply changes.");
}
