/*
 * HBT_X.core.cpp - device implementations for HBT_X module
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "HBT_X.analogfunction.h"

#include "logging.h"
#include "complex.h"
#include "matrix.h"
#include "object.h"
#include "node.h"
#include "circuit.h"
#include "net.h"
#include "component_id.h"
#include "constants.h"
#include "device.h"
#include "HBT_X.core.h"

#ifndef CIR_HBT_X
#define CIR_HBT_X -1
#endif

// external nodes
#define c 0
#define b 1
#define e 2
#define t 3
// internal nodes
#define ei 4
#define bi 5
#define bii 6
#define ci 7
#define ex 8
#define exx 9
#define cx 10
#define ni 11
#define nii 12
#define niix 13
#define niii 14
#define niiix 15
#define niv 16
#define nivx 17
#define ti 18

// useful macro definitions
#define NP(node) real (getV (node))
#define BP(pnode,nnode) (NP(pnode) - NP(nnode))
#define _load_static_residual2(pnode,nnode,current)\
	_rhs[pnode] -= current;\
	_rhs[nnode] += current;
#define _load_static_augmented_residual2(pnode,nnode,current)\
	_rhs[pnode] -= current;\
	_rhs[nnode] += current;
#define _load_static_residual1(node,current)\
	_rhs[node] -= current;
#define _load_static_augmented_residual1(node,current)\
	_rhs[node] -= current;
#define _load_static_jacobian4(pnode,nnode,vpnode,vnnode,conductance)\
	_jstat[pnode][vpnode] += conductance;\
	_jstat[nnode][vnnode] += conductance;\
	_jstat[pnode][vnnode] -= conductance;\
	_jstat[nnode][vpnode] -= conductance;\
	if (doHB) {\
	_ghs[pnode] += conductance * BP(vpnode,vnnode);\
	_ghs[nnode] -= conductance * BP(vpnode,vnnode);\
        } else {\
	_rhs[pnode] += conductance * BP(vpnode,vnnode);\
	_rhs[nnode] -= conductance * BP(vpnode,vnnode);\
	}
#define _load_static_jacobian2p(node,vpnode,vnnode,conductance)\
	_jstat[node][vpnode] += conductance;\
	_jstat[node][vnnode] -= conductance;\
	if (doHB) {\
        _ghs[node] += conductance * BP(vpnode,vnnode);\
        } else {\
        _rhs[node] += conductance * BP(vpnode,vnnode);\
	}
#define _load_static_jacobian2s(pnode,nnode,node,conductance)\
	_jstat[pnode][node] += conductance;\
	_jstat[nnode][node] -= conductance;\
	if (doHB) {\
	_ghs[pnode] += conductance * NP(node);\
	_ghs[nnode] -= conductance * NP(node);\
        } else {\
	_rhs[pnode] += conductance * NP(node);\
	_rhs[nnode] -= conductance * NP(node);\
	}
#define _load_static_jacobian1(node,vnode,conductance)\
	_jstat[node][vnode] += conductance;\
	if (doHB) {\
	_ghs[node] += conductance * NP(vnode);\
        } else {\
	_rhs[node] += conductance * NP(vnode);\
	}
#define _load_dynamic_residual2(pnode,nnode,charge)\
	if (doTR) _charges[pnode][nnode] += charge;\
	if (doHB) {\
	_qhs[pnode] -= charge;\
	_qhs[nnode] += charge;\
	}
#define _load_dynamic_residual1(node,charge)\
	if (doTR) _charges[node][node] += charge;\
	if (doHB) {\
	_qhs[node] -= charge;\
	}
#define _load_dynamic_jacobian4(pnode,nnode,vpnode,vnnode,capacitance)\
	if (doAC) {\
	_jdyna[pnode][vpnode] += capacitance;\
	_jdyna[nnode][vnnode] += capacitance;\
	_jdyna[pnode][vnnode] -= capacitance;\
	_jdyna[nnode][vpnode] -= capacitance;\
	}\
        if (doTR) {\
        _caps[pnode][nnode][vpnode][vnnode] += capacitance;\
  	}\
	if (doHB) {\
	_chs[pnode] += capacitance * BP(vpnode,vnnode);\
	_chs[nnode] -= capacitance * BP(vpnode,vnnode);\
	}
#define _load_dynamic_jacobian2s(pnode,nnode,vnode,capacitance)\
	if (doAC) {\
	_jdyna[pnode][vnode] += capacitance;\
	_jdyna[nnode][vnode] -= capacitance;\
	}\
	if (doTR) {\
	_caps[pnode][nnode][vnode][vnode] += capacitance;\
	}\
	if (doHB) {\
	_chs[pnode] += capacitance * NP(vnode);\
	_chs[nnode] -= capacitance * NP(vnode);\
	}
#define _load_dynamic_jacobian2p(node,vpnode,vnnode,capacitance)\
	if (doAC) {\
	_jdyna[node][vpnode] += capacitance;\
        _jdyna[node][vnnode] -= capacitance;\
        }\
	if (doTR) {\
        _caps[node][node][vpnode][vnnode] += capacitance;\
	}\
	if (doHB) {\
	_chs[node] += capacitance * BP(vpnode,vnnode);\
	}
#define _load_dynamic_jacobian1(node,vnode,capacitance)\
	if (doAC) {\
	_jdyna[node][vnode] += capacitance;\
	}\
	if (doTR) {\
	_caps[node][node][vnode][vnode] += capacitance;\
	}\
	if (doHB) {\
	_chs[node] += capacitance * NP(vnode);\
	}

#define _save_whitenoise1(n1,pwr,type)\
	_white_pwr[n1][n1] += pwr;
#define _save_whitenoise2(n1,n2,pwr,type)\
	_white_pwr[n1][n2] += pwr;
#define _save_flickernoise1(n1,pwr,exp,type)\
	_flicker_pwr[n1][n1] += pwr;\
	_flicker_exp[n1][n1] += exp;
#define _save_flickernoise2(n1,n2,pwr,exp,type)\
	_flicker_pwr[n1][n2] += pwr;\
	_flicker_exp[n1][n2] += exp;
#define _load_whitenoise2(n1,n2,pwr)\
	cy (n1,n2) -= pwr/kB/T0; cy (n2,n1) -= pwr/kB/T0;\
	cy (n1,n1) += pwr/kB/T0; cy (n2,n2) += pwr/kB/T0;
#define _load_whitenoise1(n1,pwr)\
	cy (n1,n1) += pwr/kB/T0;
#define _load_flickernoise2(n1,n2,pwr,exp)\
	cy (n1,n2) -= pwr*pow(_freq,-exp)/kB/T0;\
	cy (n2,n1) -= pwr*pow(_freq,-exp)/kB/T0;\
	cy (n1,n1) += pwr*pow(_freq,-exp)/kB/T0;\
	cy (n2,n2) += pwr*pow(_freq,-exp)/kB/T0;
#define _load_flickernoise1(n1,pwr,exp)\
	cy (n1,n1) += pwr*pow(_freq,-exp)/kB/T0;

// derivative helper macros
#define m00_hypot(v00,x,y)      v00 = xhypot(x,y);
#define m10_hypot(v10,v00,x,y)  v10 = (x)/(v00);
#define m11_hypot(v11,v00,x,y)  v11 = (y)/(v00);
#define m00_max(v00,x,y)        v00 = ((x)>(y))?(x):(y);
#define m10_max(v10,v00,x,y)    v10 = ((x)>(y))?1.0:0.0;
#define m11_max(v11,v00,x,y)    v11 = ((x)>(y))?0.0:1.0;
#define m00_min(v00,x,y)        v00 = ((x)<(y))?(x):(y);
#define m10_min(v10,v00,x,y)    v10 = ((x)<(y))?1.0:0.0;
#define m11_min(v11,v00,x,y)    v11 = ((x)<(y))?0.0:1.0;
#define m00_pow(v00,x,y)        v00 = pow(x,y);
#define m10_pow(v10,v00,x,y)    v10 = (x==0.0)?0.0:(v00)*(y)/(x);
#define m11_pow(v11,v00,x,y)    v11 = (x==0.0)?0.0:(log(x)*(v00));

#define m00_div(v00,v10,x,y)    double v10=1/(y); double v00=(x)*v10;
#define m10_div(v10,v00,vv,x,y)
#define m11_div(v11,v00,vv,x,y) double v11 = -v00*vv;

#define m00_mult(v00,v10,v11,x,y) double v10=(x); double v11=(y); double v00=v10*v11;
#define m00_add(v00,x,y)        double v00=(x)+(y);

#define m00_cos(v00,x)          v00 = cos(x);
#define m10_cos(v10,v00,x)      v10 = (-sin(x));
#define m00_sin(v00,x)          v00 = sin(x);
#define m10_sin(v10,v00,x)      v10 = (cos(x));
#define m00_tan(v00,x)          v00 = tan(x);
#define m10_tan(v10,v00,x)      v10 = (1.0/cos(x)/cos(x));
#define m00_cosh(v00,x)         v00 = cosh(x);
#define m10_cosh(v10,v00,x)     v10 = (sinh(x));
#define m00_sinh(v00,x)         v00 = sinh(x);
#define m10_sinh(v10,v00,x)     v10 = (cosh(x));
#define m00_tanh(v00,x)         v00 = tanh(x);
#define m10_tanh(v10,v00,x)     v10 = (1.0/cosh(x)/cosh(x));
#define m00_acos(v00,x)         v00 = acos(x);
#define m10_acos(v10,v00,x)     v10 = (-1.0/sqrt(1-x*x));
#define m00_asin(v00,x)         v00 = asin(x);
#define m10_asin(v10,v00,x)     v10 = (+1.0/sqrt(1-x*x));
#define m00_atan(v00,x)         v00 = atan(x);
#define m10_atan(v10,v00,x)     v10 = (+1.0/(1+x*x));
#define m00_logE(v00,x)         v00 = log(x);
#define m10_logE(v10,v00,x)     v10 = (1.0/x);
#define m00_log10(v00,x)        v00 = log10(x);
#define m10_log10(v10,v00,x)    v10 = (1.0/x/M_LN10);
#define m00_sqrt(v00,x)         v00 = sqrt(x);
#define m10_sqrt(v10,v00,x)     v10 = (0.5/v00);
#define m00_fabs(v00,x)         v00 = fabs(x);
#define m10_fabs(v10,v00,x)     v10 = (((x)>=0)?(+1.0):(-1.0));

#define m00_exp(v00,x)          v00 = exp(x);
#define m10_exp(v10,v00,x)      v10 = v00;

#define m00_abs(v00)            ((v00)<(0)?(-(v00)):(v00))
#define m00_floor(v00,x)        v00 = floor(x);
#define m00_limexp(v00,x)       v00 = ((x)<80.0?exp(x):exp(80.0)*(x-79.0));
#define m10_limexp(v10,v00,x)   v10 = ((x)<80.0?(v00):exp(80.0));

#define m20_logE(v00)           (-1.0/v00/v00)
#define m20_exp(v00)            exp(v00)
#define m20_limexp(v00)         ((v00)<80.0?exp(v00):0.0)
#define m20_sqrt(v00)           (-0.25/(v00)/sqrt(v00))
#define m20_fabs(v00)           0.0

#define m00_vt(x)               (kBoverQ*(x))
#define m10_vt(x)               (kBoverQ)

// $vt and $vt() functions 
#define _vt_nom                 (kBoverQ)


// simulator specific definitions
#define _modelname              "HBT_X"
#define _instancename           getName()
#define _circuit_temp	        (getPropertyDouble ("Temp")+273.15)

using namespace device;

/* Device constructor. */
HBT_X::HBT_X() : circuit (19)
{
  type = CIR_HBT_X;
}

/* Initialization of model. */
void HBT_X::initModel (void)
{
  // create internal nodes
  setNode (ei, createInternal (getName (), "ei"));
  setNode (bi, createInternal (getName (), "bi"));
  setNode (bii, createInternal (getName (), "bii"));
  setNode (ci, createInternal (getName (), "ci"));
  setNode (ex, createInternal (getName (), "ex"));
  setNode (exx, createInternal (getName (), "exx"));
  setNode (cx, createInternal (getName (), "cx"));
  setNode (ni, createInternal (getName (), "ni"));
  setNode (nii, createInternal (getName (), "nii"));
  setNode (niix, createInternal (getName (), "niix"));
  setNode (niii, createInternal (getName (), "niii"));
  setNode (niiix, createInternal (getName (), "niiix"));
  setNode (niv, createInternal (getName (), "niv"));
  setNode (nivx, createInternal (getName (), "nivx"));
  setNode (ti, createInternal (getName (), "ti"));
}

/* Initialization of DC analysis. */
void HBT_X::initDC (void)
{
  allocMatrixMNA ();
  initModel ();
  pol = 1;
  restartDC ();
  doAC = 1;
  doTR = 0;
  doHB = 0;
}

/* Run when DC is restarted (fallback algorithms). */
void HBT_X::restartDC (void)
{
}

/* Initialize Verilog-AMS code. */
void HBT_X::initVerilog (void)
{
  // initialization of noise variables
  _white_pwr[nii][nii] = 0.0;
  _white_pwr[cx][bii] = 0.0;
  _white_pwr[exx][bii] = 0.0;
  _white_pwr[ex][bii] = 0.0;
  _white_pwr[c][ci] = 0.0;
  _white_pwr[e][ei] = 0.0;
  _white_pwr[b][bi] = 0.0;
  _white_pwr[bii][bi] = 0.0;
  _white_pwr[ni][ni] = 0.0;
  _white_pwr[bii][ei] = 0.0;
  _flicker_pwr[niv][niv] = 0.0;
  _flicker_exp[niv][niv] = 0.0;
  _flicker_pwr[niii][niii] = 0.0;
  _flicker_exp[niii][niii] = 0.0;

  int i1, i2, i3, i4;

  // zero charges
  for (i1 = 0; i1 < 19; i1++) {
  for (i2 = 0; i2 < 19; i2++) {
    _charges[i1][i2] = 0.0;
  } }

  // zero capacitances
  for (i1 = 0; i1 < 19; i1++) {
  for (i2 = 0; i2 < 19; i2++) {
  for (i3 = 0; i3 < 19; i3++) {
  for (i4 = 0; i4 < 19; i4++) {
    _caps[i1][i2][i3][i4] = 0.0;
  } } } }

  // zero right hand side, static and dynamic jacobian
  for (i1 = 0; i1 < 19; i1++) {
    _rhs[i1] = 0.0;
    _qhs[i1] = 0.0;
    _chs[i1] = 0.0;
    _ghs[i1] = 0.0;
    for (i2 = 0; i2 < 19; i2++) {
      _jstat[i1][i2] = 0.0;
      _jdyna[i1][i2] = 0.0;
    }
  }
}

/* Evaluate Verilog-AMS equations. */
void HBT_X::calcVerilog (void)
{
  // get device model parameters
  int Mode = getPropertyInteger ("Mode");
  int Noise = getPropertyInteger ("Noise");
  int Debug = getPropertyInteger ("Debug");
  int DebugPlus = getPropertyInteger ("DebugPlus");
  nr_double_t Temp = getPropertyDouble ("Temp");
  nr_double_t Rth = getPropertyDouble ("Rth");
  nr_double_t Cth = getPropertyDouble ("Cth");
  int N = getPropertyInteger ("N");
  nr_double_t L = getPropertyDouble ("L");
  nr_double_t W = getPropertyDouble ("W");
  nr_double_t Jsf = getPropertyDouble ("Jsf");
  nr_double_t nf = getPropertyDouble ("nf");
  nr_double_t Vg = getPropertyDouble ("Vg");
  nr_double_t Jse = getPropertyDouble ("Jse");
  nr_double_t ne = getPropertyDouble ("ne");
  nr_double_t Rbxx = getPropertyDouble ("Rbxx");
  nr_double_t Vgb = getPropertyDouble ("Vgb");
  nr_double_t Jsee = getPropertyDouble ("Jsee");
  nr_double_t nee = getPropertyDouble ("nee");
  nr_double_t Rbbxx = getPropertyDouble ("Rbbxx");
  nr_double_t Vgbb = getPropertyDouble ("Vgbb");
  nr_double_t Jsr = getPropertyDouble ("Jsr");
  nr_double_t nr = getPropertyDouble ("nr");
  nr_double_t Vgr = getPropertyDouble ("Vgr");
  nr_double_t XCjc = getPropertyDouble ("XCjc");
  nr_double_t Jsc = getPropertyDouble ("Jsc");
  nr_double_t nc = getPropertyDouble ("nc");
  nr_double_t Rcxx = getPropertyDouble ("Rcxx");
  nr_double_t Vgc = getPropertyDouble ("Vgc");
  nr_double_t Bf = getPropertyDouble ("Bf");
  nr_double_t kBeta = getPropertyDouble ("kBeta");
  nr_double_t Br = getPropertyDouble ("Br");
  nr_double_t VAF = getPropertyDouble ("VAF");
  nr_double_t VAR = getPropertyDouble ("VAR");
  nr_double_t IKF = getPropertyDouble ("IKF");
  nr_double_t IKR = getPropertyDouble ("IKR");
  nr_double_t Mc = getPropertyDouble ("Mc");
  nr_double_t BVceo = getPropertyDouble ("BVceo");
  nr_double_t kc = getPropertyDouble ("kc");
  nr_double_t BVebo = getPropertyDouble ("BVebo");
  nr_double_t Tr = getPropertyDouble ("Tr");
  nr_double_t Trx = getPropertyDouble ("Trx");
  nr_double_t Tf = getPropertyDouble ("Tf");
  nr_double_t Tft = getPropertyDouble ("Tft");
  nr_double_t Thcs = getPropertyDouble ("Thcs");
  nr_double_t Ahc = getPropertyDouble ("Ahc");
  nr_double_t Cje = getPropertyDouble ("Cje");
  nr_double_t mje = getPropertyDouble ("mje");
  nr_double_t Vje = getPropertyDouble ("Vje");
  nr_double_t Cjc = getPropertyDouble ("Cjc");
  nr_double_t mjc = getPropertyDouble ("mjc");
  nr_double_t Vjc = getPropertyDouble ("Vjc");
  nr_double_t kjc = getPropertyDouble ("kjc");
  nr_double_t Cmin = getPropertyDouble ("Cmin");
  nr_double_t J0 = getPropertyDouble ("J0");
  nr_double_t XJ0 = getPropertyDouble ("XJ0");
  nr_double_t Rci0 = getPropertyDouble ("Rci0");
  nr_double_t Jk = getPropertyDouble ("Jk");
  nr_double_t RJk = getPropertyDouble ("RJk");
  nr_double_t Vces = getPropertyDouble ("Vces");
  nr_double_t Rc = getPropertyDouble ("Rc");
  nr_double_t Re = getPropertyDouble ("Re");
  nr_double_t Rb = getPropertyDouble ("Rb");
  nr_double_t Rb2 = getPropertyDouble ("Rb2");
  nr_double_t Lc = getPropertyDouble ("Lc");
  nr_double_t Le = getPropertyDouble ("Le");
  nr_double_t Lb = getPropertyDouble ("Lb");
  nr_double_t Cq = getPropertyDouble ("Cq");
  nr_double_t Cpb = getPropertyDouble ("Cpb");
  nr_double_t Cpc = getPropertyDouble ("Cpc");
  nr_double_t Kfb = getPropertyDouble ("Kfb");
  nr_double_t Afb = getPropertyDouble ("Afb");
  nr_double_t Ffeb = getPropertyDouble ("Ffeb");
  nr_double_t Kb = getPropertyDouble ("Kb");
  nr_double_t Ab = getPropertyDouble ("Ab");
  nr_double_t Fb = getPropertyDouble ("Fb");
  nr_double_t Kfe = getPropertyDouble ("Kfe");
  nr_double_t Afe = getPropertyDouble ("Afe");
  nr_double_t Ffee = getPropertyDouble ("Ffee");
  nr_double_t Tnom = getPropertyDouble ("Tnom");


/* ----------------- evaluate verilog analog equations -------------------- */

/* #define's for translated code */
#undef  _DDT
#define _DDT(q) q
#define _DYNAMIC
#define _DERIVATE
#define _DDX
#define _DERIVATEFORDDX

double Inivx;
#if defined(_DERIVATE)
double Inivx_Vbii_ei;
double Inivx_Tempti_GND;
double Inivx_Vbii_ci;
#endif
double Iniiix;
#if defined(_DERIVATE)
double Iniiix_Vbii_ei;
double Iniiix_Tempti_GND;
double Iniiix_Vbii_ci;
#endif
double Iniix;
#if defined(_DERIVATE)
double Iniix_Vbii_ei;
double Iniix_Tempti_GND;
double Iniix_Vbii_ci;
#endif
#if defined(_DYNAMIC)
double qbe;
#if defined(_DERIVATE)
double qbe_Vbii_ei;
double qbe_Tempti_GND;
double qbe_Vbii_ci;
double qbe_Vci_ei;
#endif
#endif
#if defined(_DYNAMIC)
double qbtf;
#if defined(_DERIVATE)
double qbtf_Tempti_GND;
double qbtf_Vbii_ei;
double qbtf_Vbii_ci;
#endif
#endif
#if defined(_DYNAMIC)
double xtff;
#if defined(_DERIVATE)
double xtff_Vbii_ei;
double xtff_Tempti_GND;
double xtff_Vbii_ci;
double xtff_Vci_ei;
#endif
#endif
#if defined(_DYNAMIC)
double Wh;
#if defined(_DERIVATE)
double Wh_Vci_ei;
double Wh_Vbii_ei;
double Wh_Tempti_GND;
double Wh_Vbii_ci;
#endif
#endif
#if defined(_DYNAMIC)
double Ih;
#if defined(_DERIVATE)
double Ih_Vci_ei;
double Ih_Vbii_ei;
double Ih_Tempti_GND;
double Ih_Vbii_ci;
#endif
#endif
#if defined(_DYNAMIC)
double AHC;
#endif
#if defined(_DYNAMIC)
double RCIO;
#endif
#if defined(_DYNAMIC)
double InvVpt;
#endif
#if defined(_DYNAMIC)
double Vlim;
#endif
#if defined(_DYNAMIC)
double qb2;
#if defined(_DERIVATE)
double qb2_Vbii_ei;
double qb2_Tempti_GND;
double qb2_Vbii_ci;
#endif
#endif
#if defined(_DYNAMIC)
double qbtra;
#if defined(_DERIVATE)
double qbtra_Vbi_ci;
double qbtra_Tempti_GND;
#endif
#endif
#if defined(_DYNAMIC)
double qbtr;
#if defined(_DERIVATE)
double qbtr_Vbii_ci;
double qbtr_Tempti_GND;
double qbtr_Vbii_ei;
#endif
#endif
#if defined(_DYNAMIC)
double qb1;
#if defined(_DERIVATE)
double qb1_Vbi_ci;
#endif
#endif
#if defined(_DYNAMIC)
double xix;
#if defined(_DERIVATE)
double xix_Vbii_ei;
double xix_Tempti_GND;
double xix_Vbii_ci;
#endif
#endif
#if defined(_DYNAMIC)
double qb2med;
#if defined(_DERIVATE)
double qb2med_Vbii_ei;
double qb2med_Tempti_GND;
double qb2med_Vbii_ci;
#endif
#endif
#if defined(_DYNAMIC)
double I00;
#endif
double Ipdiss;
#if defined(_DERIVATE)
double Ipdiss_Vbii_ei;
double Ipdiss_Tempti_GND;
double Ipdiss_Vbii_ci;
double Ipdiss_Vci_ei;
double Ipdiss_Vbi_ci;
#endif
double Icdx;
#if defined(_DERIVATE)
double Icdx_Vcx_ci;
double Icdx_Tempti_GND;
#endif
double Ibdxx;
#if defined(_DERIVATE)
double Ibdxx_Vexx_ei;
double Ibdxx_Tempti_GND;
#endif
double Ibdx;
#if defined(_DERIVATE)
double Ibdx_Vex_ei;
double Ibdx_Tempti_GND;
#endif
double Ib1;
#if defined(_DERIVATE)
double Ib1_Vbii_ei;
double Ib1_Tempti_GND;
double Ib1_Vbii_ci;
#endif
double Ib0;
#if defined(_DERIVATE)
double Ib0_Vbii_ei;
double Ib0_Tempti_GND;
double Ib0_Vbii_ci;
#endif
double EdBeta;
#if defined(_DERIVATE)
double EdBeta_Tempti_GND;
double EdBeta_Vbii_ci;
#endif
double mm;
#if defined(_DERIVATE)
double mm_Vbii_ci;
#endif
double Vbclin;
double epsi;
double Ibx;
#if defined(_DERIVATE)
double Ibx_Vbi_ci;
double Ibx_Tempti_GND;
#endif
double Ib2;
#if defined(_DERIVATE)
double Ib2_Vbii_ci;
double Ib2_Tempti_GND;
#endif
double Ic1;
#if defined(_DERIVATE)
double Ic1_Vbii_ei;
double Ic1_Tempti_GND;
double Ic1_Vbii_ci;
#endif
double Ic1r;
#if defined(_DERIVATE)
double Ic1r_Vbii_ci;
double Ic1r_Tempti_GND;
double Ic1r_Vbii_ei;
#endif
double Ic0;
#if defined(_DERIVATE)
double Ic0_Vbii_ei;
double Ic0_Tempti_GND;
double Ic0_Vbii_ci;
#endif
double qb;
#if defined(_DERIVATE)
double qb_Vbii_ei;
double qb_Vbii_ci;
double qb_Tempti_GND;
#endif
double q2;
#if defined(_DERIVATE)
double q2_Vbii_ei;
double q2_Tempti_GND;
double q2_Vbii_ci;
#endif
double q1;
#if defined(_DERIVATE)
double q1_Vbii_ei;
double q1_Vbii_ci;
#endif
double Ic1ra;
#if defined(_DERIVATE)
double Ic1ra_Vbii_ci;
double Ic1ra_Tempti_GND;
#endif
double Ic0a;
#if defined(_DERIVATE)
double Ic0a_Vbii_ei;
double Ic0a_Tempti_GND;
#endif
double TWO_Q;
double FOUR_K;
double Area;
double Tex;
#if defined(_DERIVATE)
double Tex_Tempti_GND;
#endif
double TjK;
#if defined(_DERIVATE)
double TjK_Tempti_GND;
#endif
double Tj;
#if defined(_DERIVATE)
double Tj_Tempti_GND;
#endif
double Texi;
#if defined(_DERIVATE)
double Texi_Tempti_GND;
#endif
double vcei;
#if defined(_DERIVATE)
double vcei_Vci_ei;
#endif
double vxxe;
#if defined(_DERIVATE)
double vxxe_Vexx_ei;
#endif
double vxc;
#if defined(_DERIVATE)
double vxc_Vcx_ci;
#endif
double vxe;
#if defined(_DERIVATE)
double vxe_Vex_ei;
#endif
double vbei;
#if defined(_DERIVATE)
double vbei_Vbii_ei;
#endif
double vbci;
#if defined(_DERIVATE)
double vbci_Vbii_ci;
#endif
double vbcx;
#if defined(_DERIVATE)
double vbcx_Vbi_ci;
#endif
#if defined(_DERIVATE)
vbcx_Vbi_ci=1.0;
#endif
vbcx=BP(bi,ci);
#if defined(_DERIVATE)
vbci_Vbii_ci=1.0;
#endif
vbci=BP(bii,ci);
#if defined(_DERIVATE)
vbei_Vbii_ei=1.0;
#endif
vbei=BP(bii,ei);
#if defined(_DERIVATE)
vxe_Vex_ei=1.0;
#endif
vxe=BP(ex,ei);
#if defined(_DERIVATE)
vxc_Vcx_ci=1.0;
#endif
vxc=BP(cx,ci);
#if defined(_DERIVATE)
vxxe_Vexx_ei=1.0;
#endif
vxxe=BP(exx,ei);
#if defined(_DERIVATE)
vcei_Vci_ei=1.0;
#endif
vcei=BP(ci,ei);
#if defined(_DERIVATE)
Texi_Tempti_GND=1.0;
#endif
Texi=NP(ti);
#if defined(_DERIVATE)
Tj_Tempti_GND=Texi_Tempti_GND;
#endif
Tj=(Texi+Temp);
#if defined(_DERIVATE)
TjK_Tempti_GND=Tj_Tempti_GND;
#endif
TjK=(Tj+273.15);
#if defined(_DERIVATE)
Tex_Tempti_GND=Tj_Tempti_GND;
#endif
Tex=(Tj-Tnom);
Area=(((L*W)*1.0e12)*N);
FOUR_K=(4*1.3806226e-23);
TWO_Q=(2*1.6021918e-19);
#if defined(_DERIVATE)
Ic0a_Vbii_ei=HBT_X_d_diode(vbei,Jsf,Vg,nf,Area,Tj,Tnom,vbei_Vbii_ei,0.0,0.0,0.0,0.0,0.0,0.0);
Ic0a_Tempti_GND=HBT_X_d_diode(vbei,Jsf,Vg,nf,Area,Tj,Tnom,0.0,0.0,0.0,0.0,0.0,Tj_Tempti_GND,0.0);
#endif
Ic0a=HBT_X_diode(vbei,Jsf,Vg,nf,Area,Tj,Tnom);
#if defined(_DERIVATE)
Ic1ra_Vbii_ci=HBT_X_d_diode(vbci,(XCjc*Jsr),Vgr,nr,Area,Tj,Tnom,vbci_Vbii_ci,0.0,0.0,0.0,0.0,0.0,0.0);
Ic1ra_Tempti_GND=HBT_X_d_diode(vbci,(XCjc*Jsr),Vgr,nr,Area,Tj,Tnom,0.0,0.0,0.0,0.0,0.0,Tj_Tempti_GND,0.0);
#endif
Ic1ra=HBT_X_diode(vbci,(XCjc*Jsr),Vgr,nr,Area,Tj,Tnom);
if
(((VAF>0.0)&&(VAR>0.0)))
{
#if defined(_DERIVATE)
q1_Vbii_ei=(((HBT_X_d_charge(vbei,1.0,Vje,mje,1.0,vbei_Vbii_ei,0.0,0.0,0.0,0.0)-HBT_X_d_charge(0.0,1.0,Vje,mje,1.0,0.0,0.0,0.0,0.0,0.0))/VAR)+((HBT_X_d_charge(vbci,1.0,Vjc,mjc,1.0,0.0,0.0,0.0,0.0,0.0)-HBT_X_d_charge(0.0,1.0,Vjc,mjc,1.0,0.0,0.0,0.0,0.0,0.0))/VAF));
q1_Vbii_ci=(((HBT_X_d_charge(vbei,1.0,Vje,mje,1.0,0.0,0.0,0.0,0.0,0.0)-HBT_X_d_charge(0.0,1.0,Vje,mje,1.0,0.0,0.0,0.0,0.0,0.0))/VAR)+((HBT_X_d_charge(vbci,1.0,Vjc,mjc,1.0,vbci_Vbii_ci,0.0,0.0,0.0,0.0)-HBT_X_d_charge(0.0,1.0,Vjc,mjc,1.0,0.0,0.0,0.0,0.0,0.0))/VAF));
#endif
q1=((1.0+((HBT_X_charge(vbei,1.0,Vje,mje,1.0)-HBT_X_charge(0.0,1.0,Vje,mje,1.0))/VAR))+((HBT_X_charge(vbci,1.0,Vjc,mjc,1.0)-HBT_X_charge(0.0,1.0,Vjc,mjc,1.0))/VAF));
}
else
{
if
(((VAF>0.0)&&(VAR==0.0)))
{
#if defined(_DERIVATE)
q1_Vbii_ei=0.0;
q1_Vbii_ci=((HBT_X_d_charge(vbci,1.0,Vjc,mjc,1.0,vbci_Vbii_ci,0.0,0.0,0.0,0.0)-HBT_X_d_charge(0.0,1.0,Vjc,mjc,1.0,0.0,0.0,0.0,0.0,0.0))/VAF);
#endif
q1=(1.0+((HBT_X_charge(vbci,1.0,Vjc,mjc,1.0)-HBT_X_charge(0.0,1.0,Vjc,mjc,1.0))/VAF));
}
else
{
if
(((VAF==0.0)&&(VAR>0.0)))
{
#if defined(_DERIVATE)
q1_Vbii_ei=((HBT_X_d_charge(vbei,1.0,Vje,mje,1.0,vbei_Vbii_ei,0.0,0.0,0.0,0.0)-HBT_X_d_charge(0.0,1.0,Vje,mje,1.0,0.0,0.0,0.0,0.0,0.0))/VAR);
q1_Vbii_ci=0.0;
#endif
q1=(1.0+((HBT_X_charge(vbei,1.0,Vje,mje,1.0)-HBT_X_charge(0.0,1.0,Vje,mje,1.0))/VAR));
}
else
{
#if defined(_DERIVATE)
q1_Vbii_ei=0.0;
q1_Vbii_ci=0.0;
#endif
q1=1.0;
}
}
}
if
(((IKF>0.0)&&(IKR>0.0)))
{
#if defined(_DERIVATE)
q2_Vbii_ei=(Ic0a_Vbii_ei/(Area*IKF));
q2_Tempti_GND=((Ic0a_Tempti_GND/(Area*IKF))+(Ic1ra_Tempti_GND/(Area*IKR)));
q2_Vbii_ci=(Ic1ra_Vbii_ci/(Area*IKR));
#endif
q2=((Ic0a/(Area*IKF))+(Ic1ra/(Area*IKR)));
}
else
{
if
(((IKF>0.0)&&(IKR==0.0)))
{
#if defined(_DERIVATE)
q2_Vbii_ei=(Ic0a_Vbii_ei/(Area*IKF));
q2_Tempti_GND=(Ic0a_Tempti_GND/(Area*IKF));
q2_Vbii_ci=0.0;
#endif
q2=(Ic0a/(Area*IKF));
}
else
{
if
(((IKF==0.0)&&(IKR>0.0)))
{
#if defined(_DERIVATE)
q2_Vbii_ei=0.0;
q2_Tempti_GND=(Ic1ra_Tempti_GND/(Area*IKR));
q2_Vbii_ci=(Ic1ra_Vbii_ci/(Area*IKR));
#endif
q2=(Ic1ra/(Area*IKR));
}
else
{
#if defined(_DERIVATE)
q2_Vbii_ei=0.0;
q2_Tempti_GND=0.0;
q2_Vbii_ci=0.0;
#endif
q2=0.0;
}
}
}
{
double m00_sqrt(d00_sqrt0,((q1*q1)+(4.0*q2)))
#if defined(_DERIVATE)
double m10_sqrt(d10_sqrt0,d00_sqrt0,((q1*q1)+(4.0*q2)))
#endif
#if defined(_DERIVATE)
qb_Vbii_ei=((q1_Vbii_ei+(((q1_Vbii_ei*q1)+(q1*q1_Vbii_ei))+(4.0*q2_Vbii_ei))*d10_sqrt0)/2.0);
qb_Vbii_ci=((q1_Vbii_ci+(((q1_Vbii_ci*q1)+(q1*q1_Vbii_ci))+(4.0*q2_Vbii_ci))*d10_sqrt0)/2.0);
qb_Tempti_GND=((4.0*q2_Tempti_GND)*d10_sqrt0/2.0);
#endif
qb=((q1+d00_sqrt0)/2.0);
}
#if defined(_DERIVATE)
Ic0_Vbii_ei=((Ic0a_Vbii_ei*qb-Ic0a*qb_Vbii_ei)/qb/qb);
Ic0_Tempti_GND=((Ic0a_Tempti_GND*qb-Ic0a*qb_Tempti_GND)/qb/qb);
Ic0_Vbii_ci=(-Ic0a*qb_Vbii_ci/qb/qb);
#endif
Ic0=(Ic0a/qb);
#if defined(_DERIVATE)
Ic1r_Vbii_ci=((Ic1ra_Vbii_ci*qb-Ic1ra*qb_Vbii_ci)/qb/qb);
Ic1r_Tempti_GND=((Ic1ra_Tempti_GND*qb-Ic1ra*qb_Tempti_GND)/qb/qb);
Ic1r_Vbii_ei=(-Ic1ra*qb_Vbii_ei/qb/qb);
#endif
Ic1r=(Ic1ra/qb);
#if defined(_DERIVATE)
Ic1_Vbii_ei=(Ic0_Vbii_ei-Ic1r_Vbii_ei);
Ic1_Tempti_GND=(Ic0_Tempti_GND-Ic1r_Tempti_GND);
Ic1_Vbii_ci=(Ic0_Vbii_ci-Ic1r_Vbii_ci);
#endif
Ic1=(Ic0-Ic1r);
#if defined(_DERIVATE)
Ib2_Vbii_ci=(HBT_X_d_diode(vbci,(XCjc*Jsr),Vgr,nr,Area,Tj,Tnom,vbci_Vbii_ci,0.0,0.0,0.0,0.0,0.0,0.0)/Br);
Ib2_Tempti_GND=(HBT_X_d_diode(vbci,(XCjc*Jsr),Vgr,nr,Area,Tj,Tnom,0.0,0.0,0.0,0.0,0.0,Tj_Tempti_GND,0.0)/Br);
#endif
Ib2=(HBT_X_diode(vbci,(XCjc*Jsr),Vgr,nr,Area,Tj,Tnom)/Br);
#if defined(_DERIVATE)
Ibx_Vbi_ci=(HBT_X_d_diode(vbcx,((1.0-XCjc)*Jsr),Vgr,nr,Area,Tj,Tnom,vbcx_Vbi_ci,0.0,0.0,0.0,0.0,0.0,0.0)/Br);
Ibx_Tempti_GND=(HBT_X_d_diode(vbcx,((1.0-XCjc)*Jsr),Vgr,nr,Area,Tj,Tnom,0.0,0.0,0.0,0.0,0.0,Tj_Tempti_GND,0.0)/Br);
#endif
Ibx=(HBT_X_diode(vbcx,((1.0-XCjc)*Jsr),Vgr,nr,Area,Tj,Tnom)/Br);
epsi=1.0e-6;
{
double m00_pow(d00_pow0,(1.0-epsi),(1/Mc))
Vbclin=(BVceo*d00_pow0);
}
#if defined(_DERIVATE)
mm_Vbii_ci=HBT_X_d_MM(vbci,BVceo,Mc,Vbclin,Bf,kc,vbci_Vbii_ci,0.0,0.0,0.0,0.0,0.0);
#endif
mm=HBT_X_MM(vbci,BVceo,Mc,Vbclin,Bf,kc);
if
((mm>1.0))
{
if
((kBeta>0.0))
{
if
(((Bf-(kBeta*Tex))>1e-6))
{
#if defined(_DERIVATE)
EdBeta_Tempti_GND=((-1*(-(kBeta*Tex_Tempti_GND))/(Bf-(kBeta*Tex))/(Bf-(kBeta*Tex)))/((kc*(mm-1))+1));
EdBeta_Vbii_ci=(((-(kc*mm_Vbii_ci))*((kc*(mm-1))+1)-((1/(Bf-(kBeta*Tex)))-(kc*(mm-1)))*(kc*mm_Vbii_ci))/((kc*(mm-1))+1)/((kc*(mm-1))+1));
#endif
EdBeta=(((1/(Bf-(kBeta*Tex)))-(kc*(mm-1)))/((kc*(mm-1))+1));
}
else
{
#if defined(_DERIVATE)
EdBeta_Tempti_GND=0.0;
EdBeta_Vbii_ci=(((-(kc*mm_Vbii_ci))*((kc*(mm-1))+1)-(1e6-(kc*(mm-1)))*(kc*mm_Vbii_ci))/((kc*(mm-1))+1)/((kc*(mm-1))+1));
#endif
EdBeta=((1e6-(kc*(mm-1)))/((kc*(mm-1))+1));
}
}
else
{
#if defined(_DERIVATE)
EdBeta_Tempti_GND=0.0;
EdBeta_Vbii_ci=(((-(kc*mm_Vbii_ci))*((kc*(mm-1))+1)-((1/Bf)-(kc*(mm-1)))*(kc*mm_Vbii_ci))/((kc*(mm-1))+1)/((kc*(mm-1))+1));
#endif
EdBeta=(((1/Bf)-(kc*(mm-1)))/((kc*(mm-1))+1));
}
}
else
{
if
((kBeta>0.0))
{
if
(((Bf-(kBeta*Tex))>1e-6))
{
#if defined(_DERIVATE)
EdBeta_Tempti_GND=(-1*(-(kBeta*Tex_Tempti_GND))/(Bf-(kBeta*Tex))/(Bf-(kBeta*Tex)));
EdBeta_Vbii_ci=0.0;
#endif
EdBeta=(1/(Bf-(kBeta*Tex)));
}
else
{
#if defined(_DERIVATE)
EdBeta_Tempti_GND=0.0;
EdBeta_Vbii_ci=0.0;
#endif
EdBeta=1e6;
}
}
else
{
#if defined(_DERIVATE)
EdBeta_Tempti_GND=0.0;
EdBeta_Vbii_ci=0.0;
#endif
EdBeta=(1/Bf);
}
}
#if defined(_DERIVATE)
Ib0_Vbii_ei=Ic0a_Vbii_ei*EdBeta;
Ib0_Tempti_GND=((Ic0a_Tempti_GND*EdBeta)+(Ic0a*EdBeta_Tempti_GND));
Ib0_Vbii_ci=(Ic0a*EdBeta_Vbii_ci);
#endif
Ib0=(Ic0a*EdBeta);
if
((BVebo>0))
{
#if defined(_DERIVATE)
Ib1_Vbii_ei=(Ib0_Vbii_ei-HBT_X_d_diode(((-BVebo)-vbei),Jsf,0.0,1.0,Area,0.0,0.0,(-vbei_Vbii_ei),0.0,0.0,0.0,0.0,0.0,0.0));
Ib1_Tempti_GND=(Ib0_Tempti_GND-HBT_X_d_diode(((-BVebo)-vbei),Jsf,0.0,1.0,Area,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0));
Ib1_Vbii_ci=(Ib0_Vbii_ci-HBT_X_d_diode(((-BVebo)-vbei),Jsf,0.0,1.0,Area,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0));
#endif
Ib1=(Ib0-HBT_X_diode(((-BVebo)-vbei),Jsf,0.0,1.0,Area,0.0,0.0));
}
else
{
#if defined(_DERIVATE)
Ib1_Vbii_ei=Ib0_Vbii_ei;
Ib1_Tempti_GND=Ib0_Tempti_GND;
Ib1_Vbii_ci=Ib0_Vbii_ci;
#endif
Ib1=Ib0;
}
if
(((Jse>0.0)&&(ne>0)))
{
#if defined(_DERIVATE)
Ibdx_Vex_ei=HBT_X_d_diode(vxe,Jse,Vgb,ne,Area,Tj,Tnom,vxe_Vex_ei,0.0,0.0,0.0,0.0,0.0,0.0);
Ibdx_Tempti_GND=HBT_X_d_diode(vxe,Jse,Vgb,ne,Area,Tj,Tnom,0.0,0.0,0.0,0.0,0.0,Tj_Tempti_GND,0.0);
#endif
Ibdx=HBT_X_diode(vxe,Jse,Vgb,ne,Area,Tj,Tnom);
}
else
{
#if defined(_DERIVATE)
Ibdx_Vex_ei=vxe_Vex_ei*1e-12;
Ibdx_Tempti_GND=0.0;
#endif
Ibdx=(vxe*1e-12);
}
if
(((Jsee>0.0)&&(nee>0)))
{
#if defined(_DERIVATE)
Ibdxx_Vexx_ei=HBT_X_d_diode(vxxe,Jsee,Vgbb,nee,Area,Tj,Tnom,vxxe_Vexx_ei,0.0,0.0,0.0,0.0,0.0,0.0);
Ibdxx_Tempti_GND=HBT_X_d_diode(vxxe,Jsee,Vgbb,nee,Area,Tj,Tnom,0.0,0.0,0.0,0.0,0.0,Tj_Tempti_GND,0.0);
#endif
Ibdxx=HBT_X_diode(vxxe,Jsee,Vgbb,nee,Area,Tj,Tnom);
}
else
{
#if defined(_DERIVATE)
Ibdxx_Vexx_ei=vxxe_Vexx_ei*1e-12;
Ibdxx_Tempti_GND=0.0;
#endif
Ibdxx=(vxxe*1e-12);
}
if
(((Jsc>0.0)&&(nc>0)))
{
#if defined(_DERIVATE)
Icdx_Vcx_ci=HBT_X_d_diode(vxc,Jsc,Vgc,nc,Area,Tj,Tnom,vxc_Vcx_ci,0.0,0.0,0.0,0.0,0.0,0.0);
Icdx_Tempti_GND=HBT_X_d_diode(vxc,Jsc,Vgc,nc,Area,Tj,Tnom,0.0,0.0,0.0,0.0,0.0,Tj_Tempti_GND,0.0);
#endif
Icdx=HBT_X_diode(vxc,Jsc,Vgc,nc,Area,Tj,Tnom);
}
else
{
#if defined(_DERIVATE)
Icdx_Vcx_ci=vxc_Vcx_ci*1e-12;
Icdx_Tempti_GND=0.0;
#endif
Icdx=(vxc*1e-12);
}
#if defined(_DERIVATE)
Ipdiss_Vbii_ei=(Ic1_Vbii_ei*vcei+((Ib1_Vbii_ei*vbei)+(Ib1*vbei_Vbii_ei)));
Ipdiss_Tempti_GND=(((Ic1_Tempti_GND*vcei+Ib1_Tempti_GND*vbei)+Ib2_Tempti_GND*vbci)+Ibx_Tempti_GND*vbcx);
Ipdiss_Vbii_ci=((Ic1_Vbii_ci*vcei+Ib1_Vbii_ci*vbei)+((Ib2_Vbii_ci*vbci)+(Ib2*vbci_Vbii_ci)));
Ipdiss_Vci_ei=(Ic1*vcei_Vci_ei);
Ipdiss_Vbi_ci=((Ibx_Vbi_ci*vbcx)+(Ibx*vbcx_Vbi_ci));
#endif
Ipdiss=((((Ic1*vcei)+(Ib1*vbei))+(Ib2*vbci))+(Ibx*vbcx));
if
((Ipdiss<0.0))
{
#if defined(_DERIVATE)
Ipdiss_Vbii_ei=0.0;
Ipdiss_Tempti_GND=0.0;
Ipdiss_Vbii_ci=0.0;
Ipdiss_Vci_ei=0.0;
Ipdiss_Vbi_ci=0.0;
#endif
Ipdiss=0;
}
#if defined(_DYNAMIC)
I00=(J0*Area);
#endif
#ifdef _DYNAMIC
if
(((XCjc<1.0)&&(XCjc>0.0)))
{
#ifdef _DYNAMIC
if
(((J0<=0.0)||(Ic0<0.0)))
{
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
qb2med_Vbii_ei=0.0;
qb2med_Tempti_GND=0.0;
qb2med_Vbii_ci=((XCjc*HBT_X_d_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area,vbci_Vbii_ci,0.0,0.0,0.0,0.0))+(((XCjc*Area)*Cmin)*vbci_Vbii_ci));
#endif
qb2med=((XCjc*HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area))+(((XCjc*Area)*Cmin)*vbci));
#endif
}
else
{
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
xix_Vbii_ei=(Ic0_Vbii_ei/I00);
xix_Tempti_GND=(Ic0_Tempti_GND/I00);
xix_Vbii_ci=(Ic0_Vbii_ci/I00);
#endif
xix=(Ic0/I00);
#endif
#if defined(_DYNAMIC)
{
double m00_tanh(d00_tanh0,xix)
#if defined(_DERIVATE)
double m10_tanh(d10_tanh0,d00_tanh0,xix)
#endif
#if defined(_DERIVATE)
qb2med_Vbii_ei=(((XCjc*(-xix_Vbii_ei*d10_tanh0))*(HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area)+((((1.0-XJ0)*Area)*Cmin)*vbci)))+((XCjc*(1.0-d00_tanh0))*HBT_X_d_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area,0.0,0.0,0.0,0.0,0.0)));
qb2med_Tempti_GND=(((XCjc*(-xix_Tempti_GND*d10_tanh0))*(HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area)+((((1.0-XJ0)*Area)*Cmin)*vbci)))+((XCjc*(1.0-d00_tanh0))*HBT_X_d_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area,0.0,0.0,0.0,0.0,0.0)));
qb2med_Vbii_ci=((((XCjc*(-xix_Vbii_ci*d10_tanh0))*(HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area)+((((1.0-XJ0)*Area)*Cmin)*vbci)))+((XCjc*(1.0-d00_tanh0))*(HBT_X_d_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area,vbci_Vbii_ci,0.0,0.0,0.0,0.0)+((((1.0-XJ0)*Area)*Cmin)*vbci_Vbii_ci))))+((((XJ0*XCjc)*Area)*Cmin)*vbci_Vbii_ci));
#endif
qb2med=(((XCjc*(1.0-d00_tanh0))*(HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area)+((((1.0-XJ0)*Area)*Cmin)*vbci)))+((((XJ0*XCjc)*Area)*Cmin)*vbci));
}
#endif
}
#endif /* if (...) */
}
else
{
#ifdef _DYNAMIC
if
(((J0<0.0)||(Ic0<0.0)))
{
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
qb2med_Vbii_ei=0.0;
qb2med_Tempti_GND=0.0;
qb2med_Vbii_ci=(HBT_X_d_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area,vbci_Vbii_ci,0.0,0.0,0.0,0.0)+((Area*Cmin)*vbci_Vbii_ci));
#endif
qb2med=(HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area)+((Area*Cmin)*vbci));
#endif
}
else
{
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
xix_Vbii_ei=(Ic0_Vbii_ei/I00);
xix_Tempti_GND=(Ic0_Tempti_GND/I00);
xix_Vbii_ci=(Ic0_Vbii_ci/I00);
#endif
xix=(Ic0/I00);
#endif
#if defined(_DYNAMIC)
{
double m00_tanh(d00_tanh0,xix)
#if defined(_DERIVATE)
double m10_tanh(d10_tanh0,d00_tanh0,xix)
#endif
#if defined(_DERIVATE)
qb2med_Vbii_ei=(((-xix_Vbii_ei*d10_tanh0)*(HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area)+((((1.0-XJ0)*Area)*Cmin)*vbci)))+((1.0-d00_tanh0)*HBT_X_d_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area,0.0,0.0,0.0,0.0,0.0)));
qb2med_Tempti_GND=(((-xix_Tempti_GND*d10_tanh0)*(HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area)+((((1.0-XJ0)*Area)*Cmin)*vbci)))+((1.0-d00_tanh0)*HBT_X_d_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area,0.0,0.0,0.0,0.0,0.0)));
qb2med_Vbii_ci=((((-xix_Vbii_ci*d10_tanh0)*(HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area)+((((1.0-XJ0)*Area)*Cmin)*vbci)))+((1.0-d00_tanh0)*(HBT_X_d_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area,vbci_Vbii_ci,0.0,0.0,0.0,0.0)+((((1.0-XJ0)*Area)*Cmin)*vbci_Vbii_ci))))+(((XJ0*Area)*Cmin)*vbci_Vbii_ci));
#endif
qb2med=(((1.0-d00_tanh0)*(HBT_X_charge(vbci,(Cjc-Cmin),Vjc,mjc,Area)+((((1.0-XJ0)*Area)*Cmin)*vbci)))+(((XJ0*Area)*Cmin)*vbci));
}
#endif
}
#endif /* if (...) */
}
#endif /* if (...) */
#ifdef _DYNAMIC
if
(((XCjc<1.0)&&(XCjc>0.0)))
{
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
qb1_Vbi_ci=(((1.0-XCjc)*HBT_X_d_charge(vbcx,(Cjc-Cmin),Vjc,mjc,Area,vbcx_Vbi_ci,0.0,0.0,0.0,0.0))+((((1.0-XCjc)*Area)*Cmin)*vbcx_Vbi_ci));
#endif
qb1=(((1.0-XCjc)*HBT_X_charge(vbcx,(Cjc-Cmin),Vjc,mjc,Area))+((((1.0-XCjc)*Area)*Cmin)*vbcx));
#endif
}
else
{
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
qb1_Vbi_ci=0.0;
#endif
qb1=0.0;
#endif
}
#endif /* if (...) */
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
qbtr_Vbii_ci=(Tr*Ic1r_Vbii_ci);
qbtr_Tempti_GND=(Tr*Ic1r_Tempti_GND);
qbtr_Vbii_ei=(Tr*Ic1r_Vbii_ei);
#endif
qbtr=(Tr*Ic1r);
#endif
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
qbtra_Vbi_ci=(Trx*Ibx_Vbi_ci);
qbtra_Tempti_GND=(Trx*Ibx_Tempti_GND);
#endif
qbtra=(Trx*Ibx);
#endif
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
qb2_Vbii_ei=(qb2med_Vbii_ei+qbtr_Vbii_ei);
qb2_Tempti_GND=(qb2med_Tempti_GND+qbtr_Tempti_GND);
qb2_Vbii_ci=(qb2med_Vbii_ci+qbtr_Vbii_ci);
#endif
qb2=(qb2med+qbtr);
#endif
#ifdef _DYNAMIC
if
(((Jk>0.0)&&(Rci0>0.0)))
{
#ifdef _DYNAMIC
if
((RJk>0.0))
{
#if defined(_DYNAMIC)
Vlim=((Jk*Rci0)/(1.0-(Rci0/RJk)));
#endif
#if defined(_DYNAMIC)
InvVpt=((1.0-(Rci0/RJk))/(Jk*RJk));
#endif
}
else
{
#if defined(_DYNAMIC)
Vlim=((Jk*Rci0)/1.016);
#endif
#if defined(_DYNAMIC)
InvVpt=0.0;
#endif
}
#endif /* if (...) */
}
#endif /* if (...) */
#ifdef _DYNAMIC
if
(((((Thcs>0.0)&&(Ahc>0.0))&&(Jk>0.0))&&(Ic0>0.0)))
{
#if defined(_DYNAMIC)
RCIO=(Rci0/Area);
#endif
#if defined(_DYNAMIC)
AHC=(Area*Ahc);
#endif
#ifdef _DYNAMIC
if
(((Rci0<RJk)||(RJk<=0.0)))
{
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
Ih_Vci_ei=(-(HBT_X_d_ICK(vcei,RCIO,Vlim,InvVpt,Vces,vcei_Vci_ei,0.0,0.0,0.0,0.0)/Ic0));
Ih_Vbii_ei=(-((HBT_X_d_ICK(vcei,RCIO,Vlim,InvVpt,Vces,0.0,0.0,0.0,0.0,0.0)*Ic0-HBT_X_ICK(vcei,RCIO,Vlim,InvVpt,Vces)*Ic0_Vbii_ei)/Ic0/Ic0));
Ih_Tempti_GND=(-((HBT_X_d_ICK(vcei,RCIO,Vlim,InvVpt,Vces,0.0,0.0,0.0,0.0,0.0)*Ic0-HBT_X_ICK(vcei,RCIO,Vlim,InvVpt,Vces)*Ic0_Tempti_GND)/Ic0/Ic0));
Ih_Vbii_ci=(-((HBT_X_d_ICK(vcei,RCIO,Vlim,InvVpt,Vces,0.0,0.0,0.0,0.0,0.0)*Ic0-HBT_X_ICK(vcei,RCIO,Vlim,InvVpt,Vces)*Ic0_Vbii_ci)/Ic0/Ic0));
#endif
Ih=(1.0-(HBT_X_ICK(vcei,RCIO,Vlim,InvVpt,Vces)/Ic0));
#endif
}
else
{
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
Ih_Vci_ei=(-(HBT_X_d_Vceff(vcei,Vces,vcei_Vci_ei,0.0)/(RCIO*Ic0)));
Ih_Vbii_ei=(-((HBT_X_d_Vceff(vcei,Vces,0.0,0.0)*(RCIO*Ic0)-HBT_X_Vceff(vcei,Vces)*(RCIO*Ic0_Vbii_ei))/(RCIO*Ic0)/(RCIO*Ic0)));
Ih_Tempti_GND=(-((HBT_X_d_Vceff(vcei,Vces,0.0,0.0)*(RCIO*Ic0)-HBT_X_Vceff(vcei,Vces)*(RCIO*Ic0_Tempti_GND))/(RCIO*Ic0)/(RCIO*Ic0)));
Ih_Vbii_ci=(-((HBT_X_d_Vceff(vcei,Vces,0.0,0.0)*(RCIO*Ic0)-HBT_X_Vceff(vcei,Vces)*(RCIO*Ic0_Vbii_ci))/(RCIO*Ic0)/(RCIO*Ic0)));
#endif
Ih=(1.0-(HBT_X_Vceff(vcei,Vces)/(RCIO*Ic0)));
#endif
}
#endif /* if (...) */
#if defined(_DYNAMIC)
{
double m00_sqrt(d00_sqrt0,((Ih*Ih)+AHC))
double m00_sqrt(d00_sqrt1,(1.0+AHC))
#if defined(_DERIVATE)
double m10_sqrt(d10_sqrt0,d00_sqrt0,((Ih*Ih)+AHC))
#endif
#if defined(_DERIVATE)
Wh_Vci_ei=((Ih_Vci_ei+((Ih_Vci_ei*Ih)+(Ih*Ih_Vci_ei))*d10_sqrt0)/(1.0+d00_sqrt1));
Wh_Vbii_ei=((Ih_Vbii_ei+((Ih_Vbii_ei*Ih)+(Ih*Ih_Vbii_ei))*d10_sqrt0)/(1.0+d00_sqrt1));
Wh_Tempti_GND=((Ih_Tempti_GND+((Ih_Tempti_GND*Ih)+(Ih*Ih_Tempti_GND))*d10_sqrt0)/(1.0+d00_sqrt1));
Wh_Vbii_ci=((Ih_Vbii_ci+((Ih_Vbii_ci*Ih)+(Ih*Ih_Vbii_ci))*d10_sqrt0)/(1.0+d00_sqrt1));
#endif
Wh=((Ih+d00_sqrt0)/(1.0+d00_sqrt1));
}
#endif
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
xtff_Vbii_ei=(((Thcs*Ic0_Vbii_ei)*(Wh*Wh))+((Thcs*Ic0)*((Wh_Vbii_ei*Wh)+(Wh*Wh_Vbii_ei))));
xtff_Tempti_GND=(((Thcs*Ic0_Tempti_GND)*(Wh*Wh))+((Thcs*Ic0)*((Wh_Tempti_GND*Wh)+(Wh*Wh_Tempti_GND))));
xtff_Vbii_ci=(((Thcs*Ic0_Vbii_ci)*(Wh*Wh))+((Thcs*Ic0)*((Wh_Vbii_ci*Wh)+(Wh*Wh_Vbii_ci))));
xtff_Vci_ei=((Thcs*Ic0)*((Wh_Vci_ei*Wh)+(Wh*Wh_Vci_ei)));
#endif
xtff=((Thcs*Ic0)*(Wh*Wh));
#endif
}
else
{
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
xtff_Vbii_ei=0.0;
xtff_Tempti_GND=0.0;
xtff_Vbii_ci=0.0;
xtff_Vci_ei=0.0;
#endif
xtff=0;
#endif
}
#endif /* if (...) */
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
qbtf_Tempti_GND=(((Tft*Tex_Tempti_GND)*Ic0)+((Tf+(Tft*Tex))*Ic0_Tempti_GND));
qbtf_Vbii_ei=((Tf+(Tft*Tex))*Ic0_Vbii_ei);
qbtf_Vbii_ci=((Tf+(Tft*Tex))*Ic0_Vbii_ci);
#endif
qbtf=((Tf+(Tft*Tex))*Ic0);
#endif
#if defined(_DYNAMIC)
#if defined(_DERIVATE)
qbe_Vbii_ei=((xtff_Vbii_ei+qbtf_Vbii_ei)+HBT_X_d_charge(vbei,Cje,Vje,mje,Area,vbei_Vbii_ei,0.0,0.0,0.0,0.0));
qbe_Tempti_GND=((xtff_Tempti_GND+qbtf_Tempti_GND)+HBT_X_d_charge(vbei,Cje,Vje,mje,Area,0.0,0.0,0.0,0.0,0.0));
qbe_Vbii_ci=((xtff_Vbii_ci+qbtf_Vbii_ci)+HBT_X_d_charge(vbei,Cje,Vje,mje,Area,0.0,0.0,0.0,0.0,0.0));
qbe_Vci_ei=(xtff_Vci_ei+HBT_X_d_charge(vbei,Cje,Vje,mje,Area,0.0,0.0,0.0,0.0,0.0));
#endif
qbe=((xtff+qbtf)+HBT_X_charge(vbei,Cje,Vje,mje,Area));
#endif
_load_static_residual2(bi,ci,Ibx);
#if defined(_DERIVATE)
_load_static_jacobian2s(bi,ci,ti,Ibx_Tempti_GND);
_load_static_jacobian4(bi,ci,bi,ci,Ibx_Vbi_ci);
#endif
#if defined(_DYNAMIC)
_load_dynamic_residual2(bi,ci,_DDT((qb1+qbtra)));
#if defined(_DERIVATE)
_load_dynamic_jacobian2s(bi,ci,ti,qbtra_Tempti_GND);
_load_dynamic_jacobian4(bi,ci,bi,ci,(qb1_Vbi_ci+qbtra_Vbi_ci));
#endif
#endif
_load_static_residual2(bii,ci,Ib2);
#if defined(_DERIVATE)
_load_static_jacobian2s(bii,ci,ti,Ib2_Tempti_GND);
_load_static_jacobian4(bii,ci,bii,ci,Ib2_Vbii_ci);
#endif
#if defined(_DYNAMIC)
_load_dynamic_residual2(bii,ci,_DDT(qb2));
#if defined(_DERIVATE)
_load_dynamic_jacobian4(bii,ci,bii,ci,qb2_Vbii_ci);
_load_dynamic_jacobian2s(bii,ci,ti,qb2_Tempti_GND);
_load_dynamic_jacobian4(bii,ci,bii,ei,qb2_Vbii_ei);
#endif
#endif
_load_static_residual2(bii,ei,Ib1);
#if defined(_DERIVATE)
_load_static_jacobian4(bii,ei,bii,ci,Ib1_Vbii_ci);
_load_static_jacobian2s(bii,ei,ti,Ib1_Tempti_GND);
_load_static_jacobian4(bii,ei,bii,ei,Ib1_Vbii_ei);
#endif
#if defined(_DYNAMIC)
_load_dynamic_residual2(bii,ei,_DDT(qbe));
#if defined(_DERIVATE)
_load_dynamic_jacobian4(bii,ei,ci,ei,qbe_Vci_ei);
_load_dynamic_jacobian4(bii,ei,bii,ci,qbe_Vbii_ci);
_load_dynamic_jacobian2s(bii,ei,ti,qbe_Tempti_GND);
_load_dynamic_jacobian4(bii,ei,bii,ei,qbe_Vbii_ei);
#endif
#endif
_load_static_residual2(ci,ei,Ic1);
#if defined(_DERIVATE)
_load_static_jacobian4(ci,ei,bii,ci,Ic1_Vbii_ci);
_load_static_jacobian2s(ci,ei,ti,Ic1_Tempti_GND);
_load_static_jacobian4(ci,ei,bii,ei,Ic1_Vbii_ei);
#endif
_load_static_residual2(ex,ei,Ibdx);
#if defined(_DERIVATE)
_load_static_jacobian2s(ex,ei,ti,Ibdx_Tempti_GND);
_load_static_jacobian4(ex,ei,ex,ei,Ibdx_Vex_ei);
#endif
_load_static_residual2(exx,ei,Ibdxx);
#if defined(_DERIVATE)
_load_static_jacobian2s(exx,ei,ti,Ibdxx_Tempti_GND);
_load_static_jacobian4(exx,ei,exx,ei,Ibdxx_Vexx_ei);
#endif
_load_static_residual2(cx,ci,Icdx);
#if defined(_DERIVATE)
_load_static_jacobian2s(cx,ci,ti,Icdx_Tempti_GND);
_load_static_jacobian4(cx,ci,cx,ci,Icdx_Vcx_ci);
#endif
_save_whitenoise2(bii,ei,(TWO_Q*Ib1),"Ib");
_load_static_residual1(ni,NP(ni));
#if defined(_DERIVATE)
_load_static_jacobian1(ni,ni,1.0);
#endif
_save_whitenoise1(ni,(TWO_Q*Ic0),"Ic");
_load_static_residual2(bii,ei,NP(ni));
#if defined(_DERIVATE)
_load_static_jacobian2s(bii,ei,ni,1.0);
#endif
_load_static_residual2(bii,ci,NP(ni));
#if defined(_DERIVATE)
_load_static_jacobian2s(bii,ci,ni,1.0);
#endif
_load_static_residual2(bii,bi,(BP(bii,bi)/(Rb2/N)));
#if defined(_DERIVATE)
_load_static_jacobian4(bii,bi,bii,bi,(1/(Rb2/N)));
#endif
_save_whitenoise2(bii,bi,((FOUR_K*TjK)/(Rb2/N)),"thermal");
_load_static_residual2(b,bi,(BP(b,bi)/(Rb/N)));
#if defined(_DERIVATE)
_load_static_jacobian4(b,bi,b,bi,(1/(Rb/N)));
#endif
_save_whitenoise2(b,bi,((FOUR_K*TjK)/(Rb/N)),"thermal");
_load_static_residual2(e,ei,(BP(e,ei)/(Re/N)));
#if defined(_DERIVATE)
_load_static_jacobian4(e,ei,e,ei,(1/(Re/N)));
#endif
_save_whitenoise2(e,ei,((FOUR_K*TjK)/(Re/N)),"thermal");
_load_static_residual2(c,ci,(BP(c,ci)/(Rc/N)));
#if defined(_DERIVATE)
_load_static_jacobian4(c,ci,c,ci,(1/(Rc/N)));
#endif
_save_whitenoise2(c,ci,((FOUR_K*TjK)/(Rc/N)),"thermal");
if
(((Jse>0.0)&&(ne>0)))
{
_load_static_residual2(ex,bii,(BP(ex,bii)/(Rbxx/N)));
#if defined(_DERIVATE)
_load_static_jacobian4(ex,bii,ex,bii,(1/(Rbxx/N)));
#endif
_save_whitenoise2(ex,bii,((FOUR_K*TjK)/(Rbxx/N)),"thermal");
}
else
{
_load_static_residual2(ex,bii,(BP(ex,bii)*1e-12));
#if defined(_DERIVATE)
_load_static_jacobian4(ex,bii,ex,bii,(1e-12));
#endif
}
if
(((Jsee>0.0)&&(nee>0)))
{
_load_static_residual2(exx,bii,(BP(exx,bii)/(Rbbxx/N)));
#if defined(_DERIVATE)
_load_static_jacobian4(exx,bii,exx,bii,(1/(Rbbxx/N)));
#endif
_save_whitenoise2(exx,bii,((FOUR_K*TjK)/(Rbbxx/N)),"thermal");
}
else
{
_load_static_residual2(exx,bii,(BP(exx,bii)*1e-12));
#if defined(_DERIVATE)
_load_static_jacobian4(exx,bii,exx,bii,(1e-12));
#endif
}
if
(((Jsc>0.0)&&(nc>0)))
{
_load_static_residual2(cx,bii,(BP(cx,bii)/(Rcxx/N)));
#if defined(_DERIVATE)
_load_static_jacobian4(cx,bii,cx,bii,(1/(Rcxx/N)));
#endif
_save_whitenoise2(cx,bii,((FOUR_K*TjK)/(Rcxx/N)),"thermal");
}
else
{
_load_static_residual2(cx,bii,(BP(cx,bii)*1e-12));
#if defined(_DERIVATE)
_load_static_jacobian4(cx,bii,cx,bii,(1e-12));
#endif
}
#if defined(_DYNAMIC)
_load_dynamic_residual1(b,_DDT((Cpb*NP(b))));
#if defined(_DERIVATE)
_load_dynamic_jacobian1(b,b,(Cpb));
#endif
#endif
#if defined(_DYNAMIC)
_load_dynamic_residual1(c,_DDT((Cpc*NP(c))));
#if defined(_DERIVATE)
_load_dynamic_jacobian1(c,c,(Cpc));
#endif
#endif
#if defined(_DYNAMIC)
_load_dynamic_residual2(b,c,_DDT((Cq*BP(b,c))));
#if defined(_DERIVATE)
_load_dynamic_jacobian4(b,c,b,c,(Cq));
#endif
#endif
_load_static_residual1(ti,(-Ipdiss));
#if defined(_DERIVATE)
_load_static_jacobian2p(ti,bi,ci,(-Ipdiss_Vbi_ci));
_load_static_jacobian2p(ti,ci,ei,(-Ipdiss_Vci_ei));
_load_static_jacobian2p(ti,bii,ci,(-Ipdiss_Vbii_ci));
_load_static_jacobian1(ti,ti,(-Ipdiss_Tempti_GND));
_load_static_jacobian2p(ti,bii,ei,(-Ipdiss_Vbii_ei));
#endif
if
(Rth)
{
_load_static_residual2(t,ti,(BP(t,ti)/Rth));
#if defined(_DERIVATE)
_load_static_jacobian4(t,ti,t,ti,(1/Rth));
#endif
#if defined(_DYNAMIC)
_load_dynamic_residual2(t,ti,(Cth*_DDT(BP(t,ti))));
#if defined(_DERIVATE)
_load_dynamic_jacobian4(t,ti,t,ti,(Cth));
#endif
#endif
}
else
{
_load_static_residual2(t,ti,(BP(t,ti)*1e3));
#if defined(_DERIVATE)
_load_static_jacobian4(t,ti,t,ti,(1e3));
#endif
}
if
((Ib0<=0))
{
#if defined(_DERIVATE)
Iniix_Vbii_ei=0.0;
Iniix_Tempti_GND=0.0;
Iniix_Vbii_ci=0.0;
#endif
Iniix=0;
#if defined(_DERIVATE)
Iniiix_Vbii_ei=0.0;
Iniiix_Tempti_GND=0.0;
Iniiix_Vbii_ci=0.0;
#endif
Iniiix=0;
}
if
(((Ib0+Ic1)<=0))
{
#if defined(_DERIVATE)
Inivx_Vbii_ei=0.0;
Inivx_Tempti_GND=0.0;
Inivx_Vbii_ci=0.0;
#endif
Inivx=0;
}
else
{
if
((Ab==2))
{
#if defined(_DERIVATE)
Iniix_Vbii_ei=Ib0_Vbii_ei;
Iniix_Tempti_GND=Ib0_Tempti_GND;
Iniix_Vbii_ci=Ib0_Vbii_ci;
#endif
Iniix=Ib0;
}
else
{
{
double m00_pow(d00_pow0,Ib0,(Ab*0.5))
#if defined(_DERIVATE)
double m10_pow(d10_pow0,d00_pow0,Ib0,(Ab*0.5))
#endif
#if defined(_DERIVATE)
Iniix_Vbii_ei=(d10_pow0*Ib0_Vbii_ei);
Iniix_Tempti_GND=(d10_pow0*Ib0_Tempti_GND);
Iniix_Vbii_ci=(d10_pow0*Ib0_Vbii_ci);
#endif
Iniix=d00_pow0;
}
}
if
((Afb==2))
{
#if defined(_DERIVATE)
Iniiix_Vbii_ei=Ib0_Vbii_ei;
Iniiix_Tempti_GND=Ib0_Tempti_GND;
Iniiix_Vbii_ci=Ib0_Vbii_ci;
#endif
Iniiix=Ib0;
}
else
{
{
double m00_pow(d00_pow0,Ib0,(Afb*0.5))
#if defined(_DERIVATE)
double m10_pow(d10_pow0,d00_pow0,Ib0,(Afb*0.5))
#endif
#if defined(_DERIVATE)
Iniiix_Vbii_ei=(d10_pow0*Ib0_Vbii_ei);
Iniiix_Tempti_GND=(d10_pow0*Ib0_Tempti_GND);
Iniiix_Vbii_ci=(d10_pow0*Ib0_Vbii_ci);
#endif
Iniiix=d00_pow0;
}
}
if
((Afe==2))
{
#if defined(_DERIVATE)
Inivx_Vbii_ei=(Ib0_Vbii_ei+Ic1_Vbii_ei);
Inivx_Tempti_GND=(Ib0_Tempti_GND+Ic1_Tempti_GND);
Inivx_Vbii_ci=(Ib0_Vbii_ci+Ic1_Vbii_ci);
#endif
Inivx=(Ib0+Ic1);
}
else
{
{
double m00_pow(d00_pow0,(Ib0+Ic1),(Afe*0.5))
#if defined(_DERIVATE)
double m10_pow(d10_pow0,d00_pow0,(Ib0+Ic1),(Afe*0.5))
#endif
#if defined(_DERIVATE)
Inivx_Vbii_ei=(d10_pow0*(Ib0_Vbii_ei+Ic1_Vbii_ei));
Inivx_Tempti_GND=(d10_pow0*(Ib0_Tempti_GND+Ic1_Tempti_GND));
Inivx_Vbii_ci=(d10_pow0*(Ib0_Vbii_ci+Ic1_Vbii_ci));
#endif
Inivx=d00_pow0;
}
}
}
_load_static_residual1(nii,NP(nii));
#if defined(_DERIVATE)
_load_static_jacobian1(nii,nii,1.0);
#endif
#if defined(_DYNAMIC)
_load_dynamic_residual1(nii,_DDT((NP(nii)/((2.0*3.1415)*Fb))));
#if defined(_DERIVATE)
_load_dynamic_jacobian1(nii,nii,(1/((2.0*3.1415)*Fb)));
#endif
#endif
_save_whitenoise1(nii,(Area*Kb),NULL);
_load_static_residual1(niix,NP(niix));
#if defined(_DERIVATE)
_load_static_jacobian1(niix,niix,1.0);
#endif
_load_static_residual1(niix,Iniix);
#if defined(_DERIVATE)
_load_static_jacobian2p(niix,bii,ci,Iniix_Vbii_ci);
_load_static_jacobian1(niix,ti,Iniix_Tempti_GND);
_load_static_jacobian2p(niix,bii,ei,Iniix_Vbii_ei);
#endif
_load_static_residual1(niii,NP(niii));
#if defined(_DERIVATE)
_load_static_jacobian1(niii,niii,1.0);
#endif
_save_flickernoise1(niii,(Area*Kfb),Ffeb,"Flicker noise base-emitter junction (a)");
_load_static_residual1(niiix,NP(niiix));
#if defined(_DERIVATE)
_load_static_jacobian1(niiix,niiix,1.0);
#endif
_load_static_residual1(niiix,Iniiix);
#if defined(_DERIVATE)
_load_static_jacobian2p(niiix,bii,ci,Iniiix_Vbii_ci);
_load_static_jacobian1(niiix,ti,Iniiix_Tempti_GND);
_load_static_jacobian2p(niiix,bii,ei,Iniiix_Vbii_ei);
#endif
_load_static_residual2(bii,ei,(NP(nii)*NP(niix)));
#if defined(_DERIVATE)
_load_static_jacobian2s(bii,ei,niix,(NP(nii)));
_load_static_jacobian2s(bii,ei,nii,(NP(niix)));
#endif
_load_static_residual2(bii,ei,(NP(niii)*NP(niiix)));
#if defined(_DERIVATE)
_load_static_jacobian2s(bii,ei,niiix,(NP(niii)));
_load_static_jacobian2s(bii,ei,niii,(NP(niiix)));
#endif
_load_static_residual1(niv,NP(niv));
#if defined(_DERIVATE)
_load_static_jacobian1(niv,niv,1.0);
#endif
_save_flickernoise1(niv,(Area*Kfe),Ffee,"Hooge noise of emitter resistance");
_load_static_residual1(nivx,NP(nivx));
#if defined(_DERIVATE)
_load_static_jacobian1(nivx,nivx,1.0);
#endif
_load_static_residual1(nivx,Inivx);
#if defined(_DERIVATE)
_load_static_jacobian2p(nivx,bii,ci,Inivx_Vbii_ci);
_load_static_jacobian1(nivx,ti,Inivx_Tempti_GND);
_load_static_jacobian2p(nivx,bii,ei,Inivx_Vbii_ei);
#endif
_load_static_residual2(e,ei,((NP(niv)*NP(nivx))/(Re/N)));
#if defined(_DERIVATE)
_load_static_jacobian2s(e,ei,nivx,((NP(niv))/(Re/N)));
_load_static_jacobian2s(e,ei,niv,((NP(nivx))/(Re/N)));
#endif

/* ------------------ end of verilog analog equations --------------------- */

/* ------------------ evaluate verilog noise equations -------------------- */

/* ------------------- end of verilog noise equations --------------------- */

}

/* Perform DC iteration. */
void HBT_X::calcDC (void)
{
  // evaluate Verilog code
  initVerilog ();		
  calcVerilog ();		

  // fill right hand side and static jacobian
  for (int i1 = 0; i1 < 19; i1++) {
    setI (i1, _rhs[i1]);
    for (int i2 = 0; i2 < 19; i2++) {
      setY (i1, i2, _jstat[i1][i2]);
    }
  }
}

/* Save operating points. */
void HBT_X::saveOperatingPoints (void)
{
#if DEBUG
  setOperatingPoint ("jstat[c][c]", _jstat[c][c]);
  setOperatingPoint ("jdyna[c][c]", _jdyna[c][c]);
  setOperatingPoint ("jstat[c][b]", _jstat[c][b]);
  setOperatingPoint ("jdyna[c][b]", _jdyna[c][b]);
  setOperatingPoint ("jstat[c][e]", _jstat[c][e]);
  setOperatingPoint ("jdyna[c][e]", _jdyna[c][e]);
  setOperatingPoint ("jstat[c][t]", _jstat[c][t]);
  setOperatingPoint ("jdyna[c][t]", _jdyna[c][t]);
  setOperatingPoint ("jstat[c][ei]", _jstat[c][ei]);
  setOperatingPoint ("jdyna[c][ei]", _jdyna[c][ei]);
  setOperatingPoint ("jstat[c][bi]", _jstat[c][bi]);
  setOperatingPoint ("jdyna[c][bi]", _jdyna[c][bi]);
  setOperatingPoint ("jstat[c][bii]", _jstat[c][bii]);
  setOperatingPoint ("jdyna[c][bii]", _jdyna[c][bii]);
  setOperatingPoint ("jstat[c][ci]", _jstat[c][ci]);
  setOperatingPoint ("jdyna[c][ci]", _jdyna[c][ci]);
  setOperatingPoint ("jstat[c][ex]", _jstat[c][ex]);
  setOperatingPoint ("jdyna[c][ex]", _jdyna[c][ex]);
  setOperatingPoint ("jstat[c][exx]", _jstat[c][exx]);
  setOperatingPoint ("jdyna[c][exx]", _jdyna[c][exx]);
  setOperatingPoint ("jstat[c][cx]", _jstat[c][cx]);
  setOperatingPoint ("jdyna[c][cx]", _jdyna[c][cx]);
  setOperatingPoint ("jstat[c][ni]", _jstat[c][ni]);
  setOperatingPoint ("jdyna[c][ni]", _jdyna[c][ni]);
  setOperatingPoint ("jstat[c][nii]", _jstat[c][nii]);
  setOperatingPoint ("jdyna[c][nii]", _jdyna[c][nii]);
  setOperatingPoint ("jstat[c][niix]", _jstat[c][niix]);
  setOperatingPoint ("jdyna[c][niix]", _jdyna[c][niix]);
  setOperatingPoint ("jstat[c][niii]", _jstat[c][niii]);
  setOperatingPoint ("jdyna[c][niii]", _jdyna[c][niii]);
  setOperatingPoint ("jstat[c][niiix]", _jstat[c][niiix]);
  setOperatingPoint ("jdyna[c][niiix]", _jdyna[c][niiix]);
  setOperatingPoint ("jstat[c][niv]", _jstat[c][niv]);
  setOperatingPoint ("jdyna[c][niv]", _jdyna[c][niv]);
  setOperatingPoint ("jstat[c][nivx]", _jstat[c][nivx]);
  setOperatingPoint ("jdyna[c][nivx]", _jdyna[c][nivx]);
  setOperatingPoint ("jstat[c][ti]", _jstat[c][ti]);
  setOperatingPoint ("jdyna[c][ti]", _jdyna[c][ti]);
  setOperatingPoint ("jstat[b][c]", _jstat[b][c]);
  setOperatingPoint ("jdyna[b][c]", _jdyna[b][c]);
  setOperatingPoint ("jstat[b][b]", _jstat[b][b]);
  setOperatingPoint ("jdyna[b][b]", _jdyna[b][b]);
  setOperatingPoint ("jstat[b][e]", _jstat[b][e]);
  setOperatingPoint ("jdyna[b][e]", _jdyna[b][e]);
  setOperatingPoint ("jstat[b][t]", _jstat[b][t]);
  setOperatingPoint ("jdyna[b][t]", _jdyna[b][t]);
  setOperatingPoint ("jstat[b][ei]", _jstat[b][ei]);
  setOperatingPoint ("jdyna[b][ei]", _jdyna[b][ei]);
  setOperatingPoint ("jstat[b][bi]", _jstat[b][bi]);
  setOperatingPoint ("jdyna[b][bi]", _jdyna[b][bi]);
  setOperatingPoint ("jstat[b][bii]", _jstat[b][bii]);
  setOperatingPoint ("jdyna[b][bii]", _jdyna[b][bii]);
  setOperatingPoint ("jstat[b][ci]", _jstat[b][ci]);
  setOperatingPoint ("jdyna[b][ci]", _jdyna[b][ci]);
  setOperatingPoint ("jstat[b][ex]", _jstat[b][ex]);
  setOperatingPoint ("jdyna[b][ex]", _jdyna[b][ex]);
  setOperatingPoint ("jstat[b][exx]", _jstat[b][exx]);
  setOperatingPoint ("jdyna[b][exx]", _jdyna[b][exx]);
  setOperatingPoint ("jstat[b][cx]", _jstat[b][cx]);
  setOperatingPoint ("jdyna[b][cx]", _jdyna[b][cx]);
  setOperatingPoint ("jstat[b][ni]", _jstat[b][ni]);
  setOperatingPoint ("jdyna[b][ni]", _jdyna[b][ni]);
  setOperatingPoint ("jstat[b][nii]", _jstat[b][nii]);
  setOperatingPoint ("jdyna[b][nii]", _jdyna[b][nii]);
  setOperatingPoint ("jstat[b][niix]", _jstat[b][niix]);
  setOperatingPoint ("jdyna[b][niix]", _jdyna[b][niix]);
  setOperatingPoint ("jstat[b][niii]", _jstat[b][niii]);
  setOperatingPoint ("jdyna[b][niii]", _jdyna[b][niii]);
  setOperatingPoint ("jstat[b][niiix]", _jstat[b][niiix]);
  setOperatingPoint ("jdyna[b][niiix]", _jdyna[b][niiix]);
  setOperatingPoint ("jstat[b][niv]", _jstat[b][niv]);
  setOperatingPoint ("jdyna[b][niv]", _jdyna[b][niv]);
  setOperatingPoint ("jstat[b][nivx]", _jstat[b][nivx]);
  setOperatingPoint ("jdyna[b][nivx]", _jdyna[b][nivx]);
  setOperatingPoint ("jstat[b][ti]", _jstat[b][ti]);
  setOperatingPoint ("jdyna[b][ti]", _jdyna[b][ti]);
  setOperatingPoint ("jstat[e][c]", _jstat[e][c]);
  setOperatingPoint ("jdyna[e][c]", _jdyna[e][c]);
  setOperatingPoint ("jstat[e][b]", _jstat[e][b]);
  setOperatingPoint ("jdyna[e][b]", _jdyna[e][b]);
  setOperatingPoint ("jstat[e][e]", _jstat[e][e]);
  setOperatingPoint ("jdyna[e][e]", _jdyna[e][e]);
  setOperatingPoint ("jstat[e][t]", _jstat[e][t]);
  setOperatingPoint ("jdyna[e][t]", _jdyna[e][t]);
  setOperatingPoint ("jstat[e][ei]", _jstat[e][ei]);
  setOperatingPoint ("jdyna[e][ei]", _jdyna[e][ei]);
  setOperatingPoint ("jstat[e][bi]", _jstat[e][bi]);
  setOperatingPoint ("jdyna[e][bi]", _jdyna[e][bi]);
  setOperatingPoint ("jstat[e][bii]", _jstat[e][bii]);
  setOperatingPoint ("jdyna[e][bii]", _jdyna[e][bii]);
  setOperatingPoint ("jstat[e][ci]", _jstat[e][ci]);
  setOperatingPoint ("jdyna[e][ci]", _jdyna[e][ci]);
  setOperatingPoint ("jstat[e][ex]", _jstat[e][ex]);
  setOperatingPoint ("jdyna[e][ex]", _jdyna[e][ex]);
  setOperatingPoint ("jstat[e][exx]", _jstat[e][exx]);
  setOperatingPoint ("jdyna[e][exx]", _jdyna[e][exx]);
  setOperatingPoint ("jstat[e][cx]", _jstat[e][cx]);
  setOperatingPoint ("jdyna[e][cx]", _jdyna[e][cx]);
  setOperatingPoint ("jstat[e][ni]", _jstat[e][ni]);
  setOperatingPoint ("jdyna[e][ni]", _jdyna[e][ni]);
  setOperatingPoint ("jstat[e][nii]", _jstat[e][nii]);
  setOperatingPoint ("jdyna[e][nii]", _jdyna[e][nii]);
  setOperatingPoint ("jstat[e][niix]", _jstat[e][niix]);
  setOperatingPoint ("jdyna[e][niix]", _jdyna[e][niix]);
  setOperatingPoint ("jstat[e][niii]", _jstat[e][niii]);
  setOperatingPoint ("jdyna[e][niii]", _jdyna[e][niii]);
  setOperatingPoint ("jstat[e][niiix]", _jstat[e][niiix]);
  setOperatingPoint ("jdyna[e][niiix]", _jdyna[e][niiix]);
  setOperatingPoint ("jstat[e][niv]", _jstat[e][niv]);
  setOperatingPoint ("jdyna[e][niv]", _jdyna[e][niv]);
  setOperatingPoint ("jstat[e][nivx]", _jstat[e][nivx]);
  setOperatingPoint ("jdyna[e][nivx]", _jdyna[e][nivx]);
  setOperatingPoint ("jstat[e][ti]", _jstat[e][ti]);
  setOperatingPoint ("jdyna[e][ti]", _jdyna[e][ti]);
  setOperatingPoint ("jstat[t][c]", _jstat[t][c]);
  setOperatingPoint ("jdyna[t][c]", _jdyna[t][c]);
  setOperatingPoint ("jstat[t][b]", _jstat[t][b]);
  setOperatingPoint ("jdyna[t][b]", _jdyna[t][b]);
  setOperatingPoint ("jstat[t][e]", _jstat[t][e]);
  setOperatingPoint ("jdyna[t][e]", _jdyna[t][e]);
  setOperatingPoint ("jstat[t][t]", _jstat[t][t]);
  setOperatingPoint ("jdyna[t][t]", _jdyna[t][t]);
  setOperatingPoint ("jstat[t][ei]", _jstat[t][ei]);
  setOperatingPoint ("jdyna[t][ei]", _jdyna[t][ei]);
  setOperatingPoint ("jstat[t][bi]", _jstat[t][bi]);
  setOperatingPoint ("jdyna[t][bi]", _jdyna[t][bi]);
  setOperatingPoint ("jstat[t][bii]", _jstat[t][bii]);
  setOperatingPoint ("jdyna[t][bii]", _jdyna[t][bii]);
  setOperatingPoint ("jstat[t][ci]", _jstat[t][ci]);
  setOperatingPoint ("jdyna[t][ci]", _jdyna[t][ci]);
  setOperatingPoint ("jstat[t][ex]", _jstat[t][ex]);
  setOperatingPoint ("jdyna[t][ex]", _jdyna[t][ex]);
  setOperatingPoint ("jstat[t][exx]", _jstat[t][exx]);
  setOperatingPoint ("jdyna[t][exx]", _jdyna[t][exx]);
  setOperatingPoint ("jstat[t][cx]", _jstat[t][cx]);
  setOperatingPoint ("jdyna[t][cx]", _jdyna[t][cx]);
  setOperatingPoint ("jstat[t][ni]", _jstat[t][ni]);
  setOperatingPoint ("jdyna[t][ni]", _jdyna[t][ni]);
  setOperatingPoint ("jstat[t][nii]", _jstat[t][nii]);
  setOperatingPoint ("jdyna[t][nii]", _jdyna[t][nii]);
  setOperatingPoint ("jstat[t][niix]", _jstat[t][niix]);
  setOperatingPoint ("jdyna[t][niix]", _jdyna[t][niix]);
  setOperatingPoint ("jstat[t][niii]", _jstat[t][niii]);
  setOperatingPoint ("jdyna[t][niii]", _jdyna[t][niii]);
  setOperatingPoint ("jstat[t][niiix]", _jstat[t][niiix]);
  setOperatingPoint ("jdyna[t][niiix]", _jdyna[t][niiix]);
  setOperatingPoint ("jstat[t][niv]", _jstat[t][niv]);
  setOperatingPoint ("jdyna[t][niv]", _jdyna[t][niv]);
  setOperatingPoint ("jstat[t][nivx]", _jstat[t][nivx]);
  setOperatingPoint ("jdyna[t][nivx]", _jdyna[t][nivx]);
  setOperatingPoint ("jstat[t][ti]", _jstat[t][ti]);
  setOperatingPoint ("jdyna[t][ti]", _jdyna[t][ti]);
  setOperatingPoint ("jstat[ei][c]", _jstat[ei][c]);
  setOperatingPoint ("jdyna[ei][c]", _jdyna[ei][c]);
  setOperatingPoint ("jstat[ei][b]", _jstat[ei][b]);
  setOperatingPoint ("jdyna[ei][b]", _jdyna[ei][b]);
  setOperatingPoint ("jstat[ei][e]", _jstat[ei][e]);
  setOperatingPoint ("jdyna[ei][e]", _jdyna[ei][e]);
  setOperatingPoint ("jstat[ei][t]", _jstat[ei][t]);
  setOperatingPoint ("jdyna[ei][t]", _jdyna[ei][t]);
  setOperatingPoint ("jstat[ei][ei]", _jstat[ei][ei]);
  setOperatingPoint ("jdyna[ei][ei]", _jdyna[ei][ei]);
  setOperatingPoint ("jstat[ei][bi]", _jstat[ei][bi]);
  setOperatingPoint ("jdyna[ei][bi]", _jdyna[ei][bi]);
  setOperatingPoint ("jstat[ei][bii]", _jstat[ei][bii]);
  setOperatingPoint ("jdyna[ei][bii]", _jdyna[ei][bii]);
  setOperatingPoint ("jstat[ei][ci]", _jstat[ei][ci]);
  setOperatingPoint ("jdyna[ei][ci]", _jdyna[ei][ci]);
  setOperatingPoint ("jstat[ei][ex]", _jstat[ei][ex]);
  setOperatingPoint ("jdyna[ei][ex]", _jdyna[ei][ex]);
  setOperatingPoint ("jstat[ei][exx]", _jstat[ei][exx]);
  setOperatingPoint ("jdyna[ei][exx]", _jdyna[ei][exx]);
  setOperatingPoint ("jstat[ei][cx]", _jstat[ei][cx]);
  setOperatingPoint ("jdyna[ei][cx]", _jdyna[ei][cx]);
  setOperatingPoint ("jstat[ei][ni]", _jstat[ei][ni]);
  setOperatingPoint ("jdyna[ei][ni]", _jdyna[ei][ni]);
  setOperatingPoint ("jstat[ei][nii]", _jstat[ei][nii]);
  setOperatingPoint ("jdyna[ei][nii]", _jdyna[ei][nii]);
  setOperatingPoint ("jstat[ei][niix]", _jstat[ei][niix]);
  setOperatingPoint ("jdyna[ei][niix]", _jdyna[ei][niix]);
  setOperatingPoint ("jstat[ei][niii]", _jstat[ei][niii]);
  setOperatingPoint ("jdyna[ei][niii]", _jdyna[ei][niii]);
  setOperatingPoint ("jstat[ei][niiix]", _jstat[ei][niiix]);
  setOperatingPoint ("jdyna[ei][niiix]", _jdyna[ei][niiix]);
  setOperatingPoint ("jstat[ei][niv]", _jstat[ei][niv]);
  setOperatingPoint ("jdyna[ei][niv]", _jdyna[ei][niv]);
  setOperatingPoint ("jstat[ei][nivx]", _jstat[ei][nivx]);
  setOperatingPoint ("jdyna[ei][nivx]", _jdyna[ei][nivx]);
  setOperatingPoint ("jstat[ei][ti]", _jstat[ei][ti]);
  setOperatingPoint ("jdyna[ei][ti]", _jdyna[ei][ti]);
  setOperatingPoint ("jstat[bi][c]", _jstat[bi][c]);
  setOperatingPoint ("jdyna[bi][c]", _jdyna[bi][c]);
  setOperatingPoint ("jstat[bi][b]", _jstat[bi][b]);
  setOperatingPoint ("jdyna[bi][b]", _jdyna[bi][b]);
  setOperatingPoint ("jstat[bi][e]", _jstat[bi][e]);
  setOperatingPoint ("jdyna[bi][e]", _jdyna[bi][e]);
  setOperatingPoint ("jstat[bi][t]", _jstat[bi][t]);
  setOperatingPoint ("jdyna[bi][t]", _jdyna[bi][t]);
  setOperatingPoint ("jstat[bi][ei]", _jstat[bi][ei]);
  setOperatingPoint ("jdyna[bi][ei]", _jdyna[bi][ei]);
  setOperatingPoint ("jstat[bi][bi]", _jstat[bi][bi]);
  setOperatingPoint ("jdyna[bi][bi]", _jdyna[bi][bi]);
  setOperatingPoint ("jstat[bi][bii]", _jstat[bi][bii]);
  setOperatingPoint ("jdyna[bi][bii]", _jdyna[bi][bii]);
  setOperatingPoint ("jstat[bi][ci]", _jstat[bi][ci]);
  setOperatingPoint ("jdyna[bi][ci]", _jdyna[bi][ci]);
  setOperatingPoint ("jstat[bi][ex]", _jstat[bi][ex]);
  setOperatingPoint ("jdyna[bi][ex]", _jdyna[bi][ex]);
  setOperatingPoint ("jstat[bi][exx]", _jstat[bi][exx]);
  setOperatingPoint ("jdyna[bi][exx]", _jdyna[bi][exx]);
  setOperatingPoint ("jstat[bi][cx]", _jstat[bi][cx]);
  setOperatingPoint ("jdyna[bi][cx]", _jdyna[bi][cx]);
  setOperatingPoint ("jstat[bi][ni]", _jstat[bi][ni]);
  setOperatingPoint ("jdyna[bi][ni]", _jdyna[bi][ni]);
  setOperatingPoint ("jstat[bi][nii]", _jstat[bi][nii]);
  setOperatingPoint ("jdyna[bi][nii]", _jdyna[bi][nii]);
  setOperatingPoint ("jstat[bi][niix]", _jstat[bi][niix]);
  setOperatingPoint ("jdyna[bi][niix]", _jdyna[bi][niix]);
  setOperatingPoint ("jstat[bi][niii]", _jstat[bi][niii]);
  setOperatingPoint ("jdyna[bi][niii]", _jdyna[bi][niii]);
  setOperatingPoint ("jstat[bi][niiix]", _jstat[bi][niiix]);
  setOperatingPoint ("jdyna[bi][niiix]", _jdyna[bi][niiix]);
  setOperatingPoint ("jstat[bi][niv]", _jstat[bi][niv]);
  setOperatingPoint ("jdyna[bi][niv]", _jdyna[bi][niv]);
  setOperatingPoint ("jstat[bi][nivx]", _jstat[bi][nivx]);
  setOperatingPoint ("jdyna[bi][nivx]", _jdyna[bi][nivx]);
  setOperatingPoint ("jstat[bi][ti]", _jstat[bi][ti]);
  setOperatingPoint ("jdyna[bi][ti]", _jdyna[bi][ti]);
  setOperatingPoint ("jstat[bii][c]", _jstat[bii][c]);
  setOperatingPoint ("jdyna[bii][c]", _jdyna[bii][c]);
  setOperatingPoint ("jstat[bii][b]", _jstat[bii][b]);
  setOperatingPoint ("jdyna[bii][b]", _jdyna[bii][b]);
  setOperatingPoint ("jstat[bii][e]", _jstat[bii][e]);
  setOperatingPoint ("jdyna[bii][e]", _jdyna[bii][e]);
  setOperatingPoint ("jstat[bii][t]", _jstat[bii][t]);
  setOperatingPoint ("jdyna[bii][t]", _jdyna[bii][t]);
  setOperatingPoint ("jstat[bii][ei]", _jstat[bii][ei]);
  setOperatingPoint ("jdyna[bii][ei]", _jdyna[bii][ei]);
  setOperatingPoint ("jstat[bii][bi]", _jstat[bii][bi]);
  setOperatingPoint ("jdyna[bii][bi]", _jdyna[bii][bi]);
  setOperatingPoint ("jstat[bii][bii]", _jstat[bii][bii]);
  setOperatingPoint ("jdyna[bii][bii]", _jdyna[bii][bii]);
  setOperatingPoint ("jstat[bii][ci]", _jstat[bii][ci]);
  setOperatingPoint ("jdyna[bii][ci]", _jdyna[bii][ci]);
  setOperatingPoint ("jstat[bii][ex]", _jstat[bii][ex]);
  setOperatingPoint ("jdyna[bii][ex]", _jdyna[bii][ex]);
  setOperatingPoint ("jstat[bii][exx]", _jstat[bii][exx]);
  setOperatingPoint ("jdyna[bii][exx]", _jdyna[bii][exx]);
  setOperatingPoint ("jstat[bii][cx]", _jstat[bii][cx]);
  setOperatingPoint ("jdyna[bii][cx]", _jdyna[bii][cx]);
  setOperatingPoint ("jstat[bii][ni]", _jstat[bii][ni]);
  setOperatingPoint ("jdyna[bii][ni]", _jdyna[bii][ni]);
  setOperatingPoint ("jstat[bii][nii]", _jstat[bii][nii]);
  setOperatingPoint ("jdyna[bii][nii]", _jdyna[bii][nii]);
  setOperatingPoint ("jstat[bii][niix]", _jstat[bii][niix]);
  setOperatingPoint ("jdyna[bii][niix]", _jdyna[bii][niix]);
  setOperatingPoint ("jstat[bii][niii]", _jstat[bii][niii]);
  setOperatingPoint ("jdyna[bii][niii]", _jdyna[bii][niii]);
  setOperatingPoint ("jstat[bii][niiix]", _jstat[bii][niiix]);
  setOperatingPoint ("jdyna[bii][niiix]", _jdyna[bii][niiix]);
  setOperatingPoint ("jstat[bii][niv]", _jstat[bii][niv]);
  setOperatingPoint ("jdyna[bii][niv]", _jdyna[bii][niv]);
  setOperatingPoint ("jstat[bii][nivx]", _jstat[bii][nivx]);
  setOperatingPoint ("jdyna[bii][nivx]", _jdyna[bii][nivx]);
  setOperatingPoint ("jstat[bii][ti]", _jstat[bii][ti]);
  setOperatingPoint ("jdyna[bii][ti]", _jdyna[bii][ti]);
  setOperatingPoint ("jstat[ci][c]", _jstat[ci][c]);
  setOperatingPoint ("jdyna[ci][c]", _jdyna[ci][c]);
  setOperatingPoint ("jstat[ci][b]", _jstat[ci][b]);
  setOperatingPoint ("jdyna[ci][b]", _jdyna[ci][b]);
  setOperatingPoint ("jstat[ci][e]", _jstat[ci][e]);
  setOperatingPoint ("jdyna[ci][e]", _jdyna[ci][e]);
  setOperatingPoint ("jstat[ci][t]", _jstat[ci][t]);
  setOperatingPoint ("jdyna[ci][t]", _jdyna[ci][t]);
  setOperatingPoint ("jstat[ci][ei]", _jstat[ci][ei]);
  setOperatingPoint ("jdyna[ci][ei]", _jdyna[ci][ei]);
  setOperatingPoint ("jstat[ci][bi]", _jstat[ci][bi]);
  setOperatingPoint ("jdyna[ci][bi]", _jdyna[ci][bi]);
  setOperatingPoint ("jstat[ci][bii]", _jstat[ci][bii]);
  setOperatingPoint ("jdyna[ci][bii]", _jdyna[ci][bii]);
  setOperatingPoint ("jstat[ci][ci]", _jstat[ci][ci]);
  setOperatingPoint ("jdyna[ci][ci]", _jdyna[ci][ci]);
  setOperatingPoint ("jstat[ci][ex]", _jstat[ci][ex]);
  setOperatingPoint ("jdyna[ci][ex]", _jdyna[ci][ex]);
  setOperatingPoint ("jstat[ci][exx]", _jstat[ci][exx]);
  setOperatingPoint ("jdyna[ci][exx]", _jdyna[ci][exx]);
  setOperatingPoint ("jstat[ci][cx]", _jstat[ci][cx]);
  setOperatingPoint ("jdyna[ci][cx]", _jdyna[ci][cx]);
  setOperatingPoint ("jstat[ci][ni]", _jstat[ci][ni]);
  setOperatingPoint ("jdyna[ci][ni]", _jdyna[ci][ni]);
  setOperatingPoint ("jstat[ci][nii]", _jstat[ci][nii]);
  setOperatingPoint ("jdyna[ci][nii]", _jdyna[ci][nii]);
  setOperatingPoint ("jstat[ci][niix]", _jstat[ci][niix]);
  setOperatingPoint ("jdyna[ci][niix]", _jdyna[ci][niix]);
  setOperatingPoint ("jstat[ci][niii]", _jstat[ci][niii]);
  setOperatingPoint ("jdyna[ci][niii]", _jdyna[ci][niii]);
  setOperatingPoint ("jstat[ci][niiix]", _jstat[ci][niiix]);
  setOperatingPoint ("jdyna[ci][niiix]", _jdyna[ci][niiix]);
  setOperatingPoint ("jstat[ci][niv]", _jstat[ci][niv]);
  setOperatingPoint ("jdyna[ci][niv]", _jdyna[ci][niv]);
  setOperatingPoint ("jstat[ci][nivx]", _jstat[ci][nivx]);
  setOperatingPoint ("jdyna[ci][nivx]", _jdyna[ci][nivx]);
  setOperatingPoint ("jstat[ci][ti]", _jstat[ci][ti]);
  setOperatingPoint ("jdyna[ci][ti]", _jdyna[ci][ti]);
  setOperatingPoint ("jstat[ex][c]", _jstat[ex][c]);
  setOperatingPoint ("jdyna[ex][c]", _jdyna[ex][c]);
  setOperatingPoint ("jstat[ex][b]", _jstat[ex][b]);
  setOperatingPoint ("jdyna[ex][b]", _jdyna[ex][b]);
  setOperatingPoint ("jstat[ex][e]", _jstat[ex][e]);
  setOperatingPoint ("jdyna[ex][e]", _jdyna[ex][e]);
  setOperatingPoint ("jstat[ex][t]", _jstat[ex][t]);
  setOperatingPoint ("jdyna[ex][t]", _jdyna[ex][t]);
  setOperatingPoint ("jstat[ex][ei]", _jstat[ex][ei]);
  setOperatingPoint ("jdyna[ex][ei]", _jdyna[ex][ei]);
  setOperatingPoint ("jstat[ex][bi]", _jstat[ex][bi]);
  setOperatingPoint ("jdyna[ex][bi]", _jdyna[ex][bi]);
  setOperatingPoint ("jstat[ex][bii]", _jstat[ex][bii]);
  setOperatingPoint ("jdyna[ex][bii]", _jdyna[ex][bii]);
  setOperatingPoint ("jstat[ex][ci]", _jstat[ex][ci]);
  setOperatingPoint ("jdyna[ex][ci]", _jdyna[ex][ci]);
  setOperatingPoint ("jstat[ex][ex]", _jstat[ex][ex]);
  setOperatingPoint ("jdyna[ex][ex]", _jdyna[ex][ex]);
  setOperatingPoint ("jstat[ex][exx]", _jstat[ex][exx]);
  setOperatingPoint ("jdyna[ex][exx]", _jdyna[ex][exx]);
  setOperatingPoint ("jstat[ex][cx]", _jstat[ex][cx]);
  setOperatingPoint ("jdyna[ex][cx]", _jdyna[ex][cx]);
  setOperatingPoint ("jstat[ex][ni]", _jstat[ex][ni]);
  setOperatingPoint ("jdyna[ex][ni]", _jdyna[ex][ni]);
  setOperatingPoint ("jstat[ex][nii]", _jstat[ex][nii]);
  setOperatingPoint ("jdyna[ex][nii]", _jdyna[ex][nii]);
  setOperatingPoint ("jstat[ex][niix]", _jstat[ex][niix]);
  setOperatingPoint ("jdyna[ex][niix]", _jdyna[ex][niix]);
  setOperatingPoint ("jstat[ex][niii]", _jstat[ex][niii]);
  setOperatingPoint ("jdyna[ex][niii]", _jdyna[ex][niii]);
  setOperatingPoint ("jstat[ex][niiix]", _jstat[ex][niiix]);
  setOperatingPoint ("jdyna[ex][niiix]", _jdyna[ex][niiix]);
  setOperatingPoint ("jstat[ex][niv]", _jstat[ex][niv]);
  setOperatingPoint ("jdyna[ex][niv]", _jdyna[ex][niv]);
  setOperatingPoint ("jstat[ex][nivx]", _jstat[ex][nivx]);
  setOperatingPoint ("jdyna[ex][nivx]", _jdyna[ex][nivx]);
  setOperatingPoint ("jstat[ex][ti]", _jstat[ex][ti]);
  setOperatingPoint ("jdyna[ex][ti]", _jdyna[ex][ti]);
  setOperatingPoint ("jstat[exx][c]", _jstat[exx][c]);
  setOperatingPoint ("jdyna[exx][c]", _jdyna[exx][c]);
  setOperatingPoint ("jstat[exx][b]", _jstat[exx][b]);
  setOperatingPoint ("jdyna[exx][b]", _jdyna[exx][b]);
  setOperatingPoint ("jstat[exx][e]", _jstat[exx][e]);
  setOperatingPoint ("jdyna[exx][e]", _jdyna[exx][e]);
  setOperatingPoint ("jstat[exx][t]", _jstat[exx][t]);
  setOperatingPoint ("jdyna[exx][t]", _jdyna[exx][t]);
  setOperatingPoint ("jstat[exx][ei]", _jstat[exx][ei]);
  setOperatingPoint ("jdyna[exx][ei]", _jdyna[exx][ei]);
  setOperatingPoint ("jstat[exx][bi]", _jstat[exx][bi]);
  setOperatingPoint ("jdyna[exx][bi]", _jdyna[exx][bi]);
  setOperatingPoint ("jstat[exx][bii]", _jstat[exx][bii]);
  setOperatingPoint ("jdyna[exx][bii]", _jdyna[exx][bii]);
  setOperatingPoint ("jstat[exx][ci]", _jstat[exx][ci]);
  setOperatingPoint ("jdyna[exx][ci]", _jdyna[exx][ci]);
  setOperatingPoint ("jstat[exx][ex]", _jstat[exx][ex]);
  setOperatingPoint ("jdyna[exx][ex]", _jdyna[exx][ex]);
  setOperatingPoint ("jstat[exx][exx]", _jstat[exx][exx]);
  setOperatingPoint ("jdyna[exx][exx]", _jdyna[exx][exx]);
  setOperatingPoint ("jstat[exx][cx]", _jstat[exx][cx]);
  setOperatingPoint ("jdyna[exx][cx]", _jdyna[exx][cx]);
  setOperatingPoint ("jstat[exx][ni]", _jstat[exx][ni]);
  setOperatingPoint ("jdyna[exx][ni]", _jdyna[exx][ni]);
  setOperatingPoint ("jstat[exx][nii]", _jstat[exx][nii]);
  setOperatingPoint ("jdyna[exx][nii]", _jdyna[exx][nii]);
  setOperatingPoint ("jstat[exx][niix]", _jstat[exx][niix]);
  setOperatingPoint ("jdyna[exx][niix]", _jdyna[exx][niix]);
  setOperatingPoint ("jstat[exx][niii]", _jstat[exx][niii]);
  setOperatingPoint ("jdyna[exx][niii]", _jdyna[exx][niii]);
  setOperatingPoint ("jstat[exx][niiix]", _jstat[exx][niiix]);
  setOperatingPoint ("jdyna[exx][niiix]", _jdyna[exx][niiix]);
  setOperatingPoint ("jstat[exx][niv]", _jstat[exx][niv]);
  setOperatingPoint ("jdyna[exx][niv]", _jdyna[exx][niv]);
  setOperatingPoint ("jstat[exx][nivx]", _jstat[exx][nivx]);
  setOperatingPoint ("jdyna[exx][nivx]", _jdyna[exx][nivx]);
  setOperatingPoint ("jstat[exx][ti]", _jstat[exx][ti]);
  setOperatingPoint ("jdyna[exx][ti]", _jdyna[exx][ti]);
  setOperatingPoint ("jstat[cx][c]", _jstat[cx][c]);
  setOperatingPoint ("jdyna[cx][c]", _jdyna[cx][c]);
  setOperatingPoint ("jstat[cx][b]", _jstat[cx][b]);
  setOperatingPoint ("jdyna[cx][b]", _jdyna[cx][b]);
  setOperatingPoint ("jstat[cx][e]", _jstat[cx][e]);
  setOperatingPoint ("jdyna[cx][e]", _jdyna[cx][e]);
  setOperatingPoint ("jstat[cx][t]", _jstat[cx][t]);
  setOperatingPoint ("jdyna[cx][t]", _jdyna[cx][t]);
  setOperatingPoint ("jstat[cx][ei]", _jstat[cx][ei]);
  setOperatingPoint ("jdyna[cx][ei]", _jdyna[cx][ei]);
  setOperatingPoint ("jstat[cx][bi]", _jstat[cx][bi]);
  setOperatingPoint ("jdyna[cx][bi]", _jdyna[cx][bi]);
  setOperatingPoint ("jstat[cx][bii]", _jstat[cx][bii]);
  setOperatingPoint ("jdyna[cx][bii]", _jdyna[cx][bii]);
  setOperatingPoint ("jstat[cx][ci]", _jstat[cx][ci]);
  setOperatingPoint ("jdyna[cx][ci]", _jdyna[cx][ci]);
  setOperatingPoint ("jstat[cx][ex]", _jstat[cx][ex]);
  setOperatingPoint ("jdyna[cx][ex]", _jdyna[cx][ex]);
  setOperatingPoint ("jstat[cx][exx]", _jstat[cx][exx]);
  setOperatingPoint ("jdyna[cx][exx]", _jdyna[cx][exx]);
  setOperatingPoint ("jstat[cx][cx]", _jstat[cx][cx]);
  setOperatingPoint ("jdyna[cx][cx]", _jdyna[cx][cx]);
  setOperatingPoint ("jstat[cx][ni]", _jstat[cx][ni]);
  setOperatingPoint ("jdyna[cx][ni]", _jdyna[cx][ni]);
  setOperatingPoint ("jstat[cx][nii]", _jstat[cx][nii]);
  setOperatingPoint ("jdyna[cx][nii]", _jdyna[cx][nii]);
  setOperatingPoint ("jstat[cx][niix]", _jstat[cx][niix]);
  setOperatingPoint ("jdyna[cx][niix]", _jdyna[cx][niix]);
  setOperatingPoint ("jstat[cx][niii]", _jstat[cx][niii]);
  setOperatingPoint ("jdyna[cx][niii]", _jdyna[cx][niii]);
  setOperatingPoint ("jstat[cx][niiix]", _jstat[cx][niiix]);
  setOperatingPoint ("jdyna[cx][niiix]", _jdyna[cx][niiix]);
  setOperatingPoint ("jstat[cx][niv]", _jstat[cx][niv]);
  setOperatingPoint ("jdyna[cx][niv]", _jdyna[cx][niv]);
  setOperatingPoint ("jstat[cx][nivx]", _jstat[cx][nivx]);
  setOperatingPoint ("jdyna[cx][nivx]", _jdyna[cx][nivx]);
  setOperatingPoint ("jstat[cx][ti]", _jstat[cx][ti]);
  setOperatingPoint ("jdyna[cx][ti]", _jdyna[cx][ti]);
  setOperatingPoint ("jstat[ni][c]", _jstat[ni][c]);
  setOperatingPoint ("jdyna[ni][c]", _jdyna[ni][c]);
  setOperatingPoint ("jstat[ni][b]", _jstat[ni][b]);
  setOperatingPoint ("jdyna[ni][b]", _jdyna[ni][b]);
  setOperatingPoint ("jstat[ni][e]", _jstat[ni][e]);
  setOperatingPoint ("jdyna[ni][e]", _jdyna[ni][e]);
  setOperatingPoint ("jstat[ni][t]", _jstat[ni][t]);
  setOperatingPoint ("jdyna[ni][t]", _jdyna[ni][t]);
  setOperatingPoint ("jstat[ni][ei]", _jstat[ni][ei]);
  setOperatingPoint ("jdyna[ni][ei]", _jdyna[ni][ei]);
  setOperatingPoint ("jstat[ni][bi]", _jstat[ni][bi]);
  setOperatingPoint ("jdyna[ni][bi]", _jdyna[ni][bi]);
  setOperatingPoint ("jstat[ni][bii]", _jstat[ni][bii]);
  setOperatingPoint ("jdyna[ni][bii]", _jdyna[ni][bii]);
  setOperatingPoint ("jstat[ni][ci]", _jstat[ni][ci]);
  setOperatingPoint ("jdyna[ni][ci]", _jdyna[ni][ci]);
  setOperatingPoint ("jstat[ni][ex]", _jstat[ni][ex]);
  setOperatingPoint ("jdyna[ni][ex]", _jdyna[ni][ex]);
  setOperatingPoint ("jstat[ni][exx]", _jstat[ni][exx]);
  setOperatingPoint ("jdyna[ni][exx]", _jdyna[ni][exx]);
  setOperatingPoint ("jstat[ni][cx]", _jstat[ni][cx]);
  setOperatingPoint ("jdyna[ni][cx]", _jdyna[ni][cx]);
  setOperatingPoint ("jstat[ni][ni]", _jstat[ni][ni]);
  setOperatingPoint ("jdyna[ni][ni]", _jdyna[ni][ni]);
  setOperatingPoint ("jstat[ni][nii]", _jstat[ni][nii]);
  setOperatingPoint ("jdyna[ni][nii]", _jdyna[ni][nii]);
  setOperatingPoint ("jstat[ni][niix]", _jstat[ni][niix]);
  setOperatingPoint ("jdyna[ni][niix]", _jdyna[ni][niix]);
  setOperatingPoint ("jstat[ni][niii]", _jstat[ni][niii]);
  setOperatingPoint ("jdyna[ni][niii]", _jdyna[ni][niii]);
  setOperatingPoint ("jstat[ni][niiix]", _jstat[ni][niiix]);
  setOperatingPoint ("jdyna[ni][niiix]", _jdyna[ni][niiix]);
  setOperatingPoint ("jstat[ni][niv]", _jstat[ni][niv]);
  setOperatingPoint ("jdyna[ni][niv]", _jdyna[ni][niv]);
  setOperatingPoint ("jstat[ni][nivx]", _jstat[ni][nivx]);
  setOperatingPoint ("jdyna[ni][nivx]", _jdyna[ni][nivx]);
  setOperatingPoint ("jstat[ni][ti]", _jstat[ni][ti]);
  setOperatingPoint ("jdyna[ni][ti]", _jdyna[ni][ti]);
  setOperatingPoint ("jstat[nii][c]", _jstat[nii][c]);
  setOperatingPoint ("jdyna[nii][c]", _jdyna[nii][c]);
  setOperatingPoint ("jstat[nii][b]", _jstat[nii][b]);
  setOperatingPoint ("jdyna[nii][b]", _jdyna[nii][b]);
  setOperatingPoint ("jstat[nii][e]", _jstat[nii][e]);
  setOperatingPoint ("jdyna[nii][e]", _jdyna[nii][e]);
  setOperatingPoint ("jstat[nii][t]", _jstat[nii][t]);
  setOperatingPoint ("jdyna[nii][t]", _jdyna[nii][t]);
  setOperatingPoint ("jstat[nii][ei]", _jstat[nii][ei]);
  setOperatingPoint ("jdyna[nii][ei]", _jdyna[nii][ei]);
  setOperatingPoint ("jstat[nii][bi]", _jstat[nii][bi]);
  setOperatingPoint ("jdyna[nii][bi]", _jdyna[nii][bi]);
  setOperatingPoint ("jstat[nii][bii]", _jstat[nii][bii]);
  setOperatingPoint ("jdyna[nii][bii]", _jdyna[nii][bii]);
  setOperatingPoint ("jstat[nii][ci]", _jstat[nii][ci]);
  setOperatingPoint ("jdyna[nii][ci]", _jdyna[nii][ci]);
  setOperatingPoint ("jstat[nii][ex]", _jstat[nii][ex]);
  setOperatingPoint ("jdyna[nii][ex]", _jdyna[nii][ex]);
  setOperatingPoint ("jstat[nii][exx]", _jstat[nii][exx]);
  setOperatingPoint ("jdyna[nii][exx]", _jdyna[nii][exx]);
  setOperatingPoint ("jstat[nii][cx]", _jstat[nii][cx]);
  setOperatingPoint ("jdyna[nii][cx]", _jdyna[nii][cx]);
  setOperatingPoint ("jstat[nii][ni]", _jstat[nii][ni]);
  setOperatingPoint ("jdyna[nii][ni]", _jdyna[nii][ni]);
  setOperatingPoint ("jstat[nii][nii]", _jstat[nii][nii]);
  setOperatingPoint ("jdyna[nii][nii]", _jdyna[nii][nii]);
  setOperatingPoint ("jstat[nii][niix]", _jstat[nii][niix]);
  setOperatingPoint ("jdyna[nii][niix]", _jdyna[nii][niix]);
  setOperatingPoint ("jstat[nii][niii]", _jstat[nii][niii]);
  setOperatingPoint ("jdyna[nii][niii]", _jdyna[nii][niii]);
  setOperatingPoint ("jstat[nii][niiix]", _jstat[nii][niiix]);
  setOperatingPoint ("jdyna[nii][niiix]", _jdyna[nii][niiix]);
  setOperatingPoint ("jstat[nii][niv]", _jstat[nii][niv]);
  setOperatingPoint ("jdyna[nii][niv]", _jdyna[nii][niv]);
  setOperatingPoint ("jstat[nii][nivx]", _jstat[nii][nivx]);
  setOperatingPoint ("jdyna[nii][nivx]", _jdyna[nii][nivx]);
  setOperatingPoint ("jstat[nii][ti]", _jstat[nii][ti]);
  setOperatingPoint ("jdyna[nii][ti]", _jdyna[nii][ti]);
  setOperatingPoint ("jstat[niix][c]", _jstat[niix][c]);
  setOperatingPoint ("jdyna[niix][c]", _jdyna[niix][c]);
  setOperatingPoint ("jstat[niix][b]", _jstat[niix][b]);
  setOperatingPoint ("jdyna[niix][b]", _jdyna[niix][b]);
  setOperatingPoint ("jstat[niix][e]", _jstat[niix][e]);
  setOperatingPoint ("jdyna[niix][e]", _jdyna[niix][e]);
  setOperatingPoint ("jstat[niix][t]", _jstat[niix][t]);
  setOperatingPoint ("jdyna[niix][t]", _jdyna[niix][t]);
  setOperatingPoint ("jstat[niix][ei]", _jstat[niix][ei]);
  setOperatingPoint ("jdyna[niix][ei]", _jdyna[niix][ei]);
  setOperatingPoint ("jstat[niix][bi]", _jstat[niix][bi]);
  setOperatingPoint ("jdyna[niix][bi]", _jdyna[niix][bi]);
  setOperatingPoint ("jstat[niix][bii]", _jstat[niix][bii]);
  setOperatingPoint ("jdyna[niix][bii]", _jdyna[niix][bii]);
  setOperatingPoint ("jstat[niix][ci]", _jstat[niix][ci]);
  setOperatingPoint ("jdyna[niix][ci]", _jdyna[niix][ci]);
  setOperatingPoint ("jstat[niix][ex]", _jstat[niix][ex]);
  setOperatingPoint ("jdyna[niix][ex]", _jdyna[niix][ex]);
  setOperatingPoint ("jstat[niix][exx]", _jstat[niix][exx]);
  setOperatingPoint ("jdyna[niix][exx]", _jdyna[niix][exx]);
  setOperatingPoint ("jstat[niix][cx]", _jstat[niix][cx]);
  setOperatingPoint ("jdyna[niix][cx]", _jdyna[niix][cx]);
  setOperatingPoint ("jstat[niix][ni]", _jstat[niix][ni]);
  setOperatingPoint ("jdyna[niix][ni]", _jdyna[niix][ni]);
  setOperatingPoint ("jstat[niix][nii]", _jstat[niix][nii]);
  setOperatingPoint ("jdyna[niix][nii]", _jdyna[niix][nii]);
  setOperatingPoint ("jstat[niix][niix]", _jstat[niix][niix]);
  setOperatingPoint ("jdyna[niix][niix]", _jdyna[niix][niix]);
  setOperatingPoint ("jstat[niix][niii]", _jstat[niix][niii]);
  setOperatingPoint ("jdyna[niix][niii]", _jdyna[niix][niii]);
  setOperatingPoint ("jstat[niix][niiix]", _jstat[niix][niiix]);
  setOperatingPoint ("jdyna[niix][niiix]", _jdyna[niix][niiix]);
  setOperatingPoint ("jstat[niix][niv]", _jstat[niix][niv]);
  setOperatingPoint ("jdyna[niix][niv]", _jdyna[niix][niv]);
  setOperatingPoint ("jstat[niix][nivx]", _jstat[niix][nivx]);
  setOperatingPoint ("jdyna[niix][nivx]", _jdyna[niix][nivx]);
  setOperatingPoint ("jstat[niix][ti]", _jstat[niix][ti]);
  setOperatingPoint ("jdyna[niix][ti]", _jdyna[niix][ti]);
  setOperatingPoint ("jstat[niii][c]", _jstat[niii][c]);
  setOperatingPoint ("jdyna[niii][c]", _jdyna[niii][c]);
  setOperatingPoint ("jstat[niii][b]", _jstat[niii][b]);
  setOperatingPoint ("jdyna[niii][b]", _jdyna[niii][b]);
  setOperatingPoint ("jstat[niii][e]", _jstat[niii][e]);
  setOperatingPoint ("jdyna[niii][e]", _jdyna[niii][e]);
  setOperatingPoint ("jstat[niii][t]", _jstat[niii][t]);
  setOperatingPoint ("jdyna[niii][t]", _jdyna[niii][t]);
  setOperatingPoint ("jstat[niii][ei]", _jstat[niii][ei]);
  setOperatingPoint ("jdyna[niii][ei]", _jdyna[niii][ei]);
  setOperatingPoint ("jstat[niii][bi]", _jstat[niii][bi]);
  setOperatingPoint ("jdyna[niii][bi]", _jdyna[niii][bi]);
  setOperatingPoint ("jstat[niii][bii]", _jstat[niii][bii]);
  setOperatingPoint ("jdyna[niii][bii]", _jdyna[niii][bii]);
  setOperatingPoint ("jstat[niii][ci]", _jstat[niii][ci]);
  setOperatingPoint ("jdyna[niii][ci]", _jdyna[niii][ci]);
  setOperatingPoint ("jstat[niii][ex]", _jstat[niii][ex]);
  setOperatingPoint ("jdyna[niii][ex]", _jdyna[niii][ex]);
  setOperatingPoint ("jstat[niii][exx]", _jstat[niii][exx]);
  setOperatingPoint ("jdyna[niii][exx]", _jdyna[niii][exx]);
  setOperatingPoint ("jstat[niii][cx]", _jstat[niii][cx]);
  setOperatingPoint ("jdyna[niii][cx]", _jdyna[niii][cx]);
  setOperatingPoint ("jstat[niii][ni]", _jstat[niii][ni]);
  setOperatingPoint ("jdyna[niii][ni]", _jdyna[niii][ni]);
  setOperatingPoint ("jstat[niii][nii]", _jstat[niii][nii]);
  setOperatingPoint ("jdyna[niii][nii]", _jdyna[niii][nii]);
  setOperatingPoint ("jstat[niii][niix]", _jstat[niii][niix]);
  setOperatingPoint ("jdyna[niii][niix]", _jdyna[niii][niix]);
  setOperatingPoint ("jstat[niii][niii]", _jstat[niii][niii]);
  setOperatingPoint ("jdyna[niii][niii]", _jdyna[niii][niii]);
  setOperatingPoint ("jstat[niii][niiix]", _jstat[niii][niiix]);
  setOperatingPoint ("jdyna[niii][niiix]", _jdyna[niii][niiix]);
  setOperatingPoint ("jstat[niii][niv]", _jstat[niii][niv]);
  setOperatingPoint ("jdyna[niii][niv]", _jdyna[niii][niv]);
  setOperatingPoint ("jstat[niii][nivx]", _jstat[niii][nivx]);
  setOperatingPoint ("jdyna[niii][nivx]", _jdyna[niii][nivx]);
  setOperatingPoint ("jstat[niii][ti]", _jstat[niii][ti]);
  setOperatingPoint ("jdyna[niii][ti]", _jdyna[niii][ti]);
  setOperatingPoint ("jstat[niiix][c]", _jstat[niiix][c]);
  setOperatingPoint ("jdyna[niiix][c]", _jdyna[niiix][c]);
  setOperatingPoint ("jstat[niiix][b]", _jstat[niiix][b]);
  setOperatingPoint ("jdyna[niiix][b]", _jdyna[niiix][b]);
  setOperatingPoint ("jstat[niiix][e]", _jstat[niiix][e]);
  setOperatingPoint ("jdyna[niiix][e]", _jdyna[niiix][e]);
  setOperatingPoint ("jstat[niiix][t]", _jstat[niiix][t]);
  setOperatingPoint ("jdyna[niiix][t]", _jdyna[niiix][t]);
  setOperatingPoint ("jstat[niiix][ei]", _jstat[niiix][ei]);
  setOperatingPoint ("jdyna[niiix][ei]", _jdyna[niiix][ei]);
  setOperatingPoint ("jstat[niiix][bi]", _jstat[niiix][bi]);
  setOperatingPoint ("jdyna[niiix][bi]", _jdyna[niiix][bi]);
  setOperatingPoint ("jstat[niiix][bii]", _jstat[niiix][bii]);
  setOperatingPoint ("jdyna[niiix][bii]", _jdyna[niiix][bii]);
  setOperatingPoint ("jstat[niiix][ci]", _jstat[niiix][ci]);
  setOperatingPoint ("jdyna[niiix][ci]", _jdyna[niiix][ci]);
  setOperatingPoint ("jstat[niiix][ex]", _jstat[niiix][ex]);
  setOperatingPoint ("jdyna[niiix][ex]", _jdyna[niiix][ex]);
  setOperatingPoint ("jstat[niiix][exx]", _jstat[niiix][exx]);
  setOperatingPoint ("jdyna[niiix][exx]", _jdyna[niiix][exx]);
  setOperatingPoint ("jstat[niiix][cx]", _jstat[niiix][cx]);
  setOperatingPoint ("jdyna[niiix][cx]", _jdyna[niiix][cx]);
  setOperatingPoint ("jstat[niiix][ni]", _jstat[niiix][ni]);
  setOperatingPoint ("jdyna[niiix][ni]", _jdyna[niiix][ni]);
  setOperatingPoint ("jstat[niiix][nii]", _jstat[niiix][nii]);
  setOperatingPoint ("jdyna[niiix][nii]", _jdyna[niiix][nii]);
  setOperatingPoint ("jstat[niiix][niix]", _jstat[niiix][niix]);
  setOperatingPoint ("jdyna[niiix][niix]", _jdyna[niiix][niix]);
  setOperatingPoint ("jstat[niiix][niii]", _jstat[niiix][niii]);
  setOperatingPoint ("jdyna[niiix][niii]", _jdyna[niiix][niii]);
  setOperatingPoint ("jstat[niiix][niiix]", _jstat[niiix][niiix]);
  setOperatingPoint ("jdyna[niiix][niiix]", _jdyna[niiix][niiix]);
  setOperatingPoint ("jstat[niiix][niv]", _jstat[niiix][niv]);
  setOperatingPoint ("jdyna[niiix][niv]", _jdyna[niiix][niv]);
  setOperatingPoint ("jstat[niiix][nivx]", _jstat[niiix][nivx]);
  setOperatingPoint ("jdyna[niiix][nivx]", _jdyna[niiix][nivx]);
  setOperatingPoint ("jstat[niiix][ti]", _jstat[niiix][ti]);
  setOperatingPoint ("jdyna[niiix][ti]", _jdyna[niiix][ti]);
  setOperatingPoint ("jstat[niv][c]", _jstat[niv][c]);
  setOperatingPoint ("jdyna[niv][c]", _jdyna[niv][c]);
  setOperatingPoint ("jstat[niv][b]", _jstat[niv][b]);
  setOperatingPoint ("jdyna[niv][b]", _jdyna[niv][b]);
  setOperatingPoint ("jstat[niv][e]", _jstat[niv][e]);
  setOperatingPoint ("jdyna[niv][e]", _jdyna[niv][e]);
  setOperatingPoint ("jstat[niv][t]", _jstat[niv][t]);
  setOperatingPoint ("jdyna[niv][t]", _jdyna[niv][t]);
  setOperatingPoint ("jstat[niv][ei]", _jstat[niv][ei]);
  setOperatingPoint ("jdyna[niv][ei]", _jdyna[niv][ei]);
  setOperatingPoint ("jstat[niv][bi]", _jstat[niv][bi]);
  setOperatingPoint ("jdyna[niv][bi]", _jdyna[niv][bi]);
  setOperatingPoint ("jstat[niv][bii]", _jstat[niv][bii]);
  setOperatingPoint ("jdyna[niv][bii]", _jdyna[niv][bii]);
  setOperatingPoint ("jstat[niv][ci]", _jstat[niv][ci]);
  setOperatingPoint ("jdyna[niv][ci]", _jdyna[niv][ci]);
  setOperatingPoint ("jstat[niv][ex]", _jstat[niv][ex]);
  setOperatingPoint ("jdyna[niv][ex]", _jdyna[niv][ex]);
  setOperatingPoint ("jstat[niv][exx]", _jstat[niv][exx]);
  setOperatingPoint ("jdyna[niv][exx]", _jdyna[niv][exx]);
  setOperatingPoint ("jstat[niv][cx]", _jstat[niv][cx]);
  setOperatingPoint ("jdyna[niv][cx]", _jdyna[niv][cx]);
  setOperatingPoint ("jstat[niv][ni]", _jstat[niv][ni]);
  setOperatingPoint ("jdyna[niv][ni]", _jdyna[niv][ni]);
  setOperatingPoint ("jstat[niv][nii]", _jstat[niv][nii]);
  setOperatingPoint ("jdyna[niv][nii]", _jdyna[niv][nii]);
  setOperatingPoint ("jstat[niv][niix]", _jstat[niv][niix]);
  setOperatingPoint ("jdyna[niv][niix]", _jdyna[niv][niix]);
  setOperatingPoint ("jstat[niv][niii]", _jstat[niv][niii]);
  setOperatingPoint ("jdyna[niv][niii]", _jdyna[niv][niii]);
  setOperatingPoint ("jstat[niv][niiix]", _jstat[niv][niiix]);
  setOperatingPoint ("jdyna[niv][niiix]", _jdyna[niv][niiix]);
  setOperatingPoint ("jstat[niv][niv]", _jstat[niv][niv]);
  setOperatingPoint ("jdyna[niv][niv]", _jdyna[niv][niv]);
  setOperatingPoint ("jstat[niv][nivx]", _jstat[niv][nivx]);
  setOperatingPoint ("jdyna[niv][nivx]", _jdyna[niv][nivx]);
  setOperatingPoint ("jstat[niv][ti]", _jstat[niv][ti]);
  setOperatingPoint ("jdyna[niv][ti]", _jdyna[niv][ti]);
  setOperatingPoint ("jstat[nivx][c]", _jstat[nivx][c]);
  setOperatingPoint ("jdyna[nivx][c]", _jdyna[nivx][c]);
  setOperatingPoint ("jstat[nivx][b]", _jstat[nivx][b]);
  setOperatingPoint ("jdyna[nivx][b]", _jdyna[nivx][b]);
  setOperatingPoint ("jstat[nivx][e]", _jstat[nivx][e]);
  setOperatingPoint ("jdyna[nivx][e]", _jdyna[nivx][e]);
  setOperatingPoint ("jstat[nivx][t]", _jstat[nivx][t]);
  setOperatingPoint ("jdyna[nivx][t]", _jdyna[nivx][t]);
  setOperatingPoint ("jstat[nivx][ei]", _jstat[nivx][ei]);
  setOperatingPoint ("jdyna[nivx][ei]", _jdyna[nivx][ei]);
  setOperatingPoint ("jstat[nivx][bi]", _jstat[nivx][bi]);
  setOperatingPoint ("jdyna[nivx][bi]", _jdyna[nivx][bi]);
  setOperatingPoint ("jstat[nivx][bii]", _jstat[nivx][bii]);
  setOperatingPoint ("jdyna[nivx][bii]", _jdyna[nivx][bii]);
  setOperatingPoint ("jstat[nivx][ci]", _jstat[nivx][ci]);
  setOperatingPoint ("jdyna[nivx][ci]", _jdyna[nivx][ci]);
  setOperatingPoint ("jstat[nivx][ex]", _jstat[nivx][ex]);
  setOperatingPoint ("jdyna[nivx][ex]", _jdyna[nivx][ex]);
  setOperatingPoint ("jstat[nivx][exx]", _jstat[nivx][exx]);
  setOperatingPoint ("jdyna[nivx][exx]", _jdyna[nivx][exx]);
  setOperatingPoint ("jstat[nivx][cx]", _jstat[nivx][cx]);
  setOperatingPoint ("jdyna[nivx][cx]", _jdyna[nivx][cx]);
  setOperatingPoint ("jstat[nivx][ni]", _jstat[nivx][ni]);
  setOperatingPoint ("jdyna[nivx][ni]", _jdyna[nivx][ni]);
  setOperatingPoint ("jstat[nivx][nii]", _jstat[nivx][nii]);
  setOperatingPoint ("jdyna[nivx][nii]", _jdyna[nivx][nii]);
  setOperatingPoint ("jstat[nivx][niix]", _jstat[nivx][niix]);
  setOperatingPoint ("jdyna[nivx][niix]", _jdyna[nivx][niix]);
  setOperatingPoint ("jstat[nivx][niii]", _jstat[nivx][niii]);
  setOperatingPoint ("jdyna[nivx][niii]", _jdyna[nivx][niii]);
  setOperatingPoint ("jstat[nivx][niiix]", _jstat[nivx][niiix]);
  setOperatingPoint ("jdyna[nivx][niiix]", _jdyna[nivx][niiix]);
  setOperatingPoint ("jstat[nivx][niv]", _jstat[nivx][niv]);
  setOperatingPoint ("jdyna[nivx][niv]", _jdyna[nivx][niv]);
  setOperatingPoint ("jstat[nivx][nivx]", _jstat[nivx][nivx]);
  setOperatingPoint ("jdyna[nivx][nivx]", _jdyna[nivx][nivx]);
  setOperatingPoint ("jstat[nivx][ti]", _jstat[nivx][ti]);
  setOperatingPoint ("jdyna[nivx][ti]", _jdyna[nivx][ti]);
  setOperatingPoint ("jstat[ti][c]", _jstat[ti][c]);
  setOperatingPoint ("jdyna[ti][c]", _jdyna[ti][c]);
  setOperatingPoint ("jstat[ti][b]", _jstat[ti][b]);
  setOperatingPoint ("jdyna[ti][b]", _jdyna[ti][b]);
  setOperatingPoint ("jstat[ti][e]", _jstat[ti][e]);
  setOperatingPoint ("jdyna[ti][e]", _jdyna[ti][e]);
  setOperatingPoint ("jstat[ti][t]", _jstat[ti][t]);
  setOperatingPoint ("jdyna[ti][t]", _jdyna[ti][t]);
  setOperatingPoint ("jstat[ti][ei]", _jstat[ti][ei]);
  setOperatingPoint ("jdyna[ti][ei]", _jdyna[ti][ei]);
  setOperatingPoint ("jstat[ti][bi]", _jstat[ti][bi]);
  setOperatingPoint ("jdyna[ti][bi]", _jdyna[ti][bi]);
  setOperatingPoint ("jstat[ti][bii]", _jstat[ti][bii]);
  setOperatingPoint ("jdyna[ti][bii]", _jdyna[ti][bii]);
  setOperatingPoint ("jstat[ti][ci]", _jstat[ti][ci]);
  setOperatingPoint ("jdyna[ti][ci]", _jdyna[ti][ci]);
  setOperatingPoint ("jstat[ti][ex]", _jstat[ti][ex]);
  setOperatingPoint ("jdyna[ti][ex]", _jdyna[ti][ex]);
  setOperatingPoint ("jstat[ti][exx]", _jstat[ti][exx]);
  setOperatingPoint ("jdyna[ti][exx]", _jdyna[ti][exx]);
  setOperatingPoint ("jstat[ti][cx]", _jstat[ti][cx]);
  setOperatingPoint ("jdyna[ti][cx]", _jdyna[ti][cx]);
  setOperatingPoint ("jstat[ti][ni]", _jstat[ti][ni]);
  setOperatingPoint ("jdyna[ti][ni]", _jdyna[ti][ni]);
  setOperatingPoint ("jstat[ti][nii]", _jstat[ti][nii]);
  setOperatingPoint ("jdyna[ti][nii]", _jdyna[ti][nii]);
  setOperatingPoint ("jstat[ti][niix]", _jstat[ti][niix]);
  setOperatingPoint ("jdyna[ti][niix]", _jdyna[ti][niix]);
  setOperatingPoint ("jstat[ti][niii]", _jstat[ti][niii]);
  setOperatingPoint ("jdyna[ti][niii]", _jdyna[ti][niii]);
  setOperatingPoint ("jstat[ti][niiix]", _jstat[ti][niiix]);
  setOperatingPoint ("jdyna[ti][niiix]", _jdyna[ti][niiix]);
  setOperatingPoint ("jstat[ti][niv]", _jstat[ti][niv]);
  setOperatingPoint ("jdyna[ti][niv]", _jdyna[ti][niv]);
  setOperatingPoint ("jstat[ti][nivx]", _jstat[ti][nivx]);
  setOperatingPoint ("jdyna[ti][nivx]", _jdyna[ti][nivx]);
  setOperatingPoint ("jstat[ti][ti]", _jstat[ti][ti]);
  setOperatingPoint ("jdyna[ti][ti]", _jdyna[ti][ti]);
  setOperatingPoint ("white_pwr[nii][nii]", _white_pwr[nii][nii]);
  setOperatingPoint ("white_pwr[cx][bii]", _white_pwr[cx][bii]);
  setOperatingPoint ("white_pwr[exx][bii]", _white_pwr[exx][bii]);
  setOperatingPoint ("white_pwr[ex][bii]", _white_pwr[ex][bii]);
  setOperatingPoint ("white_pwr[c][ci]", _white_pwr[c][ci]);
  setOperatingPoint ("white_pwr[e][ei]", _white_pwr[e][ei]);
  setOperatingPoint ("white_pwr[b][bi]", _white_pwr[b][bi]);
  setOperatingPoint ("white_pwr[bii][bi]", _white_pwr[bii][bi]);
  setOperatingPoint ("white_pwr[ni][ni]", _white_pwr[ni][ni]);
  setOperatingPoint ("white_pwr[bii][ei]", _white_pwr[bii][ei]);
  setOperatingPoint ("flicker_pwr[niv][niv]", _flicker_pwr[niv][niv]);
  setOperatingPoint ("flicker_pwr[niii][niii]", _flicker_pwr[niii][niii]);
#endif /* DEBUG */
}

/* Load operating points. */
void HBT_X::loadOperatingPoints (void)
{
}

/* Calculate operating points. */
void HBT_X::calcOperatingPoints (void)
{
}

/* Initialization of AC analysis. */
void HBT_X::initAC (void)
{
  allocMatrixMNA ();
}

/* Perform AC calculations. */
void HBT_X::calcAC (nr_double_t frequency)
{
  setMatrixY (calcMatrixY (frequency));
}

/* Compute Y-matrix for AC analysis. */
matrix HBT_X::calcMatrixY (nr_double_t frequency)
{
  _freq = frequency;
  saveOperatingPoints ();
  matrix y (19);

  for (int i1 = 0; i1 < 19; i1++) {
    for (int i2 = 0; i2 < 19; i2++) {
      y (i1,i2) = rect (_jstat[i1][i2], _jdyna[i1][i2] * 2 * M_PI * _freq);
    }
  }

  return y;
}

/* Initialization of S-parameter analysis. */
void HBT_X::initSP (void)
{
  allocMatrixS ();
}

/* Perform S-parameter calculations. */
void HBT_X::calcSP (nr_double_t frequency)
{
  setMatrixS (ytos (calcMatrixY (frequency)));   
}

/* Initialization of transient analysis. */
void HBT_X::initTR (void)
{
  setStates (2 * 19 * 19);
  initDC ();
}

/* Perform transient analysis iteration step. */
void HBT_X::calcTR (nr_double_t)
{
  doHB = 0;
  doAC = 1;
  doTR = 1;
  calcDC ();

  int i1, i2, i3, i4, state;

  // 2-node charge integrations
  for (i1 = 0; i1 < 19; i1++) {
  for (i2 = 0; i2 < 19; i2++) {
    state = 2 * (i2 + 19 * i1);
    if (i1 != i2)
    if (_charges[i1][i2] != 0.0)
      transientCapacitanceQ (state, i1, i2, _charges[i1][i2]);
  } }

  // 1-node charge integrations
  for (i1 = 0; i1 < 19; i1++) {
    state = 2 * (i1 + 19 * i1);
    if (_charges[i1][i1] != 0.0)
      transientCapacitanceQ (state, i1, _charges[i1][i1]);
  }

  // charge: 2-node, voltage: 2-node
  for (i1 = 0; i1 < 19; i1++) {
  for (i2 = 0; i2 < 19; i2++) {
  if (i1 != i2)
  for (i3 = 0; i3 < 19; i3++) {
  for (i4 = 0; i4 < 19; i4++) {
    if (i3 != i4)
    if (_caps[i1][i2][i3][i4] != 0.0)
      transientCapacitanceC (i1, i2, i3, i4, _caps[i1][i2][i3][i4], BP(i3,i4));
  } } } }

  // charge: 2-node, voltage: 1-node
  for (i1 = 0; i1 < 19; i1++) {
  for (i2 = 0; i2 < 19; i2++) {
  if (i1 != i2)
  for (i3 = 0; i3 < 19; i3++) {
    if (_caps[i1][i2][i3][i3] != 0.0)
      transientCapacitanceC2Q (i1, i2, i3, _caps[i1][i2][i3][i3], NP(i3));
  } } }

  // charge: 1-node, voltage: 2-node
  for (i1 = 0; i1 < 19; i1++) {
  for (i3 = 0; i3 < 19; i3++) {
  for (i4 = 0; i4 < 19; i4++) {
    if (i3 != i4)
    if (_caps[i1][i1][i3][i4] != 0.0)
      transientCapacitanceC2V (i1, i3, i4, _caps[i1][i1][i3][i4], BP(i3,i4));
  } } }
     
  // charge: 1-node, voltage: 1-node
  for (i1 = 0; i1 < 19; i1++) {
  for (i3 = 0; i3 < 19; i3++) {
    if (_caps[i1][i1][i3][i3] != 0.0)
      transientCapacitanceC (i1, i3, _caps[i1][i1][i3][i3], NP(i3));
  } }
}

/* Compute Cy-matrix for AC noise analysis. */
matrix HBT_X::calcMatrixCy (nr_double_t frequency) 
{
  _freq = frequency;
  matrix cy (19);

  _load_flickernoise1 (niv,_flicker_pwr[niv][niv],_flicker_exp[niv][niv]);
  _load_flickernoise1 (niii,_flicker_pwr[niii][niii],_flicker_exp[niii][niii]);
  _load_whitenoise1 (nii,_white_pwr[nii][nii]);
  _load_whitenoise2 (cx,bii,_white_pwr[cx][bii]);
  _load_whitenoise2 (exx,bii,_white_pwr[exx][bii]);
  _load_whitenoise2 (ex,bii,_white_pwr[ex][bii]);
  _load_whitenoise2 (c,ci,_white_pwr[c][ci]);
  _load_whitenoise2 (e,ei,_white_pwr[e][ei]);
  _load_whitenoise2 (b,bi,_white_pwr[b][bi]);
  _load_whitenoise2 (bii,bi,_white_pwr[bii][bi]);
  _load_whitenoise1 (ni,_white_pwr[ni][ni]);
  _load_whitenoise2 (bii,ei,_white_pwr[bii][ei]);

  return cy;
}

/* Perform AC noise computations. */
void HBT_X::calcNoiseAC (nr_double_t frequency) 
{
  setMatrixN (calcMatrixCy (frequency));
}

/* Perform S-parameter noise computations. */
void HBT_X::calcNoiseSP (nr_double_t frequency) 
{
  setMatrixN (cytocs (calcMatrixCy (frequency) * z0, getMatrixS ()));
}

/* Initialization of HB analysis. */
void HBT_X::initHB (int)
{
  initDC ();
  allocMatrixHB ();
}

/* Perform HB analysis. */
void HBT_X::calcHB (int)
{
  doHB = 1;
  doAC = 1;
  doTR = 0;

  // jacobian dI/dV and currents get filled
  calcDC ();
  saveOperatingPoints ();

  // fill in HB matrices
  for (int i1 = 0; i1 < 19; i1++) {
    setQ  (i1, _qhs[i1]); // charges
    setCV (i1, _chs[i1]); // jacobian dQ/dV * V
    setGV (i1, _ghs[i1]); // jacobian dI/dV * V
    for (int i2 = 0; i2 < 19; i2++) {
      setQV (i1, i2, _jdyna[i1][i2]); // jacobian dQ/dV
    }
  }
}
