% File src/library/base/man/table.Rd
% Part of the R package, http://www.R-project.org
% Copyright 1995-2007 R Core Development Team
% Distributed under GPL 2 or later

\name{table}
\title{Cross Tabulation and Table Creation}
\alias{table}
\alias{summary.table}
\alias{print.summary.table}%% {print.table} is in ./print.Rd
\alias{as.data.frame.table}
\alias{as.table}
\alias{as.table.default}
\alias{is.table}

\concept{counts}
\concept{frequencies}
\concept{occurrences}
\concept{contingency table}

\description{
  \code{table} uses the cross-classifying factors to build a contingency
  table of the counts at each combination of factor levels.
}
\usage{
table(\dots, exclude = c(NA, NaN), dnn = list.names(...),
      deparse.level = 1)
as.table(x, \dots)
is.table(x)

\method{as.data.frame}{table}(x, row.names = NULL, \dots,
              responseName = "Freq")
}
\arguments{
  \item{\dots}{one of more objects which can be interpreted as factors
    (including character strings), or a list (or data frame) whose
    components can be so interpreted.  (For \code{as.table} and
    \code{as.data.frame}, arguments passed to specific methods.)}
  \item{exclude}{values to use in the exclude argument of \code{\link{factor}}
    when interpreting non-factor objects; if specified, levels to remove
    from all factors in \code{\dots}.}
  \item{dnn}{the names to be given to the dimensions in the result (the
    \emph{dimnames names}).}
  \item{deparse.level}{controls how the default \code{dnn} is
    constructed.  See details.}
  \item{x}{an arbitrary \R object, or an object inheriting from class
    \code{"table"} for the \code{as.data.frame} method.}
  \item{row.names}{a character vector giving the row names for the data
    frame.}
  \item{responseName}{The name to be used for the column of
    table entries, usually counts.}
}
\details{
  If the argument \code{dnn} is not supplied, the internal function
  \code{list.names} is called to compute the \sQuote{dimname names}.  If the
  arguments in \code{\dots} are named, those names are used.  For the
  remaining arguments, \code{deparse.level = 0} gives an empty name,
  \code{deparse.level = 1} uses the supplied argument if it is a symbol,
  and \code{deparse.level = 2} will deparse the argument.

  Only when \code{exclude} is specified (i.e., not by default), will
  \code{table} potentially drop levels of factor arguments.

  Note (again!) that \code{exclude} has no effect on factor arguments;
  if these contain \code{\link{NA}}'s and you want to tabulate them, use
  \code{table(factor(., exclude=NULL), ...)}; see the examples.

  The \code{summary} method for class \code{"table"} (used for objects
  created by \code{table} or \code{\link{xtabs}}) which gives basic
  information and performs a chi-squared test for independence of
  factors (note that the function \code{\link{chisq.test}} currently
  only handles 2-d tables).
}
\value{
  \code{table()} returns a \emph{contingency table}, an object of
  \code{\link[base]{class}} \code{"table"}, an array of integer values.
  Note that unlike S the result is always an array, a 1D array if one
  factor is given.

  \code{as.table} and \code{is.table} coerce to and test for contingency
  table, respectively.

  The \code{as.data.frame} method for objects inheriting from class
  \code{"table"} can be used to convert the array-based representation
  of a contingency table to a data frame containing the classifying
  factors and the corresponding entries (the latter as component
  named by \code{responseName}).  This is the inverse of \code{\link{xtabs}}.
}
\seealso{
  \code{\link{tabulate}} is the underlying function and allows finer
  control. 

  Use \code{\link{ftable}} for printing (and more) of
  multidimensional tables.  \code{\link{margin.table}},
  \code{\link{prop.table}}, \code{\link{addmargins}}.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\examples{
require(stats) # for rpois and xtabs
## Simple frequency distribution
table(rpois(100,5))
## Check the design:
with(warpbreaks, table(wool, tension))
table(state.division, state.region)

# simple two-way contingency table
with(airquality, table(cut(Temp, quantile(Temp)), Month))

a <- letters[1:3]
table(a, sample(a))                    # dnn is c("a", "")
table(a, sample(a), deparse.level = 0) # dnn is c("", "")
table(a, sample(a), deparse.level = 2) # dnn is c("a", "sample(a)")

## xtabs() <-> as.data.frame.table() :
UCBAdmissions ## already a contingency table
DF <- as.data.frame(UCBAdmissions)
class(tab <- xtabs(Freq ~ ., DF)) # xtabs & table
## tab *is* "the same" as the original table:
all(tab == UCBAdmissions)
all.equal(dimnames(tab), dimnames(UCBAdmissions))

a <- rep(c(NA, 1/0:3), 10)
table(a)
table(a, exclude=NULL)
b <- factor(rep(c("A","B","C"), 10))
table(b)
table(b, exclude="B")
d <- factor(rep(c("A","B","C"), 10), levels=c("A","B","C","D","E"))
table(d, exclude="B")
print(table(b,d), zero.print = ".")

## NA counting:
is.na(d) <- 3:4
d. <- factor(d, exclude=NULL)
d.[1:7]
table(d.) # ", exclude = NULL" is not needed
## i.e., if you want to count the NA's of 'd', use
table(factor(d, exclude = NULL))
}
\keyword{category}
