#
#  Ordinal: A Library of Ordinal Models
#  Copyright (C) 1998, 1999, 2000, 2001 P.J. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public Licence as published by
#  the Free Software Foundation; either version 2 of the Licence, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public Licence for more details.
#
#  You should have received a copy of the GNU General Public Licence
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#  plot.ordinal(z,ccov=NULL,main=NULL,xlab=NULL,ylab=NULL,xlim=NULL,
#               ylim=NULL,lty=NULL,add=F,axes=F,bty="n",at=NULL,
#               touch=F,...)
#  moprofile(z,...)
#  moprofile.default(z,curve.type="probability")
#  moprofile.lcr(z,curve.type="probability")
#  moprofile.kalordinal(z,curve.type="probability")
#  plot.moprofile(z,nind=1,observed=T,main=NULL,xlab=NULL,ylab=NULL,
#                 xlim=NULL,ylim=NULL,lty=NULL,pch=NULL,add=F,axes=F,
#                 bty="n",at=NULL,touch=F,...)
#  ioprofile(z,...)
#  ioprofile.default(z,curve.type="mean")
#  ioprofile.kalordinal(z,curve.type="mean")
#  plot.ioprofile(z,nind=1,observed=T,main=NULL,xlab=NULL,ylab=NULL,
#                 xlim=NULL,ylim=NULL,lty=NULL,pch=NULL,add=F,axes=F,
#                 bty="n",at=NULL,touch=F,...)
#  poprofile(mu,...)
#  poprofile.default(mu,pintercept,preg,pinitial=NULL,
#                    depend="independence",times=NULL,
#                    distribution="multinomial",
#                    curve.type="probability")
#  plot.poprofile(z,main="Predicted profile",xlab=NULL,ylab=NULL,
#                 xlim=NULL,ylim=NULL,lty=NULL,add=F,axes=F,bty="n",
#                 at=NULL,touch=F,...)
#  hist.repeated(z,ccov=NULL,breaks=NULL,nclass=NULL,col=NULL,
#                border=par("fg"),levels=NULL,side=F,main=NULL,
#                xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,add=F,
#                axes=F,bty="n",at=NULL,touch=F,...)
#  hist.ordinal(z,ccov=NULL,breaks=NULL,nclass=NULL,col=NULL,
#               border=par("fg"),levels=F,side=F,main=NULL,
#               xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,add=F,
#               axes=F,bty="n",at=NULL,touch=F,...)
#
#  DESCRIPTION
#
#    Utility functions for ploting ordinal repeated measurements (the
#  response, marginal profiles, individual profiles, predicted
#  profiles, and respectively the equivalent histograms).
#

plot.ordinal <- function(z,ccov=NULL,main=NULL,xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,lty=NULL,add=F,axes=F,bty="n",at=NULL,touch=F,...) {
  if(inherits(z,"response")) {
    if(!is.null(ccov))
      warning("No covariates available in response objects.")
    ccov <- NULL
    z <- rmna(z)
  }
  if(!inherits(z,"repeated"))
    stop("The object z must either be a response or a repeated object.")
  nc <- length(unique(z$response$y))-1
  if(is.null(z$response$times))
    stop("No times availlable.")
  if(is.null(xlim))
    xlim <- range(z$response$times)
  if(is.null(ylim))
    ylim <- c(0,1)
  if(axes)
    xaxt <- yaxt <- "s"
  else
    xaxt <- yaxt <- "n"
  if(touch) {
    xaxs <- yaxs <- "i"
    xaxt <- yaxt <- "n"
  }
  else
    xaxs <- yaxs <- "r"
  if(!is.null(lty))
    if(length(lty)!=1&&length(lty)!=nc) {
      warning(paste("Dropping contents of option lty as it must have lentgh ",nc,".",sep=""))
      lty <- NULL
    }
    else
      if(length(lty)==1)
        lty <- rep(lty,nc)
  if(!is.null(ccov)) {
    if(length(ccov)!=1) {
      warning("Only using first covariate.")
      ccov <- ccov[1]
    }
    cov.name <- ccov
    ccov <- covariates(z,names=ccov)
    if(is.data.frame(ccov))
      ccov <- codes(ccov)
    np <- unique(ccov)
    ns <- length(np)
    if(ns>25)
      stop("The covariate chosen has more than 25 levels (too many categories for plotting).")
    if(length(ccov)==length(nobs(z)))
      ccov <- rep(ccov,nobs(z))
    if(is.null(main))
      main <- paste(cov.name," ",np,".",sep="")
    else
      if(length(main)!=1&&length(main)!=ns) {
        warning(paste("Dropping contents of option main as it must have lentgh ",ns,".",sep=""))
        main <- paste(cov.name," ",np,".",sep="")
      }
      else
        if(length(main)==1)
          main <- rep(main,ns)
    if(is.null(xlab))
      xlab <- rep("Time",ns)
    else
      if(length(xlab)!=1&&length(xlab)!=ns) {
        warning(paste("Dropping contents of option xlab as it must have lentgh ",ns,".",sep=""))
        xlab <- rep("Time",ns)
      }
      else
        if(length(xlab)==1)
          xlab <- rep(xlab,ns)
    if(is.null(ylab))
      ylab <- rep("Cumulative probabilities",ns)
    else
      if(length(ylab)!=1&&length(ylab)!=ns) {
        warning(paste("Dropping contents of option ylab as it must have lentgh ",ns,".",sep=""))
        ylab <- rep("Cumulative probabilities",ns)
      }
      else
        if(length(ylab)==1)
          ylab <- rep(ylab,ns)
  }
  else {
    ccov <- np <- ns <- 1
    if(is.null(main))
      main <- "Cumulative time profile."
    else
      if(length(main)!=1) {
        warning("Dropping contents of option main as it must have lentgh 1.")
        main <- "Cumulative time profile."
      }
    if(is.null(xlab))
      xlab <- "Time"
    else
      if(length(xlab)!=1) {
        warning("Dropping contents of option xlab as it must have lentgh 1.")
        xlab <- "Time"
      }
    if(is.null(ylab))
      ylab <- "Cumulative probabilities"
    else
      if(length(ylab)!=1) {
        warning("Dropping contents of option ylab as it must have lentgh 1.")
        ylab <- "Cumulative probabilities"
      }
  }
  if(ns!=1&&!add)
    if(ns==2)
      oldpar <- par(mfrow=c(1,2))
    else {
      ng <- ceiling(sqrt(ns))
      ng <- c(ceiling(ns/ng),ng)
      oldpar <- par(mfrow=ng)
    }
  i <- 0
  for(k in np) {
    i <- i+1
    times <- sort(unique(times(z)[ccov==k]))
    ntimes <- length(times)
    if(is.null(weights(z)))
      tab <- table(response(z)[ccov==k],times(z)[ccov==k])
    else
      tab <- tapply(weights(z)[ccov==k],list(as.vector(response(z)[ccov==k]),times(z)[ccov==k]),sum)
    lt <- if(is.null(lty)) 1 else lty[1]
    plot(times,apply(matrix(tab[1,],ncol=ntimes),2,sum)/apply(tab,2,sum),main=main[i],xlim=xlim,ylim=ylim,ylab=ylab[i],xlab=xlab[i],type="l",lty=lt,
         xaxt=xaxt,yaxt=yaxt,xaxs=xaxs,yaxs=yaxs,bty=bty,...)
    if(!axes||touch) {
      if(!axes)
        if(is.null(at))
          axis(1,at=times,...)
        else
          axis(1,at=at,...)
      else
        axis(1,at=pretty(xlim),...)
      if(touch)
        axis(2,at=pretty(ylim),...)
      else
        axis(2,...)
    }
    if(nc>1)
      for(j in 2:nc) {
        lt <- if(is.null(lty)) lt%%6+1 else lty[j]
        lines(times,apply(matrix(tab[1:j,],ncol=ntimes),2,sum)/apply(tab,2,sum),lty=lt,...)
      }
  }
  if(ns!=1&&!add)
    par(oldpar)
}

moprofile <- function(z,...) UseMethod("moprofile")

moprofile.default <- function(z,curve.type="probability") {
  if(inherits(z$response,"response"))
    stop("Object z must contain the response.")
  tmp <- c("probability","cumulative")
  curve.type <- match.arg(curve.type,tmp)
  if(curve.type=="probability")
    if(is.null(z$pred))
      stop("Highest probabilities not available (pred).")
    else
      if(is.null(z$cpred))
        stop("Cumulative probabilities not available (cpred).")
  z$curve.type <- curve.type
  class(z) <- c("moprofile",class(z))
  invisible(z)
}

moprofile.lcr <- function(z,curve.type="probability") {
  z$curve.type <- match.arg(curve.type,c("probability","cumulative"))
  class(z) <- c("moprofile",class(z))
  invisible(z)
}

moprofile.kalordinal <- function(z,curve.type="probability") {
  z$curve.type <- match.arg(curve.type,c("probability","cumulative"))
  class(z) <- c("moprofile",class(z))
  invisible(z)
}

plot.moprofile <- function(z,nind=1,observed=T,main=NULL,xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,lty=NULL,pch=NULL,add=F,axes=F,bty="n",at=NULL,touch=F,...) {
  if(max(nind)>length(nobs(z))||min(nind)<1)
    stop("No such individual.")
  nc <- length(unique(z$response$y))-1
  if(is.null(z$response$times))
    stop("Times are required to perform a marginal ordinal profile.")
  ii <- covind(z)
  ns <- length(nind)
  if(is.null(xlim))
    xlim <- range(z$response$times)
  if(axes)
    xaxt <- yaxt <- "s"
  else
    xaxt <- yaxt <- "n"
  if(touch) {
    xaxs <- yaxs <- "i"
    xaxt <- yaxt <- "n"
  }
  else
    xaxs <- yaxs <- "r"
  if(z$curve.type=="probability") {
    if(is.null(main))
      if(ns==1)
        main <- paste("Individual ",nind,".",sep="")
      else
        main <- paste("Individuals ",paste(nind,collapse=", "),".",sep="")
    else
      if(length(main)!=1) {
        warning("Dropping contents of option main as it must have lentgh 1.")
        if(ns==1)
          main <- paste("Individual ",nind,".",sep="")
        else
          main <- paste("Individuals ",paste(nind,collapse=", "),".",sep="")
      }
    if(is.null(xlab))
      xlab <- "Time"
    else
      if(length(xlab)!=1) {
        warning("Dropping contents of option xlab as it must have lentgh 1.")
        xlab <- "Time"
      }
    if(is.null(ylab))
      ylab <- "Highest probabilities"
    else
      if(length(ylab)!=1) {
        warning("Dropping contents of option ylab as it must have lentgh 1.")
        ylab <- "Highest probabilities"
      }
    if(is.null(ylim))
      ylim <- c(0,nc)
    if(!is.null(lty))
      if(length(lty)!=1&&length(lty)!=ns) {
        warning(paste("Dropping contents of option lty as it must have lentgh ",ns,".",sep=""))
        lty <- NULL
      }
      else
        if(length(lty)==1)
          lty <- rep(lty,ns)
    if(!is.null(pch))
      if(length(pch)!=1&&length(pch)!=ns) {
        warning(paste("Dropping contents of option pch as it must have lentgh ",ns,".",sep=""))
        pch <- NULL
      }
      else
        if(length(pch)==1)
          pch <- rep(pch,ns)
    first <- !add
    lt <- pc <- j <- 0
    for(i in nind) {
      j <- j+1
      lt <- if(is.null(lty)) lt%%6+1 else lty[j]
      if(first) {
        plot(z$response$times[ii==i],z$pred[ii==i],main=main,xlim=xlim,ylim=ylim,xlab=xlab,ylab=ylab,type="l",lty=lt,xaxt=xaxt,yaxt=yaxt,xaxs=xaxs,yaxs=yaxs,
             bty=bty,...)
        if(!axes||touch) {
          if(!axes)
            if(is.null(at))
              axis(1,at=z$response$times[ii==i],...)
            else
              axis(1,at=at,...)
          else
            axis(1,at=pretty(xlim),...)
          if(touch)
            axis(2,at=pretty(ylim),...)
          else
            axis(2,...)
        }
        first <- F
      }
      else
        lines(z$response$times[ii==i],z$pred[ii==i],lty=lt,...)
      if(observed) {
        pc <- if(is.null(pch)) pc%%6+1 else pch[j]
        points(z$response$times[ii==i],z$response$y[ii==i],pch=pc,...)
      }
    }
  }
  else {
    if(ns>25)
      stop("Too many individuals were chosen to be plotted at once (maximum 25).")
    if(ns!=1&&!add)
      if(ns==2)
        oldpar <- par(mfrow=c(1,2))
      else {
        ng <- ceiling(sqrt(ns))
        ng <- c(ceiling(ns/ng),ng)
        oldpar <- par(mfrow=ng)
      }
    if(is.null(main))
      main <- paste("Individual ",nind,".",sep="")
    else
      if((length(main)!=1&&length(main)!=ns)) {
        warning(paste("Dropping contents of option main as it must have lentgh ",ns,".",sep=""))
        main <- paste("Individual ",nind,".",sep="")
      }
      else
        if(length(main)==1)
          main <- rep(main,ns)
    if(is.null(xlab))
      xlab <- rep("Time",ns)
    else
      if((length(xlab)!=1&&length(xlab)!=ns)){
        warning(paste("Dropping contents of option xlab as it must have lentgh ",ns,".",sep=""))
        xlab <- rep("Time",ns)
      }
      else
        if(length(xlab)==1)
          xlab <- rep(xlab,ns)
    if(is.null(ylab))
      ylab <- rep("Cumulative probabilities",ns)
    else
      if((length(ylab)!=1&&length(ylab)!=ns)) {
        warning(paste("Dropping contents of option ylab as it must have lentgh ",ns,".",sep=""))
        ylab <- rep("Cumulative probabilities",ns)
      }
      else
        if(length(ylab)==1)
          ylab <- rep(ylab,ns)
    if(is.null(ylim))
      ylim <- c(0,1)
    if(!is.null(lty))
      if(length(lty)!=1&&length(lty)!=nc) {
        warning(paste("Dropping contents of option lty as it must have lentgh ",nc,".",sep=""))
        lty <- NULL
      }
      else
        if(length(lty)==1)
          lty <- rep(lty,nc)
    pred <- matrix(z$cpred,nrow=nc,byrow=T)
    j <- 0
    for(i in nind) {
      j <- j+1
      lt <- if(is.null(lty)) 1 else lty[1]
      plot(z$response$times[ii==i],pred[1,ii==i],main=main[j],xlim=xlim,ylim=ylim,xlab=xlab[j],ylab=ylab[j],type="l",lty=lt,
           xaxt=xaxt,yaxt=yaxt,xaxs=xaxs,yaxs=yaxs,bty=bty,...)
      if(!axes||touch) {
        if(!axes)
          if(is.null(at))
            axis(1,at=z$response$times[ii==i],...)
          else
            axis(1,at=at,...)
        else
          axis(1,at=pretty(xlim),...)
        if(touch)
          axis(2,at=pretty(ylim),...)
        else
          axis(2,...)
      }
      if(nc>1)
        for(k in 2:nc) {
          lt <- if(is.null(lty)) lt%%6+1 else lty[k]
          lines(z$response$times[ii==i],pred[k,ii==i],lty=lt,...)
        }
      if(observed)
        mtext(z$response$y[ii==i],at=z$response$times[ii==i],...)
    }
    if(ns!=1&&!add)
      par(oldpar)
  }
}

ioprofile <- function(z,...) UseMethod("ioprofile")

ioprofile.default <- function(z,curve.type="mean") {
  if(inherits(z$response,"response"))
    stop("Object z must contain the response.")
  tmp <- c("mean","probability","both","cumulative")
  curve.type <- match.arg(curve.type,tmp)
  if(!inherits(z,"recursive"))
    stop("The object must have class, recursive")
  else {
    if(curve.type=="mean"||curve.type=="both") {
      if(is.null(z$rpred))
        stop("Recursive means not available (rpred).")
    }
    if(curve.type=="probability"||curve.type=="both") {
      if(is.null(z$ppred))
        stop("Recursive highest probabilities not available (ppred).")
    }
    if(curve.type=="cumulative") {
      if(is.null(z$cprob))
        stop("Recursive cumulative probabilities not available (cprob).")
    }
  }
  z$curve.type <- curve.type
  class(z) <- c("ioprofile",class(z))
  invisible(z)
}

ioprofile.kalordinal <- function(z,curve.type="mean") {
  z$curve.type <- match.arg(curve.type,c("mean","probability","both","cumulative"))
  class(z) <- c("ioprofile",class(z))
  invisible(z)
}

plot.ioprofile <- function(z,nind=1,observed=T,main=NULL,xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,lty=NULL,pch=NULL,add=F,axes=F,bty="n",at=NULL,touch=F,...) {
  if(max(nind)>length(nobs(z))||min(nind)<1)
    stop("No such individual.")
  nc <- length(unique(z$response$y))-1
  if(is.null(z$response$times))
    stop("Times are required to perform an individual ordinal profile.")
  ii <- covind(z)
  ns <- length(nind)
  if(is.null(xlim))
    xlim <- range(z$response$times)
  if(axes)
    xaxt <- yaxt <- "s"
  else
    xaxt <- yaxt <- "n"
  if(touch) {
    xaxs <- yaxs <- "i"
    xaxt <- yaxt <- "n"
  }
  else
    xaxs <- yaxs <- "r"
  if(z$curve.type=="cumulative") {
    if(is.null(ylim))
      ylim <- c(0,1)
    if(!is.null(lty))
      if(length(lty)!=1&&length(lty)!=nc) {
        warning(paste("Dropping contents of option lty as it must have lentgh ",nc,".",sep=""))
        lty <- NULL
      }
      else
        if(length(lty)==1)
          lty <- rep(lty,nc)
  }
  else {
    if(is.null(ylim))
      ylim <- c(0,nc)
    if(z$curve.type=="both") {
      if(is.null(lty))
        lty <- 1:2
      else
        if((length(lty)!=1&&length(lty)!=2)) {
          warning("Dropping contents of option lty as it must have lentgh 2.")
          lty <- 1:2
        }
        else
          if(length(lty)==1)
            lty <- c(lty,lty+1)
      if(is.null(pch))
        pch <- 1
      else
        if(length(pch)!=1) {
          warning("Dropping contents of option pch as it must have lentgh 1.")
          pch <- 1
        }
    }
    else {
      if(is.null(main))
        if(ns==1)
          main <- paste("Individual ",nind,".",sep="")
        else
          main <- paste("Individuals ",paste(nind,collapse=", "),".",sep="")
      else
        if(length(main)!=1) {
          warning("Dropping contents of option main as it must have lentgh 1.")
          if(ns==1)
            main <- paste("Individual ",nind,".",sep="")
          else
            main <- paste("Individuals ",paste(nind,collapse=", "),".",sep="")
        }
      if(is.null(xlab))
        xlab <- "Time"
      else
        if(length(xlab)!=1) {
          warning("Dropping contents of option xlab as it must have lentgh 1.")
          xlab <- "Time"
        }
      if(!is.null(lty))
        if(length(lty)!=1&&length(lty)!=ns) {
          warning(paste("Dropping contents of option lty as it must have lentgh ",ns,".",sep=""))
          lty <- NULL
        }
        else
          if(length(lty)==1)
            lty <- rep(lty,ns)
      if(!is.null(pch))
        if(length(pch)!=1&&length(pch)!=ns) {
          warning(paste("Dropping contents of option pch as it must have lentgh ",ns,".",sep=""))
          pch <- NULL
        }
        else
          if(length(pch)==1)
            pch <- rep(pch,ns)
    }
  }
  if(z$curve.type=="mean") {
    if(is.null(ylab))
      ylab <- "Recursive means"
    else
      if(length(ylab)!=1) {
        warning("Dropping contents of option ylab as it must have lentgh 1.")
        ylab <- "Recursive means"
      }
    first <- !add
    lt <- pc <- j <- 0
    for(i in nind) {
      j <- j+1
      lt <- if(is.null(lty)) lt%%6+1 else lty[j]
      if(first) {
        plot(z$response$times[ii==i],z$rpred[ii==i],main=main,xlim=xlim,ylim=ylim,xlab=xlab,ylab=ylab,type="l",lty=lt,
             xaxt=xaxt,yaxt=yaxt,xaxs=xaxs,yaxs=yaxs,bty=bty,...)
        if(!axes||touch) {
          if(!axes)
            if(is.null(at))
              axis(1,at=z$response$times[ii==i],...)
            else
              axis(1,at=at,...)
          else
            axis(1,at=pretty(xlim),...)
          if(touch)
            axis(2,at=pretty(ylim),...)
          else
            axis(2,...)
        }
        first <- F
      }
      else
        lines(z$response$times[ii==i],z$rpred[ii==i],lty=lt,...)
      if(observed) {
        pc <- if(is.null(pch)) pc%%6+1 else pch[j]
        points(z$response$times[ii==i],z$response$y[ii==i],pch=pc,...)
      }
    }
  }
  else
    if(z$curve.type=="probability") {
      if(is.null(ylab))
        ylab <- "Recursive highest probabilities"
      else
        if(length(ylab)!=1) {
          warning("Dropping contents of option ylab as it must have lentgh 1.")
          ylab <- "Recursive highest probabilities"
        }
      first <- !add
      lt <- pc <- j <- 0
      for(i in nind) {
        j <- j+1
        lt <- if(is.null(lty)) lt%%6+1 else lty[j]
        if(first) {
          plot(z$response$times[ii==i],z$ppred[ii==i],main=main,xlim=xlim,ylim=ylim,xlab=xlab,ylab=ylab,type="l",lty=lt,
               xaxt=xaxt,yaxt=yaxt,xaxs=xaxs,yaxs=yaxs,bty=bty,...)
          if(!axes||touch) {
            if(!axes)
              if(is.null(at))
                axis(1,at=z$response$times[ii==i],...)
              else
                axis(1,at=at,...)
            else
              axis(1,at=pretty(xlim),...)
            if(touch)
              axis(2,at=pretty(ylim),...)
            else
              axis(2,...)
          }
          first <- F
        }
        else
          lines(z$response$times[ii==i],z$ppred[ii==i],lty=lt,...)
        if(observed) {
          pc <- if(is.null(pch)) pc%%6+1 else pch[j]
          points(z$response$times[ii==i],z$response$y[ii==i],pch=pc,...)
        }
      }
    }
    else {
      if(ns>25)
        stop("Too many individuals were chosen to be plotted at once (maximum 25).")
      if(ns!=1&&!add)
        if(ns==2)
          oldpar <- par(mfrow=c(1,2))
        else {
          ng <- ceiling(sqrt(ns))
          ng <- c(ceiling(ns/ng),ng)
          oldpar <- par(mfrow=ng)
        }
      if(is.null(main))
        main <- paste("Individual ",nind,".",sep="")
      else
        if((length(main)!=1&&length(main)!=ns)) {
          warning(paste("Dropping contents of option main as it must have lentgh ",ns,".",sep=""))
          main <- paste("Individual ",nind,".",sep="")
        }
        else
          if(length(main)==1)
            main <- rep(main,ns)
      if(is.null(xlab))
        xlab <- rep("Time",ns)
      else
        if((length(xlab)!=1&&length(xlab)!=ns)){
          warning(paste("Dropping contents of option xlab as it must have lentgh ",ns,".",sep=""))
          xlab <- rep("Time",ns)
        }
        else
          if(length(xlab)==1)
            xlab <- rep(xlab,ns)
      if(z$curve.type=="both") {
        if(is.null(ylab))
          ylab <- rep("Recursive predictions",ns)
        else
          if((length(ylab)!=1&&length(ylab)!=ns)) {
            warning(paste("Dropping contents of option ylab as it must have lentgh ",ns,".",sep=""))
            ylab <- rep("Recursive predictions",ns)
          }
          else
            if(length(ylab)==1)
              ylab <- rep(ylab,ns)
        j <- 0
        for(i in nind) {
          j <- j+1
          plot(z$response$times[ii==i],z$rpred[ii==i],main=main[j],xlim=xlim,ylim=ylim,xlab=xlab[j],ylab=ylab[j],type="l",lty=lty[1],
               xaxt=xaxt,yaxt=yaxt,xaxs=xaxs,yaxs=yaxs,bty=bty,...)
          if(!axes||touch) {
            if(!axes)
              if(is.null(at))
                axis(1,at=z$response$times[ii==i],...)
              else
                axis(1,at=at,...)
            else
              axis(1,at=pretty(xlim),...)
            if(touch)
              axis(2,at=pretty(ylim),...)
            else
              axis(2,...)
          }
          lines(z$response$times[ii==i],z$ppred[ii==i],lty=lty[2],...)
          if(observed)
            points(z$response$times[ii==i],z$response$y[ii==i],pch=pch,...)
        }
      }
      else {
        if(is.null(ylab))
          ylab <- rep("Recursive cumulative probabilities",ns)
        else
          if((length(ylab)!=1&&length(ylab)!=ns)) {
            warning(paste("Dropping contents of option ylab as it must have lentgh ",ns,".",sep=""))
            ylab <- rep("Recursive cumulative probabilities",ns)
          }
          else
            if(length(ylab)==1)
              ylab <- rep(ylab,ns)
        pred <- matrix(z$cprob,nrow=nc,byrow=T)
        j <- 0
        for(i in nind) {
          j <- j+1
          lt <- if(is.null(lty)) 1 else lty[1]
          plot(z$response$times[ii==i],pred[1,ii==i],main=main[j],xlim=xlim,ylim=ylim,xlab=xlab[j],ylab=ylab[j],type="l",lty=lt,
               xaxt=xaxt,yaxt=yaxt,xaxs=xaxs,yaxs=yaxs,bty=bty,...)
          if(!axes||touch) {
            if(!axes)
              if(is.null(at))
                axis(1,at=z$response$times[ii==i],...)
              else
                axis(1,at=at,...)
            else
              axis(1,at=pretty(xlim),...)
            if(touch)
              axis(2,at=pretty(ylim),...)
            else
              axis(2,...)
          }
          if(nc>1)
            for(k in 2:nc) {
              lt <- if(is.null(lty)) lt%%6+1 else lty[k]
              lines(z$response$times[ii==i],pred[k,ii==i],lty=lt,...)
            }
          if(observed)
            mtext(z$response$y[ii==i],at=z$response$times[ii==i],...)
        }
      }
      if(ns!=1&&!add)
        par(oldpar)
    }
}

poprofile <- function(mu,...) UseMethod("poprofile")

poprofile.default <- function(mu,pintercept,preg,pinitial=NULL,depend="independence",times=NULL,distribution="multinomial",curve.type="probability") {
  call <- sys.call()
  tmp <- c("binary","multinomial","continuation-ratio","proportional-odds")
  mdl <- match(distribution <- match.arg(distribution,tmp),tmp)
  tmp <- c("independence","Markov","serial","frailty")
  dep <- match(depend <- match.arg(depend,tmp),tmp)-1
  tmp <- c("probability","cumulative")
  curve.type <- match.arg(curve.type,tmp)
  if(is.null(times)&&(depend=="serial"||depend=="Markov"))
    stop("No times. Serial and Markov dependence cannot be fitted.")
  init <- !is.null(pinitial)
  if(init)
    if(pinitial<=0)
      stop("Estimate of initial parameter must greater than 0")
    else
      pinitial <- log(pinitial)
  else
    if(depend!="independence")
      stop(paste("To introduce a ",depend," dependence, pinit must have an initial estimate.",sep=""))
  nc <- length(pintercept)+1
  b <- mu(preg)
  n <- length(b)
  p <- c(pintercept,preg,pinitial)
  x <- .C("pkord",
          p=as.double(p),
          nbs=as.integer(n),
          nc=as.integer(nc),
          init=as.integer(init),
          model=as.integer(mdl),
          dep=as.integer(dep),
          preg=as.integer(length(preg)),
          cpred=double(n*(nc-1)),
          pred=double(n),
          bbb=as.double(b),
          beta=double((nc-1)),
          bt=double((nc-1)),
          bt2=double((nc-1)),
          bet=double((nc-1)),
          cum=double((nc-1)),
          DUP=F)
  z <- list(call=call,
            curve.type=curve.type,
            distribution=distribution,
            mdl=mdl,
            mu=mu,
            npr=nc-1+length(preg),
            depend=depend,
            response=list(y=0:(nc-1),times=times,nobs=n),
            cpred=x$cpred,
            pred=x$pred,
            init=init,
            coefficients=p)
  class(z) <- "poprofile"
  invisible(z)
}

plot.poprofile <- function(z,main="Predicted profile",xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,lty=NULL,add=F,axes=F,bty="n",at=NULL,touch=F,...)
  plot.moprofile(z,nind=1,observed=F,main=main,xlab=xlab,ylab=ylab,xlim=xlim,ylim=ylim,lty=lty,pch=NULL,add=add,axes=axes,bty=bty,at=at,touch=touch,...)

hist.repeated <- function(z,ccov=NULL,breaks=NULL,nclass=NULL,col=NULL,border=par("fg"),levels=NULL,side=F,main=NULL,xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,add=F,
                          axes=F,bty="n",at=NULL,touch=F,...) {
  if(z$response$type!="ordinal")
    stop("To perform a histogram, the response in the `repeated' object must be of type `ordinal'.")
  hist.ordinal(z=z,ccov=ccov,breaks=breaks,nclass=nclass,col=col,border=border,levels=levels,side=side,main=main,xlab=xlab,ylab=ylab,xlim=xlim,ylim=ylim,add=add,
               axes=axes,bty=bty,at=at,touch=touch,...)
}

hist.ordinal <- function(z,ccov=NULL,breaks=NULL,nclass=NULL,col=NULL,border=par("fg"),levels=F,side=F,main=NULL,xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,add=F,
                         axes=F,bty="n",at=NULL,touch=F,...) {
  if(inherits(z,"response")) {
    if(!is.null(ccov))
      warning("No covariates available in response objects.")
    ccov <- NULL
    z <- rmna(z)
  }
  if(!inherits(z,"repeated"))
    stop("The object z must either be a response or a repeated object.")
  nc <- length(unique(z$response$y))
  if(is.null(times(z))) {
    all.times <- 1
    m <- 1
  }
  else {
    all.times <- sort(unique(times(z)))
    m <- length(all.times)
  }
  use.br <- !is.null(breaks)
  if(use.br) {
    if(!is.null(nclass))
      warning("`nclass' not used when `breaks' specified")
  }
  else
    if(!is.null(nclass)&&length(nclass)==1)
      breaks <- nclass
  rm(nclass)
  use.br <- use.br&&length(breaks)>1
  if(use.br) {
    if(any(is.na(breaks))|length(breaks)!=m+1)
      stop(paste("Invalid or wrong number of breaks, no NAs are allowed and ",m+1," breaks are required.",sep=""))
    breaks <- sort(breaks)
    if(length(unique(diff(breaks)))!=1)
      stop("The breaks are invalid as they must all delimit the same interval.")
    nB <- length(breaks)
  }
  else {
    if(is.null(breaks))
      breaks <- all.times
    else
      if(is.na(breaks)|breaks!=m)
        stop(paste("Invalid number of breaks, ",m," are required.",sep=""))
      else
        breaks <- all.times
    if(m>1)
      breaks <- c(breaks,max(breaks)+unique(diff(breaks)))
    else
      breaks <- c(all.times,all.times+1)
    breaks <- breaks-(unique(diff(breaks))/2)
    nB <- length(breaks)
  }
  if(is.logical(side)&&side)
    side <- "l"
  else
    if(!is.logical(side)&&side!="l"&&side!="r") {
      warning("Droping contents of option `side' as it must either be `l' (for left side) or `r' (for right side).")
      side <- "l"
    }
  if(is.null(levels))
    levels <- 1:nc
  else
    if(length(levels)!=nc) {
      warning(paste("Wrong number of levels, vector `levels' should be of length ",nc,".",sep=""))
      levels <- 1:nc
    }
  if(is.null(col))
    col <- gray(seq(1,0,length=nc+1))[-(nc+1)]
  else
    if(length(col)!=nc) {
      warning(paste("Wrong number of colors, vector `col' should be of length ",nc,".",sep=""))
      col <- gray(seq(1,0,length=nc+1))[-(nc+1)]
    }
  if(is.null(xlim))
    xlim <- c(min(breaks)-unique(diff(breaks))/2,max(breaks)+unique(diff(breaks))/2)
  if(is.null(ylim))
    ylim <- 0:1
  if(axes)
    xaxt <- yaxt <- "s"
  else
    xaxt <- yaxt <- "n"
  if(touch) {
    xaxs <- yaxs <- "i"
    xaxt <- yaxt <- "n"
  }
  else
    xaxs <- yaxs <- "r"
  if(!is.null(ccov)) {
    if(length(ccov)!=1) {
      warning("Only using first covariate.")
      ccov <- ccov[1]
    }
    cov.name <- ccov
    ccov <- covariates(z,names=ccov)
    if(is.data.frame(ccov))
      ccov <- codes(ccov)
    np <- unique(ccov)
    ns <- length(np)
    if(ns>25)
      stop("The covariate chosen has more than 25 levels (too many categories for plotting).")
    if(length(ccov)==length(nobs(z)))
      ccov <- rep(ccov,nobs(z))
    if(is.null(main))
      main <- paste(cov.name," ",np,".",sep="")
    else
      if(length(main)!=1&&length(main)!=ns) {
        warning(paste("Dropping contents of option main as it must have lentgh ",ns,".",sep=""))
        main <- paste(cov.name," ",np,".",sep="")
      }
      else
        if(length(main)==1)
          main <- rep(main,ns)
    if(is.null(xlab))
      xlab <- rep("Time",ns)
    else
      if(length(xlab)!=1&&length(xlab)!=ns) {
        warning(paste("Dropping contents of option xlab as it must have lentgh ",ns,".",sep=""))
        xlab <- rep("Time",ns)
      }
      else
        if(length(xlab)==1)
          xlab <- rep(xlab,ns)
    if(is.null(ylab))
      ylab <- rep("Cumulative probabilities",ns)
    else
      if(length(ylab)!=1&&length(ylab)!=ns) {
        warning(paste("Dropping contents of option ylab as it must have lentgh ",ns,".",sep=""))
        ylab <- rep("Cumulative probabilities",ns)
      }
      else
        if(length(ylab)==1)
          ylab <- rep(ylab,ns)
  }
  else {
    ccov <- np <- ns <- 1
    if(is.null(main))
      main <- "Cumulative histograms over time."
    if(is.null(xlab))
      xlab <- "Time"
    if(is.null(ylab))
      ylab <- "Cumulative probabilities"
  }
  if(ns!=1&&!add)
    if(ns==2)
      oldpar <- par(mfrow=c(1,2))
    else {
      ng <- ceiling(sqrt(ns))
      ng <- c(ceiling(ns/ng),ng)
      oldpar <- par(mfrow=ng)
    }
  i <- 0
  for(k in np) {
    i <- i+1
    times <- sort(unique(times(z)[ccov==k]))
    ntimes <- length(times)
    if(is.null(weights(z)))
      tab <- table(response(z)[ccov==k],times(z)[ccov==k])
    else
      tab <- tapply(weights(z)[ccov==k],list(as.vector(response(z)[ccov==k]),times(z)[ccov==k]),sum)
    plot.new()
    plot.window(xlim=xlim,ylim=ylim,xaxt=xaxt,yaxt=yaxt,xaxs=xaxs,yaxs=yaxs,bty=bty,...)
    title(main=main[i],xlab=xlab[i],ylab=ylab[i],...)
    if(!axes||touch) {
      if(!axes)
        if(is.null(at))
          axis(1,at=times,...)
        else
          axis(1,at=at,...)
      else
        axis(1,at=pretty(xlim),...)
      if(touch)
        axis(2,at=pretty(ylim),...)
      else
        axis(2,...)
    }
    bottom <-  rep(0,m)
    for(j in 1:nc) {
      x <- apply(matrix(tab[1:j,],ncol=ntimes),2,sum)/apply(tab,2,sum)
      if(side=="l")
        height <- (bottom[1]+x[1])/2
      else
        if(side=="r")
          height <- (bottom[m]+x[m])/2
      rect(breaks[-nB],bottom,breaks[-1],x,col=col[j],border=border)
      if(side=="l")
        text(x=xlim[1],y=height,label=levels[j])
      else
        if(side=="r")
          text(x=xlim[2],y=height,label=levels[j])
      bottom <- x
    }
  }
  if(ns!=1&&!add)
    par(oldpar)
}
