/****h* ROBODoc/Links
 * FUNCTION
 *  This module contains functions to manipulate links.
 *  Links are derived from headers.  They are used to create
 *  links in the documentation between a word and the part of
 *  the documentation that explains something about that word.
 *  (For instance a function name or variable name).
 *  In addition to the links derived from the headers links are
 *  also derived from the names of all the sourcefiles.
 * MODIFICATION HISTORY
 *   ????-??-??   Frans Slothouber  V1.0 
 *   2003-02-03   Frans Slothouber  Refactoring
 *******
 * $Header: /cvsroot/robodoc/robo/Source/links.c,v 1.24 2003/12/30 17:39:36 gumpu Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include "globals.h"
#include "robodoc.h"
#include "headers.h"
#include "util.h"
#include "links.h"
#include "folds.h"
#include "document.h"
#include "part.h"
#include "file.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif


/* Local functions */

static struct RB_link *RB_Alloc_Link( char *label_name, char *object_name,
                                      char *file_name );


/* Can't get quicksort to work on Win32. So for now
 * this function is obsolete.
 */

#if 0
int
link_cmp( const void *l1, const void *l2 )
{
    const struct RB_link *link1 = l1;
    const struct RB_link *link2 = l2;

    return strcmp( link2->label_name, link1->label_name );
}
#endif

/****f* Links/RB_CollectLinks
 * FUNCTION
 *   Convert header information into link information.
 *   RB_header -> RB_link conversion
 * SYNOPSIS
 *   void RB_CollectLinks(struct RB_Document* document, 
 *         struct RB_header** headers, long count)
 * INPUTS
 *   document -- 
 *   headers  -- the array with headers.
 *   count    -- number of headers in the array
 * OUTPUT
 *   link_index -- an array with links
 *   link_index_size -- the number of links in the array.
 * SOURCE
 */

void
RB_CollectLinks( struct RB_Document *document, struct RB_header **headers,
                 unsigned long count )
{
    unsigned int        i, j;
    struct RB_Part     *i_part;

    link_index_size = count;

    if ( document->actions & DO_MULTIDOC )
    {
        for ( i_part = document->parts; i_part; i_part = i_part->next )
        {
            link_index_size++;
        }
    }

    link_index =
        ( struct RB_link ** ) calloc( link_index_size,
                                      sizeof( struct RB_link ** ) );

    for ( i = 0; i < count; ++i )
    {
        struct RB_link     *link;
        struct RB_header   *header;

        header = headers[i];
        assert( header->unique_name );
        assert( header->file_name );
        link = RB_Alloc_Link( header->unique_name, header->function_name,
                              header->file_name );
        link->htype = header->htype;
        link->is_internal = header->is_internal;
        link_index[i] = link;
    }

    /* If we are in multidoc mode, we also create links
     * for all the source files.
     */

    if ( document->actions & DO_MULTIDOC )
    {
        for ( i_part = document->parts; i_part; i_part = i_part->next )
        {
            struct RB_link     *link;

            link =
                RB_Alloc_Link( i_part->filename->name, i_part->filename->name,
                               RB_Get_FullDocname( i_part->filename ) );
            i_part->filename->link = link;
//            link->type = NO_HEADER;
            link->htype = RB_FindHeaderType( HT_SOURCEHEADERTYPE );
            link_index[i] = link;
            ++i;
        }
    }

    /* Sort all the links so we can use a binary search */

/* Doesn't work :( (On Win32)  qsort( link_index, link_index_size,
 * sizeof(struct RB_link*), link_cmp ); */

    for ( i = 0; i < link_index_size; ++i )
    {
        for ( j = i + 1; j < link_index_size; ++j )
        {
            char               *name1 = ( link_index[i] )->object_name;
            char               *name2 = ( link_index[j] )->object_name;

            if ( strcmp( name1, name2 ) > 0 )
            {
                struct RB_link     *link = link_index[j];

                link_index[j] = link_index[i];
                link_index[i] = link;
            }
        }
    }
}

/*****/


/****f* Links/RB_Free_Links
 * FUNCTION
 *   Deallocate all the memory used to store the links.
 * INPUTS
 *   o link_index_size
 *   o link_index[]
 * BUGS
 *   Should use RB_Free_Link instead of doing
 *   everything by it self.
 * SOURCE
 */

void
RB_Free_Links( void )
{
    struct RB_link     *cur_link;
    unsigned int        i;

    for ( i = 0; i < link_index_size; ++i )
    {
        cur_link = link_index[i];
        free( cur_link->object_name );
        free( cur_link->label_name );
        free( cur_link->file_name );
        free( cur_link );
    }
    free( link_index );
}

/*******/

/* TODO Documentation */

int RB_Number_Of_Links( struct RB_HeaderType* header_type, char* file_name, int internal )
{
    struct RB_link     *cur_link;
    int                 n = 0;
    unsigned int        i;

    for ( i = 0; i < link_index_size; ++i )
    {
        cur_link = link_index[i];
        if ( RB_CompareHeaderTypes( cur_link->htype, header_type ) &&
             ( ( cur_link->is_internal && internal ) ||
               ( !cur_link->is_internal && !internal ) ) )
        {
            if ( file_name )
            {
                if ( strcmp( file_name, cur_link->file_name ) == 0 )
                {
                    n++;
                }
            }
            else
            {
                n++;
            }
        }
    }
    return n;
}


/****f* Links/RB_Find_Link [3.0h]
 * NAME
 *   RB_Find_Link -- try to match word with a link
 * SYNOPSIS
 *   result = RB_Find_Link (word_begin, label_name, file_name)
 *   int      RB_Find_Link (char *,     char **,    char **)
 * FUNCTION
 *   Searches for the given word in the list of links and
 *   headers.  There are three passes (or four, when the C option
 *   is selected). Each pass uses a different definition of "word".
 *   In the first pass it is any thing that ends with a 'space', a '.' 
 *   or a ','.
 *   In the second pass it is any string that consists of alpha
 *   numerics, '_', ':', '.', or '-'.  
 *   In the third pass (for C) it is any string that consists 
 *   of alpha numerics or '_'.
 *   In the last pass it is any string that consists of alpha
 *   numerics.
 * INPUTS
 *   word_begin  - pointer to a word (a string).
 *   object_name  - pointer to a pointer to a string
 *   file_name   - pointer to a pointer to a string
 * SIDE EFFECTS
 *   label_name & file_name are modified
 * RESULT
 *   object_name   -- points to the object if a match was found,
 *                    NULL otherwise.
 *   file_name     -- points to the file name if a match was found,
 *                    NULL otherwise.
 *   TRUE          -- a match was found.
 *   FALSE         -- no match was found.
 * NOTES
 *   This is a rather sensitive algorithm. Don't mess with it
 *   too much.
 * SOURCE
 */

int
RB_Find_Link( char *word_begin, char **object_name, char **label_name,
              char **file_name )
{
    char               *cur_char, old_char;
    int                 low_index, high_index, cur_index, state, pass;


    for ( pass = 0; pass < 4; pass++ )
    {
        switch ( pass )
        {
        case 0:
            {
                for ( cur_char = word_begin;
                      isalnum( *cur_char ) || ispunct( *cur_char );
                      cur_char++ );
                if ( ( ( *( cur_char - 1 ) ) == ',' )
                     || ( ( *( cur_char - 1 ) ) == '.' ) )
                    cur_char--;
                break;
            }
        case 1:
            {
                for ( cur_char = word_begin;
                      isalnum( *cur_char ) || ( *cur_char == '_' ) ||
                      ( *cur_char == '-' ) || ( *cur_char == '.' ) ||
                      ( *cur_char == ':' ); cur_char++ );
                break;
            }
        case 2:
            {
                if ( extra_flags & C_MODE )
                {
                    for ( cur_char = word_begin;
                          isalnum( *cur_char ) || ( *cur_char == '_' );
                          cur_char++ );
                    break;
                }
                else
                    continue;
            }
        case 3:
            {
                for ( cur_char = word_begin;
                      isalnum( *cur_char ); cur_char++ );
                break;
            }
        }

        old_char = *cur_char;
        *cur_char = '\0';       /*
                                   * End the word with a '\0' 
                                 */
        /*
         * RB_Say ("Testing \"%s\"\n", word_begin); 
         */

        /*
         * Search in the link table 
         */
        for ( cur_index = 0, low_index = 0, high_index =
              link_index_size - 1; high_index >= low_index; )
        {
            cur_index = ( high_index - low_index ) / 2 + low_index;
            state = strcmp( word_begin, link_index[cur_index]->object_name );
            if ( state < 0 )
            {
                high_index = cur_index - 1;
            }
            else if ( state == 0 )
            {
                *object_name = link_index[cur_index]->object_name;
                *label_name = link_index[cur_index]->label_name;
                *file_name = link_index[cur_index]->file_name;
                RB_Say( "linking \"%s\"->\"%s\" form \"%s\"\n",
                        word_begin, *object_name, *file_name );
                *cur_char = old_char;
                return ( TRUE );
            }
            else if ( state > 0 )
            {
                low_index = cur_index + 1;
            }
        }
        *cur_char = old_char;
        *file_name = NULL;
        *object_name = NULL;
        *label_name = NULL;
    }
    return ( FALSE );
}

/*****/


/****f* Links/RB_Alloc_Link [2.01]
 * NAME
 *   RB_Alloc_Link              -- oop
 * SYNOPSIS
 *   struct RB_link *RB_Alloc_Link( char *label_name, char *file_name )
 * FUNCTION
 *   allocate struct + strings
 * INPUTS
 *   char *label_name -- strings to copy into the link
 *   char *file_name
 * RESULT
 *   struct RB_link *  -- ready-to-use
 * AUTHOR
 *   Koessi
 * SEE ALSO
 *   RB_StrDup(), RB_Free_Link()
 * SOURCE
 */

static struct RB_link *
RB_Alloc_Link( char *label_name, char *object_name, char *file_name )
{
    struct RB_link     *new_link;

    assert( object_name );
    assert( label_name );
    assert( file_name );
    new_link = malloc( sizeof( struct RB_link ) );
    memset( new_link, 0, sizeof( struct RB_link ) );

    new_link->file_name = RB_StrDup( file_name );
    new_link->object_name = RB_StrDup( object_name );
    new_link->label_name = RB_StrDup( label_name );
    return ( new_link );
}

/*****/

/****f* Links/RB_Free_Link
 * NAME
 *   RB_Free_Link               -- oop
 * SYNOPSIS
 *   void RB_Free_Link( struct RB_link *link )
 * FUNCTION
 *   free struct + strings
 * INPUTS
 *   struct RB_link *link
 * AUTHOR
 *   Koessi
 * SEE ALSO
 *   RB_Alloc_Link(), RB_Close_The_Shop()
 * SOURCE
 */

void
RB_Free_Link( struct RB_link *link )
{
    if ( link )
    {
        if ( link->label_name )
        {
            free( link->label_name );
        }
        if ( link->file_name )
        {
            free( link->file_name );
        }
        free( link );
    }
}

/******/
