# Video streaming via Gst::Playbin.

require 'gst0.10'

Thread.abort_on_exception = true

Gst.init


class Discoverer < Gst::Pipeline
  type_register
  attr_reader :mimetype, :audiolength, :videolength, :videowidth, :videoheight, :videorate, :audiorate, :audiochannels, :audiowidth, :audiodepth, :is_audio, :is_video, :tags
  signal_new("discovered",
    GLib::Signal::RUN_FIRST,
    nil,
    nil,
    GLib::Type["gboolean"]
  )

  def signal_do_discovered(result)
    #puts "discover signal"
  end

  def initialize(filename)
    super()
    @mimetype = nil

    @audiocaps = nil
    @videocaps = nil

    @videowidth = 0
    @videoheight = 0
    @videorate = nil

    @audiofloat = false
    @audiorate = 0
    @audiodepth = 0
    @audiowidth = 0
    @audiochannels = 0

    @audiolength = 0
    @videolength = 0

    @is_video = false
    @is_audio = false

    @otherstreams = nil

    @finished = false
    @tags = {}
    @_success = false

    @_timeoutid = 0
        
        
    # the initial elements of the pipeline
    @source = Gst::ElementFactory.make('filesrc')
    @source.location = filename
    @source.blocksize = 1000000
    @decodebin = Gst::ElementFactory.make('decodebin')
    add(@source, @decodebin)
    @source >> @decodebin
    
    @typefind = @decodebin.get_by_name("typefind")

    # callbacks
    @decodebin.signal_connect("new-decoded-pad") { | dbin, pad, is_last |
      new_decoded_pad_cb(dbin, pad, is_last)
    }

    @typefind.signal_connect("have-type") { | typefind, prob, caps | @mimetype=caps.to_string }
    #    self.dbin.connect("unknown-type", self._unknown_type_cb)
  end

  def new_decoded_pad_cb(dbin, pad, is_last)
    caps = pad.caps
    capsstr = caps.to_string
        
    if capsstr.include? "audio"
      @is_audio = true
    elsif capsstr.include? "video"
      @is_video = true
    end
    if is_last and not @is_audio and not @is_video
      _finished(false)
      return 
    end
    queue = Gst::ElementFactory.make('queue')
    fakesink = Gst::ElementFactory.make('fakesink')
    add(queue, fakesink)
    queue >> fakesink
    sinkpad = fakesink.get_pad("sink")
    queuepad = queue.get_pad("sink")

                
    sinkpad.signal_connect("notify::caps") { | pad, args |
      caps = pad.negotiatedcaps
      
      if caps
        # these caps are now fixed
        # We can now get the total length of stream
        #query = Gst::Query.new_duration(Gst::Format::TIME)
        peer = pad.peer
        length = peer.query_duration(Gst::Format::TIME)
        capsstr = caps.to_string
        if capsstr.include? "audio"
          @audiocaps = caps
          @audiolength = length
          first_struct = @audiocaps.get_structure(0)
          @audiorate = first_struct["rate"]
          @audiowidth = first_struct["width"]
          @audiodepth = first_struct["depth"]
          @audiochannels = first_struct["channels"]
          if capsstr.include? "x-raw-float"
            @audiofloat = true
          else
            @audiofloat = false
          end
        elsif capsstr.include? "video"
          @videocaps = caps
          @videolength = length
          first_struct = @videocaps.get_structure(0)
          @videowidth = first_struct["width"]
          @videoheight = first_struct["height"]
	  @videorate = first_struct["framerate"]
          if (not @is_audio) or @audiocaps
            _finished(true)
          end
        end
      end

    }
    pad.link(queuepad)
    queue.play
    fakesink.play
  end

  def discover
    if @finished
      signal_emit('discovered', false)
      return
    end

    @buswatchid = bus.add_watch { | message | bus_message_cb(message) }

    # 3s timeout
    @_timeoutid = GLib::Timeout.add(3000) { _finished(false) }
        
    if not play
      _finished(false)
    end
  end
  
  def _finished(success)
    @_success = success
    bus.remove_watch(@buswatchid)
    if @_timeoutid 
      @_timeoutid = nil
      GLib::Idle.add { _stop }
      return false
    end
  end

  def _stop
    @finished = true
    ready
    signal_emit("discovered", true)
  end

  def bus_message_cb(message)
    case message.get_type
      when Gst::Message::MessageType::ERROR then
        _finished(false)
      when Gst::Message::MessageType::EOS then
        _finished(false)
      when Gst::Message::MessageType::TAG then
        tag = message.parse_tag
	tag.each do |key,value|
		if key != "name"
			@tags[key] = value
		end
	end
    end
    true
  end

  def _time_to_string(value)
     if value == -1
       return "Unknown"
     end
     ms = value / Gst::MSECOND
     sec = ms / 1000
     ms = ms % 1000
     min = sec / 60
     sec = sec % 60
     return "#{min}m #{sec}s #{ms}ms"
  end

  def print_info
    if not @finished:
      return
    end
    if not @mimetype:
      print "Unknown media type\n"
      return
    else
      print "Mime Type : #{@mimetype}\n"
    end
    if not @is_video and not @is_audio
      return
    else
      #print "Length : #{self._time_to_string(max(self.audiolength, self.videolength))
      print "Audio length: #{_time_to_string(@audiolength)} Video length: #{_time_to_string(@videolength)}\n"
    end
    if @is_video #and @videorate:
      print "Video :\n"
      print "   #{@videowidth} x #{@videoheight} @ #{@videorate.inspect}fps\n" # @ %d/%d fps" % (self.videowidth,                                            self.videoheight,self.videorate.num, self.videorate.denom)
      if @tags["video-codec"]
      	print "   Codec: #{@tags['video-codec']}\n"
      end
    end
    if @is_audio
      print "Audio :\n"
      print "    #{@audiochannels} channels(s) : #{@audiorate}Hz @ #{@audiowidth}bits\n"

      if @tags["audio-codec"]
        print "    Codec: #{@tags['audio-codec']}\n"
      end
    end
    print "Additional information :\n"
    @tags.each do |key,value|
      if not key.include? "codec"
        print "#{key}: #{value}\n"
      end

    end
  end
end

def usage
    puts "Usage: #{__FILE__} file"
    exit
end

def discovered_cb(status, pipeline, mainloop)
  pipeline.print_info
  GLib::Idle.add { mainloop.quit }
end

usage if ARGV.length < 1

    
mainloop = GLib::MainLoop.new(GLib::MainContext.default, true)
    
@pipeline = Discoverer.new(ARGV.first)
@pipeline.signal_connect("discovered") { | obj, status | discovered_cb(status, @pipeline, mainloop) }
@pipeline.discover
begin
  mainloop.run
rescue Interrupt
  ensure
    @pipeline.stop
end
