# Samizdat engine deployment functions
#
#   Copyright (c) 2002-2006  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'


# hack to get through to CGI environment
#
# fixme: not thread-safe
#
class RequestSingleton
  include Singleton

  def initialize
    reset
  end

  # unset all variables
  #
  # make sure to call this before processing new request
  #
  def reset
    @env = {}
    @host = nil
    @site_name = nil
  end

  attr_accessor :env, :site_name

  # return requested host name
  #
  # following variables are tried in that order:
  #
  #  HTTP_X_FORWARDED_HOST
  #  SERVER_NAME
  #  HTTP_HOST
  #
  def host
    @host ||= (@env['HTTP_X_FORWARDED_HOST'] or
      @env['SERVER_NAME'] or
      @env['HTTP_HOST'])
  end
end


# todo: document this class
# 
class ConfigHash < Hash

  # pre-load with supplied hash
  #
  def initialize(hash)
    super()
    self.merge!(hash)
  end

  # translate to option or ConfigHash of suboptions
  #
  def method_missing(name)
    value = self[name.to_s]
    case value
    when nil then nil
    when Hash then ConfigHash.new(value)
    else value
    end
  end

  # traverse the tree of sub-hashes and only update modified values
  #
  def deep_update!(hash)
    hash.each do |key, value|
      if value.kind_of? Hash then
        case self[key]
        when ConfigHash then self[key].deep_update!(value)
        when Hash then self[key] = ConfigHash.new(self[key]).deep_update!(value)
        else self[key] = value   # scalar or nil is discarded in favor of hash
        end
      else
        self[key] = value
      end
    end
    self
  end
end


# helper method to load YAML data from file
#
def load_yaml_file(filename)
  File.open(filename) {|f| YAML.load(f) }
end

# load and operate mapping of server prefixes to Samizdat site names
#
class SamizdatSites
  include Singleton

  SITES_MAP = '/etc/samizdat/sites.yaml'

  # loads Samizdat sites mapping from /etc/samizdat/sites.yaml
  #
  def initialize
    if File.readable?(SITES_MAP)
      @sites = load_yaml_file(SITES_MAP)
    end
  end

  # determine :site_name or :prefix_uri (depending on result_type) from CGI
  # variables
  #
  # prefixes sorted by descending length so that more specific prefixes are
  # tried before shorter ones
  #
  # returns nil if site is not found
  #
  def find(result_type)
    server_name = RequestSingleton.instance.host
    request_uri = RequestSingleton.instance.env['REQUEST_URI']

    # fall back to config.yaml in current directory
    return nil unless @sites.kind_of? Hash and @sites[server_name].kind_of? Hash

    # optimize: pre-generate sorted URI prefix lists, pre-compile Regexps
    @sites[server_name].keys.sort_by {|p| -p.size }.each do |prefix|
      next unless request_uri =~ /\A#{Regexp.escape(prefix)}/
      # config files are always trusted
      case result_type
      when :site_name then return @sites[server_name][prefix].dup.untaint
      when :uri_prefix then return prefix.dup.untaint
      end
    end

    nil
  end
end

# name of site being accessed by current request
#
# can be overridden by SAMIZDAT_SITE environment variable, falls back to
# config.yaml in the current directory on the server to identify the site
#
def site_name
  ENV['SAMIZDAT_SITE'] or
    RequestSingleton.instance.site_name ||=
      (SamizdatSites.instance.find(:site_name) or Dir.pwd + '/config.yaml')
end

# URI prefix of site being accessed by current request
#
# can be overridden by SAMIZDAT_URI environment variable, falls back to
# script's location (can be problematic if you mod_rewrite your scripts to a
# different location)
#
def uri_prefix
  ENV['SAMIZDAT_URI'] or SamizdatSites.instance.find(:uri_prefix) or
    RequestSingleton.instance.env['SCRIPT_NAME'].sub(%r{/[^/]\z}, '')
end


# load, merge in, and cache site configuration from rdf.yaml, defaults.yaml,
# and site.yaml (in that order), cache xhtml.yaml
#
class SiteConfig < SimpleDelegator

  # look up config files in one of given directories, by default look in /etc,
  # /usr/share, /usr/local/share, and current directory
  CONFIG_DIRS = [ '/etc/samizdat/',
                  Config::CONFIG['datadir'] + '/samizdat/',
                  '/usr/local/share/samizdat/',
                  '' ]

  # find config file in one of the CONFIG_DIRS
  #
  def SiteConfig.find(file, dirs=CONFIG_DIRS)
    dirs.each do |dir|
      if File.readable?(dir + file)
        return(dir + file)
      end
    end
    nil
  end

  # global RDF-relational mapping
  RDF_CONFIG = 'rdf.yaml'

  # default settings common for all sites
  DEFAULT_CONFIG = 'defaults.yaml'

  # XHTML options for Samizdat::Sanitize
  XHTML_CONFIG = 'xhtml.yaml'

  # location of site-specific configs
  SITES_DIR = '/etc/samizdat/sites/'

  def initialize(site)
    @@rdf ||= load_yaml_file(SiteConfig.find(RDF_CONFIG))
    @@defaults ||= load_yaml_file(SiteConfig.find(DEFAULT_CONFIG))

    @config = ConfigHash.new(@@rdf)
    @config.deep_update!(@@defaults)

    site_config =
      if site =~ %r{/.*config\.yaml} and File.readable? site
        site   # config.yaml in current directory
      else
        SITES_DIR + site + '.yaml'
      end

    @config.deep_update!(load_yaml_file(site_config))

    if @config.cache and @config.cache =~ /\Adruby:/ then
      @drb = @config.cache
    end

    super @config
  end

  attr_reader :drb

  # pre-loaded XHTML validation configuration for Samizdat::Sanitize
  #
  def xhtml
    @@xhtml ||= load_yaml_file(SiteConfig.find(XHTML_CONFIG))
  end

  # location of content directory relative to host root
  #
  def content_location
    if @config['site']['content'].kind_of? String then
      uri_prefix + @config['site']['content']
    end
  end

  # check if all parameters necessary to send out emails are set
  #
  def email_enabled?
    @config['email'] and @config['email']['address'] and @config['email']['sendmail']
  end
end

class SiteConfigSingleton
  include Singleton

  def initialize
    @config = {}
  end

  # cache parsed SiteConfig by site_name
  #
  def config
    @config[site_name] ||= SiteConfig.new(site_name)
  end
end

# shortcut access to SiteConfig
#
def config
  SiteConfigSingleton.instance.config
end


# wraps Cache methods to prepend cache key with site_name
#
class SiteCache
  def initialize(cache)
    @cache = cache
  end

  def flush(site=Regexp.new('\A'+Regexp.escape(site_name)))
    @cache.flush(site)
  end

  def []=(key, value)
    @cache[site_name + key] = value
  end

  def has_key?(key)
    @cache.has_key?(site_name + key)
  end

  def [](key)
    @cache[site_name + key]
  end

  def fetch_or_add(key, &p)
    @cache.fetch_or_add(site_name + key, &p)
  end
end

# puts a SiteCache wrapper around cache objects
#
class CacheSingleton
  include Singleton

  # size limit for in-process cache
  LOCAL_SIZE = 2000

  def initialize
    @local = SiteCache.new(Samizdat::Cache.new(nil, LOCAL_SIZE))
    @drb = {}
    @cache = {}
  end

  # explicitly request an in-process cache (e.g. for singleton classes)
  attr_accessor :local

  # cache DRb connections by per-site DRb URI, keep same SiteCache-wrapped
  # in-process local cache
  #
  def cache
    @cache[site_name] ||=
      if config.drb then
        @drb[config.drb] ||= SiteCache.new(DRbObject.new(nil, config.drb))
      else
        @local
      end
  end
end

# shortcut access to Cache object
#
def cache
  CacheSingleton.instance.cache
end

# shortcut to explicitly request an in-process cache
#
def local_cache
  CacheSingleton.instance.local
end


# database connection management
#
# permanently keeps DB connections in in-process cache
#
def db
  # todo: verify concurrency in db and storage
  # todo: check connection timeouts
  # optimize: generate connection pool
  local_cache.fetch_or_add('db:' + site_name) do
    db = DBI.connect(config['db']['dsn'].untaint,
      (ENV['USER'] or config['db']['user']),
      ENV['USER']? nil : config['db']['passwd'])
    begin
      db['AutoCommit'] = false
    rescue DBI::NotSupportedError
      # no need to disable if it's not there
    end
    db
  end
end

# rdf storage access shortcut
#
def rdf
  local_cache.fetch_or_add('rdf:' + site_name) do
    Samizdat::RDF.new(db, config)
  end
end

