#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc GridUpdate {} {
    global grid
    global current

    if {$current(frame) != ""} {
	SetWatchCursor
	if {$grid(view) && [$current(frame) has fits]} {
	    AdjustGridOptions
	    $current(frame) grid create $grid(system) $grid(sky) \
		$grid(skyformat) $grid(type) [GridOptions]
	} else {
	    $current(frame) grid delete
	}

	UnsetWatchCursor
    }
}	    

proc GridUpdateZoom {} {
    global grid

    if {$grid(type) == "publication"} {
	GridUpdate
    }
}

proc AdjustGridOptions {} {
    global grid
    global current

    if {$current(frame) != ""} {
	# change values if needed
	AdjustCoord $current(frame) grid(system)

	# adjust units
	set grid(gapunit1) pixels
	set grid(gapunit2) pixels
	switch -- $grid(system) {
	    image -
	    physical -
	    amplifier -
	    detector {}
	    default {
		if {$current(frame) != ""} {
		    if {[$current(frame) has wcs equatorial $grid(system)]} {
			set grid(gapunit1) degrees
			set grid(gapunit2) degrees
		    }
		}
	    }
	}
    } else {
	set grid(gapunit1) pixels
	set grid(gapunit2) pixels
    }
}

proc GridOptions {} {
    global grid
    global current

    set opt "\""

    # Grid
    append opt " Grid=$grid(grid),"
    append opt " Colour(grid)=$grid(grid,color),"
    append opt " Width(grid)=$grid(grid,width),"
    append opt " Style(grid)=$grid(grid,style),"

    # Axes
    append opt " DrawAxes=$grid(axes),"
    append opt " Colour(axes)=$grid(axes,color),"
    append opt " Width(axes)=$grid(axes,width),"
    append opt " Style(axes)=$grid(axes,style),"

    # Format
    switch $grid(system) {
	image -
	physical -
	detector -
	amplifier {}
	default {
	    if {[$current(frame) has wcs equatorial $grid(system)]} {
		switch $grid(sky) {
		    "fk4" -
		    "fk5" -
		    "icrs" {
			switch $grid(skyformat) {
			    degrees {
				append opt " Format(1)=d.3,"
				append opt " Format(2)=d.3,"
			    }
			    sexagesimal {
				append opt " Format(1)=hms.1,"
				append opt " Format(2)=dms.1,"
			    }
			    hms {
				append opt " Format(1)=lhms.1,"
				append opt " Format(2)=ldms.1,"
			    }
			}
		    }
		    "galactic" -
		    "ecliptic" {
			switch $grid(skyformat) {
			    degrees {
				append opt " Format(1)=d.3,"
				append opt " Format(2)=d.3,"
			    }
			    sexagesimal {
				append opt " Format(1)=dms.1,"
				append opt " Format(2)=dms.1,"
			    }
			    hms {
				append opt " Format(1)=ldms.1,"
				append opt " Format(2)=ldms.1,"
			    }
			}
		    }
		}
	    }
	}
    }

    # Ticks
    if {!$grid(tick)} {
	append opt " MajTickLen=0,"
	append opt " MinTick(1)=0,"
	append opt " MinTick(2)=0,"
    }
    append opt " Colour(ticks)=$grid(tick,color),"
    append opt " Width(ticks)=$grid(tick,width),"
    append opt " Style(ticks)=$grid(tick,style),"

    # Border
    append opt " Border=$grid(border),"
    append opt " Colour(border)=$grid(border,color),"
    append opt " Width(border)=$grid(border,width),"
    append opt " Style(border)=$grid(border,style),"

    # Labels
    append opt " Labelling=$grid(labelling),"
    append opt " LabelUp=$grid(labelup),"

    append opt " Edge(1)=$grid(edge,1),"
    append opt " Edge(2)=$grid(edge,2),"

    # NumLab
    append opt " NumLab=$grid(numlab),"
    set opt \
"$opt Font(numlab)=[GridFont2Ast $grid(numlab,font) $grid(numlab,style)],"
    append opt " Size(numlab)=$grid(numlab,size),"
    append opt " Colour(numlab)=$grid(numlab,color),"

    # TextLab
    switch -- $grid(type) {
	analysis {append opt " TextLab(1)=0,"}
	publication {append opt " TextLab(1)=$grid(textlab),"}
    }
    if {!$grid(textlab,def1)} {
	append opt " Label(1)=[GridStripComma $grid(label1)] ,"
    }

    switch -- $grid(type) {
	analysis {append opt " TextLab(2)=0,"}
	publication {append opt " TextLab(2)=$grid(textlab),"}
    }
    if {!$grid(textlab,def2)} {
	append opt " Label(2)=[GridStripComma $grid(label2)] ,"
    }

    set opt \
"$opt Font(textlab)=[GridFont2Ast $grid(textlab,font) $grid(textlab,style)],"
    append opt " Size(textlab)=$grid(textlab,size),"
    append opt " Colour(textlab)=$grid(textlab,color),"

    # Title
    switch -- $grid(type) {
	analysis {append opt " DrawTitle=0,"}
	publication {append opt " DrawTitle=$grid(title),"}
    }

    if {$grid(title,def)} {
	set t [GridStripComma "[$current(frame) get fits object name]"]
	if {$t != ""} {
	    append opt " Title=$t ,"
	}
    } else {
	set t [GridStripComma "$grid(title,text)"]
	if {$t != ""} {
	    append opt " Title=$t ,"
	}
    }

    set opt \
"$opt Font(title)=[GridFont2Ast $grid(title,font) $grid(title,style)],"
    append opt " Size(title)=$grid(title,size),"
    append opt " Colour(title)=$grid(title,color),"

    # Grid Spacing
    if {$grid(gap1) != ""} {
	if {$grid(gapunit1) == "degrees"} {
	    append opt " Gap(1)=[expr 3.14159/180.*$grid(gap1)],"
	} else {
	    append opt " Gap(1)=$grid(gap1),"
	}
    }

    if {$grid(gap2) != ""} {
	if {$grid(gapunit2) == "degrees"} {
	    append opt " Gap(2)=[expr 3.14159/180.*$grid(gap2)],"
	} else {
	    append opt " Gap(2)=$grid(gap2),"
	}
    }

    # axes numerics
    set flip 0
    switch -- $grid(type) {
	analysis {
	    set numx -.03
	    set numy -.03
	}
	publication {
	    set numx -.02
	    set numy -.01
	    switch -- $grid(labelling) {
		interior {}
		exterior {
		    switch -- $grid(type,numlab) {
			interior {}
			exterior {set flip 1}
		    }
		}
	    }
	}
    }

    # override
    if {$grid(numlab,gap1) != {}} {
	set numx [expr -$grid(numlab,gap1)/100.]
    }
    if {$grid(numlab,gap2) != {}} {
	set numy [expr -$grid(numlab,gap2)/100.]
    }
    if {$flip} {
	set numx [expr -$numx]
	set numy [expr -$numy]
    }
    append opt " NumLabGap(1)=[expr $numx/[lindex $current(zoom) 0]],"
    append opt " NumLabGap(2)=[expr $numy/[lindex $current(zoom) 1]],"

    # Label gaps
    switch -- $grid(type) {
	analysis {
	    set axisx 0
	    set axisy 0
	    set title 0
	}
	publication {
	    if {$grid(textlab,gap1) != {}} {
		set axisx [expr $grid(textlab,gap1)/100.]
	    } else {
		set axisx .04
	    }
	    if {$grid(textlab,gap2) != {}} {
		set axisy [expr $grid(textlab,gap2)/100.]
	    } else {
		set axisy .04
	    }

	    if {$grid(title,gap) != {}} {
		set title [expr $grid(title,gap)/100.]
	    } else {
		set title .04
	    }
	}
    }

    append opt " TextLabGap(1)=[expr $axisx/[lindex $current(zoom) 0]],"
    append opt " TextLabGap(2)=[expr $axisy/[lindex $current(zoom) 1]],"
    append opt " TitleGap=[expr -1-($title/[lindex $current(zoom) 0])],"

    # Orientation
    append opt " Edge(1)=top,"
    append opt " Edge(2)=left,"

    # The End
    append opt " \""

    global debug

    if {$debug(tcl,grid)} {
	puts "GridOptions"
	puts "$opt"
    }

    return $opt
}

proc GridParseOptions {opt} {
    global grid
    global current
    global debug

    if {$debug(tcl,grid)} {
	puts "GridParseOptions"
	puts "$opt"
    }

    # Grid
    strtok " Grid=%d," opt grid(grid)
    strtok " Colour(grid)=%d," opt grid(grid,color)
    strtok " Width(grid)=%d," opt grid(grid,width)
    strtok " Style(grid)=%d," opt grid(grid,style)

    # Axes
    strtok " DrawAxes=%d," opt grid(axes)
    strtok " Colour(axes)=%d," opt grid(axes,color)
    strtok " Width(axes)=%d," opt grid(axes,width)
    strtok " Style(axes)=%d," opt grid(axes,style)

    # Format
    set format1 ""
    set format2 ""
    strtok { Format(1)=%[^,]} opt format1
    strtok { Format(2)=%[^,]} opt format2

    # Ticks
    set grid(tick) 1
    strtok " MajTickLen=%d," opt grid(tick)
    strtok " MinTick(1)=%d," opt grid(tick)
    strtok " MinTick(2)=%d," opt grid(tick)

    strtok " Colour(ticks)=%d," opt grid(tick,color)
    strtok " Width(ticks)=%d," opt  grid(tick,width)
    strtok " Style(ticks)=%d," opt grid(tick,style)

    # Border
    strtok " Border=%d," opt grid(border)
    strtok " Colour(border)=%d," opt grid(border,color)
    strtok " Width(border)=%d," opt grid(border,width)
    strtok " Style(border)=%d," opt grid(border,style)

    # Labels
    strtok { Labelling=%[^,],} opt grid(labelling)
    strtok " LabelUp=%d," opt grid(labelup)

    strtok { Edge(1)=%[^,],} opt grid(edge,1)
    strtok { Edge(2)=%[^,],} opt grid(edge,2)

    # Numlab
    strtok " NumLab=%d," opt grid(numlab)

    set numlabFont 0
    strtok " Font(numlab)=%d," opt numlabFont
    GridAst2Font $numlabFont grid(numlab,font) grid(numlab,style)
    
    strtok " Size(numlab)=%d," opt grid(numlab,size)
    strtok " Colour(numlab)=%d," opt grid(numlab,color)

    # Textlab
    switch -- $grid(type) {
	analysis {strtok " TextLab(1)=%d," opt dummy}
	publication {strtok " TextLab(1)=%d," opt grid(textlab)}
    }
    set grid(textlab,def1) [expr ![strtok { Label(1)=%[^,],} opt grid(label1)]]

    switch -- $grid(type) {
	analysis {strtok " TextLab(2)=%d," opt dummy}
	publication {strtok " TextLab(2)=%d," opt grid(textlab)}
    }
    set grid(textlab,def2) [expr ![strtok { Label(2)=%[^,],} opt grid(label2)]]

    set textlabFont 0
    strtok " Font(textlab)=%d," opt textlabFont
    GridAst2Font $textlabFont grid(textlab,font) grid(textlab,style)

    strtok " Size(textlab)=%d," opt grid(textlab,size)
    strtok " Colour(textlab)=%d," opt grid(textlab,color)

    # Title
    switch -- $grid(type) {
	analysis {strtok " DrawTitle=%d," opt dummy}
	publication {strtok " DrawTitle=%d," opt grid(title)}
    }
    strtok { Title=%[^,],} opt grid(title,text)

    set titleFont 0
    strtok " Font(title)=%d," opt titleFont
    GridAst2Font $titleFont grid(title,font) grid(title,style)

    strtok " Size(title)=%d," opt grid(title,size)
    strtok " Colour(title)=%d" opt grid(title,color)

    # Spacing
    set gap1 ""
    strtok { Gap(1)=%[^,]} opt gap1
    if {($gap1 != "") && ($grid(gapunit1) == "degrees")} {
	set grid(gap1) [expr 180.*$gap1/3.14159]
    } else {
	set grid(gap1) $gap1
    }

    set gap2 ""
    strtok { Gap(2)=%[^,]} opt gap2
    if {($gap2 != "") && ($grid(gapunit2) == "degrees")} {
	set grid(gap2) [expr 180.*$gap2/3.14159]
    } else {
	set grid(gap2) $gap2
    }

    # Gaps
    set dummy {}
    strtok " NumLabGap(1)=%f," opt dummy
    strtok " NumLabGap(2)=%f," opt dummy
    strtok " TextLabGap(1)=%f," opt dummy
    strtok " TextLabGap(2)=%f," opt dummy
    strtok " TitleGap=%f," opt dummy

    # Orientation
    set dummy {}
    strtok { Edge(1)=%[^,],} opt dummy
    strtok { Edge(2)=%[^,],} opt dummy
}

proc strtok {tmp strvar varvar} {
    upvar $strvar str
    upvar $varvar var
    global debug

    set r [scan $str $tmp var]
    if {$r} {
	set i [string first "," $str]
	set str [string range $str [expr $i+1] end]

	if {$debug(tcl,grid)} {
	    puts "Grid strtok: Success parse $tmp"
	}
    } else {
	if {$debug(tcl,grid)} {
	    puts "Grid strtok: Failed to parse $tmp"
	}
    }

    return $r
}

proc GridAst2Font {ast fnvar fsvar} {
    upvar $fnvar fn
    upvar $fsvar fs

    switch -- $ast {
	0 {set fn "system"; set fs "plain"}
	1 {set fn "times"; set fs "plain"}
	2 {set fn "helvetica"; set fs "plain"}
	3 {set fn "symbol"; set fs "plain"}
	4 {set fn "courier"; set fs "plain"}
	10 {set fn "system"; set fs "bold"}
	11 {set fn "times"; set fs "bold"}
	12 {set fn "helvetica"; set fs "bold"}
	13 {set fn "symbol"; set fs "bold"}
	14 {set fn "courier"; set fs "bold"}
	20 {set fn "system"; set fs "italic"}
	21 {set fn "times"; set fs "italic"}
	22 {set fn "helvetica"; set fs "italic"}
	23 {set fn "symbol"; set fs "italic"}
	24 {set fn "courier"; set fs "italic"}
	default {set fn "system"; set fs "plain"}
    }
}

proc GridFont2Ast {fn fs} {
    if {$fn == "system" && $fs == "plain"} {
	return 0;
    } elseif {$fn == "times" && $fs == "plain"} {
	return 1;
    } elseif {$fn == "helvetica" && $fs == "plain"} {
	return 2;
    } elseif {$fn == "symbol" && $fs == "plain"} {
	return 3;
    } elseif {$fn == "courier" && $fs == "plain"} {
	return 4;
    } elseif {$fn == "system" && $fs == "bold"} {
	return 10;
    } elseif {$fn == "times" && $fs == "bold"} {
	return 11;
    } elseif {$fn == "helvetica" && $fs == "bold"} {
	return 12;
    } elseif {$fn == "symbol" && $fs == "bold"} {
	return 13;
    } elseif {$fn == "courier" && $fs == "bold"} {
	return 14;
    } elseif {$fn == "system" && $fs == "italic"} {
	return 20;
    } elseif {$fn == "times" && $fs == "italic"} {
	return 21;
    } elseif {$fn == "helvetica" && $fs == "italic"} {
	return 22;
    } elseif {$fn == "symbol" && $fs == "italic"} {
	return 23;
    } elseif {$fn == "courier" && $fs == "italic"} {
	return 24;
    } else {
	return 0;
    }
}

proc GridDialog {} {
    global grid
    global current
    global menu
    global gr
    global ds9

    # see if we already have a window visible

    if [winfo exist $grid(top)] {
	raise $grid(top)
	return
    }

    set w $grid(top)
    set title "Coordinate Grid"

    # create the window

    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW GridDestroyDialog

    $w configure -menu $grid(mb)

    menu $grid(mb) -tearoff 0

    GridFileMenu
    GridTypeMenu
    GridCoordMenu
    GridViewMenu
    GridColorMenu
    GridLineMenu
    GridFontMenu

    set length 300

    frame $w.buttons -relief groove -borderwidth 2
    frame $w.label -relief groove -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    pack $w.label $w.param $w.buttons -fill x -expand true -ipadx 4 -ipady 4

    # Labels

    label $w.label.label -text "Title"
    entry $w.label.title -textvariable grid(title,text) \
	-width 28 
    checkbutton $w.label.default -text "Default" \
	-variable grid(title,def) -selectcolor $menu(selectcolor)
    label $w.label.label1 -text "Axis 1"
    entry $w.label.title1 -textvariable grid(label1) \
	-width 28 
    checkbutton $w.label.default1 -text "Default" \
	-variable grid(textlab,def1) -selectcolor $menu(selectcolor)
    label $w.label.label2 -text "Axis 2"
    entry $w.label.title2 -textvariable grid(label2) \
	-width 28 
    checkbutton $w.label.default2 -text "Default" \
	-variable grid(textlab,def2) -selectcolor $menu(selectcolor)

    grid $w.label.label $w.label.title $w.label.default \
	-padx 4 -pady 1 -sticky w
    grid $w.label.label1 $w.label.title1 $w.label.default1 \
	-padx 4 -pady 1 -sticky w
    grid $w.label.label2 $w.label.title2 $w.label.default2 \
	-padx 4 -pady 1 -sticky w

    # Params

    label $w.param.lspace -text "Label\nSpacing%"
    label $w.param.ngap -text "Numerics\nSpacing%"
    label $w.param.lgap -text "Grid Gap"
    label $w.param.lgapunit -text "Grid Gap\nUnits"

    label $w.param.titlet -text "Title"
    entry $w.param.spacet -textvariable grid(title,gap) \
	-width 8 

    label $w.param.title1 -text "Axis 1"
    entry $w.param.tspace1 -textvariable grid(textlab,gap1) -width 8 
    entry $w.param.nspace1 -textvariable grid(numlab,gap1) -width 8 
    entry $w.param.gap1 -textvariable grid(gap1) -width 8 
    label $w.param.gapunit1 -textvariable grid(gapunit1) \
	-relief groove -width 8 -padx 4

    label $w.param.title2 -text "Axis 2"
    entry $w.param.tspace2 -textvariable grid(textlab,gap2) -width 8 
    entry $w.param.nspace2 -textvariable grid(numlab,gap2) -width 8 
    entry $w.param.gap2 -textvariable grid(gap2) -width 8 
    label $w.param.gapunit2 -textvariable grid(gapunit2) \
	-relief groove -width 8 -padx 4

    grid x $w.param.lspace $w.param.ngap $w.param.lgap \
	$w.param.lgapunit -padx 4 -pady 1 -sticky news
    grid $w.param.titlet $w.param.spacet -padx 4 -pady 1 -sticky w
    grid $w.param.title1 $w.param.tspace1 $w.param.nspace1 \
	$w.param.gap1 $w.param.gapunit1 -padx 4 -pady 1 -sticky w
    grid $w.param.title2 $w.param.tspace2 $w.param.nspace2 \
	$w.param.gap2 $w.param.gapunit2 -padx 4 -pady 1 -sticky w

    button $w.buttons.apply -text "Apply" -command GridApplyDialog
    button $w.buttons.clear -text "Clear" -command GridClearDialog
    button $w.buttons.close -text "Close" -command GridDestroyDialog
    pack $w.buttons.apply $w.buttons.clear $w.buttons.close \
	-side left -padx 10 -expand true

    bind $w <Return> "GridApplyDialog"

    # some window managers need a hint
    raise $w

    UpdateGridDialog
}

proc GridApplyDialog {} {
    global grid

    set grid(view) 1
    GridUpdate
}

proc GridClearDialog {} {
    global grid

    set grid(view) 0
    GridUpdate
}

proc GridDestroyDialog {} {
    global grid

    if {[winfo exist $grid(top)]} {
	destroy $grid(top)
	destroy $grid(mb)
    }
}

proc UpdateGridMenu {} {
    global grid
    global current

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateGridMenu"
    }

    # set menu

    if {($current(frame) != "") && [$current(frame) has fits]} {
	set grid(view) [$current(frame) has grid]
    }

    # reassign system and format

    if {($current(frame) != "") && [$current(frame) has fits]} {
	if [$current(frame) has grid] {
	    set l  [$current(frame) get grid]
	    set grid(system) [lindex $l 0]
	    set grid(sky) [lindex $l 1]
	    set grid(skyformat) [lindex $l 2]
	    set grid(type) [lindex $l 3]

	    GridParseOptions [$current(frame) get grid option]
	}
    }
}

proc UpdateGridDialog {} {
    global current
    global grid

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateGridDialog"
    }

    AdjustGridOptions

    if {[winfo exist $grid(top)]} {
	# enable/disable menu items
	UpdateCoordMenu "$current(frame)" $grid(mb).coord
	if {$current(frame) != ""} {
	    if {[$current(frame) has wcs equatorial $grid(system)]} {
		$grid(mb).coord entryconfig "HMS/DMS" -state normal
	    } else {
		$grid(mb).coord entryconfig "HMS/DMS" -state disabled
	    }
	} else {
	    $grid(mb).coord entryconfig "HMS/DMS" -state normal
	}
    }
}

proc GridFileMenu {} {
    global grid
    global menu

    $grid(mb) add cascade -label File -menu $grid(mb).file

    menu $grid(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $grid(mb).file add command -label Apply -command GridApplyDialog
    $grid(mb).file add command -label Clear -command GridClearDialog
    $grid(mb).file add separator
    $grid(mb).file add command -label "Load Configuration..." \
	-command GridLoadDialog
    $grid(mb).file add command -label "Save Configuration..." \
	-command GridSaveDialog
    $grid(mb).file add separator
    $grid(mb).file add command -label Close -command GridDestroyDialog
}

proc GridCoordMenu {} {
    global grid
    global menu
    global ds9

    $grid(mb) add cascade -label Coordinate -menu $grid(mb).coord

    menu $grid(mb).coord -tearoff 0 -selectcolor $menu(selectcolor)
    $grid(mb).coord add radiobutton -label "WCS" \
	-variable grid(system) -value wcs -command UpdateGridDialog
    $grid(mb).coord add cascade -label "Multiple WCS" \
	-menu $grid(mb).coord.wcs
    $grid(mb).coord add separator
    $grid(mb).coord add radiobutton -label "Image" \
	-variable grid(system) -value image -command UpdateGridDialog
    $grid(mb).coord add radiobutton -label "Physical" \
	-variable grid(system) -value physical -command UpdateGridDialog
    if {$ds9(amp,det)} {
	$grid(mb).coord add radiobutton -label "Amplifier" \
	    -variable grid(system) -value amplifier -command UpdateGridDialog
	$grid(mb).coord add radiobutton -label "Detector" \
	    -variable grid(system) -value detector -command UpdateGridDialog
    }
    $grid(mb).coord add separator
    $grid(mb).coord add radiobutton -label "Equatorial B1950" \
	-variable grid(sky) -value fk4 -command UpdateGridDialog
    $grid(mb).coord add radiobutton -label "Equatorial J2000" \
	-variable grid(sky) -value fk5 -command UpdateGridDialog
    $grid(mb).coord add radiobutton -label "ICRS" \
	-variable grid(sky) -value icrs -command UpdateGridDialog
    $grid(mb).coord add radiobutton -label "Galactic" \
	-variable grid(sky) -value galactic -command UpdateGridDialog
    $grid(mb).coord add radiobutton -label "Ecliptic" \
	-variable grid(sky) -value ecliptic -command UpdateGridDialog
    $grid(mb).coord add separator
    $grid(mb).coord add radiobutton -label "Degrees" \
	-variable grid(skyformat) -value degrees -command UpdateGridDialog
    $grid(mb).coord add radiobutton -label "Sexagesimal" \
	-variable grid(skyformat) -value sexagesimal \
	-command UpdateGridDialog
    $grid(mb).coord add radiobutton -label "HMS/DMS" \
	-variable grid(skyformat) -value hms -command UpdateGridDialog

    menu $grid(mb).coord.wcs -tearoff 0 -selectcolor $menu(selectcolor)
    foreach l {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$grid(mb).coord.wcs add radiobutton -label "WCS $l" \
	    -variable grid(system) -value "wcs$l" \
	    -command UpdateGridDialog
    }
}

proc GridTypeMenu {} {
    global grid
    global menu

    $grid(mb) add cascade -label Type -menu $grid(mb).type

    menu $grid(mb).type -tearoff 0 -selectcolor $menu(selectcolor)
    $grid(mb).type add radiobutton -label "Analysis" \
	-variable grid(type) -value analysis
    $grid(mb).type add radiobutton -label "Publication" \
	-variable grid(type) -value publication
    $grid(mb).type add separator
    $grid(mb).type add radiobutton -label "Interior Axes" \
	-variable grid(labelling) -value interior
    $grid(mb).type add radiobutton -label "Exterior Axes" \
	-variable grid(labelling) -value exterior
    $grid(mb).type add separator
    $grid(mb).type add radiobutton -label "Interior Numerics" \
	-variable grid(type,numlab) -value interior
    $grid(mb).type add radiobutton -label "Exterior Numerics" \
	-variable grid(type,numlab) -value exterior
}

proc GridViewMenu {} {
    global grid
    global menu

    $grid(mb) add cascade -label View -menu $grid(mb).view

    menu $grid(mb).view -tearoff 0 -selectcolor $menu(selectcolor)
    $grid(mb).view add checkbutton -label Grid -variable grid(grid)
    $grid(mb).view add checkbutton -label Axes -variable grid(axes)
    $grid(mb).view add checkbutton -label "Axes Numbers" -variable grid(numlab)
    $grid(mb).view add checkbutton -label "Axes Tickmarks" -variable grid(tick)
    $grid(mb).view add checkbutton -label "Axes Label" -variable grid(textlab)
    $grid(mb).view add checkbutton -label Title -variable grid(title)
    $grid(mb).view add checkbutton -label Border -variable grid(border)
    $grid(mb).view add separator
    $grid(mb).view add checkbutton -label "Vertical Text" \
	-variable grid(labelup)
}

proc GridColorMenu {} {
    global grid
    global menu

    $grid(mb) add cascade -label Color -menu $grid(mb).color

    menu $grid(mb).color -tearoff 0 -selectcolor $menu(selectcolor)
    $grid(mb).color add cascade -label Grid -menu $grid(mb).color.grid
    $grid(mb).color add cascade -label Axes -menu $grid(mb).color.axes
    $grid(mb).color add cascade -label "Axes Numbers" \
	-menu $grid(mb).color.numlab
    $grid(mb).color add cascade -label "Axes Label" \
	-menu $grid(mb).color.textlab
    $grid(mb).color add cascade -label Title -menu $grid(mb).color.title
    $grid(mb).color add cascade -label Tickmarks -menu $grid(mb).color.tick
    $grid(mb).color add cascade -label Border -menu $grid(mb).color.border

    GridCreateColorMenu $grid(mb).color.grid grid(grid,color)
    GridCreateColorMenu $grid(mb).color.axes grid(axes,color)
    GridCreateColorMenu $grid(mb).color.title grid(title,color)
    GridCreateColorMenu $grid(mb).color.textlab grid(textlab,color)
    GridCreateColorMenu $grid(mb).color.numlab grid(numlab,color)
    GridCreateColorMenu $grid(mb).color.tick grid(tick,color)
    GridCreateColorMenu $grid(mb).color.border grid(border,color)
}

proc GridCreateColorMenu {which varname} {
    global menu
    global grid

    menu $which -tearoff 0 -selectcolor $menu(selectcolor)
    $which add radiobutton -label "Black" -variable $varname -value 0
    $which add radiobutton -label "White" -variable $varname -value 1
    $which add radiobutton -label "Red" -variable $varname -value 2
    $which add radiobutton -label "Green" -variable $varname -value 3
    $which add radiobutton -label "Blue" -variable $varname -value 4
    $which add radiobutton -label "Cyan" -variable $varname -value 5
    $which add radiobutton -label "Magenta" -variable $varname -value 6
    $which add radiobutton -label "Yellow" -variable $varname -value 7
}

proc GridLineMenu {} {
    global grid
    global menu

    $grid(mb) add cascade -label Line -menu $grid(mb).line

    menu $grid(mb).line -tearoff 0 -selectcolor $menu(selectcolor)
    $grid(mb).line add cascade -label Grid -menu $grid(mb).line.grid
    $grid(mb).line add cascade -label Axes -menu $grid(mb).line.axes
    $grid(mb).line add cascade -label Tickmarks -menu $grid(mb).line.tick
    $grid(mb).line add cascade -label Border -menu $grid(mb).line.border

    GridCreateLineMenu $grid(mb).line.grid grid(grid,width) grid(grid,style)
    GridCreateLineMenu $grid(mb).line.axes grid(axes,width) grid(axes,style)
    GridCreateLineMenu $grid(mb).line.tick grid(tick,width) grid(tick,style)
    GridCreateLineMenu $grid(mb).line.border \
	grid(border,width) grid(border,style)
}

proc GridCreateLineMenu {which var1 var2} {
    global grid
    global menu

    menu $which -tearoff 0 -selectcolor $menu(selectcolor)
    $which add radiobutton -label "Thin" -variable $var1 -value 0
    $which add radiobutton -label "1" -variable $var1 -value 1
    $which add radiobutton -label "2" -variable $var1 -value 2
    $which add radiobutton -label "3" -variable $var1 -value 3
    $which add radiobutton -label "4" -variable $var1 -value 4
    $which add separator
    $which add radiobutton -label Solid -variable $var2 -value 0
    $which add radiobutton -label Dash -variable $var2 -value 1
}

proc GridFontMenu {} {
    global grid
    global menu

    $grid(mb) add cascade -label Font -menu $grid(mb).font

    menu $grid(mb).font -tearoff 0 -selectcolor $menu(selectcolor)
    $grid(mb).font add cascade -label "Axes Numbers" \
	-menu $grid(mb).font.numlab
    $grid(mb).font add cascade -label "Axes Label" \
	-menu $grid(mb).font.textlab
    $grid(mb).font add cascade -label "Title" \
	-menu $grid(mb).font.title

    GridCreateFontMenu $grid(mb).font.title \
	grid(title,font) grid(title,size) grid(title,style)
    GridCreateFontMenu $grid(mb).font.textlab \
	grid(textlab,font) grid(textlab,size) grid(textlab,style)
    GridCreateFontMenu $grid(mb).font.numlab \
	grid(numlab,font) grid(numlab,size) grid(numlab,style)
}

proc GridCreateFontMenu {which var1 var2 var3} {
    global grid
    global menu

    menu $which -tearoff 0 -selectcolor $menu(selectcolor)
    $which add radiobutton -label Times -variable $var1 -value times
    $which add radiobutton -label Helvetica -variable $var1 -value helvetica
    $which add radiobutton -label Symbol -variable $var1 -value symbol
    $which add radiobutton -label Courier -variable $var1 -value courier
    $which add separator
    $which add radiobutton -label 9 -variable $var2 -value 9
    $which add radiobutton -label 10 -variable $var2 -value 10
    $which add radiobutton -label 12 -variable $var2 -value 12
    $which add radiobutton -label 14 -variable $var2 -value 14
    $which add radiobutton -label 18 -variable $var2 -value 18
    $which add radiobutton -label 24 -variable $var2 -value 24
    $which add radiobutton -label 30 -variable $var2 -value 30
    $which add radiobutton -label 36 -variable $var2 -value 36
    $which add separator
    $which add radiobutton -label Plain -variable $var3 -value plain
    $which add radiobutton -label Bold -variable $var3 -value bold
    $which add radiobutton -label Italic -variable $var3 -value italic
}

proc GridSaveDialog {} {
    GridSave [SaveFileDialog gridfbox]
}

proc GridSave {filename} {
    global grid

    if {$filename != {}} {
	set file [open $filename w]

	puts $file "global grid"

	# Coordinate system
	puts $file "set grid(system) $grid(system)"
	puts $file "set grid(sky) $grid(sky)"
	puts $file "set grid(skyformat) $grid(skyformat)"

	# Type
	puts $file "set grid(type) $grid(type)"
	puts $file "set grid(type,numlab) $grid(type,numlab)"

	# Grid
	puts $file "set grid(grid) $grid(grid)"
	puts $file "set grid(grid,color) $grid(grid,color)"
	puts $file "set grid(grid,width) $grid(grid,width)"
	puts $file "set grid(grid,style) $grid(grid,style)"

	puts $file "set grid(gap1) \{$grid(gap1)\}"
	puts $file "set grid(gapunit1) $grid(gapunit1)"
	puts $file "set grid(gap2) \{$grid(gap2)\}"
	puts $file "set grid(gapunit2) $grid(gapunit2)"

	# Axes
	puts $file "set grid(axes) $grid(axes)"
	puts $file "set grid(axes,color) $grid(axes,color)"
	puts $file "set grid(axes,width) $grid(axes,width)"
	puts $file "set grid(axes,style) $grid(axes,style)"

	# Ticks
	puts $file "set grid(tick) $grid(tick)"
	puts $file "set grid(tick,color) $grid(tick,color)"
	puts $file "set grid(tick,width) $grid(tick,width)"
	puts $file "set grid(tick,style) $grid(tick,style)"

	# Border
	puts $file "set grid(border) $grid(border)"
	puts $file "set grid(border,color) $grid(border,color)"
	puts $file "set grid(border,width) $grid(border,width)"
	puts $file "set grid(border,style) $grid(border,style)"

	# Labels
	puts $file "set grid(labelling) $grid(labelling)"
	puts $file "set grid(labelup) $grid(labelup)"

	# Numlab
	puts $file "set grid(numlab) $grid(numlab)"
	puts $file "set grid(numlab,font)  $grid(numlab,font)"
	puts $file "set grid(numlab,style) $grid(numlab,style)"
	puts $file "set grid(numlab,size)  $grid(numlab,size)"
	puts $file "set grid(numlab,color) $grid(numlab,color)"
	puts $file "set grid(numlab,gap1) \{$grid(numlab,gap1)\}"
	puts $file "set grid(numlab,gap2) \{$grid(numlab,gap2)\}"

	# Textlab
	puts $file "set grid(textlab) $grid(textlab)"
	puts $file "set grid(textlab,def1) $grid(textlab,def1)"
	puts $file "set grid(label1) \{$grid(label1)\} "
	puts $file "set grid(textlab,gap1) \{$grid(textlab,gap1)\}"

	puts $file "set grid(textlab,def2) $grid(textlab,def2)"
	puts $file "set grid(label2) \{$grid(label2)\} "
	puts $file "set grid(textlab,gap2) \{$grid(textlab,gap2)\}"

	puts $file "set grid(textlab,font)  $grid(textlab,font)"
	puts $file "set grid(textlab,style) $grid(textlab,style)"
	puts $file "set grid(textlab,size)  $grid(textlab,size)"
	puts $file "set grid(textlab,color) $grid(textlab,color)"

	# Title
	puts $file "set grid(title) $grid(title)"
	puts $file "set grid(title,def) $grid(title,def)"
	puts $file "set grid(title,text) \{$grid(title,text)\} "
	puts $file "set grid(title,gap) \{$grid(title,gap)\}"

	puts $file "set grid(title,font)  $grid(title,font)"
	puts $file "set grid(title,style) $grid(title,style)"
	puts $file "set grid(title,size)  $grid(title,size)"
	puts $file "set grid(title,color) $grid(title,color)"

	close $file
    }
}

proc GridLoadDialog {} {
    GridLoad [OpenFileDialog gridfbox]
}

proc GridLoad {filename} {
    global grid

    if {$filename != {}} {
	source $filename
    }

    # backward compatibility
    switch -- $grid(type) {
	1 {
	    set grid(type) analysis
	    set grid(labelling) interior
	    set grid(type,numlab) interior
	}
	2 {
	    set grid(type) analysis
	    set grid(labelling) exterior
	    set grid(type,numlab) interior
	}
	3 {
	    set grid(type) publication
	    set grid(labelling) exterior
	    set grid(type,numlab) interior
	}
	4 {
	    set grid(type) publication
	    set grid(labelling) exterior
	    set grid(type,numlab) exterior
	}
    }
}

proc GridStripComma {str} {
    # strip ','
    set t ""
    regsub -all "," "$str" " " t
    return $t
}

proc ProcessGridCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global grid
    switch -- [string tolower [lindex $var $i]] {
	load {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    GridLoad $fn
	    FileLastFull gridfbox $fn
	    GridUpdate
	}
	save {
	    incr i
	    set fn [file normalize [lindex $var $i]]
	    GridSave $fn
	    FileLastFull gridfbox $fn
	}

	system {incr i; set grid(system) [lindex $var $i]; GridUpdate}
	sky {incr i; set grid(sky) [lindex $var $i]; GridUpdate}
	skyformat {incr i; set grid(skyformat) [lindex $var $i]; GridUpdate}

	type {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		axes {incr i; set grid(labelling) [lindex $var $i]}
		numerics {incr i; set grid(type,numlab) [lindex $var $i]}
		default {set grid(type) [lindex $var $i]}
	    }
	    GridUpdate
	}

	view {
	    incr i
	    switch -- [string tolower [lindex $var $i]] {
		grid {incr i; set grid(grid) [FromYesNo [lindex $var $i]]}
		axes {
		    incr i
		    switch -- [string tolower [lindex $var $i]] {
			numbers {incr i; set grid(numlab) \
				     [FromYesNo [lindex $var $i]]}
			tickmarks {incr i; set grid(tick) \
				       [FromYesNo [lindex $var $i]]}
			label {incr i; set grid(textlab) \
				   [FromYesNo [lindex $var $i]]}
			default {set grid(axes) [FromYesNo [lindex $var $i]]}
		    }
		}
		title {incr i; set grid(title) [FromYesNo [lindex $var $i]]}
		border {incr i; set grid(border) [FromYesNo [lindex $var $i]]}
		vertical {incr i; set grid(labelup) \
			      [FromYesNo [lindex $var $i]]}
	    }
	    GridUpdate
	}

	yes -
	true -
	1 -
	no -
	false -
	0 {
	    set grid(view) [FromYesNo [lindex $var $i]]
	    GridUpdate
	}
	default {
	    set grid(view) 1
	    GridUpdate
	    incr i -1
	}
    }
}
