#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# Public Proceduress

proc HV {varname title url font extmenu init} {
    upvar #0 $varname var
    global $varname

    global debug
    global ds9
    global menu

    if {$debug(tcl,hv)} {
	puts "HV $varname $title $url"
    }

    set var(top) ".${varname}"
    set var(mb) ".${varname}mb"

    set w $var(top)

    # see if we already have a window visible
    
    if {[winfo exist $w]} {
	raise $w
    } else {
	set var(widget) {}
	set var(frame) new
	set var(file,mode) { }
	set var(save) 0
	set var(title) "$title"

	set var(active) 0
	set var(index) 0
	set var(font,size) $font
	set var(init) $init

	set var(img,forward) ${varname}forward
	set var(img,back) ${varname}back
	set var(img,reload) ${varname}reload
	set var(img,stop) ${varname}stop
	set var(img,gray) ${varname}gray

	HVClearAll $varname
	set var(delete) 0

	# create window
	# Note: we want this window to have its own colormap
	toplevel $w
	wm title $w $title
	wm iconname $w $title
	wm group $w $ds9(top)
	wm protocol $w WM_DELETE_WINDOW "HVDestroyCmd $varname"

	$w configure -menu $var(mb)

	menu $var(mb) -tearoff 0 -selectcolor $menu(selectcolor)
	$var(mb) add cascade -label File -menu $var(mb).file
	$var(mb) add cascade -label Edit -menu $var(mb).edit
	$var(mb) add cascade -label View -menu $var(mb).view
	$var(mb) add cascade -label Frame -menu $var(mb).frame
	$var(mb) add cascade -label "Open Fits As" -menu $var(mb).open

	menu $var(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
	if {$debug(tcl,hv) || $extmenu} {
	    $var(mb).file add command -label "Open URL" \
		-command "HVURLDialogCmd $varname"
	    $var(mb).file add command -label "Open File" \
		-command "HVFileDialogCmd $varname"
	    $var(mb).file add separator
	    $var(mb).file add command -label Clear -command "HVClearCmd $varname"
	    $var(mb).file add command -label "Page Source" \
		-command "HVPageSourceCmd $varname"
	    $var(mb).file add separator
	}
	$var(mb).file add command -label Reload -command "HVReloadCmd $varname"
	$var(mb).file add separator
	$var(mb).file add command -label Close -command "HVDestroyCmd $varname"

	menu $var(mb).edit -tearoff 0 -selectcolor $menu(selectcolor)
	$var(mb).edit add command -label Cut -state disabled
	$var(mb).edit add command -label Copy -state disabled
	$var(mb).edit add command -label Paste -state disabled
	$var(mb).edit add command -label Clear -state disabled

	menu $var(mb).view -tearoff 0 -selectcolor $menu(selectcolor)
	$var(mb).view add command -label Back -command "HVBackCmd $varname"
	$var(mb).view add command -label Forward -command "HVForwardCmd $varname"
	$var(mb).view add separator
	$var(mb).view add cascade -label "Text Size" \
	    -menu $var(mb).view.font
	$var(mb).view add separator
	$var(mb).view add command -label Stop -command "HVStopCmd $varname" \
	    -state disabled

	menu $var(mb).view.font -tearoff 0 -selectcolor $menu(selectcolor)
	$var(mb).view.font add radiobutton -label Smaller \
	    -variable ${varname}(font,size) -value -2 \
	    -command "HVFontCmd $varname"
	$var(mb).view.font add radiobutton -label Normal \
	    -variable ${varname}(font,size) -value 0 \
	    -command "HVFontCmd $varname"
	$var(mb).view.font add radiobutton -label Larger \
	    -variable ${varname}(font,size) -value 2 \
	    -command "HVFontCmd $varname"
	$var(mb).view.font add radiobutton -label Largest \
	    -variable ${varname}(font,size) -value 4 \
	    -command "HVFontCmd $varname"

	menu $var(mb).frame -tearoff 0 -selectcolor $menu(selectcolor)
	$var(mb).frame add checkbutton -label "Save FITS on download" \
	    -variable ${varname}(save)
	$var(mb).frame add separator
	$var(mb).frame add radiobutton \
	    -label "Create New Frame on Download" \
	    -variable ${varname}(frame) -value new
	$var(mb).frame add radiobutton \
	    -label "Use Current Frame on Download" \
	    -variable ${varname}(frame) -value current

	menu $var(mb).open -tearoff 0 -selectcolor $menu(selectcolor)
	$var(mb).open add radiobutton -label "Fits" \
	    -variable ${varname}(file,mode) -value { }
	$var(mb).open add separator
	$var(mb).open add radiobutton -label "RGB Fits Image" \
	    -variable ${varname}(file,mode) -value {rbg image}
	$var(mb).open add radiobutton -label "RGB Fits Cube" \
	    -variable ${varname}(file,mode) -value {rbg cube}
	$var(mb).open add separator
	$var(mb).open add radiobutton -label "Multi Ext Data Cube" \
	    -variable ${varname}(file,mode) -value {data cube}
	$var(mb).open add separator
	$var(mb).open add radiobutton -label "Mosaic IRAF" \
	    -variable ${varname}(file,mode) -value {mosaic image iraf}
	$var(mb).open add radiobutton -label "Mosaic IRAF Segment" \
	    -variable ${varname}(file,mode) -value {mosaic iraf}
	$var(mb).open add separator
	$var(mb).open add radiobutton -label "Mosaic WCS" \
	    -variable ${varname}(file,mode) -value {mosaic image wcs}
	$var(mb).open add radiobutton -label "Mosaic WCS Next" \
	    -variable ${varname}(file,mode) -value {mosaic image next wcs}
	$var(mb).open add radiobutton -label "Mosaic WCS Segment" \
	    -variable ${varname}(file,mode) -value {mosaic wcs}
	$var(mb).open add separator
	$var(mb).open add radiobutton -label "Mosaic Image WFPC2" \
	    -variable ${varname}(file,mode) -value {mosaic image wfpc2}

	image create photo $var(img,back) -data {R0lGODlhDwANAKL/AM///8DAwJD//y/I/y+X/y9n/wAAAAAAACH5BAEAAAEALAAAAAAPAA0AAAM0GLq2/qE0+AqYVFmB6eZFKEoRIAyCaaYCYWxDLM9uYBAxoe/7dA8ug3AoZOg6mRsyuUxmEgA7}
	image create photo $var(img,forward) -data {R0lGODlhDwANAKL/AM///8DAwJD//y/I/y+X/y9n/wAAAAAAACH5BAEAAAEALAAAAAAPAA0AAAM3GLpa/K8YSMuYlBVwV/kgCAhdsAFoig7ktA1wLA9SQdw4DkuB4f8/Ag2TMRB4GYUBmewRm09FAgA7}
	image create photo $var(img,stop) -data {R0lGODlhDQANALP/AP///1Lq81I5Of+EhCEAAHsAAMYAAP+UQv9zCHuMjP8AMf8AKf+MnK1CSv8QIQAAACH5BAEAAAEALAAAAAANAA0AAARWMMjUTC1J6ubOQYdiCBuIIMuiiCT1OWu6Ys05AMPC4ItBGB8dYMdI+RoHR4qY6v1CwlvRcEQ4brndwFAgJAwIRdPIzVTEYiqXJBEU1FQCW5Mg2O0ZSQQAOw==}
	image create photo $var(img,reload) -data {R0lGODlhDAANALP/AP///zk5OVJSUoSEhKWlpcDAwP//1v//xr3erZTOezGcEFKtSimce3NzezkxOQAAACH5BAEAAAUALAAAAAAMAA0AAARRcJBJyRilEMC5AcjQaB1wHMYkCFuXLKDQONsBLIuynEBAGAcJAnYy0AyGBOLENPg4qGUISTMdEIoEg4A6ohK6BND4YyqBqCdyve453vB44BEBADs=}

	image create photo $var(img,gray) -data {R0lGODdhPAA+APAAALi4uAAAACwAAAAAPAA+AAACQISPqcvtD6OctNqLs968+w+G4kiW5omm6sq27gvH8kzX9o3n+s73/g8MCofEovGITCqXzKbzCY1Kp9Sq9YrNFgsAO}

	frame $w.b -relief groove -borderwidth 2
	pack $w.b -side top -fill x

	button $w.b.back -image $var(img,back) -width 15 -height 13 \
	    -command "HVBackCmd $varname"
	button $w.b.forward -image $var(img,forward) -width 15 -height 13 \
	    -command "HVForwardCmd $varname"
	button $w.b.stop -image $var(img,stop) -width 15 -height 13 \
	    -command "HVStopCmd $varname"
	button $w.b.reload -image $var(img,reload) -width 15 -height 13 \
	    -command "HVReloadCmd $varname"
	pack $w.b.back $w.b.forward $w.b.stop $w.b.reload -side left

	frame $w.h
	grid rowconfigure $w.h 0 -weight 1
	grid columnconfigure $w.h 0 -weight 1
	pack $w.h -side top -fill both -expand true

	set var(widget) [html $w.h.html \
				-yscrollcommand "$w.h.yscroll set" \
				-xscrollcommand "$w.h.xscroll set" \
				-padx 5 \
				-pady 9 \
				-formcommand "HVFormCB $varname" \
				-imagecommand "HVImageCB $varname" \
				-scriptcommand "HVScriptCB $varname"\
				-appletcommand "HVAppletCB $varname" \
				-framecommand "HVFrameCB $varname" \
				-underlinehyperlinks 1 \
				-bg white \
				-width 640 \
				-height 512 \
				-fontcommand "HVFontCB $varname" \
				-tablerelief raised]

	$var(widget) token handler {NOSCRIPT} "HVNoScriptCB $varname"
	$var(widget) token handler {/NOSCRIPT} "HVNoScriptCB $varname"

	scrollbar $w.h.yscroll -orient vertical -command "$w.h.html yview"
	scrollbar $w.h.xscroll -orient horizontal -command "$w.h.html xview"

	grid $w.h.html $w.h.yscroll -sticky news
	grid $w.h.xscroll -stick news

	frame $w.s -relief groove -borderwidth 2
	pack $w.s -side bottom -fill x

	label $w.s.status -text "" -font {helvetica 10} -width 120 -anchor w
	pack $w.s.status -side left

	bind $var(widget).x <1> "HVLinkBind $varname %x %y"
	bind $var(widget).x <2> "HVLinkNewBind $varname %x %y"
	bind $var(widget).x <Motion> "HVMotionBind $varname %x %y" 

	# we have a problem with the html widget. first time thur, some
	# structures are not allocated/initialized. if we first display
	# a blank page, all seems ok
	$var(widget) clear
	$var(widget) parse "<html>\n<body>\n<form method=\"get\" action=\"foo\">\n</form>\n</body>\n</html>"

	global debug
	if {$debug(tcl,idletasks)} {
	    puts "HV"
	}
	update idletasks
    }

    if {$url != ""} {
	# no need to resolve
	HVLoadURL $varname $url {}
    }
}

# Commands

proc HVDestroyCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVDestroyCmd"
    }

    if {$var(active)} {
	if {$debug(tcl,hv)} {
	    puts "HVDestroyCmd aborted-- still active"
	}
	return
    }

    # this sometimes will not cancel pending transactions
    HVCancel $varname

    # clear any pending tmp files
    HVClearTmpFile $varname

    # clear the widge and all images
    HVClearWidget $varname

    # delete all button images
    image delete $var(img,forward)
    image delete $var(img,back)
    image delete $var(img,reload)
    image delete $var(img,stop)
    image delete $var(img,gray)

    # destroy the window and menubar
    destroy $var(top)
    destroy $var(mb)

    # stop any refresh
    if {$var(refresh,id)>0} {
	after cancel $var(refresh,id)
	set var(refresh,id) 0
    }

    unset $varname
}

proc HVURLDialogCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    set url "$var(url)"
    if {[EntryDialog "URL" "Enter World Wide Web Location (URL):" 80 url]} {
	if {[string length $url] == 0} {
	    return
	}

	ParseURL $url r
	switch -- $r(scheme) {
	    {} {
		# append 'http://' if needed
		if {[string range $r(path) 0 0] == "/"} {
		    set url "http:/$url"
		} else {
		    set url "http://$url"
		}
		
		if {$debug(tcl,hv)} {
		    puts "HVURLDialogCmd new $url"
		}
	    }
	}

	# clear the base
	$var(widget) config -base {}

	HVClearIndex $varname 0
	HVClearAll $varname
	# no need to resolve
	HVLoadURL $varname $url {}
    }
}

proc HVFileDialogCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    set fn [OpenFileDialog hvhtmlfbox]
    if {"$fn" != ""} {
	# clear the base
	$var(widget) config -base {}

	HVClearIndex $varname 0
	HVClearAll $varname
	# no need to resolve
	HVLoadURL $varname "$fn" {}
    }
}

proc HVClearCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    HVClearAll $varname
    set var(active) 0

    HVClearWidget $varname

    HVClearStatus $varname
    HVClearIndex $varname 0
    HVUpdateDialog $varname
}

proc HVBackCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVBackCmd index $var(index)"
    }

    incr ${varname}(index) -1
    if {[info exists ${varname}(index,$var(index))]} {
	set url [lindex $var(index,$var(index)) 0]
	set query [lindex $var(index,$var(index)) 1]
	if {$debug(tcl,hv)} {
	    puts "HVBackCmd $var(index) $url $query"
	}
	# clear the base
	$var(widget) config -base {}

	# HVGotoHTML will incr the index again
	incr ${varname}(index) -1
	# no need to resolve
	HVLoadURL $varname $url $query
    } else {
	incr ${varname}(index)
    }
}

proc HVForwardCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVForwardCmd $var(index)"
    }

    incr ${varname}(index)
    if {[info exists ${varname}(index,$var(index))]} {
	set url [lindex $var(index,$var(index)) 0]
	set query [lindex $var(index,$var(index)) 1]
	if {$debug(tcl,hv)} {
	    puts "HVForwardCmd $var(index) $url $query"
	}
	# clear the base
	$var(widget) config -base {}

	# HVGotoHTML will incr the index again
	incr ${varname}(index) -1
	# no need to resolve
	HVLoadURL $varname $url $query
    } else {
	incr ${varname}(index) -1
    }
}

proc HVReloadCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVReloadCmd"
    }

    # clear the base
    $var(widget) config -base {}

    # clear previous
    set var(previous) {}

    # HVGotoHTML will incr the index again
    incr ${varname}(index) -1
    # no need to resolve
    HVLoadURL $varname $var(url) $var(query)
}

proc HVStopCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "\n*** HVStopCmd ***\n"
    }

    set var(previous) {}
    HVClearStatus $varname
    HVCancel $varname
}

proc HVPageSourceCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVPageSourceCmd"
    }

    SimpleTextDialog ${varname}st $var(url) 80 20 insert top $var(html)
}

proc HVFontCmd {varname} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVFontCmd"
    }

    HVRefresh $varname
}

proc HVArchUserCmd {varname title url} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVArchUserCmd"
    }

    if {[string length $url] == 0} {
	return
    }

    ParseURL $url r
    switch -- $r(scheme) {
	{} {
	    # append 'http://' if needed
	    if {[string range $r(path) 0 0] == "/"} {
		set url "http:/$url"
	    } else {
		set url "http://$url"
	    }
	    
	    if {$debug(tcl,hv)} {
		puts "HVArchUserCmd new $url"
	    }
	}
    }
    HV $varname $title $url 2 0 {}
}

proc HVAnalysisCmd {varname title url} {
    upvar #0 $varname var
    global $varname

    global debug

    if {$debug(tcl,hv)} {
	puts "HVAnalysisCmd $varname $title $url"
    }

    if {[string length $url] == 0} {
	HV $varname "$title" {} 2 1 {}
    } else {
	ParseURL $url r
	switch -- $r(scheme) {
	    {} {
		# append 'http://' if needed
		if {[string range $r(path) 0 0] == "/"} {
		    set url "http:/$url"
		} else {
		    set url "http://$url"
		}
		
		if {$debug(tcl,hv)} {
		    puts "HVURLDialogCmd new $url"
		}
	    }
	}
	HV $varname "$title" $url 2 1 {} 
    }
}

proc ProcessWebCmd {varname iname} {
    global debug
    global hvweb

    set w hvweb

    upvar $varname var
    upvar $iname i

    set url [lindex $var $i]
    if {[string length $url] == 0} {
	HV $w Web {} 2 1 {}
    } else {
	ParseURL $url r
	switch -- $r(scheme) {
	    {} {
		# append 'http://' if needed
		if {[string range $r(path) 0 0] == "/"} {
		    set url "http:/$url"
		} else {
		    set url "http://$url"
		}
		
		if {$debug(tcl,hv)} {
		    puts "HVURLDialogCmd new $url"
		}
	    }
	}
	HV $w Web $url 2 1 {} 
    }
}

# Archive Servers
# Optical

proc HVArchMAST {} {
    global current

    set value {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set value "[lindex $coord 0], [lindex $coord 1]"
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 target \{$value\}"
    }

    global hvmast
    HV hvmast MAST http://stdatu.stsci.edu 2 1 $l
}

proc HVArchNSA {} {
    global current

    set ra {}
    set dec {}
    set size {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    set ra [lindex $coord 0]
	    set dec [lindex $coord 1]

	    set s [$current(frame) get fits size wcs arcmin]
	    set size [lindex $s 0]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 ra \{$ra\}"
	lappend l "1 dec \{$dec\}"
	lappend l "1 width \{$size\}"
    }

    global hvnsa
    HV hvnsa NSA http://archive.noao.edu/nsa/ 2 1 $l
}

proc HVArchSSS {} {
    global current

    set ra {}
    set dec {}
    set size {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec

	    set s [$current(frame) get fits size wcs arcmin]
	    set size [lindex $s 0]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 coords \{$ra $dec\}"
	lappend l "1 size \{$size\}"
    }

    global hvsss
    HV hvsss {SuperCOSMOS Sky Survey} \
	http://www-wfau.roe.ac.uk/sss/pixel.html 2 1 $l
}

# Infrad

proc HVArchIRAS {} {
    global current

    set value {}
    set size {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set value [$current(frame) get cursor wcs fk5 degrees]
	    set size [lindex [$current(frame) get fits size wcs arcsec] 0]
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "2 locstr \{$value\}"
	lappend l "2 subsz \{$size\}"
	lappend l "2 objstr \{$value\}"
	lappend l "2 size \{$size\}"
    }

    global hvmass
    HV hvmass {IRAS} http://irsa.ipac.caltech.edu/ 2 1 $l
}

# High Energy

proc HVArchChandraChaser {} {
    global current

    set coord {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set size \
		[expr [lindex [$current(frame) get fits size wcs arcmin] 0]/2.]
	}
    }

    set l {}
    if {[string length $coord] != 0} {
	lappend l "1 lon [lindex $coord 0]"
	lappend l "1 lat [lindex $coord 1]"
	lappend l "1 radius $size"
    }

    global hvchandrachaser
    HV hvchandrachaser {Chandra Chaser} http://cda.harvard.edu/chaser/mainEntry.do 2 1 $l
}

proc HVArchChandraPop {} {
    global current

    set coord {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set size \
		[expr [lindex [$current(frame) get fits size wcs arcmin] 0]/2.]
	}
    }

    set l {}
    if {[string length $coord] != 0} {
	lappend l "1 lon [lindex $coord 0]"
	lappend l "1 lat [lindex $coord 1]"
	lappend l "1 radius $size"
	lappend l "1 searchBy position"
    }

    global hvchandrapop
    HV hvchandrapop {Chandra Popular} http://cda.harvard.edu/pop/mainEntry.do 2 1 $l
}

proc HVArchChandraFTP {} {
    global current

    set ra {}
    set dec {}
    set wid {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    set ra [lindex $coord 0]
	    set dec [lindex $coord 1]

	    set wid [lindex [$current(frame) get fits size wcs degrees] 0]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 ra \{$ra\}"
	lappend l "1 dec \{$dec\}"
	lappend l "1 wid \{$wid\}"
    }

    global hvchandraftp
    HV hvchandraftp {Chandra FTP} \
	http://cfa-www.harvard.edu/archive/chandra/search 2 1 $l
}

proc HVArchRosat {} {
    global current

    set ra {}
    set dec {}
    set cprd {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    set ra [split [lindex $coord 0] :]
	    set dec [split [lindex $coord 1] :]
	    set raa "[lindex $ra 0]h[lindex $ra 1]m[lindex $ra 2]s"
	    set decc "[lindex $dec 0]d[lindex $dec 1]m[lindex $dec 2]s"
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 lon \{$raa\}"
	lappend l "1 lat \{$decc\}"
    }
    lappend l "1 cprd im1 \{photon image 0.1-2.4 keV (fits)\}"

    global hvrosat
    HV hvrosat {Rosat All-Sky} \
	http://www.xray.mpe.mpg.de/cgi-bin/rosat/rosat-survey 2 1 $l
}

proc HVArchSkyView {} {
    global current

    set value {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set value "[lindex $coord 0], [lindex $coord 1]"
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 VCOORD \{$value\}"
    }

    global hvskyview
    HV hvskyview SkyView http://skyview.gsfc.nasa.gov/cgi-bin/skvbasic.pl 2 1 $l
}

proc HVArchW3Browse {} {
    global current

    set value {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set value "[lindex $coord 0], [lindex $coord 1]"
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "3 Entry \{$value\}"
    }

    global hvw3browse
    HV hvw3browse W3Browse \
	http://heasarc.gsfc.nasa.gov/db-perl/W3Browse/w3browse.pl 2 1 $l
}

# Radio

proc HVArchNVSS {} {
    global current

    set ra {}
    set dec {}
    set sra 1
    set sdec 1

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec

	    set s [$current(frame) get fits size wcs degrees]
	    set sra [lindex $s 0]
	    set sdec [lindex $s 1]
	    if {$sra > 2} {
		set sra 2
	    }
	    if {$sdec > 2} {
		set sdec 2
	    }
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 RA \{$ra\}"
	lappend l "1 Dec \{$dec\}"
	lappend l "1 Size \{$sra $sdec\}"
    }
    lappend l "1 Type \{image/x-fits\} \{FITS Image\}"

    global hvnvss
    HV hvnvss NVSS http://www.cv.nrao.edu/nvss/postage.shtml 2 1 $l
}

proc HVArch4MASS {} {
    global current

    set ra {}
    set dec {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 RA \{$ra\}"
	lappend l "1 Dec \{$dec\}"
    }

    global hvmass4
    HV hvmass4 4MASS http://www.cv.nrao.edu/4mass/findFITS.shtml 2 1 $l
}

proc HVArchSIRTF {} {
    global current

    set ra {}
    set dec {}
    set sra 1
    set sdec 1

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec

	    set s [$current(frame) get fits size wcs arcmin]
	    set sra [lindex $s 0]
	    set sdec [lindex $s 1]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 RA \{$ra\}"
	lappend l "1 Dec \{$dec\}"
	lappend l "1 Size \{$sra $sdec\}"
    }
    lappend l "1 Type \{image/x-fits\} \{FITS Image\}"

    global hvsirtf
    HV hvsirtf SIRTF http://www.cv.nrao.edu/sirtf_fls/SFpostage.shtml 2 1 $l
}

proc HVArchFirst {} {
    global current

    set value {}
    set size {4.5}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set value [$current(frame) get cursor wcs fk5 sexagesimal]
	    set size [lindex [$current(frame) get fits size wcs arcmin] 0]
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 RA \{$value\}"
	lappend l "1 ImageSize \{$size\}"
	lappend l "1 ImageType \{FITS Image\}"
    }

    global hvfirst
    HV hvfirst First http://third.ucllnl.org/cgi-bin/firstcutout 2 1 $l
}

# Other

proc HVArchDataScope {} {
    global current

    set value {}
    set size {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 degrees]
	    set value "[lindex $coord 0], [lindex $coord 1]"
	    set size [lindex [$current(frame) get fits size wcs degrees] 0]
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 position \{$value\}"
	lappend l "1 delta \{$size\}"
    }

    global hvnvo
    HV hvnvo {NVO DataScope}  http://heasarc.gsfc.nasa.gov/vo 2 1 $l
}

proc HVArchNED {} {
    global current

    set ra {}
    set dec {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} [lindex $coord 0] { } ra
	    regsub -all {:} [lindex $coord 1] { } dec
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 lon \{$ra\}"
	lappend l "1 lat \{$dec\}"
    }

    global hvned
    HV hvned {NED} http://nedwww.ipac.caltech.edu/ 2 1 $l
}

proc HVArchSIMBADSAO {} {
    global hvsimbadsao
    HVArchSIMBAD hvsimbadsao {SIMBAD@SAO} http://simbad.harvard.edu/Simbad
}

proc HVArchSIMBADCDS {} {
    global hvsimbadcds
    HVArchSIMBAD hvsimbadcds {SIMBAD@CDS} http://simbad.u-strasbg.fr/Simbad
}

proc HVArchSIMBAD {varname title url} {
    upvar #0 $varname var
    global $varname

    global current

    set value {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set value [$current(frame) get cursor wcs fk5 sexagesimal]
	    regsub -all {:} $value { } value
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 Ident \{$value\}"
    }

    HV $varname $title $url 2 1 $l
}

proc HVArchADSSAO {} {
    global hvadssao
    HVArchADS hvadssao {ADS@SAO} http://adswww.harvard.edu/
}

proc HVArchADSCDS {} {
    global hvadscds
    HVArchADS hvadscds {ADS@CDS} http://cdsads.u-strasbg.fr/
}

proc HVArchADS {varname title url} {
    upvar #0 $varname var
    global $varname

    global current

    set value {}
    set size {}
    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set value [$current(frame) get cursor wcs fk5 sexagesimal]
	    set size [lindex [$current(frame) get fits size wcs arcmin] 0]
	    regsub -all {:} $value { } value
	    # limit size to 1 arcmin, otherwise you get too many responces
	    if {$size < .1} {
		append value " : $size"
	    } else {
		append value " : .1"
	    }
	}
    }

    set l {}
    if {[string length $value] != 0} {
	lappend l "1 object \{$value\}"
    }

    HV $varname $title $url 2 1 $l
}

proc HVArchSAOTDC {} {
    global current

    set ra {}
    set dec {}
    set rad {}

    if {$current(frame) != ""} {
	if {[$current(frame) has wcs equatorial wcs]} {
	    set coord [$current(frame) get cursor wcs fk5 sexagesimal]
	    set ra [lindex $coord 0]
	    set dec [lindex $coord 1]

	    set s [lindex [$current(frame) get fits size wcs arcsec] 0]
	    set rad [expr $s/2.]
	}
    }

    set l {}
    if {[string length $ra] != 0} {
	lappend l "1 ra \{$ra\}"
	lappend l "1 dec \{$dec\}"
	lappend l "1 rad \{$rad\}"
    }

    global hvsaotdc
    HV hvsaotdc {SAO Telescope Data Center} \
	http://tdc-www.harvard.edu/archive/ 2 1 $l
}

proc HVArchUserMenu {} {
    global hv
    global ds9
    global debug
    global menu

    # clear menu
    if {[$ds9(mb).analysis.arch index end] > [expr $menu(size,arch)-1]} {
	$ds9(mb).analysis.arch delete $menu(size,arch) end
    }

    foreach n {1 2 3 4} {
	if {[string length $hv(archive,url,$n)] > 0} {
	    $ds9(mb).analysis.arch add command -label "$hv(archive,menu,$n)" \
		-command "global hvuser$n; HVArchUserCmd hvuser$n \{$hv(archive,menu,$n)\} $hv(archive,url,$n)"
	}
    }
}

# Preferences

proc HVArchPref {n} {
    global hv
    global ds9
    global ed

    set w ".hvpref"

    set ed(ok) 0
    set ed(label) $hv(archive,menu,$n)
    set ed(url) $hv(archive,url,$n)

    DialogCreate $w "Archive Preference" -borderwidth 2
    frame $w.ans  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ans $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ans.title -text "Enter URL to added to Archive Menu:"

    label $w.ans.mtitle -text "Menu:"
    entry $w.ans.mvalue -textvariable ed(label) -width 30

    label $w.ans.utitle -text "URL:"
    entry $w.ans.uvalue -textvariable ed(url) -width 60

    grid rowconfigure $w.ans 0 -pad 4
    grid rowconfigure $w.ans 1 -pad 4
    grid rowconfigure $w.ans 2 -pad 4

    grid $w.ans.title - -padx 4 -sticky w
    grid $w.ans.mtitle $w.ans.mvalue -padx 4 -sticky w
    grid $w.ans.utitle $w.ans.uvalue -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.clear -text "Clear" \
	-command "set ed(label) {}; set ed(url) {}"
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok $w.buttons.clear $w.buttons.cancel \
	-side left -expand 1 -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ans.mvalue select range 0 end
    DialogWait $w ed(ok) $w.ans.mvalue
    DialogDismiss $w

    if {$ed(ok)} {
	set hv(archive,menu,$n) $ed(label)
	set hv(archive,url,$n) $ed(url)

	HVArchUserMenu
    }
    
    set r $ed(ok)
    unset ed
    return $r
}
