// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "framergbtrue16.h"
#include "fitsimage.h"
#include "util.h"

// Tk Canvas Widget Function Declarations

int FrameRGBTrueColor16CreateProc(Tcl_Interp*, Tk_Canvas, Tk_Item*, int, 
				  Tcl_Obj *const []);

// FrameRGBTrueColor16 Specs

static Tk_CustomOption tagsOption = {
  Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, NULL
};

static Tk_ConfigSpec frameRGBTrueColor16Specs[] = {

  {TK_CONFIG_STRING, "-command", NULL, NULL, "framergb",
   Tk_Offset(WidgetOptions, cmdName), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-x", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, x), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-y", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, y), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-width", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, width), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-height", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, height), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_ANCHOR, "-anchor", NULL, NULL, "nw",
   Tk_Offset(WidgetOptions, anchor), 0, NULL},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL, NULL,
   0, TK_CONFIG_NULL_OK, &tagsOption},

  {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0, NULL},
};

// Tk Static Structure

static Tk_ItemType frameRGBTrueColor16Type = {
  "framergbtruecolor16",           // name
  sizeof(WidgetOptions),        // item size
  FrameRGBTrueColor16CreateProc,   // configProc
  frameRGBTrueColor16Specs,        // configSpecs
  WidgetConfigProc,             // configProc
  WidgetCoordProc,              // coordProc
  WidgetDeleteProc,             // deleteProc
  WidgetDisplayProc,            // displayProc
  0,                            // alwaysRedraw
  WidgetPointProc,              // pointProc
  WidgetAreaProc,               // areaProc
  WidgetPostscriptProc,         // postscriptProc
  WidgetScaleProc,              // scaleProc
  WidgetTranslateProc,          // translateProc
  (Tk_ItemIndexProc*)NULL,      // indexProc
  WidgetICursorProc,            // icursorProc
  (Tk_ItemSelectionProc*)NULL,  // selectionProc
  (Tk_ItemInsertProc*)NULL,     // insertProc
  (Tk_ItemDCharsProc*)NULL,     // dCharsProc
  (Tk_ItemType*)NULL            // nextPtr
};

// Non-Member Functions

int FrameRGBTrueColor16_Init(Tcl_Interp* interp)
{
  Tk_CreateItemType(&frameRGBTrueColor16Type);
  return TCL_OK;
}

int FrameRGBTrueColor16CreateProc(Tcl_Interp* interp, Tk_Canvas canvas, 
				  Tk_Item* item, int argc, 
				  Tcl_Obj *const argv[])
{
  FrameRGBTrueColor16* frame = new FrameRGBTrueColor16(interp, canvas, item);

  // and set default configuration

  if (frame->configure(argc, (const char**)argv, 0) != TCL_OK) {
    delete frame;
    Tcl_AppendResult(interp, " error occured while creating frame.", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

// FrameRGBTrueColor16 Member Functions

FrameRGBTrueColor16::FrameRGBTrueColor16(Tcl_Interp* i, Tk_Canvas c, 
					 Tk_Item* item)
  : FrameRGBTrueColor(i, c, item)
{
  configSpecs = frameRGBTrueColor16Specs;  // frame configure options
}

FrameRGBTrueColor16::~FrameRGBTrueColor16()
{
  // we must do this at this level, because updateColorScale is called
  unloadAllFits();
}

void FrameRGBTrueColor16::encodeTrueColor(XColor* src, char* dest)
{
  if (!baseXImage)
    return;

  int msb = baseXImage->byte_order;
  int rs,gs,bs;
#ifndef _WIN32
  unsigned short rm = decodeMask((unsigned short)visual->red_mask, &rs);
  unsigned short gm = decodeMask((unsigned short)visual->green_mask, &gs);
  unsigned short bm = decodeMask((unsigned short)visual->blue_mask, &bs);
#else
  unsigned short rm = decodeMask((unsigned short)0x7C00, &rs);
  unsigned short gm = decodeMask((unsigned short)0x03E0, &gs);
  unsigned short bm = decodeMask((unsigned short)0x001F, &bs);
#endif

  // we need to check to byteswap when we have cross platforms

  unsigned short r = (unsigned char)src->red;
  unsigned short g = (unsigned char)src->green;
  unsigned short b = (unsigned char)src->blue;
  unsigned short a = 0;
  a |= rs>0 ? ((r & rm) << rs) : ((r & rm) >> -rs);
  a |= gs>0 ? ((g & gm) << gs) : ((g & gm) >> -gs);
  a |= bs>0 ? ((b & bm) << bs) : ((b & bm) >> -bs);

  if ((!msb && lsb()) || (msb && !lsb()))
    memcpy(dest, &a, 2);
  else {
    unsigned char* rr = (unsigned char*)(&a);
    *(dest) = *(rr+1);
    *(dest+1) = *(rr);
  }
}

void FrameRGBTrueColor16::buildXImage(XImage* xmap,
				      int x0, int y0, int x1, int y1,
				      double* (FitsImage::*getMatrix)())
{
  // we need a colorScale before we can render
  if (!validColorScale())
    return;

  // create img
  unsigned char* img = fillRGBImage(xmap->width, xmap->height, 
				    x0, y0, x1, y1, getMatrix);

  fillXImage16(img, xmap);

  // clean up
  if (img)
    delete [] img;
}

void FrameRGBTrueColor16::buildColormapXM(const unsigned char* img)
{
  int& height = colormapXM->height;
  memset(colormapXM->data, 255, colormapXM->bytes_per_line * height);

  fillXImage16(img, colormapXM);
}

void FrameRGBTrueColor16::fillXImage16(const unsigned char* img, XImage* xmap)
{
  int& width = xmap->width;
  int& height = xmap->height;

  int rs,gs,bs;
#ifndef _WIN32
  unsigned short rm = decodeMask((unsigned short)visual->red_mask, &rs);
  unsigned short gm = decodeMask((unsigned short)visual->green_mask, &gs);
  unsigned short bm = decodeMask((unsigned short)visual->blue_mask, &bs);
#else
  unsigned short rm = decodeMask((unsigned short)0x7C00, &rs);
  unsigned short gm = decodeMask((unsigned short)0x03E0, &gs);
  unsigned short bm = decodeMask((unsigned short)0x001F, &bs);
#endif

  // we need to check to byteswap when we have cross platforms
  const unsigned char* ptr = img;
  int msb = xmap->byte_order;

  if ((!msb && lsb()) || (msb && !lsb())) {
    for (int j=0; j<height; j++) {
      // the line may be padded at the end
      char* dest = xmap->data + j*xmap->bytes_per_line;

      for (int i=0; i<width; i++, dest+=2, ptr+=3) {
	unsigned short r = ptr[0];
	unsigned short g = ptr[1];
	unsigned short b = ptr[2];
	unsigned short a = 0;
	a |= rs>0 ? ((r & rm) << rs) : ((r & rm) >> -rs);
	a |= gs>0 ? ((g & gm) << gs) : ((g & gm) >> -gs);
	a |= bs>0 ? ((b & bm) << bs) : ((b & bm) >> -bs);

	memcpy(dest, &a, 2);
      }
    }
  }
  else {
    for (int j=0; j<height; j++) {
      // the line may be padded at the end
      char* dest = xmap->data + j*xmap->bytes_per_line;

      for (int i=0; i<width; i++, dest+=2, ptr+=3) {
	unsigned short r = ptr[0];
	unsigned short g = ptr[1];
	unsigned short b = ptr[2];
	unsigned short a = 0;
	a |= rs>0 ? ((r & rm) << rs) : ((r & rm) >> -rs);
	a |= gs>0 ? ((g & gm) << gs) : ((g & gm) >> -gs);
	a |= bs>0 ? ((b & bm) << bs) : ((b & bm) >> -bs);

	unsigned char* rr = (unsigned char*)(&a);
	*(dest) = *(rr+1);
	*(dest+1) = *(rr);
      }
    }
  }
}




