// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "projection.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Projection::Projection(const Projection& a) : BaseLine(a)
{
  width = a.width;
  method = a.method;

  system = a.system;
  skyframe = a.skyframe;
}

Projection::Projection(FrameBase* p, const Vector& ptr1, const Vector& ptr2, 
		       double wd,
		       const char* mvcb, const char* delcb,
		       const char* clr, int wth, 
		       const char* fnt, const char* txt, 
		       unsigned short prop, const char* cmt,
		       const List<Tag>& tg, const List<CallBack>& cb) 
  : BaseLine(p, ptr1, ptr2, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  width = wd;
  method = AVERAGE;

  system = PHYSICAL;
  skyframe = FK5;

  strcpy(type,"projection");
  handle = new Vector[3];
  numHandle = 3;

  if (mvcb && *mvcb) {
    addCallBack(CallBack::MOVECB, mvcb, parent->options->cmdName);
    addCallBack(CallBack::EDITCB, mvcb, parent->options->cmdName);
    addCallBack(CallBack::UPDATECB, mvcb, parent->options->cmdName);
  }
  if (delcb && *delcb)
    addCallBack(CallBack::DELETECB, delcb, parent->options->cmdName);

  updateBBox();
}

void Projection::updateBBox()
{
  center = (p2-p1)/2 + p1;
  angle = (p2-p1).angle();
  if (parent->isIIS())
    angle = -angle;

  // bound marker

  Matrix imm = bckCanvasMatrix();
  Matrix mm = fwdCanvasMatrix();

  Vector a = p1*imm;
  Vector b = p2*imm;
  Vector c = Vector(0,-width);

  Vector ll = a*mm;
  Vector lr = b*mm;
  Vector ul = (a+c)*mm;
  Vector ur = (b+c)*mm;
  Vector h = (((b-a)/2+a)+c)*mm;

  // bbox in canvas coords

  bbox = BBox(ll,lr);
  bbox.bound(ul);
  bbox.bound(ur);

  // generate handles in canvas coords

  handle[0] = ll;
  handle[1] = lr;
  handle[2] = h;

  // make room for handles

  bbox.expand(3);

  // calculate overall bbox

  calcAllBBox();
}

void Projection::edit(const Vector& v, int h)
{
  Matrix mm = bckCanvasMatrix();
  switch (h) {
  case 1:
    p1 = v;
    break;
  case 2:
    p2 = v;
    break;
  case 3:
    width = -(v * mm)[1];
    if (width<0)
      width = 0;
    break;
  }

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Projection::ps(int mode)
{
  Marker::ps(mode);

  {
    Vector a = p1 * parent->refToCanvas;
    Vector b = p2 * parent->refToCanvas;

    ostringstream str;
    str << "newpath " 
	<< a.TkCanvasPs(parent->canvas) << "moveto"
	<< b.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  if (width>0) {
    psLineDash();

    Matrix imm = bckCanvasMatrix();
    Matrix mm = fwdCanvasMatrix();

    Vector a = p1*imm;
    Vector b = p2*imm;
    Vector c = Vector(0,-width);

    Vector ll = a*mm;
    Vector lr = b*mm;
    Vector ul = (a+c)*mm;
    Vector ur = (b+c)*mm;

    ostringstream str;
    str << "newpath " 
	<< lr.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< ul.TkCanvasPs(parent->canvas) << "lineto"
	<< ll.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}

int Projection::isIn(const Vector& v)
{
  Vector zz = parent->zoom();
  if (width * zz.length() > 3) {
    Matrix mm = bckRefMatrix();
    Matrix imm = bckCanvasMatrix();

    Vector a = p1*imm;
    Vector b = p2*imm;

    Vector vv = -(v * mm);
    return (vv[0]>a[0] && vv[0]<b[0] && vv[1]>0 && vv[1]<width);
  }
  else {
    Vector l1 = p1 * parent->refToCanvas;
    Vector l2 = p2 * parent->refToCanvas;
    double a = (l2-l1).angle();
    Matrix m = Translate(-l1) * Rotate(a);

    Vector vv = v * m;
    Vector end = l2 * m;
    return (vv[0]>0 && vv[0]<end[0] && vv[1]>-6 && vv[1]<6);
  }
}

void Projection::set(const Vector& v1, const Vector& v2, double wd, int m)
{
  p1 = v1;
  p2 = v2;
  width = wd;
  method = (Method)m;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Projection::setWidth(double wd)
{
  width = wd;
  if (width<0)
    width = 0;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

// Private

void Projection::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  {
    switch (mode) {
    case SRC:
      XSetForeground(display, gc, color);
      XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
      setLineNoDash();
      break;
    case XOR:
      XSetForeground(display, gc, parent->getWhiteColor());
      XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
      setLineDash();
      break;
    }

    Vector a = (p1 * mx).round();
    Vector b = (p2 * mx).round();
    XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], 
	      (int)b[0], (int)b[1]);
  }

  if (width>0) {
    switch (mode) {
    case SRC:
      XSetForeground(display, gc, color);
      XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
      setLineDash();
      break;
    case XOR:
      XSetForeground(display, gc, parent->getWhiteColor());
      XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
      setLineDash();
      break;
    }

    Matrix imm = bckCanvasMatrix();
    Matrix mm = fwdRefMatrix() * mx;

    Vector a = p1*imm;
    Vector b = p2*imm;
    Vector c = Vector(0,-width);

    Vector ll = (a*mm).round();
    Vector lr = (b*mm).round();
    Vector ul = ((a+c)*mm).round();
    Vector ur = ((b+c)*mm).round();

    XDRAWLINE(display, drawable, gc, (int)lr[0], (int)lr[1], 
	      (int)ur[0], (int)ur[1]);
    XDRAWLINE(display, drawable, gc, (int)ur[0], (int)ur[1], 
	      (int)ul[0], (int)ul[1]);
    XDRAWLINE(display, drawable, gc, (int)ul[0], (int)ul[1], 
	      (int)ll[0], (int)ll[1]);
  }
}

// list

void Projection::list(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, int conj, int strip)
{
  if (!strip) {
    FitsImage* ptr = parent->findFits(center);
    listPre(str, sys, sky, ptr, strip, 1);

    switch (sys) {
    case IMAGE:
    case PHYSICAL:
    case DETECTOR:
    case AMPLIFIER:
      {
	Vector v1 = ptr->mapFromRef(p1,sys);
	Vector v2 = ptr->mapFromRef(p2,sys);
	str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	    << v2[0] << ',' << v2[1] << ',' 
	    << ptr->mapLenFromRef(width,sys) << ')';
      }
      break;
    default:
      if (ptr->hasWCS(sys)) {
	if (ptr->hasWCSEqu(sys)) {
	  switch (format) {
	  case DEGREES:
	    {
	      Vector v1 = ptr->mapFromRef(p1,sys,sky);
	      Vector v2 = ptr->mapFromRef(p2,sys,sky);
	      str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
		  << ',' << v2[0] << ',' << v2[1] << ',' 
		  << ptr->mapLenFromRef(width,sys,ARCSEC) << '"' << ')';
	    }
	    break;
	  case SEXAGESIMAL:
	    {
	      char buf[64];
	      char ra1[16], ra2[16];
	      char dec1[16], dec2[16];
	      {
		ptr->mapFromRef(p1,sys,sky,format,buf,64);
		string x(buf);
		istringstream wcs(x);
		wcs >> ra1 >> dec1;
	      }
	      {
		ptr->mapFromRef(p2,sys,sky,format,buf,64);
		string x(buf);
		istringstream wcs(x);
		wcs >> ra2 >> dec2;
	      }
	      str << type << '(' << ra1 << ',' << dec1 << ',' 
		  << ra2 << ',' << dec2 << ','
		  << ptr->mapLenFromRef(width,sys,ARCSEC) << '"' << ')';
	    }
	    break;
	  }
	}
	else {
	  Vector v1 = ptr->mapFromRef(p1,sys);
	  Vector v2 = ptr->mapFromRef(p2,sys);
	  str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
	      << ',' << v2[0] << ',' << v2[1] << ',' 
	      << ptr->mapLenFromRef(width,sys) << ')';
	}
      }
    }

    listPost(str, conj, strip);
  }
}

void Projection::listPost(ostream& str, int conj, int strip)
{
  if (conj)
    str << " ||";

  listProperties(str, 0);
}


