;;;; SBCL-specific parts of the condition system, i.e. parts which
;;;; don't duplicate/clobber functionality already provided by the
;;;; cross-compilation host Common Lisp

;;;; This software is part of the SBCL system. See the README file for
;;;; more information.
;;;;
;;;; This software is derived from the CMU CL system, which was
;;;; written at Carnegie Mellon University and released into the
;;;; public domain. The software is in the public domain and is
;;;; provided with absolutely no warranty. See the COPYING and CREDITS
;;;; files for more information.

(in-package "SB!KERNEL")

;;; not sure this is the right place, but where else?
(defun style-warn (format-control &rest format-arguments)
  (/show0 "entering STYLE-WARN")
  (/show format-control format-arguments)
  (with-sane-io-syntax
      (warn 'simple-style-warning
            :format-control format-control
            :format-arguments format-arguments)))

;;; a utility for SIGNAL, ERROR, CERROR, WARN, COMPILER-NOTIFY and
;;; INVOKE-DEBUGGER: Parse the hairy argument conventions into a
;;; single argument that's directly usable by all the other routines.
(defun coerce-to-condition (datum arguments default-type fun-name)
  (cond ((typep datum 'condition)
         (when (and arguments (not (eq fun-name 'cerror)))
           (cerror "Ignore the additional arguments."
                   'simple-type-error
                   :datum arguments
                   :expected-type 'null
                   :format-control "You may not supply additional arguments ~
                                    when giving ~S to ~S."
                   :format-arguments (list datum fun-name)))
         datum)
        ((symbolp datum) ; roughly, (SUBTYPEP DATUM 'CONDITION)
         (apply #'make-condition datum arguments))
        ((or (stringp datum) (functionp datum))
         (make-condition default-type
                         :format-control datum
                         :format-arguments arguments))
        (t
         (error 'simple-type-error
                :datum datum
                :expected-type '(or symbol string)
                :format-control "bad argument to ~S: ~S"
                :format-arguments (list fun-name datum)))))

(define-condition layout-invalid (type-error)
  ()
  (:report
   (lambda (condition stream)
     (format stream
             "~@<invalid structure layout: ~
              ~2I~_A test for class ~4I~_~S ~
              ~2I~_was passed the obsolete instance ~4I~_~S~:>"
             (classoid-proper-name (type-error-expected-type condition))
             (type-error-datum condition)))))

(define-condition case-failure (type-error)
  ((name :reader case-failure-name :initarg :name)
   (possibilities :reader case-failure-possibilities :initarg :possibilities))
  (:report
    (lambda (condition stream)
      (format stream "~@<~S fell through ~S expression. ~
                      ~:_Wanted one of ~:S.~:>"
              (type-error-datum condition)
              (case-failure-name condition)
              (case-failure-possibilities condition)))))

(define-condition compiled-program-error (program-error)
  ((message :initarg :message :reader program-error-message)
   (source :initarg :source :reader program-error-source))
  (:report (lambda (condition stream)
             (format stream "Execution of a form compiled with errors.~%~
                             Form:~%  ~A~%~
                             Compile-time error:~%  ~A"
                       (program-error-source condition)
                       (program-error-message condition)))))

(define-condition interpreted-program-error
    (program-error encapsulated-condition)
  ;; Unlike COMPILED-PROGRAM-ERROR, we don't need to dump these, so
  ;; storing the original condition and form is OK.
  ((form :initarg :form :reader program-error-form))
  (:report (lambda (condition stream)
             (format stream "~&Evaluation of~%  ~S~%~
                             caused error:~%  ~A~%"
                     (program-error-form condition)
                     (encapsulated-condition condition)))))

(define-condition simple-control-error (simple-condition control-error) ())
(define-condition simple-file-error    (simple-condition file-error)    ())
(define-condition simple-program-error (simple-condition program-error) ())
(define-condition simple-stream-error  (simple-condition stream-error)  ())
(define-condition simple-parse-error   (simple-condition parse-error)   ())

(define-condition character-coding-error (error) ())
(define-condition character-encoding-error (character-coding-error)
  ((code :initarg :code :reader character-encoding-error-code)))
(define-condition character-decoding-error (character-coding-error)
  ((octets :initarg :octets :reader character-decoding-error-octets)))
(define-condition stream-encoding-error (stream-error character-encoding-error)
  ()
  (:report
   (lambda (c s)
     (let ((stream (stream-error-stream c))
           (code (character-encoding-error-code c)))
       (format s "~@<encoding error on stream ~S (~S ~S): ~2I~_~
                  the character with code ~D cannot be encoded.~@:>"
               stream ':external-format (stream-external-format stream)
               code)))))
(define-condition stream-decoding-error (stream-error character-decoding-error)
  ()
  (:report
   (lambda (c s)
     (let ((stream (stream-error-stream c))
           (octets (character-decoding-error-octets c)))
       (format s "~@<decoding error on stream ~S (~S ~S): ~2I~_~
                  the octet sequence ~S cannot be decoded.~@:>"
               stream ':external-format (stream-external-format stream)
               octets)))))

(define-condition c-string-encoding-error (character-encoding-error)
  ((external-format :initarg :external-format :reader c-string-encoding-error-external-format))
  (:report
   (lambda (c s)
     (format s "~@<c-string encoding error (:external-format ~S): ~2I~_~
                  the character with code ~D cannot be encoded.~@:>"
               (c-string-encoding-error-external-format c)
               (character-encoding-error-code c)))))

(define-condition c-string-decoding-error (character-decoding-error)
  ((external-format :initarg :external-format :reader c-string-decoding-error-external-format))
  (:report
   (lambda (c s)
     (format s "~@<c-string decoding error (:external-format ~S): ~2I~_~
                  the octet sequence ~S cannot be decoded.~@:>"
             (c-string-decoding-error-external-format c)
             (character-decoding-error-octets c)))))

(define-condition control-stack-exhausted (storage-condition)
  ()
  (:report
    (lambda (condition stream)
      (declare (ignore condition))
      (format stream
             "Control stack exhausted (no more space for function call frames).  This is probably due to heavily nested or infinitely recursive function calls, or a tail call that SBCL cannot or has not optimized away."))))

(define-condition heap-exhausted-error (storage-condition)
  ()
  (:report
   (lambda (condition stream)
     (declare (special *heap-exhausted-error-available-bytes*
                       *heap-exhausted-error-requested-bytes*))
     ;; See comments in interr.lisp -- there is a method to this madness.
     (if (and (boundp '*heap-exhausted-error-available-bytes*)
              (boundp '*heap-exhausted-error-requested-bytes*))
         (format stream
                 "Heap exhausted: ~D bytes available, ~D requested. PROCEED WITH CAUTION!"
                 *heap-exhausted-error-available-bytes*
                 *heap-exhausted-error-requested-bytes*)
         (print-unreadable-object (condition stream))))))

(define-condition system-condition (condition)
  ((address :initarg :address :reader system-condition-address :initform nil)
   (context :initarg :context :reader system-condition-context :initform nil)))

(define-condition memory-fault-error (system-condition error) ()
  (:report
   (lambda (condition stream)
     (format stream "Unhandled memory fault at #x~X."
             (system-condition-address condition)))))

(define-condition breakpoint-error (system-condition error) ()
  (:report
   (lambda (condition stream)
     (format stream "Unhandled breakpoint/trap at #x~X."
             (system-condition-address condition)))))

(define-condition interactive-interrupt (system-condition serious-condition) ()
  (:report
   (lambda (condition stream)
     (format stream "Interactive interrupt at #x~X."
             (system-condition-address condition)))))
