/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef SRKMAINVIEW_H
#define SRKMAINVIEW_H

/**
 * This view handles the actual opening of files and processes calls to the 
 * appropriate objects. It acts as an interface/proxy that delegates user input,
 * coming from the main window.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>

#include <qwidget.h>
#include <qfile.h>

#include <libinstrudeo/isdcommentbox.h>
#include <libinstrudeo/isdexporter.h>
#include <libinstrudeo/isdprogresscallback.h>

#include "srkerror.h"

using namespace std;

#define WEBSERVICE_MAXIMAL_RESOLUTION_W 800
#define WEBSERVICE_MAXIMAL_RESOLUTION_H 600

class KProgressDialog;
class KProcess;
class SRKVideoPanel;
class QFile;
class KTempFile;
class ISDRecording;
class QCString;

class SRKMainView : public QWidget
{
    Q_OBJECT

 public:
    //-----CONSTRUCTORS-----
    SRKMainView(QWidget* parent = 0, const char* name = 0);
    
    virtual ~SRKMainView();

    //-----METHODS-----
    /**
     * Fires up a new VNC client window that records to a temp file.
     * That file is then processed and imported to a new recording.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode newRecording();

    /**
     * Loads a recording-file from disk and extracts initial data.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode loadFile(QString& fileName);

    /**
     * Imports the specified file into a new Instrudeo recording
     * and opens that recording.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode importFile(QString& fileName);

    /**
     * Saves the current recording to the specified file.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode saveRecording(QString& fileName);
    
    /**
     * Fires up the meta info dialog and saves the new meta info if 
     * necessary.
     * 
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode editMetaInfo();

    /**
     * Fires up the transmit recording dialog then transmits the recording
     * to the Instrudeo webservice.
     *
     * @param fileName The filename of the recording on disk.
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode transmitRecording(QString& fileName);

    /**
     * This is called from the transmitrecordingdialog and performs a client-side
     * check on the recording before it is actually transmitted.
     * For example, the meta-information must be filled in.
     *
     * @return Returns true if all went well.
     */
    bool applyTransmitFilter();

    /**
     * Export the current recording to the file.
     * The extension of the file is used to specify the format of the export.
     *
     * @param currentFileName The current recordings file name.
     * @param filename The file name to export to.
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode exportRecording(QString& currentFileName, QString& fileName);
    
    /**
     * Closes the current recording.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode closeRecording();

    /**
     * Adds a commentbox to the recording at this position.
     *
     * @param type The type of the commentbox.
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode addComment(ISDCommentbox::commentboxType type);
    
    /**
     * Deletes the currently active commentbox. If none is active, nothing happens.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode deleteActiveComment();

    /**
     * Edit the properties of the currently active commentbox. If none is active, nothing happens.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode editActiveCommentProperties();

    /**
     * End the currently active commentbox on the current position. If none is active, nothing happens.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode endActiveCommentHere();
    
    /**
     * Starts a new VNC server with resolution (xRes x yRes) at display disp.
     * 
     * @param xRes The X resolution of the VNC server.
     * @param xRes The Y resolution of the VNC server.
     * @param disp The display-number of the VNC server.
     * @param killOnExit If set to true, the server is killed in the destructor of this class.
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode startVncServer(int xRes, int yRes, int disp, bool killOnExit);

    /**
     * Kills the VNC server on the specified display.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode stopVncServer(int disp);

    /**
     * Change the zoom factor of the displayed video.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode changeZoomFactor(int);

    /**
     * Change the language in which the comments are made.
     *
     * @param lang The new language (the code).
     */
    SRKError::SRKErrorCode changeCommentsLanguage(QString& lang);

    /**
     * Checks if the recording is dirty (modified after last save, load, etc.)
     *
     * @return Returns true if dirty, false otherwise.
     */
    bool recIsDirty();

 signals:
    void fireFileOpened(bool state);
    void firePopupMenu(QString& name, QPoint& pos);
    void fireAddActiveLanguage(QString langCode);
    void fireClearActiveLanguages();
    void fireSyncActiveLanguage();

 public slots:
     /**
      * Use this function to aggregate back a popup-call to the main window.
      *
      * @param name The name of the menu in the XMLUI file.
      * @param pos The global position of the mouseclick.
      */
     void popupMenu(QString& name, QPoint& pos);
   
     /**
      * Callback function for the other widgets to signal a dirty-action.
      */
     void markDirty();

 protected slots:
    /**
     * This slot is called when the window of the vncViewer process is closed.
     * It handles the various exit states of the viewer and if nothing went wrong,
     * the recorded session is imported in a new recording using the instrudeo library.
     */
    void recordingFinished(KProcess* proc);

    /**
     * This slot is called when the recording-exporting is done.
     */
    void exportFinished(KProcess* proc);

    /*
     * This slot is called when the export process writes something to stdout.
     */
    void updateExportDialog(KProcess*, char*, int);
 
    /**
     * Checks the password files or asks it and launches the viewer.
     * Note: we use this as slot for the processExited signal of a process,
     * that's why we need the argument.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode startVncViewer(KProcess* process = NULL);

 protected:
    //----CHILD CLASS-----
    /**
     * This class is used to monitor the progress of the import routines.
     * It subclasses ISDProgressCallback, and implements
     * its procentDone() method, that is called regularly to report the procent
     * of imported data.
     */
    class SRKImportProgressCallback : public ISDProgressCallback {
    public:
	SRKImportProgressCallback(KProgressDialog* progressDialog);
	virtual ~SRKImportProgressCallback();
	virtual void procentDone(float procent);
	virtual bool wasCancelled();
    protected:
	KProgressDialog* progressDialog;
    };

    //-----METHODS-----
    /**
     * Checks if a new VNC server must be started by reading the config file.
     * Calls startVncServer() to start one if necessary.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode checkStartVncServer();
    /**
     * Checks if the VNC server, started at startup, must be killed.
     *
     * @return Returns a code that indicates success or failure.
     */
    SRKError::SRKErrorCode checkStopVncServer();

    //-----VARIABLES-----
    KProcess* vncServer;
    SRKVideoPanel* videoPanel;
    KProcess* vncViewer;
    int keepPass;
    QFile vncPassFile;
    KTempFile* vncTempFile;
    ISDRecording* recording;
    bool dirty;
    QString currentCommentsLanguage;
    KProcess* exportProc;
    KProgressDialog* exportDialog;
    bool fileOpened;
};

#endif
