;;; sense-region.el - minor mode to toggle region and rectangle.
;;; $Id: sense-region.el,v 1.8 2002/05/23 20:27:15 komatsu Exp $
;;;
;;; AUTHOR:  Hiroyuki KOMATSU <komatsu@taiyaki.org>
;;; LICENSE: GPL2
;;; ORIGINAL-SOURCE: http://www.taiyaki.org/elisp/sense-region/ (in Japanese)
;;;

;;; ------------------------------------------------------------
;;; mell (basic)
;;; ------------------------------------------------------------

;;; Checking Emacs or XEmacs.
(if (not (boundp 'running-xemacs))
    (defconst running-xemacs nil))

(defun mell-check-value (value)
  (and (boundp value)
       (symbol-value value)))

(defun mell-require (feature &optional filename noerror)
  (or (featurep feature)
      (if noerror
	  (condition-case nil
	      (require feature filename)
	    (file-error nil)
	    )
	(require feature filename)
	)))

(defun mell-column-at-point (point &optional buffer)
  (save-excursion
    (and buffer (set-buffer buffer))
    (goto-char point)
    (current-column)
    ))

(defun mell-point-at-column (column &optional point buffer)
  (save-excursion
    (and buffer (set-buffer buffer))
    (and point (goto-char point))
    (move-to-column column)
    (point)
    ))

;;; add-local-hook
(or (fboundp 'add-local-hook)
    (defun add-local-hook (hook function &optional append)
      (make-local-hook hook)
      (add-hook hook function append t))
    )

;;; remove-local-hook
(or (fboundp 'remove-local-hook)
    (defun remove-local-hook (hook function)
      (if (local-variable-p hook (current-buffer))
	  (remove-hook hook function t)))
    )

;; mell-marker
(defun mell-marker-make (&optional position buffer type)
  (let ((marker (make-marker)))
    (or position
	(setq position (point)))
    (set-marker marker position buffer)
    (set-marker-insertion-type marker type)
    marker
    ))

(defun mell-marker-set (marker &optional position buffer type)
  (or (markerp (eval marker))
      (set marker (make-marker)))
  (or position
      (setq position (point)))
  (set-marker (eval marker) position buffer)
  (set-marker-insertion-type (eval marker) type)
  (eval marker)
  )

;;; ------------------------------------------------------------
;;; mell-mode
;;; ------------------------------------------------------------

;;; This function requires mell-alist.
(defun mell-set-minor-mode (name modeline &optional key-map)
  (make-variable-buffer-local name)
  (setq minor-mode-alist
	(mell-alist-add minor-mode-alist (list name modeline)))
  (and key-map
       (setq minor-mode-map-alist
	     (mell-alist-add minor-mode-map-alist (cons name key-map)))
       )
  )

;; ------------------------------------------------------------
;; mell-alist
;; ------------------------------------------------------------
(defun mell-alist-add! (alist new-cons)
  (if (null alist)
      (error "mell-alist-add! can not deal nil as an alist.")
    (let ((current-cons (assoc (car new-cons) alist)))
      (if current-cons
	  (setcdr current-cons (cdr new-cons))
	(if (car alist)
	    (nconc alist (list new-cons))
	  (setcar alist new-cons))
	)
      alist)))
  
(defun mell-alist-add (alist new-cons)
  (if (null alist)
      (list new-cons)
    (let ((return-alist (copy-alist alist)))
      (mell-alist-add! return-alist new-cons)
      return-alist)))
  
(defun mell-alist-delete (alist key)
  (if key
      (let (return-alist)
	(mapcar '(lambda (x)
		   (or (equal key (car x))
		       (setq return-alist (cons x return-alist))))
		alist)
	(if return-alist
	    (reverse return-alist)
	  (list nil)))
    alist)
  )

;;; ------------------------------------------------------------
;;; mell-sign
;;; ------------------------------------------------------------
(mell-require 'overlay nil t)

(defun mell-sign-region-highlight (start end &optional buffer face)
  (save-excursion
    (or buffer (setq buffer (current-buffer)))
    (prog1
	(setq overlay (make-overlay start end buffer nil t))
      (overlay-put overlay 'face (or face 'highlight))
      (overlay-put overlay 'evaporate t)
      )))

(defun mell-sign-region-highlight-off (overlay)
  (delete-overlay overlay)
  )

(defun mell-sign-rectangle-highlight (start end &optional buffer face)
  (save-excursion
    (or buffer (setq buffer (current-buffer)))
    (mapcar
     '(lambda (region)
	(prog1
	    (setq overlay
		  (make-overlay (car region) (cdr region) buffer nil t))
	  (overlay-put overlay 'face (or face 'highlight))
	  (overlay-put overlay 'evaporate t)
	  ))
     (mell-region-get-visible-rectangle-list start end))
;     (mell-region-get-rectangle-list start end))
    ))

(defun mell-sign-rectangle-highlight-off (overlay-list)
  (mapcar
   '(lambda (overlay)
      (delete-overlay overlay))
   overlay-list)
  )
  

(defun mell-sign-reset-face (face)
  (if running-xemacs
      (reset-face face)
    (set-face-foreground face nil)
    (set-face-background face nil)
    (set-face-background-pixmap face nil)
    (set-face-underline-p face nil)
    (set-face-stipple face nil)
    ))

;;; ------------------------------------------------------------
;;; mell-region
;;; ------------------------------------------------------------

;; mell-region-face
(if running-xemacs
    (defconst mell-region-face 'zmacs-region)
  (defconst mell-region-face 'region)
  )

;; mell-region-active-p
(if running-xemacs
    (defun mell-region-active-p ()
      (region-active-p))
  (defun mell-region-active-p ()
    (mell-check-value 'mark-active))
  )

;; mell-transient-mode-p
(if running-xemacs
    (defun mell-transient-mode-p ()
      (mell-check-value 'zmacs-regions))
  (defun mell-transient-mode-p ()
    (mell-check-value 'transient-mark-mode))
  )

;; Define mell-transient-region-active-p
(defun mell-transient-region-active-p ()
  (and (mell-transient-mode-p)
       (mell-region-active-p)))

(defun mell-transient-region-stay ()
  (and running-xemacs
       (setq zmacs-region-stays t))
  )

(defun mell-transient-region-deactivate ()
  (or running-xemacs
      (setq deactivate-mark t))
  )

(defun mell-region-get-rectangle-list (start end &optional buffer)
  (save-excursion
    (and buffer (set-buffer buffer))
    (let* (rectangle-alist
	   (column-min (min (mell-column-at-point start)
			    (mell-column-at-point end)))
	   (column-max (max (mell-column-at-point start)
			    (mell-column-at-point end)))
	   (point-min (min (mell-point-at-column column-min start)
			   (mell-point-at-column column-min end)))
	   (point-max (max (mell-point-at-column column-max start)
			   (mell-point-at-column column-max end)))
	   )
      (goto-char point-min)
      (while (< (point) point-max)
	(move-to-column column-min)
	(setq rectangle-alist 
	      (cons (cons (point) (mell-point-at-column column-max))
		    rectangle-alist))
	(forward-line 1)
	)
      (reverse rectangle-alist)
      )))

(defun mell-region-get-visible-rectangle-list (start end &optional buffer)
  (save-excursion
    (and buffer (set-buffer buffer))
    (let* (rectangle-alist
	   (column-min (min (mell-column-at-point start)
			    (mell-column-at-point end)))
	   (column-max (max (mell-column-at-point start)
			    (mell-column-at-point end)))
	   (point-min (min (mell-point-at-column column-min start)
			   (mell-point-at-column column-min end)))
	   (point-max (max (mell-point-at-column column-max start)
			   (mell-point-at-column column-max end)))
	   )
      (goto-char (max point-min (window-start)))
      (while (< (point) (min point-max (window-end)))
	(move-to-column column-min)
	(setq rectangle-alist 
	      (cons (cons (point) (mell-point-at-column column-max))
		    rectangle-alist))
	(forward-line 1)
	)
      (reverse rectangle-alist)
      )))
      
(put 'mell-region-rectangle-while 'lisp-indent-function 1)
(defmacro mell-region-rectangle-while (rectangle &rest body)
  `(let ((rectangle-markers
	  (mell-region-get-rectangle-marker-list
	   (nth 0 ,rectangle) (nth 1 ,rectangle) (nth 2 ,rectangle)))
	 )
     (mapcar
      (lambda (region)
	(let ((line-beginning (car region))
	      (line-end (cdr region)))
	  ,@body
	  ))
      rectangle-markers)
     (mapcar
      (lambda (region)
	(set-marker (car region) nil)
	(set-marker (cdr region) nil)
	)
      rectangle-markers)
     ))

(defun mell-region-get-rectangle-marker-list (start end &optional buffer)
  (mapcar
   '(lambda (region)
      (cons (mell-marker-make (car region)) (mell-marker-make (cdr region)))
      )
   (mell-region-get-rectangle-list start end buffer))
  )

(defun mell-region-rectangle-right-edge-p (start end)
  (save-excursion
    (let ((list (mell-region-get-rectangle-list start end))
	  (result t))
      (while (and list
		  (progn (goto-char (cdr (car list)))
			 (eolp)))
	(setq list (cdr list))
	)
      (null list)
      )))

;;; ------------------------------------------------------------
;;; sense-region-mode
;;; ------------------------------------------------------------

(defcustom sense-region-on-hook nil
  "Function or functions called when sense-region-on is executed.")
(defcustom sense-region-off-hook nil
  "Function or functions called when sense-region-off is executed.")
(defcustom sense-region-adviced-functions
  '((set-mark-command . sense-region-set-mark)
    kill-ring-save kill-region yank
    comment-region indent-for-tab-command query-replace query-replace-regexp)
  "List of defadviced functions for sense-region."
  )
  
(defvar sense-region-mode             nil)
(defvar sense-region-overlay-list     nil)
(defvar sense-region-face             nil)
(defvar sense-region-status           'region)
(defvar sense-region-last-status      'region)
(defvar sense-region-top-of-kill-ring nil)
(defvar sense-region-track-status nil)


(defadvice set-mark-command (around sense-region-set-mark disable)
  (if (and (mell-transient-region-active-p)
	   sense-region-mode)
      (if (and (eq last-command this-command)
	       (or (eq (region-beginning) (region-end))
		   sense-region-track-status))
	  (sense-region-track)
	(setq sense-region-track-status nil)
	(sense-region-toggle))
    (setq sense-region-track-status nil)
    ad-do-it
    )
  (mell-transient-region-stay)
  )

(defun sense-region-track (&optional position)
  (cond
   ((eq sense-region-track-status nil)
    (sense-region-set-word position)
    (setq sense-region-track-status 'word))
   ((eq sense-region-track-status 'word)
    (cond ((sense-region-set-url position)
	   (setq sense-region-track-status 'url))
	  ((sense-region-set-email position)
	   (setq sense-region-track-status 'url))
	  (t
	   (sense-region-set-next-word position)
	   (setq sense-region-track-status 'next-word))
	  ))
   ((eq sense-region-track-status 'url)
    'url)
   ((eq sense-region-track-status 'email)
    'url)
   ((eq sense-region-track-status 'next-word)
    (sense-region-set-next-word position)
    'next-word)
   (t
    (message (format "sense-region-track: Wrong type status `%S'"
		     sense-region-track-status))
    (sense-region-set-word position)
    (setq sense-region-track-status 'word))
   )
  (mell-transient-region-stay)
  )

(defun sense-region-set-word (&optional position) ; mell $B9T$-$+$J(B?
  (interactive)
  (or position (setq position (point)))
  (goto-char position)

  (if (>= position (progn (backward-word 1) (forward-word 1) (point)))
      (goto-char position)
    (goto-char position)
    (backward-word 1)
    )
  (set-mark (point))
  (forward-word 1)
  (mell-transient-region-stay)
  t
  )

(defun sense-region-set-next-word (&optional position) ; mell $B9T$-$+$J(B?
  (interactive)
  (or position (setq position (region-end)))
  (goto-char position)

  (set-mark (region-beginning))
  (forward-word 1)
  (mell-transient-region-stay)
  t
  )

(defun sense-region-set-url (&optional position) ; mell $B9T$-$+$J(B?
  (interactive)
  (or position (setq position (point)))
  (goto-char position)

  (mell-transient-region-stay)
  (save-match-data
    (if (and (< (skip-chars-backward "-a-zA-Z0-9.:#~_+/") 0)
	     (looking-at "http:"))
	(progn
	  (set-mark (point))
	  (re-search-forward "[-a-zA-Z0-9.:#~_+/]+")
	  t)
      (goto-char position)
      nil)
    ))

(defun sense-region-set-email (&optional position) ; mell $B9T$-$+$J(B?
  (interactive)
  (or position (setq position (point)))
  (goto-char position)

  (mell-transient-region-stay)
  (save-match-data
    (if (and (< (skip-chars-backward "-a-zA-Z0-9.:@_") 0)
	     (looking-at "[-a-zA-Z0-9.:_]+@[-a-zA-Z0-9.:_]+"))
	(progn
	  (set-mark (point))
	  (re-search-forward "[-a-zA-Z0-9.:@_]+")
	  t)
      (goto-char position)
      nil)
    ))

(defun sense-region-redisplay ()
  (if (window-minibuffer-p (selected-window))
      nil ; Do nothing.
    (mell-sign-rectangle-highlight-off sense-region-overlay-list)
    (if (and (mell-transient-region-active-p)
	     (eq sense-region-status 'rectangle))
	(setq sense-region-overlay-list
	      (mell-sign-rectangle-highlight
	       (region-beginning) (region-end)
	       nil 'sense-region-face))
      (sense-region-to-region)
    )))

(defun sense-region-on ()
  (interactive)
  (setq sense-region-mode t)
  (copy-face mell-region-face 'sense-region-face)
  (add-hook 'post-command-hook 'sense-region-redisplay)

  (mapcar 
   '(lambda (function)
      (if (consp function)
	  (progn
	    (ad-enable-advice (car function) 'around (cdr function))
	    (ad-activate (car function))
	    )
	(ad-enable-advice function 'around 
			  (intern (format "sense-region-%S" function)))
	(ad-activate function)
	))
   sense-region-adviced-functions)
  (run-hooks 'sense-region-on-hook)
  (mell-transient-region-stay)
  )

(defun sense-region-off ()
  (interactive)
  (setq sense-region-mode nil)
  (mell-sign-rectangle-highlight-off sense-region-overlay-list)
  (remove-hook 'post-command-hook 'sense-region-redisplay)
  (copy-face 'sense-region-face mell-region-face)
  (mapcar 
   '(lambda (function)
      (if (consp function)
	  (progn
	    (ad-disable-advice (car function) 'around (cdr function))
	    (ad-activate (car function))
	    )
	(ad-disable-advice function 'around 
			  (intern (format "sense-region-%S" function)))
	(ad-activate function)
	))
   sense-region-adviced-functions)
  (run-hooks 'sense-region-off-hook)
  (setq sense-region-status 'region)
  )


(defun sense-region-toggle ()
  (interactive)
  (cond
   ((eq sense-region-status 'region)
    (sense-region-to-rectangle))
   ((eq sense-region-status 'rectangle)
    (setq sense-region-status 'region)
    (sense-region-to-region))
   ))

(defun sense-region-to-rectangle ()
  (interactive)
  (setq sense-region-status 'rectangle)
  (copy-face mell-region-face 'sense-region-face)
;  (copy-face 'default mell-region-face)
  (mell-sign-reset-face mell-region-face)
  )  
  
(defun sense-region-to-region ()
  (interactive)
  (setq sense-region-status 'region)
  (mell-sign-rectangle-highlight-off sense-region-overlay-list)
  (setq sense-region-overlay-list nil)
  (copy-face 'sense-region-face mell-region-face)
  )

;;; ------------------------------------------------------------

(defadvice kill-ring-save (around sense-region-kill-ring-save disable)
  (if (mell-transient-region-active-p)
      (cond ((eq sense-region-status 'region)
	     ad-do-it
	     (setq sense-region-last-status 'region))
	    ((eq sense-region-status 'rectangle)
	     (setq sense-region-top-of-kill-ring (car kill-ring))
	     (setq killed-rectangle
		   (extract-rectangle (region-beginning) (region-end)))
	     (setq sense-region-last-status 'rectangle))
	    (t
	     (message (concat "Unkown sense-region-status: "
			      (symbol-name sense-region-status)))
	     ad-do-it
	     (setq sense-region-last-status 'region))
	     )
    ad-do-it
    (setq sense-region-last-status 'region))
  (mell-transient-region-deactivate)
  (setq sense-region-status 'region)
  )

(defadvice kill-region (around sense-region-kill-region disable)
  (if (mell-transient-region-active-p)
      (cond ((eq sense-region-status 'region)
	     ad-do-it
	     (setq sense-region-last-status 'region))
	    ((eq sense-region-status 'rectangle)
	     (call-interactively 'kill-rectangle)
	     (setq sense-region-top-of-kill-ring (car kill-ring))
	     (setq sense-region-last-status 'rectangle))
	    (t
	     (message (concat "Unkown sense-region-status: "
			      (symbol-name sense-region-status)))
	     ad-do-it
	     (setq sense-region-last-status 'region))
	    )
    ad-do-it
    (setq sense-region-last-status 'region))
  (mell-transient-region-deactivate)
  (setq sense-region-status 'region)
  )

(defadvice yank (around sense-region-yank disable)
  (let ((begin (point))
	status overlay)
    (setq status
	  (cond ((eq sense-region-last-status 'region)
		 'region)
		((eq sense-region-last-status 'rectangle)
		 (if (string= (car kill-ring) sense-region-top-of-kill-ring)
		     'rectangle
		   'region))
		(t
		 (message (concat "Unkown sense-region-status: "
				  (symbol-name sense-region-status)))
		 'region)
		))
    (cond ((eq status 'region)
	   ad-do-it
	   (setq overlay (mell-sign-region-highlight begin (point)))
	   (sit-for 0.5)
	   (mell-sign-region-highlight-off overlay)
	   )
	  ((eq status 'rectangle)
	   (call-interactively 'yank-rectangle)
	   (setq overlay (mell-sign-rectangle-highlight begin (point)))
	   (sit-for 0.5)
	   (mell-sign-rectangle-highlight-off overlay)
	   )
	  )
    (mell-transient-region-deactivate)
    (setq sense-region-status 'region)
    ))

;;; ------------------------------------------------------------

(defadvice comment-region (around sense-region-comment-region disable)
  (ad-disable-advice 'comment-region 'around 'sense-region-comment-region)
  (ad-activate 'comment-region)
  (if (mell-transient-region-active-p)
      (cond ((eq sense-region-status 'region)
	     ad-do-it)
	    ((eq sense-region-status 'rectangle)
	     (call-interactively 'sense-region-comment-rectangle))
	    (t
	     (message (concat "Unkown sense-region-status: "
			      (symbol-name sense-region-status)))
	     ad-do-it)
	    )
    ad-do-it)
  (mell-transient-region-deactivate)
  (setq sense-region-status 'region)
  (ad-enable-advice 'comment-region 'around 'sense-region-comment-region)
  (ad-activate 'comment-region)
  )

(defun sense-region-comment-rectangle (start end &optional arg)
  (interactive "r\nP")
  (let* ((original-ce comment-end)
	 (comment-end "")
	 indent-next-line-p)
    (if (and (string= original-ce "")
	     (not (mell-region-rectangle-right-edge-p start end))
	     (y-or-n-p "Insert enter and indent? "))
	(progn
	  (setq indent-next-line-p t)
	  (if (and (not running-xemacs)
		   (>= (string-to-number emacs-version) 21)) ; FIX-ME: ADHOC!!!
	      (setq comment-end "")
	    (setq comment-end "\n")
	    ))
      (setq comment-end original-ce)
      )

    (mell-region-rectangle-while (list start end)
      (or (= (marker-position line-beginning) (marker-position line-end))
	  (comment-region line-beginning line-end arg))
      (if indent-next-line-p
	  (progn
	    (goto-char line-beginning)
	    (forward-line 1)
	    (indent-for-tab-command)
	    )
	))
    ))

(defadvice indent-for-tab-command (around sense-region-indent-for-tab-command
					  disable)
  (if (mell-transient-region-active-p)
      (cond ((eq sense-region-status 'region)
	     (call-interactively 'indent-region))
	    ((eq sense-region-status 'rectangle)
	     (if (functionp 'table-rectangle)
		 (call-interactively 'table-rectangle)
	       nil))
	    (t
	     (message (concat "Unkown sense-region-status: "
			      (symbol-name sense-region-status)))
	     ad-do-it)
	    )
    ad-do-it)
  (mell-transient-region-deactivate)
  (setq sense-region-status 'region)
  )

(defadvice indent-for-tab-command (around sense-region-indent-for-tab-command
					  disable)
  (if (mell-transient-region-active-p)
      (cond ((eq sense-region-status 'region)
	     (call-interactively 'indent-region))
	    ((eq sense-region-status 'rectangle)
	     (if (functionp 'table-rectangle)
		 (call-interactively 'table-rectangle)
	       nil))
	    (t
	     (message (concat "Unkown sense-region-status: "
			      (symbol-name sense-region-status)))
	     ad-do-it)
	    )
    ad-do-it)
  (mell-transient-region-deactivate)
  (setq sense-region-status 'region)
  )

(defadvice query-replace-regexp (around sense-region-query-replace-regexp
					disable)
  (if (mell-transient-region-active-p)
      (cond ((eq sense-region-status 'region)
	     ad-do-it)
	    ((eq sense-region-status 'rectangle)
	     (mell-region-rectangle-while
		 (list (region-beginning) (region-end))
	       (save-excursion
		 (set-mark line-beginning)
		 (goto-char line-end)
		 (and (fboundp 'zmacs-activate-region)
		      (zmacs-activate-region))
		 (if (or running-xemacs ; FIX-ME: ADHOC!!!
			 (< (string-to-number emacs-version) 21))
		     ad-do-it
		   (ad-disable-advice 'query-replace-regexp 'around
				      'sense-region-query-replace-regexp)
		   (ad-activate 'query-replace-regexp)
		   (eval '(query-replace (ad-get-arg 0) (ad-get-arg 1) nil
					 line-beginning line-end))
		   (ad-enable-advice 'query-replace-regexp 'around
				     'sense-region-query-replace-regexp)
		   (ad-activate 'query-replace-regexp)
		   )))
	     )
	    (t
	     (message (concat "Unkown sense-region-status: "
			      (symbol-name sense-region-status)))
	     ad-do-it)
	    )
    ad-do-it)
  (mell-transient-region-deactivate)
  (setq sense-region-status 'region)
  )

(defadvice query-replace (around sense-region-query-replace disable)
  (if (mell-transient-region-active-p)
      (cond ((eq sense-region-status 'region)
	     ad-do-it)
	    ((eq sense-region-status 'rectangle)
	     (mell-region-rectangle-while
		 (list (region-beginning) (region-end))
	       (save-excursion
		 (set-mark line-beginning)
		 (goto-char line-end)
		 (and (fboundp 'zmacs-activate-region)
		      (zmacs-activate-region))
		 (if (or running-xemacs ; FIX-ME: ADHOC!!!
			 (< (string-to-number emacs-version) 21))
		     ad-do-it
		   (ad-disable-advice 'query-replace 'around
				      'sense-region-query-replace)
		   (ad-activate 'query-replace)
		   (eval '(query-replace (ad-get-arg 0) (ad-get-arg 1) nil
					 line-beginning line-end))
		   (ad-enable-advice 'query-replace 'around
				     'sense-region-query-replace)
		   (ad-activate 'query-replace)
		   )))
	     )
	    (t
	     (message (concat "Unkown sense-region-status: "
			      (symbol-name sense-region-status)))
	     ad-do-it)
	    )
    ad-do-it)
  (mell-transient-region-deactivate)
  (setq sense-region-status 'region)
  )

(provide 'sense-region)