/**************************************************************************
 FreeSockets - Portable C++ classes for IP(sockets) applications. (v0.3)
 Copyright (C) 2000-2001 Rafael Guterres Jeffman
           (C) 2002-2006 Alistair Riddoch

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

**************************************************************************/

/**
 * This software package has been extensively modified by members of the
 * Worldforge Project. See the file ChangeLog for details.
 *
 * $Id: skstream.h,v 1.52 2006/10/12 01:10:12 alriddoch Exp $
 *
 */
#ifndef RGJ_FREE_STREAM_H_
#define RGJ_FREE_STREAM_H_

#include <iostream>

#include <skstream/sksocket.h>

/////////////////////////////////////////////////////////////////////////////
// class socketbuf
/////////////////////////////////////////////////////////////////////////////
class socketbuf : public std::streambuf {
private:
  char *_buffer;

protected:
  SOCKET_TYPE _socket;

  timeval _timeout;

private:
  /// Not implemented. Copying a socket buffer is not permited.
  socketbuf(const socketbuf&);
  /// Not implemented. Copying a socket buffer is not permited.
  socketbuf& operator=(const socketbuf&);

protected:
  bool Timeout;

public:
  /** Make a new socket buffer from an existing socket, with optional
   *  buffer sizes.
   */
  explicit socketbuf(SOCKET_TYPE sock, unsigned insize=0x8000,
                                       unsigned outsize=0x8000);
  /** Make a new socket buffer from an existing socket, with an existing
   *  buffer.
   */
  socketbuf(SOCKET_TYPE sock, char* buf, int length);

  /// Destroy the socket buffer.
  virtual ~socketbuf();

  /// Set the existing socket that this buffer should use.
  void setSocket(SOCKET_TYPE sock);

  /// Get the socket that this buffer uses.
  SOCKET_TYPE getSocket() const {
      return _socket; 
  }

  /** Set up a timeout value after which an error flag is set if the socket
   *  is not ready for a read or write.
   */
  void setTimeout(unsigned sec, unsigned usec=0) {
    _timeout.tv_sec  = sec;
    _timeout.tv_usec = usec;
  }

  /// Return the flag indicating whether a timeout has occured.
  bool timeout() const {
    return Timeout;
  }

protected:
  /// Handle writing data from the buffer to the socket.
  virtual int overflow(int nCh=EOF) = 0;
  /// Handle reading data from the socket to the buffer.
  virtual int underflow() = 0;

  /// Flush the output buffer.
  int sync();

  /** Set the buffer area this stream buffer uses. Only works if not already
   *  set.
   */
  std::streambuf * setbuf(std::streambuf::char_type * buf, std::streamsize len);
};

class stream_socketbuf : public socketbuf {
public:
  /** Make a new socket buffer from an existing socket, with optional
   *  buffer sizes.
   */
  explicit stream_socketbuf(SOCKET_TYPE sock, unsigned insize=0x8000,
                                              unsigned outsize=0x8000);
  /** Make a new socket buffer from an existing socket, with an existing
   *  buffer.
   */
  stream_socketbuf(SOCKET_TYPE sock, char* buf, int length);

  /// Destroy the socket buffer.
  virtual ~stream_socketbuf();

protected:
  /// Handle writing data from the buffer to the socket.
  virtual int overflow(int nCh=EOF);
  /// Handle reading data from the socket to the buffer.
  virtual int underflow();

};

class dgram_socketbuf : public socketbuf {
public:
  /** Make a new socket buffer from an existing socket, with optional
   *  buffer sizes.
   */
  explicit dgram_socketbuf(SOCKET_TYPE sock, unsigned insize=0x8000,
                                             unsigned outsize=0x8000);
  /** Make a new socket buffer from an existing socket, with an existing
   *  buffer.
   */
  dgram_socketbuf(SOCKET_TYPE sock, char* buf, int length);

  /// Destroy the socket buffer.
  virtual ~dgram_socketbuf();

  bool setTarget(const std::string& address, unsigned port, int proto);

  void setOutpeer(const sockaddr_storage & peer) { 
    out_peer = peer; 
  }

  const sockaddr_storage & getOutpeer() const {
    return out_peer; 
  }

  const sockaddr_storage & getInpeer() const { 
    return in_peer; 
  }

  SOCKLEN getOutpeerSize() const {
    return out_p_size;
  }

  SOCKLEN getInpeerSize() const {
    return in_p_size;
  }

protected:
  /// Target address of datagrams sent via this stream
  sockaddr_storage out_peer;
  /// Source address of last datagram received by this stream
  sockaddr_storage in_peer;
  /// Size of target address
  SOCKLEN out_p_size;
  /// Size of source address
  SOCKLEN in_p_size;

  /// Handle writing data from the buffer to the socket.
  virtual int overflow(int nCh=EOF);
  /// Handle reading data from the socket to the buffer.
  virtual int underflow();

};

/////////////////////////////////////////////////////////////////////////////
// Enumerations
/////////////////////////////////////////////////////////////////////////////
// Supported Protocols
namespace FreeSockets {
  enum IP_Protocol {
    proto_IP   = IPPROTO_IP,
    proto_ICMP = IPPROTO_ICMP,
#ifndef _WIN32 
    proto_IGMP = IPPROTO_IGMP, 
#else 
    proto_IGMP = IPPROTO_GGP, 
#endif 
    proto_TCP  = IPPROTO_TCP,
    proto_PUP  = IPPROTO_PUP,
    proto_UDP  = IPPROTO_UDP,
    proto_IDP  = IPPROTO_IDP,
    proto_RAW  = IPPROTO_RAW
  };
  // Well known ports
  enum IP_Service {
    echo        =       7, //
    daytime     =      13, //
    ftp         =      21, //
    ssh         =      22, //
    telnet      =      23, //
    smtp        =      25, //       mail
    time        =      37, //       timserver
    name        =      42, //       nameserver
    nameserver  =      53, //       domain        # name-domain server
    finger      =      79, //
    http        =      80, //
    pop         =     109, //       postoffice
    pop2        =     109, //                     # Post Office
    pop3        =     110, //       postoffice
    nntp        =     119  //       usenet        # Network News Transfer
  };
};

/////////////////////////////////////////////////////////////////////////////
// class socket_stream
/////////////////////////////////////////////////////////////////////////////
class basic_socket_stream : public basic_socket, public std::iostream {
protected:
  socketbuf & _sockbuf;
  int m_protocol;

public:
  /// Make a socket stream.
  basic_socket_stream(socketbuf & buffer, int proto = FreeSockets::proto_IP);

  // Destructor
  virtual ~basic_socket_stream();

  bool fail();

  bool operator!() {
    return fail();
  }

  bool timeout() const {
    return _sockbuf.timeout();
  }

  virtual SOCKET_TYPE getSocket() const;

  // Needs to be virtual to handle in-progress connect()'s for
  // tcp sockets
  virtual void close();

  void shutdown();

  void setSocket(SOCKET_TYPE sock) {
    _sockbuf.setSocket(sock);
  }

  void setTimeout(unsigned sec, unsigned usec=0) { 
    _sockbuf.setTimeout(sec,usec); 
  }

  int getProtocol() const { 
    return m_protocol; 
  }
};

/////////////////////////////////////////////////////////////////////////////
// class tcp_socket_stream
/////////////////////////////////////////////////////////////////////////////

struct addrinfo;

class tcp_socket_stream : public basic_socket_stream {
private:
  tcp_socket_stream(const tcp_socket_stream&);

  tcp_socket_stream& operator=(const tcp_socket_stream& socket);

  SOCKET_TYPE _connecting_socket;
  struct addrinfo * _connecting_address;
  struct addrinfo * _connecting_addrlist;

  stream_socketbuf & stream_sockbuf;

public:
  tcp_socket_stream();
  tcp_socket_stream(SOCKET_TYPE socket);

  tcp_socket_stream(const std::string& address, int service,
                    bool nonblock = false);

  tcp_socket_stream(const std::string& address, int service,
                    unsigned int milliseconds);

  virtual ~tcp_socket_stream();

  void open(const std::string& address, int service, bool nonblock = false);
  void open(const std::string& address, int service, unsigned int milliseconds);

  virtual void close();
  virtual SOCKET_TYPE getSocket() const;

  const std::string getRemoteHost(bool lookup = false) const;
  const std::string getRemoteService(bool lookup = false) const;
  bool isReady(unsigned int milliseconds = 0);
};

class dgram_socket_stream : public basic_socket_stream {
private:
  dgram_socket_stream(const dgram_socket_stream&);

  dgram_socket_stream& operator=(const dgram_socket_stream& socket);

protected:
  dgram_socketbuf & dgram_sockbuf;

  int bindToIpService(int service, int type, int protocol);

public:
  dgram_socket_stream();

  virtual ~dgram_socket_stream();

  bool setTarget(const std::string& address, unsigned port) { 
    return dgram_sockbuf.setTarget(address, port, m_protocol); 
  }

  void setOutpeer(const sockaddr_storage& peer) { 
    return dgram_sockbuf.setOutpeer(peer); 
  }

  const sockaddr_storage & getOutpeer() const { 
    return dgram_sockbuf.getOutpeer(); 
  }

  const sockaddr_storage & getInpeer() const { 
    return dgram_sockbuf.getInpeer(); 
  }

  SOCKLEN getOutpeerSize() const {
    return dgram_sockbuf.getOutpeerSize();
  }

  SOCKLEN getInpeerSize() const {
    return dgram_sockbuf.getInpeerSize();
  }
};


/////////////////////////////////////////////////////////////////////////////
// class udp_socket_stream
/////////////////////////////////////////////////////////////////////////////

class udp_socket_stream : public dgram_socket_stream {
private:
  udp_socket_stream(const udp_socket_stream&);

  udp_socket_stream& operator=(const udp_socket_stream& socket);

public:
  udp_socket_stream();

  virtual ~udp_socket_stream();

  int open(int service);
};

/////////////////////////////////////////////////////////////////////////////
// class raw_socket_stream
/////////////////////////////////////////////////////////////////////////////
#ifdef SOCK_RAW
class raw_socket_stream : public dgram_socket_stream {
private:
  raw_socket_stream(const raw_socket_stream&);

  raw_socket_stream& operator=(const raw_socket_stream& socket);

public:
  raw_socket_stream(FreeSockets::IP_Protocol proto=FreeSockets::proto_RAW);

  virtual ~raw_socket_stream();

  void setProtocol(FreeSockets::IP_Protocol proto);

  bool setTarget(const std::string& address, unsigned port) { 
    return dgram_sockbuf.setTarget(address, port, m_protocol); 
  }

  bool setBroadcast(bool opt=false);
};
#endif // SOCK_RAW

#endif // RGJ_FREE_STREAM_H_

