package SliMP3::Buttons::Browse;

# $Id: Browse.pm,v 1.104 2003/09/12 18:39:17 blackketter Exp $

# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.

use strict;
use File::Spec::Functions qw(:ALL);
use File::Spec::Functions qw(updir);
use SliMP3::Buttons::Block;
use SliMP3::Buttons::Common;
use SliMP3::Buttons::Playlist;
use SliMP3::Buttons::TrackInfo;
use SliMP3::Misc;
use SliMP3::Strings qw (string);
use SliMP3::Scan;

# Each button on the remote has a function:

my %functions = (
	'up' => sub  {
		my $client = shift;
		my $button = shift;
		my $inc = shift || 1;
		my $count = $client->numberOfDirItems();
		if ($count < 2) {
			SliMP3::Animation::bumpUp($client);
		} else {
			$inc = ($inc =~ /\D/) ? -1 : -$inc;
			my $newposition = SliMP3::Buttons::Common::scroll($client, $inc, $count, $client->currentDirItem());
			$client->currentDirItem($newposition);
			SliMP3::Display::update($client);
		}
	},
	'down' => sub  {
		my $client = shift;
		my $button = shift;
		my $inc = shift || 1;
		my $count = $client->numberOfDirItems();
		if ($count < 2) {
			SliMP3::Animation::bumpDown($client);
		} else {
			if ($inc =~ /\D/) {$inc = 1}
			my $newposition = SliMP3::Buttons::Common::scroll($client, $inc, $client->numberOfDirItems(), $client->currentDirItem());
			$client->currentDirItem($newposition);
			SliMP3::Display::update($client);
		}
	},
	'left' => sub  {
		my $client = shift;
		if ($client->pwd() =~ m|^[/\\]?$| || $client->pwd() eq "__playlists") {
			# go to the home directory
			SliMP3::Buttons::Common::popModeRight($client);
			$client->lastSelection('', $client->currentDirItem());
		} else {
			# move up one level
			my @oldlines = SliMP3::Display::curLines($client);
			loadDir($client, updir(), "left", \@oldlines);
		}
	},
	'right' => sub  {
		my $client = shift;
		if (!$client->numberOfDirItems()) {
			# don't do anything if the list is empty
			SliMP3::Animation::bumpRight($client);
		} else {
			my $currentItem = $client->dirItems($client->currentDirItem());
			$::d_files && msg("currentItem == $currentItem\n");
			my @oldlines = SliMP3::Display::curLines($client);
			if (SliMP3::Info::isList($currentItem)) {
				# load the current item if it's a list (i.e. directory or playlist)
				# treat playlist files as directories.
				# ie - list the contents
				loadDir($client, $currentItem, "right", \@oldlines);
			} elsif (SliMP3::Info::isSong($currentItem) || SliMP3::Info::isHTTPURL($currentItem)) {
				# enter the trackinfo mode for the track in $currentitem
				SliMP3::Buttons::Common::pushMode($client, 'trackinfo', {'track' => $currentItem});
				SliMP3::Animation::pushLeft($client, @oldlines, SliMP3::Display::curLines($client));
			} else {
				warn "Error attempting to descend directory or open file: $currentItem\n";
			}
		}
	},
	'numberScroll' => sub  {
		my $client = shift;
		my $button = shift;
		my $digit = shift;
		my $i = SliMP3::Buttons::Common::numberScroll($client, $digit, $client->dirItems, SliMP3::Info::isDir(SliMP3::Misc::virtualToAbsolute($client->pwd())),
			sub {
				my $j = $client->dirItems(shift);
				if (SliMP3::Prefs::get('filesort')) {
					return SliMP3::Info::plainTitle($j);
				} elsif (SliMP3::Info::trackNumber($j)) {
					return SliMP3::Info::trackNumber($j);
				} else {
					return SliMP3::Info::title($j);
				}
			}
			);

		$client->currentDirItem($i);
		SliMP3::Display::update($client);
	},
	'add' => sub  {
		my $client = shift;
		my $currentItem = $client->dirItems($client->currentDirItem());
		my $line1 = string('ADDING_TO_PLAYLIST');
		my $line2 = SliMP3::Info::standardTitle($client, $currentItem);
		$::d_files && msg("currentItem == $currentItem\n");
		if (SliMP3::Info::isList($currentItem)) {
			# we are looking at an playlist file or directory
			SliMP3::Buttons::Block::block($client, $line1, $line2);
			SliMP3::Command::execute($client, ["playlist", "add", $currentItem], \&playDone, [$client]);
		} elsif (SliMP3::Info::isSong($currentItem) || SliMP3::Info::isHTTPURL($currentItem)) {
			SliMP3::Animation::showBriefly($client, $line1, $line2, undef, 1);
			# we are looking at a song file, play it and all the other songs in the directory after
			SliMP3::Command::execute($client, ["playlist", "append", $currentItem]);
		} else {
			warn "Error attempting to add directory or file to playlist.\n";
			return;
		}
	},
	'insert' => sub  {
		my $client = shift;
		my $currentItem = $client->dirItems($client->currentDirItem());
		my $line1 = string('INSERT_TO_PLAYLIST');
		my $line2 = SliMP3::Info::standardTitle($client, $currentItem);
		$::d_files && msg("currentItem == $currentItem\n");
		if (SliMP3::Info::isList($currentItem)) {
			# we are looking at an playlist file or directory
			SliMP3::Buttons::Block::block($client, $line1, $line2);
			SliMP3::Command::execute($client, ["playlist", "insertlist", $currentItem], \&playDone, [$client]);
		} elsif (SliMP3::Info::isSong($currentItem) || SliMP3::Info::isHTTPURL($currentItem)) {
			SliMP3::Animation::showBriefly($client, $line1, $line2, undef, 1);
			# we are looking at a song file, play it and all the other songs in the directory after
			SliMP3::Command::execute($client, ["playlist", "insert", $currentItem]);
		} else {
			warn "Error attempting to add directory or file to playlist.\n";
			return;
		}
	},
	'play' => sub  {
		my $client = shift;
		my $currentItem = $client->dirItems($client->currentDirItem());
		my $line1;
		my $line2 = SliMP3::Info::standardTitle($client, $currentItem);
		my $shuffled = 0;
		if (SliMP3::Playlist::shuffle($client)) {
			$line1 = string('PLAYING_RANDOMLY_FROM');
			$shuffled = 1;
		} else {
			$line1 = string('NOW_PLAYING_FROM');
		
		}
		if (SliMP3::Info::isList($currentItem)) {
			# we are looking at an playlist file or directory
			SliMP3::Buttons::Block::block($client,$line1, $line2);
			SliMP3::Command::execute($client, ["playlist", "load", $currentItem], \&playDone, [$client]);
		} elsif (SliMP3::Info::isSong($currentItem) || SliMP3::Info::isHTTPURL($currentItem)) {
			# put all the songs at this level on the playlist and start playing the selected one.
			SliMP3::Animation::showBriefly($client, $line1, $line2, undef, 1);
			if (SliMP3::Prefs::get('playtrackalbum') && !SliMP3::Info::isHTTPURL($currentItem)) {
				SliMP3::Command::execute($client, ["playlist", "clear"]);
				SliMP3::Command::execute($client, ["playlist", "shuffle" , 0]);
				my $startindex = 0;
				my $startindexcounter = 0;
				my $dirref = $client->dirItems;
				if (SliMP3::Info::isPlaylist(SliMP3::Misc::virtualToAbsolute($client->pwd()))) {
					$startindex = $client->currentDirItem();
					SliMP3::Command::execute($client, ["playlist", "load", $client->pwd()], \&playDone, [$client, $startindex, $shuffled]);
				} else {
					foreach my $song (@$dirref) {
						if (SliMP3::Info::isSong($song)) {
							if ($song eq $currentItem) { $startindex = $startindexcounter; }
							SliMP3::Command::execute($client, ["playlist", "append", $song]);
							$startindexcounter++;
						}
					}
					playDone($client, $startindex, $shuffled);
				}
			} else {
				SliMP3::Command::execute($client, ["playlist", "play", $currentItem]);
			}
		} else {
			warn "Error attempting to play directory or open file.\n";
		}
	}
);

sub playDone {
	my $client = shift;
	my $startindex = shift;
	my $shuffled = shift;
	
	SliMP3::Buttons::Block::unblock($client);
	
	#The following commented out to allow showBriefly to finish
	#SliMP3::Display::update($client);
	if (defined($startindex)) { SliMP3::Command::execute($client, ["playlist", "jump", $startindex]); }
	if (defined($shuffled)) { SliMP3::Command::execute($client, ["playlist", "shuffle" , $shuffled]); }

}

sub getFunctions {
	return \%functions;
}

sub setMode {
	my $client = shift;
	$client->lines(\&lines);
}

# create a directory listing, and append it to dirItems
sub loadDir {
	my ($client,$dir, $direction, $oldlinesref) = @_;

	my $pwd;
	my $oldpwd = $client->pwd();
	my $abspwd;

	if (defined($oldpwd)) {
		$client->lastSelection($oldpwd, $client->currentDirItem());
		$::d_files && msg("saving lastSelection for: $oldpwd as " . $client->currentDirItem() . "\n");
	}

	if ($dir eq "__playlists") {
		$::d_files && msg("loaddir: dir eq __playlists\n");
		$pwd = $dir;
	} elsif (!defined($dir) || $dir eq "") {
		$::d_files && msg("loaddir: !defined(dir) || dir eq ''\n");
		$pwd = '';
	} elsif ($dir eq updir()) {
		$::d_files && msg("loaddir: dir eq updir()\n");
		$pwd = SliMP3::Misc::ascendVirtual($oldpwd);
	} else {
		$::d_files && msg("loaddir: normal descend()\n");
		$pwd = SliMP3::Misc::descendVirtual($oldpwd
			,$client->dirItems($client->currentDirItem()),$client->currentDirItem());
	}

	@{$client->dirItems}=();

	$abspwd = SliMP3::Misc::virtualToAbsolute($pwd);
	$::d_files && msg("virtual directory: $pwd\nabsolute directory: $abspwd\n");

	unless (defined($abspwd) && 
			(SliMP3::Info::isHTTPURL($abspwd) || 
			 SliMP3::Info::isITunesPlaylistURL($abspwd) || 
			 (-e $abspwd)
			)
		   ) {
		opendir_done($client, $pwd, $direction, $oldlinesref, 0);
		$::d_files && msg("No such file or dir: [$pwd] removed out from under?\n");
		return;
	}

	$::d_files && msg("debug: Opening dir: [$pwd]\n");

	if (SliMP3::Info::isList($abspwd)) {
		my $itemCount = 0;
		$::d_files && msg("getting playlist " . $pwd . " as directory\n");
		SliMP3::Buttons::Block::block($client,@$oldlinesref);

		SliMP3::Scan::addToList($client->dirItems, $abspwd, 0, undef, \&opendir_done, $client, $pwd, $direction, $oldlinesref);
		# addToList will call &opendir_done when it finishes.

	} else {
		warn "Trying to loadDir on a non directory or playlist: $pwd";
	}
}

#
# this is the callback from addToList when we open a directory:
#
sub opendir_done {
	my ($client, $pwd, $direction, $oldlinesref, $itemCount) = @_;
	$::d_files && msg("opendir_done($client, $pwd, $itemCount)\n");

	if ($pwd eq '__playlists' && SliMP3::iTunes::useiTunesLibrary()) {
		$::d_files && msg("adding itunes playlists\n");
		push @{$client->dirItems}, @{SliMP3::iTunes::playlists()};
	} 

	# in case we were blocked...
	SliMP3::Buttons::Block::unblock($client);

	$client->numberOfDirItems(scalar @{$client->dirItems});

	##############
	# check for the user's last selection
	if ($::d_files and defined $client->lastSelection($pwd)) {
		msg("\$lastselection{$pwd} == ".$client->lastSelection($pwd)."\n");
	}

	if (defined($pwd)) {
		if (defined $client->lastSelection($pwd) && ($client->lastSelection($pwd) < $client->numberOfDirItems())) {
			$client->currentDirItem($client->lastSelection($pwd));
		} else {
			$client->currentDirItem(0);
		}

		$client->pwd($pwd);
	}

	if (defined $direction) {
		if ($direction eq 'left') {
			SliMP3::Animation::pushRight($client, @$oldlinesref, SliMP3::Display::curLines($client));
		} else {
			SliMP3::Animation::pushLeft($client, @$oldlinesref, SliMP3::Display::curLines($client));
		}
	}

	return 1;
}

#
# figure out the lines to be put up to display the directory
#
sub lines {
	my $client = shift;
	my ($line1, $line2, $overlay);
	my $showArrow = 0;
	my $showSong = 0;

	$line1 = line1($client);
	$line2 = line2($client);
	$overlay = overlay($client);

	return ($line1, $line2, undef, $overlay);
}

sub line1 {
	my $client = shift;
	my $line1;

	if ($client->pwd() eq "__playlists") {
		$line1 = string('SAVED_PLAYLISTS');
	} elsif ((defined $client->pwd() and $client->pwd() =~ m|^[/\\]?$|) || !SliMP3::Prefs::get("mp3dir")) {
		$line1 = string('MUSIC');
	} else {
		my $dir;
		# show only deepest two levels past the root dir
		$dir = $client->pwd();
		my @components = splitdir($dir);
		if ($components[0] eq "__playlists") { shift @components; };
		my $shortpwd;

		foreach my $path (@components) {
			if (SliMP3::Info::isURL($path)) { $path = SliMP3::Info::standardTitle($client, $path); }
		}

		if (@components > 1) {
			$shortpwd = join('/', grep {!/^\s*$/} splice(@components, -2));
		} else {
			$shortpwd = $components[0];
		}

		$line1 = $shortpwd;
	}

	if ($client->numberOfDirItems()) {
		$line1 .= sprintf(" (%d ".string('OUT_OF')." %s)", $client->currentDirItem + 1, $client->numberOfDirItems());
	}

	return $line1;
}

sub line2 {
	my $client = shift;
	my $line2;

	if (!$client->numberOfDirItems()) {
		$line2 = string('EMPTY');
	} else {
		my $fullpath;
		$fullpath = $client->dirItems($client->currentDirItem());

		# try and use the id3 tag
		$line2 = SliMP3::Info::standardTitle($client, $fullpath);
	}

	return $line2;
}

sub overlay {
	my $client = shift;
	my $fullpath;
	$fullpath = $client->dirItems($client->currentDirItem());

	if ($fullpath && SliMP3::Info::isSong($fullpath) || SliMP3::Info::isHTTPURL($fullpath)) {
		return SliMP3::VFD::symbol('notesymbol');
	}

	if ($fullpath && SliMP3::Info::isList($fullpath)) {
		return SliMP3::VFD::symbol('rightarrow');
	}

	return undef;
}

1;

__END__
