/***************************************************************************
    smb4kpreviewitem  -  A container for previews of a remote share
                             -------------------
    begin                : Mo Mai 28 2007
    copyright            : (C) 2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KPREVIEWITEM_H
#define SMB4KPREVIEWITEM_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qstring.h>
#include <qmap.h>
#include <qpair.h>

// application specific includes
#include "smb4knetworkitems.h"

typedef QPair<int, QString> ContentsItem;

/**
 * This class provides a container for the preview of the contents of a remote
 * SMB share.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class Smb4KPreviewItem
{
  public:
    /**
     * The default constructor.
     *
     * @param item              The share for that a preview should be collected.
     *
     * @param ip                The IP address of the host where the share
     *                          is located.
     *
     * @param path              The path for that the preview should be collected.
     */
    Smb4KPreviewItem( Smb4KShareItem *item, const QString &ip = QString::null, const QString &path = QString::null );

    /**
     * The empty constructor.
     */
    Smb4KPreviewItem() {}

    /**
     * The destructor.
     */
    ~Smb4KPreviewItem();

    /**
     * Returns the workgroup where the host is located
     */
    const QString &workgroup() const { return m_workgroup; }

    /**
     * Return the name of the host where the share is located.
     */
    const QString &host() const { return m_host; }

    /**
     * Return the name of the share that is to be previewed.
     */
    const QString &share() const { return m_share; }

    /**
     * With this function you can set the share name if this item
     * represents a homes share. In all other cases it will do just
     * nothing.
     *
     * @param share             The new share name
     */
    void setShare( const QString &share );

    /**
     * Return the path that is to be previewed.
     */
    const QString &path() const { return m_path; }

    /**
     * Returns the IP address of the host where the share
     * is located.
     */
    const QString &ip() const { return m_ip; }

    /**
     * Set the IP address of the host.
     *
     * @param ip                The IP address of the host.
     */
    void setIP( const QString &ip );

    /**
     * Set the path for which the preview.
     *
     * @param path              The path
     *
     * @note As soon as this function is used, the list of files and directories
     * will be cleared.
     */
    void setPath( const QString &path );

    /**
     * Returns the current location in the form //HOST/SHARE/PATH.
     * It can be used for displaying in a preview dialog or for checks.
     *
     * @returns                 The current location
     */
    const QString &location() const { return m_location; }

    /**
     * This enumeration is used for the contents. It determines if
     * an item is a file, a hidden file, a directory, or a hidden
     * directory.
     */
    enum Contents { File, HiddenFile, Directory, HiddenDirectory };

    /**
     * Returns the contents of the location.
     *
     * @returns a map of (hidden) files and (hidden) directories.
     */
    const QValueList<ContentsItem> &contents() const { return m_contents; }

    /**
     * Add a file or directory to the contents.
     *
     * @param item              A ContentsItem object. This is a QPair<int,QString>
     *                          with the integer being a value from the Contents
     *                          enumeration and the string being the full path of
     *                          the file or directory.
     *
     * @see Smb4KPreviewItem::setPath() or Smb4KPreviewItem::clearContents() for how
     * the list of files and directories is cleared.
     */
    void addContents( const ContentsItem &item );

    /**
     * Clears the contents.
     */
    void clearContents();

  private:
    /**
     * The workgroup of the host
     */
    QString m_workgroup;

    /**
     * The host's name
     */
    QString m_host;

    /**
     * The share name
     */
    QString m_share;

    /**
     * The IP address of the host
     */
    QString m_ip;

    /**
     * The path that has to be previewed.
     */
    QString m_path;

    /**
     * The current location
     */
    QString m_location;

    /**
     * This map stores the contents of the current
     * location.
     */
    QValueList<ContentsItem> m_contents;

    /**
     * This function checks if the IP address is valid, i.e. the
     * IP address is either IP v4 or IP v6. It returns either TRUE
     * or FALSE.
     *
     * @param ip            The IP address that's going to be checked.
     *
     * @returns TRUE if the IP address is valid and FALSE otherwise.
     */
    bool ipIsValid( const QString &ip );
};

#endif
