/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* $Id: fg-protocol-xml.c,v 1.1.1.1 2005/11/04 07:19:34 tkitame Exp $ 
 *
 * Copyright (c) 2005 VA Linux Systems Japan, K.K. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdlib.h>

#include <smtpguard/smtpguard.h>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#define CONTENT_LENGTH "content-length"

/**
 * fg_write_protocl:
 * @ioc: a GIOChannel
 * @xmlbuff: xmlstring
 * @len: length of xmlstring
 * @error: a GError object pointer.
 *
 * Generate protocol header and write it and xmlstring into GIOChannel.
 * */
void
fg_write_protocol (GIOChannel *ioc, const gchar *xmlbuff, gsize len,
		   GError **error)
{
	GIOStatus status;
	GError *we = NULL;
	gchar *header;

	g_return_if_fail (xmlbuff != NULL && len > 0);

	header = g_strdup_printf ("%s: %d\n\n", CONTENT_LENGTH, len);

	status = g_io_channel_write_chars (ioc, (const gchar *)header,
					   -1, NULL, &we);
	g_free (header);
	if (status != G_IO_STATUS_NORMAL && we != NULL) {
		g_propagate_error (error, we);
		return;
	}

	status = g_io_channel_write_chars (ioc,
				  (const gchar *)xmlbuff,
				  len, NULL, &we);
	if (status != G_IO_STATUS_NORMAL && we != NULL) {
		g_propagate_error (error, we);
		return;
	}

	status = g_io_channel_flush(ioc, &we);
	if (status != G_IO_STATUS_NORMAL && we != NULL) {
		g_propagate_error (error, we);
		return;
	}
}

/**
 * fg_protocl_error_get_xml:
 * @ioc: a GIOChannel
 * @report: error which will be reported to client
 * @error: a GError object pointer.
 *
 * Generate Error xml protocol string from GError.
 * */
gchar *
fg_protocol_error_get_xml (GError *report, GError **error)
{
	xmlNodePtr rootNode;
	xmlNodePtr nPtr;
	xmlDocPtr doc;
	xmlChar *xmlbuff;
	gchar tmpbuf[BUFSIZ];
	gint buffersize;

	g_return_val_if_fail (report != NULL, NULL);

	/* Make a XML Doc buffer */
	doc = xmlNewDoc (BAD_CAST "1.0");

	/* nPtr0 is Root */
	rootNode = xmlNewNode (NULL, BAD_CAST FG_PROTO_ELM_ROOT);
	xmlDocSetRootElement (doc, rootNode);

	nPtr = xmlNewChild (rootNode, NULL, BAD_CAST FG_PROTO_ELM_ERROR,
			    BAD_CAST report->message);
	xmlNewProp (nPtr, BAD_CAST FG_PROTO_ATTR_ERROR_DOMAIN,
		    BAD_CAST g_quark_to_string(report->domain));

	g_snprintf(tmpbuf, sizeof(tmpbuf), "%d", report->code);
	xmlNewProp (nPtr, BAD_CAST FG_PROTO_ATTR_ERROR_CODE,
		    BAD_CAST tmpbuf);

	xmlDocDumpFormatMemory (doc, &xmlbuff, &buffersize, 1);
	xmlFreeDoc (doc);
	if (buffersize == 0) {
		g_set_error (error, FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_GET_XML_FAILED,
			     "Could not generate XML." );
		return NULL;
	}

	return (gchar *)xmlbuff;
}

/**
 * fg_read_parse:
 * @ioc: a GIOChannel
 * @error: a GError object pointer.
 *
 * read protocol header from GIOChannel and parse it.
 * when no errors detected in header, fg_read_parse will continue read body
 * and make new allocated FGSmtpInfo Object from body.
 *
 * Return value: A new allocated FGSmtpInfo.
 * */
FGSmtpInfo *
fg_read_parse (GIOChannel *ioc, GError **error)
{
	gchar **header;
	gchar *buf = NULL;
	GError *re = NULL;
	gsize len = 0;
	FGSmtpInfo *fsi;
	GIOStatus status;

	/* first */
	for (;;) {
		status = g_io_channel_read_line (ioc, &buf, NULL, NULL, &re);
		if (re != NULL) { /* error in read_line () */
			g_propagate_error (error, re);
			return NULL;
		}
		if (status == G_IO_STATUS_EOF) { /* EOF */
			g_set_error (error, FG_SMTP_INFO_ERROR,
				     FG_SMTP_INFO_ERROR_PROTOCOL_ERROR,
				     "Invalid protocol: read unexpected EOF.");
			if (buf) g_free (buf);
			return NULL;
		} else if (status != G_IO_STATUS_NORMAL) { /* AGAIN or ERROR */
			g_set_error (error, FG_SMTP_INFO_ERROR,
				     FG_SMTP_INFO_ERROR_PROTOCOL_ERROR,
				     "Read error: could not read from socket.");
			return NULL;
		}
		if (buf && buf[0] == '\n') { /* Detect End of header */
			g_free (buf); buf = NULL;
			if (len == 0) { /* Length is required */
				g_set_error (error, FG_SMTP_INFO_ERROR,
					     FG_SMTP_INFO_ERROR_PROTOCOL_ERROR,
					     "Invalid protocol header: unexpected end of header.");
				return NULL;
			} else
				break;
		}
		if (buf) { /* parse line */
			buf = g_strchomp (buf);
			header = g_strsplit (buf, ":", 2);
			g_free (buf); buf = NULL;

			if (header == NULL || header[0] == NULL || header[1] == NULL) {
				g_set_error (error, FG_SMTP_INFO_ERROR,
					     FG_SMTP_INFO_ERROR_PROTOCOL_ERROR,
					     "Invalid protocol header format.");
				return NULL;
			}

			/* search Content-Length */
			if (g_ascii_strcasecmp (CONTENT_LENGTH, header[0]) == 0)
				len = atoi (header[1]);

			g_free (header[0]);
			g_free (header[1]);
		}
	}
	buf = g_malloc(len);
	if (!buf) {
		g_set_error (error, FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_ALLOC_FAILED,
			     "malloc: Cannot allocate memory.");
		return NULL;
	}
	status = g_io_channel_read_chars (ioc, buf, len, NULL, &re);
	if (re != NULL) {
		g_propagate_error (error, re);
		g_free (buf);
		return NULL;
	}
	buf[len - 1] = 0;
	fsi = fg_smtp_info_new_from_xml (FALSE, buf, len, &re);
	g_free (buf);
	if (re != NULL) {
		g_propagate_error (error, re);
		return NULL;
	}
	return fsi;
}
