/*
 * ----------------------------------------------------
 *
 * Emulation of the ARM CPU, Initialization and 
 * main loop
 *
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------
 */

#include <unistd.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <string.h>
#include <setjmp.h>
#include <coprocessor.h>
#include <arm9cpu.h>
#include <instructions.h>
#include <idecode.h>
#include <thumb_decode.h>
#include <bus.h>
#include <mmu.h>
#include <fio.h>
#include <xy_tree.h>
#include <time.h>
#include <sys/time.h>
#include <cycletimer.h>
#include <configfile.h>
#include <debug.h>
#include <mainloop_events.h>

#ifdef DEBUG
#define dbgprintf(x...) { if(unlikely(debugflags & DEBUG_INSTRUCTIONS)) { fprintf(stderr,x);fflush(stderr); } }
#else
#define dbgprintf(x...)
#endif

ARM9 gcpu;

uint32_t debugflags=0;
//uint32_t debugflags = DEBUG_LOG_PC;
uint32_t mmu_high_vectors=0;
uint32_t do_alignment_check=0;

/*
 * ----------------------------------------------
 * First the operations for gdb access to the
 * ARM system. GDB expects registers to be in
 * target byteorder
 * ----------------------------------------------
 */
static int 
gdb_getreg(void *clientData,uint8_t *data,int index,int maxlen) 
{
	uint32_t value;
	if(maxlen < 4)
		return -1;
	if(index<15) {
		value = ARM9_ReadReg(index);
	} else if(index==15) {
		value = GET_REG_PC - 4; /* ???? No clue if this is Ok */ 
		dbgprintf("Read reg PC %08x\n",value); 
	} else if(index < 25){
		value = 0;
	} else if (index == 25) {
		value = REG_CPSR;
	} else {
		return 0;
	}	
	if(MMU_Byteorder() == TARGET_BYTEORDER) {
		*((uint32_t*)data) = host32_to_target(value);
	} else {
		*((uint32_t*)data) = swap32(host32_to_target(value));
	}
	return 4;
}
static void 
gdb_setreg(void *clientData,const uint8_t *data,int index,int len) 
{
	uint32_t value;
	if(len != 4)
		return;
	if(MMU_Byteorder() == TARGET_BYTEORDER) {
		value = target32_to_host(*((uint32_t*)data));
	} else {
		value = swap32(target32_to_host(*((uint32_t*)data)));
	}
	if(index<16) {
		ARM9_WriteReg(value,index);
	} else if(index < 25){
		return;
	} else if (index == 25) {
		SET_REG_CPSR(value);
	} else {
		return;
	}	
	return; 
}

/*
 * ---------------------------------------------------------------------
 * get_bkpt_ins
 * 	returns the operation code of the breakpoint instruction.
 *	Needed by the debugger to insert breakpoints
 * ---------------------------------------------------------------------
 */
static void 
gdb_get_bkpt_ins(void *clientData,uint8_t *ins,uint64_t addr,int len) 
{
	if(len == 2) {
		uint16_t value = 0xbe00;
		if(MMU_Byteorder() == TARGET_BYTEORDER) {
			*((uint16_t*)ins) = host16_to_target(value);
		} else {
			*((uint16_t*)ins) = swap16(host16_to_target(value));
		}
	} else if(len == 4) {
		uint32_t value = 0xe1200070;
		if(MMU_Byteorder() == TARGET_BYTEORDER) {
			*((uint32_t*)ins) = host32_to_target(value);
		} else {
			*((uint32_t*)ins) = swap32(host32_to_target(value));
		}
	}
}

static int 
gdb_stop(void *clientData) 
{

	gcpu.dbg_mode = DBG_MODE_STOP;	
	ARM_PostRestartLoop();
	return -1;
}

static int
gdb_cont(void *clientData)
{
	//fprintf(stderr,"ARM cont\n");
	gcpu.dbg_mode = DBG_MODE_OFF;
	return 0;
}

static int 
gdb_step(void *clientData,uint32_t addr,int use_addr) 
{
	if(use_addr)  {
		SET_REG_PC(addr);		
	}
	gcpu.dbg_steps = 1;
	gcpu.dbg_mode = DBG_MODE_STEP;

	ARM_PostRestartLoop();
	return -1;
}

static int
gdb_get_status(void *clientData)
{
	if(gcpu.dbg_mode == DBG_MODE_STOPPED) {
		return SIGINT;
	} else {
		return -1;
	}
}

/*
 * --------------------------------------------------------------------
 * Gdb getmem backend 
 * --------------------------------------------------------------------
 */

static int 
gdb_getmem(void *clientData,uint8_t *data,uint64_t addr,uint32_t len) 
{
	int count;
	/* catch exceptions from MMU */
	count = 0;
	MMU_SetDebugMode(1);
	if(setjmp(gcpu.abort_jump)) {
		MMU_SetDebugMode(0);
		return count;
	}
	for(;len >= 4;len-=4,count+=4,data+=4) {
		uint32_t value = MMU_Read32(addr+count);
		if(MMU_Byteorder() == TARGET_BYTEORDER) {
			*((uint32_t*)data) = host32_to_target(value);
		} else {
			*((uint32_t*)data) = swap32(host32_to_target(value));
		}
	}
	for(;len > 2;len-=2,count+=2,data+=2) {
		uint16_t value = MMU_Read16(addr+count);
		if(MMU_Byteorder() == TARGET_BYTEORDER) {
			*((uint16_t*)data) = host16_to_target(value);
		} else {
			*((uint16_t*)data) = swap16(host16_to_target(value));
		}
	}
	for(;len > 0;len--,count++,data++) {
		uint8_t value = MMU_Read8(addr+count);
		*data = value;
	}
	MMU_SetDebugMode(0);
	return count;
}

static int 
gdb_setmem(void *clientData,const uint8_t *data,uint64_t addr,uint32_t len) 
{
	int count=0;
	//fprintf(stderr,"ARM readmem\n");
	/* catch exceptions from MMU */
	MMU_SetDebugMode(1);
	if(setjmp(gcpu.abort_jump)) {
		MMU_SetDebugMode(0);
		return count;
	}
	for(;len >= 4;len-=4,count+=4,data+=4) {
		uint32_t value = *((uint32_t*)data);
		if(MMU_Byteorder() == TARGET_BYTEORDER) {
			MMU_Write32(target32_to_host(value),addr+count);
		} else {
			MMU_Write32(swap32(target32_to_host(value)),addr+count);
		}
	}
	for(;len > 2;len-=2,count+=2,data+=2) {
		uint16_t value = *((uint32_t*)data);
		if(MMU_Byteorder() == TARGET_BYTEORDER) {
			MMU_Write16(target16_to_host(value),addr);
		} else {
			MMU_Write16(swap16(target16_to_host(value)),addr+count);
		}
	}
	for(;len > 0;len--,count++,data++) {
		uint8_t value  = *data;
		MMU_Write8(value,addr+count);
	}
	MMU_SetDebugMode(0);
	return count;
}


void 
ARM_set_reg_cpsr(uint32_t val) 
{
	uint32_t bank = val & 0x1f;
	gcpu.reg_cpsr=(val);
	if(gcpu.reg_bank != bank) {
		if(likely(gcpu.reg_bank != MODE_FIQ)) {
			uint32_t *rp;
			*gcpu.regSet[gcpu.reg_bank].r13 = gcpu.registers[13];
			*gcpu.regSet[gcpu.reg_bank].r14 = gcpu.registers[14];
			rp = gcpu.regSet[gcpu.reg_bank].spsr;
			if(rp)
				*rp = gcpu.registers[16];
		} else {
			gcpu.r8_fiq = gcpu.registers[8];
			gcpu.r9_fiq = gcpu.registers[9];
			gcpu.r10_fiq = gcpu.registers[10];
			gcpu.r11_fiq = gcpu.registers[11];
			gcpu.r12_fiq = gcpu.registers[12];
			gcpu.r13_fiq = gcpu.registers[13];
			gcpu.r14_fiq = gcpu.registers[14];
			gcpu.spsr_fiq = gcpu.registers[16];
		}
		if(likely(bank != MODE_FIQ)) {
			uint32_t *rp;
			gcpu.registers[13] = *gcpu.regSet[bank].r13;
			gcpu.registers[14] = *gcpu.regSet[bank].r14;
			rp = gcpu.regSet[bank].spsr;
			if(rp)
				gcpu.registers[16]=*rp;
		} else {
			gcpu.registers[8] = gcpu.r8_fiq;
			gcpu.registers[9] = gcpu.r9_fiq;
			gcpu.registers[10] = gcpu.r10_fiq;
			gcpu.registers[11] = gcpu.r11_fiq;
			gcpu.registers[12] = gcpu.r12_fiq;
			gcpu.registers[13] = gcpu.r13_fiq;
			gcpu.registers[14] = gcpu.r14_fiq;
			gcpu.registers[16] = gcpu.spsr_fiq;
		}
		gcpu.signaling_mode = gcpu.reg_bank = bank;
	}
	if(unlikely(val & FLAG_I)) {
		gcpu.signal_mask &= ~ARM_SIG_IRQ;	
	} else {
		gcpu.signal_mask |= ARM_SIG_IRQ;
	}
	if(unlikely(val & FLAG_F)) {
		gcpu.signal_mask &= ~ARM_SIG_FIQ; 
	} else {
		gcpu.signal_mask |= ARM_SIG_FIQ;
	}
	gcpu.signals = gcpu.signals_raw & gcpu.signal_mask;
	if(gcpu.signals) 
		mainloop_event_pending=1;
}

/* 
 * -------------------------------------------------------------
 * Setup register Pointers to the mode dependent register sets 
 * -------------------------------------------------------------
 */
static void
ARM9_InitRegs(ARM9 *arm) {
	int i;
	arm->regSet[MODE_USER].r0 = &arm->r0;	
	arm->regSet[MODE_FIQ].r0 = &arm->r0;	
	arm->regSet[MODE_IRQ].r0 = &arm->r0;	
	arm->regSet[MODE_SVC].r0 = &arm->r0;	
	arm->regSet[MODE_ABORT].r0 = &arm->r0;	
	arm->regSet[MODE_UNDEFINED].r0 = &arm->r0;	
	arm->regSet[MODE_SYSTEM].r0 = &arm->r0;	
	
	arm->regSet[MODE_USER].r1 = &arm->r1;	
	arm->regSet[MODE_FIQ].r1 = &arm->r1;	
	arm->regSet[MODE_IRQ].r1 = &arm->r1;	
	arm->regSet[MODE_SVC].r1 = &arm->r1;	
	arm->regSet[MODE_ABORT].r1 = &arm->r1;	
	arm->regSet[MODE_UNDEFINED].r1 = &arm->r1;	
	arm->regSet[MODE_SYSTEM].r1 = &arm->r1;	

	arm->regSet[MODE_USER].r2 = &arm->r2;	
	arm->regSet[MODE_FIQ].r2 = &arm->r2;	
	arm->regSet[MODE_IRQ].r2 = &arm->r2;	
	arm->regSet[MODE_SVC].r2 = &arm->r2;	
	arm->regSet[MODE_ABORT].r2 = &arm->r2;	
	arm->regSet[MODE_UNDEFINED].r2 = &arm->r2;	
	arm->regSet[MODE_SYSTEM].r2 = &arm->r2;	

	arm->regSet[MODE_USER].r3 = &arm->r3;	
	arm->regSet[MODE_FIQ].r3 = &arm->r3;	
	arm->regSet[MODE_IRQ].r3 = &arm->r3;
	arm->regSet[MODE_SVC].r3 = &arm->r3;
	arm->regSet[MODE_ABORT].r3 = &arm->r3;
	arm->regSet[MODE_UNDEFINED].r3 = &arm->r3;
	arm->regSet[MODE_SYSTEM].r3 = &arm->r3;

	arm->regSet[MODE_USER].r4 = &arm->r4;	
	arm->regSet[MODE_FIQ].r4 = &arm->r4;	
	arm->regSet[MODE_IRQ].r4 = &arm->r4;	
	arm->regSet[MODE_SVC].r4 = &arm->r4;	
	arm->regSet[MODE_ABORT].r4 = &arm->r4;	
	arm->regSet[MODE_UNDEFINED].r4 = &arm->r4;	
	arm->regSet[MODE_SYSTEM].r4 = &arm->r4;	

	arm->regSet[MODE_USER].r5 = &arm->r5;
	arm->regSet[MODE_FIQ].r5 = &arm->r5;
	arm->regSet[MODE_IRQ].r5 = &arm->r5;
	arm->regSet[MODE_SVC].r5 = &arm->r5;
	arm->regSet[MODE_ABORT].r5 = &arm->r5;
	arm->regSet[MODE_UNDEFINED].r5 = &arm->r5;
	arm->regSet[MODE_SYSTEM].r5 = &arm->r5;

	arm->regSet[MODE_USER].r6 = &arm->r6;
	arm->regSet[MODE_FIQ].r6 = &arm->r6;
	arm->regSet[MODE_IRQ].r6 = &arm->r6;
	arm->regSet[MODE_SVC].r6 = &arm->r6;
	arm->regSet[MODE_ABORT].r6 = &arm->r6;
	arm->regSet[MODE_UNDEFINED].r6 = &arm->r6;
	arm->regSet[MODE_SYSTEM].r6 = &arm->r6;

	arm->regSet[MODE_USER].r7 = &arm->r7;
	arm->regSet[MODE_FIQ].r7 = &arm->r7;
	arm->regSet[MODE_IRQ].r7 = &arm->r7;
	arm->regSet[MODE_SVC].r7 = &arm->r7;
	arm->regSet[MODE_ABORT].r7 = &arm->r7;
	arm->regSet[MODE_UNDEFINED].r7 = &arm->r7;
	arm->regSet[MODE_SYSTEM].r7 = &arm->r7;

	arm->regSet[MODE_USER].r8 = &arm->r8;
	arm->regSet[MODE_IRQ].r8 = &arm->r8;
	arm->regSet[MODE_SVC].r8 = &arm->r8;
	arm->regSet[MODE_ABORT].r8 = &arm->r8;
	arm->regSet[MODE_UNDEFINED].r8 = &arm->r8;
	arm->regSet[MODE_SYSTEM].r8 = &arm->r8;
	arm->regSet[MODE_FIQ].r8 = &arm->r8_fiq;	

	arm->regSet[MODE_USER].r9 = &arm->r9;
	arm->regSet[MODE_IRQ].r9 = &arm->r9;
	arm->regSet[MODE_SVC].r9 = &arm->r9;
	arm->regSet[MODE_ABORT].r9 = &arm->r9;
	arm->regSet[MODE_UNDEFINED].r9 = &arm->r9;
	arm->regSet[MODE_SYSTEM].r9 = &arm->r9;
	arm->regSet[MODE_FIQ].r9 = &arm->r9_fiq;	
	
	arm->regSet[MODE_USER].r10 = &arm->r10;
	arm->regSet[MODE_IRQ].r10 = &arm->r10;
	arm->regSet[MODE_SVC].r10 = &arm->r10;
	arm->regSet[MODE_ABORT].r10 = &arm->r10;
	arm->regSet[MODE_UNDEFINED].r10 = &arm->r10;
	arm->regSet[MODE_SYSTEM].r10 = &arm->r10;
	arm->regSet[MODE_FIQ].r10 = &arm->r10_fiq;
	
	arm->regSet[MODE_USER].r11 = &arm->r11;
	arm->regSet[MODE_IRQ].r11 = &arm->r11;
	arm->regSet[MODE_SVC].r11 = &arm->r11;
	arm->regSet[MODE_ABORT].r11 = &arm->r11;
	arm->regSet[MODE_UNDEFINED].r11 = &arm->r11;
	arm->regSet[MODE_SYSTEM].r11 = &arm->r11;
	arm->regSet[MODE_FIQ].r11 = &arm->r11_fiq;

	arm->regSet[MODE_USER].r12 = &arm->r12;
	arm->regSet[MODE_IRQ].r12 = &arm->r12;
	arm->regSet[MODE_SVC].r12 = &arm->r12;
	arm->regSet[MODE_ABORT].r12 = &arm->r12;
	arm->regSet[MODE_UNDEFINED].r12 = &arm->r12;
	arm->regSet[MODE_SYSTEM].r12 = &arm->r12;
	arm->regSet[MODE_FIQ].r12 = &arm->r12_fiq;
	
        arm->regSet[MODE_USER].r13 = &arm->r13;
        arm->regSet[MODE_SYSTEM].r13 = &arm->r13;
        arm->regSet[MODE_IRQ].r13 = &arm->r13_irq;
        arm->regSet[MODE_SVC].r13 = &arm->r13_svc;
        arm->regSet[MODE_ABORT].r13 = &arm->r13_abt;
        arm->regSet[MODE_UNDEFINED].r13 = &arm->r13_und;
        arm->regSet[MODE_FIQ].r13 = &arm->r13_fiq; 	
	
        arm->regSet[MODE_USER].r14 = &arm->r14;
        arm->regSet[MODE_SYSTEM].r14 = &arm->r14;
        arm->regSet[MODE_IRQ].r14 = &arm->r14_irq;
        arm->regSet[MODE_SVC].r14 = &arm->r14_svc;
        arm->regSet[MODE_ABORT].r14 = &arm->r14_abt;
        arm->regSet[MODE_UNDEFINED] .r14= &arm->r14_und;
        arm->regSet[MODE_FIQ].r14 = &arm->r14_fiq;

        arm->regSet[MODE_USER].pc=&REG_PC;
        arm->regSet[MODE_SYSTEM].pc=&REG_PC;
        arm->regSet[MODE_IRQ].pc=&REG_PC;
        arm->regSet[MODE_SVC].pc=&REG_PC;
        arm->regSet[MODE_ABORT].pc=&REG_PC;
        arm->regSet[MODE_UNDEFINED].pc=&REG_PC;
        arm->regSet[MODE_FIQ].pc=&REG_PC; 	

        arm->regSet[MODE_IRQ].spsr = &arm->spsr_irq;
        arm->regSet[MODE_SVC].spsr = &arm->spsr_svc;
        arm->regSet[MODE_ABORT].spsr = &arm->spsr_abt;
        arm->regSet[MODE_UNDEFINED].spsr = &arm->spsr_und;
        arm->regSet[MODE_FIQ].spsr = &arm->spsr_fiq; 	
	/* To avoid segfault if CPU is in illegal mode initialize with some nonsense */
	for(i=0;i<32;i++) {
		if(arm->regSet[i].r0==NULL) {
			arm->regSet[i].r0 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r1==NULL) {
			arm->regSet[i].r1 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r2==NULL) {
			arm->regSet[i].r2 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r3==NULL) {
			arm->regSet[i].r3 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r4==NULL) {
			arm->regSet[i].r4 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r5==NULL) {
			arm->regSet[i].r5 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r6==NULL) {
			arm->regSet[i].r6 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r7==NULL) {
			arm->regSet[i].r7 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r8==NULL) {
			arm->regSet[i].r8 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r9==NULL) {
			arm->regSet[i].r9 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r10==NULL) {
			arm->regSet[i].r10 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r11==NULL) {
			arm->regSet[i].r11 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r12==NULL) {
			arm->regSet[i].r12 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r13==NULL) {
			arm->regSet[i].r13 = &arm->reg_dummy;
		}
		if(arm->regSet[i].r14==NULL) {
			arm->regSet[i].r14 = &arm->reg_dummy;
		}
		if(arm->regSet[i].pc==NULL) {
			arm->regSet[i].pc = &arm->reg_dummy;
		}
	}
	fprintf(stderr,"- Register Pointers initialized\n");
}

CycleTimer htimer;

void 
hello_proc(void *cd) {
	struct timeval *tv_start;
	struct timeval tv_now;
	unsigned int time;
	gettimeofday(&tv_now,NULL);
	tv_start=&gcpu.starttime;
        time=(tv_now.tv_sec-tv_start->tv_sec)*1000 
		+ ((tv_now.tv_usec -tv_start->tv_usec)/1000);
	dbgprintf("\nSimulator speed %d kHz\n",(int) (CycleCounter_Get()/time));
#ifdef PROFILE
	exit(0);
#endif
//	CycleTimer_Add(&htimer,10000000000LL,hello_proc,NULL);
}

static int
irq_change(SigNode *node,int value,void *clientData)
{
	if((value == SIG_LOW) || (value == SIG_PULLDOWN)) {
		ARM_PostIrq();
	} else {
		ARM_UnPostIrq();
	}
	return 0;
}

static int
fiq_change(SigNode *node,int value,void *clientData)
{
	if((value == SIG_LOW) || (value == SIG_PULLDOWN)) {
		ARM_PostFiq();
	} else {
		ARM_UnPostFiq();
	}	
	return 0;
}

/*
 * -----------------------------------------------------
 * Create a new ARM9 CPU
 * -----------------------------------------------------
 */
ARM9* 
ARM9_New() {
	char *nodename = alloca(50);
	int32_t cpu_clock=200000000;
	ARM9 *arm=&gcpu;
	Config_ReadInt32(&cpu_clock,"global","cpu_clock");
	fprintf(stderr,"Creating ARM9 CPU with clock %d HZ\n",cpu_clock);
	memset(arm,0,sizeof(ARM9));
	IDecoder_New();
	ARM9_InitRegs(&gcpu); 
	InitInstructions();
	SET_REG_CPSR(MODE_SVC | FLAG_F | FLAG_I); 
	CycleTimers_Init(cpu_clock);
	CycleTimer_Add(&htimer,285000000,hello_proc,NULL);
	sprintf(nodename,"arm.irq");
	arm->irqNode = SigNode_New(nodename);
	sprintf(nodename,"arm.fiq");
	arm->fiqNode = SigNode_New(nodename);
	if(!arm->irqNode || !arm->fiqNode) {
		fprintf(stderr,"Can not create interrupt nodes for ARM CPU\n");
		exit(2);
	}
	arm->irqTrace = SigNode_Trace(arm->irqNode,irq_change,arm);
	arm->irqTrace = SigNode_Trace(arm->fiqNode,fiq_change,arm);
	arm->gdbops.getreg = gdb_getreg;
	arm->gdbops.setreg = gdb_setreg;
	arm->gdbops.stop = gdb_stop;
	arm->gdbops.step = gdb_step;
	arm->gdbops.cont = gdb_cont;
	arm->gdbops.get_status = gdb_get_status;
	arm->gdbops.getmem = gdb_getmem;
	arm->gdbops.setmem = gdb_setmem;
	arm->gdbops.step = gdb_step;
	arm->gdbops.get_bkpt_ins = gdb_get_bkpt_ins;
	arm->gserv = GdbServer_New(&arm->gdbops,arm);
	gcpu.signal_mask |= ARM_SIG_RESTART_LOOP;
	return arm;
}

void dump_stack() {
	int i;
	uint32_t sp,value;
	sp=ARM9_ReadReg(0xd);
	for(i=0;i<(72/4);i++) {
		if((i&3)==0) {
			dbgprintf("%08x:  ",sp+4*i);
		}
		value=MMU_Read32(sp+4*i);
		dbgprintf("%08x ",value);
		if((i&3)==3) {
			dbgprintf("\n");
		}
	}
	dbgprintf("\n");

}
UNUSED static void 
dump_regs() {
	int i;
	for(i=0;i<16;i++) {
		uint32_t value;
		value=ARM9_ReadReg(i);
		dbgprintf("R%02d: %08x    ",i,value);
		if((i&3)==3) {
			dbgprintf("\n");
		}
	}
	dbgprintf("CPSR: %08x \n",REG_CPSR);
	fflush(stderr);
}


/*
 * ----------------------------------------------------
 * Generate an exception and call the handler
 * ----------------------------------------------------
 */
void 
ARM9_Exception(int exception,int pc_offset) {
	uint32_t retaddr=(GET_REG_PC-8)+pc_offset;
	uint32_t old_cpsr=REG_CPSR;
	int new_mode = EX_TO_MODE(exception);
	uint32_t new_pc = EX_TO_PC(exception);

#if 0
	fprintf(stderr,"EXCEPTION %08x at %08x\n",exception,GET_REG_PC-8+pc_offset);
	exit(1);
	//ARM_Break();
#endif

	if(mmu_high_vectors) {
		new_pc|=0xffff0000;
	}
	/* Save CPSR to SPSR in the bank of the new mode */ 

	/* clear Thumb, set 0-4 to mode , disable IRQ */
	SET_REG_CPSR((REG_CPSR & (0xffffffe0 & ~FLAG_T)) | new_mode | FLAG_I);
	if(new_mode==MODE_FIQ) {
		  REG_CPSR=REG_CPSR | FLAG_F; 
	}
	ARM9_WriteReg(old_cpsr,REG_NR_SPSR); 
	REG_LR=retaddr;
	/* jump to exception vector address */
	SET_REG_PC(new_pc);
}


/*
 * -----------------------------------------------------------------------
 * Check Signals
 * 	Called after every instruction to check for queued IO-jobs
 *	and for the signals IRQ and FIQ
 * -----------------------------------------------------------------------
 */
static inline void 
CheckSignals() {
	if(unlikely(mainloop_event_pending)) {
		mainloop_event_pending = 0;
		if(mainloop_event_io) {
			FIO_HandleInput();		
		}
		if(gcpu.signals) {
			if(likely(gcpu.signals & ARM_SIG_IRQ)) {
				ARM9_Exception(EX_IRQ,8); 
			}
			if(unlikely(gcpu.signals & ARM_SIG_FIQ)) {
				ARM9_Exception(EX_FIQ,8); 
			}
			if(unlikely(gcpu.signals & ARM_SIG_RESTART_LOOP)) {
				if(likely(gcpu.dbg_mode== DBG_MODE_OFF)) { 
					ARM_RestartLoop();
				}  else if(gcpu.dbg_mode == DBG_MODE_STEP) {
					gcpu.dbg_steps--;
					if(gcpu.dbg_steps==0) {
						gcpu.dbg_mode = DBG_MODE_STOPPED;
						if(gcpu.gserv) {
							GdbServer_Signal(gcpu.gserv,SIGINT);
						}
						ARM_RestartLoop();
					} else {
						mainloop_event_pending = 1;
					}
				} else if(gcpu.dbg_mode == DBG_MODE_STOP) {
					gcpu.dbg_mode = DBG_MODE_STOPPED;
					if(gcpu.gserv) {
						GdbServer_Signal(gcpu.gserv,SIGINT);
					}
					ARM_RestartLoop();
				} else if(gcpu.dbg_mode == DBG_MODE_BREAK) {
					if(gcpu.gserv) {
						if(GdbServer_Signal(gcpu.gserv,SIGINT)>0) {
							gcpu.dbg_mode = DBG_MODE_STOPPED;
							ARM_RestartLoop();
#if 1
						}   else {
						// jk
							gcpu.dbg_mode = DBG_MODE_STOPPED;
							ARM_RestartLoop();
#endif
						}
					}
					ARM9_Exception(EX_PABT,8);
					gcpu.dbg_mode = DBG_MODE_OFF;
				} else {
					fprintf(stderr,"Unknown restart signal reason\n");
				}
			}
		}
	}
}

static inline void
debug_print_instruction(uint32_t icode) {
#ifdef DEBUG
		if(unlikely(debugflags)) {
			if(debugflags&1) {
				Instruction *instr;
				instr=InstructionFind(icode);
					dbgprintf("Instruction %08x, name %s at %08x\n",icode,instr->name,REG_PC);
					
			}
			if(debugflags&2) {
				dump_regs();
				dump_stack();
				//usleep(10000);
			}
		}
#endif
}

static uint32_t *circ_buf=NULL;
static unsigned int circ_wp=0;

#define LOGBUF_ENTRIES (1*1024)
static inline void 
circular_log(uint32_t data) {
	if(unlikely(debugflags & DEBUG_LOG_PC)) {
		if(circ_buf == NULL) {
			circ_buf = (uint32_t *)malloc(4*LOGBUF_ENTRIES);
			if(!circ_buf)  {
				fprintf(stderr,"Out of memory\n");
				exit(1);
			}
		}
		circ_buf[circ_wp] = data;	
		circ_wp = (circ_wp +1) % LOGBUF_ENTRIES;
	}
}

#if 0
static void
circbuf_dump(int n_entries) {
	uint32_t index = (circ_wp - n_entries) % LOGBUF_ENTRIES;
	fprintf(stderr,"dumping PC history\n");
	do {
		index = (index +1) % LOGBUF_ENTRIES;
		fprintf(stderr,"0x%08x\n",circ_buf[index]);
	} while (index != circ_wp);
}
#endif

static inline void 
log_uint32(uint32_t data) {
#if 0 
	circular_log(data);
	if(data==0xffff0400) {
		circbuf_dump(128);
		ARM_Break();
	}

	static int fd=-1;
	char *buf = (char*)&data;
	int count=0;
	int result;
	if(unlikely(debugflags & DEBUG_LOG_PC)) {
		if(fd<0) {
			fd=open("uint32_logfile",O_RDWR|O_CREAT,0644);
			if(fd<0) {
				perror("Can't open logfile");
				return;
			}
		}	
		while(count!=4) {
			result=write(fd,buf+count,4-count);
			if(result>0) { 
				count+=result;
			} else {
				perror("write failed");
				return;
			}
		}
	}
#endif
}

static void
Thumb_Loop() {
	ThumbInstructionProc *iproc;
	fprintf(stderr,"Entering Thumb loop\n");
	setjmp(gcpu.abort_jump);
	while(1) {
		ICODE=MMU_IFetch16(REG_PC);
		REG_PC+=2;
		iproc=ThumbInstructionProc_Find(ICODE);
		iproc();
		CycleCounter+=2;
		CycleTimers_Check();
	}
}

/*
 * ---------------------------------------------
 * The main loop for 32Bit instruction set
 * ---------------------------------------------
 */
static void
ARM9_Loop32() {
	InstructionProc *iproc;
	/* Exceptions use goto (longjmp) */
	setjmp(gcpu.abort_jump);
	while(1) {
		log_uint32(REG_PC);
		ICODE=MMU_IFetch(REG_PC);
		REG_PC+=4;
		iproc=InstructionProcFind(ICODE);
		debug_print_instruction(ICODE);
		iproc();
		CycleCounter+=6;
		CycleTimers_Check();
		CheckSignals();

		log_uint32(REG_PC);
		ICODE=MMU_IFetch(REG_PC);
		REG_PC+=4;
		iproc=InstructionProcFind(ICODE);
		debug_print_instruction(ICODE);
		iproc();
		CheckSignals();

		log_uint32(REG_PC);
		ICODE=MMU_IFetch(REG_PC);
		REG_PC+=4;
		iproc=InstructionProcFind(ICODE);
		debug_print_instruction(ICODE);
		iproc();
		CheckSignals();
	}
}

void
ARM9_Run() {
	uint32_t addr=0;
	uint32_t gdbwait;
	if(Config_ReadUInt32(&addr,"global","start_address")<0) {
		addr=0;
	}
	if(Config_ReadUInt32(&gdbwait,"global","gdbwait")<0) {
		gdbwait=0;
	}
	if(gdbwait) {
		fprintf(stderr,"CPU is waiting for gdb connection at %08x\n",addr);
		gcpu.dbg_mode = DBG_MODE_STOPPED;
	} else {
		fprintf(stderr,"Starting CPU at %08x\n",addr);
	}
	gettimeofday(&gcpu.starttime,NULL);
	REG_PC=addr;
	/* A long jump to this label redecides which main loop is used  */
	setjmp(gcpu.restart_loop_jump);
	gcpu.signals &= ~ARM_SIG_RESTART_LOOP;
	while(1) {
		if(gcpu.dbg_mode == DBG_MODE_STOPPED) {
			struct timespec tout;
                        tout.tv_nsec=0;
                        tout.tv_sec=10000;
                        FIO_WaitEventTimeout(&tout);
		} else {
			if(REG_CPSR & FLAG_T) {
				Thumb_Loop();
			} else {
				ARM9_Loop32(); 
			}
		}
	}
}
