/*
 * ----------------------------------------------------
 *
 * PowerPC Instruction Set 
 * (C) 2005 Jochen Karrer 
 *   Author: Jochen Karrer
 *
 * With some code from PearPC
 *
 * ----------------------------------------------------
 */
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <debug.h>
#include <instructions_ppc.h>
#include <byteorder.h>
#include <cpu_ppc.h>
#include <mmu_ppc.h>

#ifdef DEBUG
#define dprintf(x...) { if(unlikely(debugflags&DEBUG_INSTRUCTIONS)) { fprintf(stderr,x); fflush(stderr); } }
#else
#define dprintf(x...)
#endif

static inline int
count_leading_zeros(uint32_t value) {
	int i;
	for(i=0;i<32;i++) {
		if(value & 0x80000000) {
			return i;
		}
		value=value<<1;
	}
	return 32;
}

static inline void
update_cr0(uint32_t result) {
	CR=CR &~(CR_LT | CR_GT | CR_EQ | CR_SO);
	if(!result) {
		CR=CR|CR_EQ;
	} else if (ISNEG(result)) {
		CR=CR|CR_LT;
	} else {
		CR=CR|CR_GT;
	}
	if (XER & XER_SO) { 
		CR |= CR_SO;
	}
}
/*
 * ---------------------------------------
 * Carry and overflow Flag Helper 
 * functions for add and sub
 * ---------------------------------------
 */
static inline uint32_t
sub_carry(uint32_t op1,uint32_t op2,uint32_t result) {
        if( ((ISNEG(op1) && ISNOTNEG(op2))
          || (ISNEG(op1) && ISNOTNEG(result))
          || (ISNOTNEG(op2) && ISNOTNEG(result)))) {
                        return 1;
        } else {
                return 0;
        }
}

static inline uint32_t
add_carry(uint32_t op1,uint32_t op2,uint32_t result) {
        if( ((ISNEG(op1) && ISNEG(op2))
          || (ISNEG(op1) && ISNOTNEG(result))
          || (ISNEG(op2) && ISNOTNEG(result)))) {
                        return 1;
        } else {
                return 0;
        }
}
static inline uint32_t
sub_overflow(uint32_t op1,uint32_t op2,uint32_t result) {
        if ((ISNEG (op1) && ISNOTNEG (op2) && ISNOTNEG (result))
          || (ISNOTNEG (op1) && ISNEG (op2) && ISNEG (result))) {
                return 1;
        } else {
                return 0;
        }
}

static inline uint32_t
add_overflow(uint32_t op1,uint32_t op2,uint32_t result) {
        if ((ISNEG (op1) && ISNEG (op2) && ISNOTNEG (result))
          || (ISNOTNEG (op1) && ISNOTNEG (op2) && ISNEG (result))) {
                return 1;
        } else {
                return 0;
        }
}


void 
i_tdi(uint32_t icode) {
	fprintf(stderr,"instr i_tdi(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------------------------
 * twi UISA Form D 
 *	Trap word Immediate
 * ------------------------------------------------------------
 */
void 
i_twi(uint32_t icode) {
	int to = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t imm = icode & 0xffff;
	int32_t eimm = imm;
	int32_t A = GPR(a);
	if(((int32_t)A < (int32_t)eimm) && (to & (1<<0))) {
		// Exception
	}
	if(((int32_t)A > (int32_t)eimm) && (to & (1<<1))) {
		// Exception
	}
	if((A==eimm) && (to & (1<<2))) {
		// Exception
	}
	if(((uint32_t)A < (uint32_t)eimm) && (to & (1<<3))) {
		// Exception
	}
	if(((uint32_t)A > (uint32_t)eimm) && (to & (1<<4))) {
		// Exception
	}
	fprintf(stderr,"instr i_twi(%08x) not implemented\n",icode);
}

/* 
 * -------------------------------------
 * mulli UISA Form D 
 *	Multiply Low Immediate
 * -------------------------------------
 */
void 
i_mulli(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;	
	int rc = icode & 1;
	uint32_t result;
	int16_t simm = (icode & 0xffff);
	result = GPR(d) = (int64_t)GPR(a) * (int64_t)simm;
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_mulli(%08x)\n",icode);
}

/*
 * ------------------------------------------------
 * subfic UISA Form D
 * ------------------------------------------------
 */
void 
i_subfic(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;	
	int16_t imm = icode & 0xffff;
	uint32_t result;
	uint32_t op2 = GPR(a);
	result = GPR(d) = imm - op2; 
	if(sub_carry(imm,op2,result)) {
		XER = XER | XER_CA;
	} else {
		XER = XER & ~XER_CA;
	}
	dprintf("instr i_subfic(%08x)\n",icode);
}

/*
 * ------------------
 * cmpli UISA Form D
 * v1
 * ------------------
 */
void 
i_cmpli(uint32_t icode) {
	uint32_t crfd=7-((icode>>23)&7);
	uint32_t a=(icode>>16)&0x1f;
	uint16_t uimm = icode &0xffff;
	uint32_t Ra = GPR(a);
	int L=(icode>>21)&1;
	uint32_t c;
	if(L) {
		fprintf(stderr,"Invalid instruction format for cmpli\n");
		return;
	}
	if(Ra < uimm) {
		c=8;
	} else if(Ra > uimm) {
		c=4;
	} else {
		c=2;
	}
	if (XER & XER_SO)  {
		c |= 1;	
	}
	CR &= 0xffffffff ^ (0xf<<(crfd<<2));
	CR |= c<<(crfd<<2);
	dprintf("instr i_cmpli(%08x)\n",icode);
}

/*
 * ----------------------
 * cmpi UISA Form D 
 * v1
 * ----------------------
 */

void 
i_cmpi(uint32_t icode) {
	uint32_t crfd=7-((icode>>23)&7);
	uint32_t a=(icode>>16)&0x1f;
	int16_t simm = icode &0xffff;
	int32_t Ra = GPR(a);
	int L=(icode>>21)&1;
	uint32_t c;
	if(L) {
		fprintf(stderr,"invalid instruction format\n");
	}
	if(Ra<simm) {
		c=8;
	} else if(Ra>simm) {
		c=4;
	} else {
		c=2;
	}
	if (XER & XER_SO)  {
		c |= 1;	
	}
	CR &= 0xffffffff ^ (0xf<<(crfd<<2));
	CR |= c<<(crfd<<2);
	dprintf("instr i_cmpi(%08x)\n",icode);
}

/*
 * -----------------------------
 * addic UISA form D
 * v1
 * -----------------------------
 */
void 
i_addic(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f; 
	int16_t simm = (icode&0xffff);
	uint32_t op1;
	uint32_t result;
	op1=GPR(a);
	GPR(d) = result = op1 + simm;
	if(add_carry(op1,simm,result)) {
		XER=XER | XER_CA;
	} else {
		XER=XER & ~XER_CA;
	}
	dprintf("instr i_addic(%08x)\n",icode);
}

/* 
 * --------------------------------
 * addic. UISA Form D 
 * v1
 * --------------------------------
 */
void 
i_addic_(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f; 
	int16_t simm = (icode&0xffff);
	uint32_t op1;
	uint32_t result;
	op1=GPR(a);
	GPR(d) = result = op1+simm;
	if(add_carry(op1,simm,result)) {
		XER=XER | XER_CA;
	} else {
		XER=XER & ~XER_CA;
	}
	update_cr0(result);
	dprintf("instr i_addic_(%08x)\n",icode);
}
/*
 * -------------------------
 * addi UISA form D
 * v1
 * -------------------------
 */
void 
i_addi(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f; 
	int16_t simm = (icode&0xffff);
	if(a==0) {
		GPR(d) = simm;
	} else {
		GPR(d) = GPR(a)+simm;
	}
	dprintf("instr i_addi(%08x)\n",icode);
}

/*
 * --------------------------------
 * addis UISA Form D
 * v1
 * --------------------------------
 */
void 
i_addis(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f; 
	int32_t simm = (icode&0xffff)<<16;
	if(a==0) {
		GPR(d) = simm;
	} else {
		GPR(d) = GPR(a)+simm;
	}
	dprintf("instr i_addis(%08x)\n",icode);
}

/*
 * -------------------------------------
 * bcx UISA Form D
 * v1
 * -------------------------------------
 */
void 
i_bcx(uint32_t icode) {
	uint32_t bo = (icode>>21)&0x1f;
	uint32_t bi = (icode>>16)&0x1f;
	int16_t bd = (icode)&0xfffc;
	int aa = (icode&2);
	int lk = (icode&1);
	int ctr_ok;
	uint32_t cond_ok;
	if(!(bo & (1<<(4-2) ))) {
		CTR = CTR - 1;
	}
	ctr_ok = ((bo>>2) & 1) | ((CTR!=0) != ((bo>>(4-3))&1));
	cond_ok = (bo & (1<<(4-0))) | (((CR>>(31-bi))&1)==((bo>>(4-1))&1));	
	if(ctr_ok && cond_ok) {
		if(lk) {
			LR=CIA+4; 		
		}
		if(aa) {
			NIA=bd;
		} else  {
			NIA=CIA+bd;
		}
	}
	dprintf("instr i_bcx(%08x)\n",icode);
}

/*
 * ----------------------------------------
 * sc
 * Systemcall 
 * not complete
 * ----------------------------------------
 */

void 
i_sc(uint32_t icode) {
	uint32_t mask = 0x87c0ff73;
	SRR0 = CIA + 4;
	SRR1 = SRR1 & ~0x783f0000;
	SRR1 = (SRR1 & ~mask ) | ( MSR & mask);
//	MSR = bla;
//	NIA = bla;
//	Exception(SYSCALL);
	fprintf(stderr,"instr i_sc(%08x) not implemented\n",icode);
}

/*
 * ----------------------------------
 * bx  UISA Form I
 * ----------------------------------
 */
void 
i_bx(uint32_t icode) {
	int32_t li;
	int aa = icode & (1<<1);
	int lk = icode & 1;
	if(icode&0x02000000) {
                li = (icode & 0x03fffffc) | 0xfc000000;
        } else {
		li = icode&  0x03fffffc;
        }
	if(lk) {
		LR=CIA+4; 		
	}
	if(aa) {
		NIA=li;
	} else  {
		NIA=CIA+li;
	}
	dprintf("instr i_bx(%08x)\n",icode);
}

/*
 * ---------------------------------------
 * mcrf UISA Form XL
 * Move condition register field
 * v1
 * ---------------------------------------
 */
void 
i_mcrf(uint32_t icode) {
	uint32_t mask;
	int crfd = 7-((icode>>23)&7); // shit PPC bit counting 
	int crfs = 7-((icode>>18)&7);
	uint32_t setbits;
	setbits=((CR>>(4*crfs))&0xf) << (4*crfd);
	mask = ~(0xf << (4*crfd));
	CR = (CR & mask) | setbits;
	dprintf("instr i_mcrf(%08x)\n",icode);
}

/*
 * ------------------------------------------
 * bclrx UISA Form XL
 * v1
 * ------------------------------------------
 */

void 
i_bclrx(uint32_t icode) {
	uint32_t bo = (icode>>21)&0x1f;
	uint32_t bi = (icode>>16)&0x1f;
	int lk = icode&1;
	int ctr_ok;
	int cond_ok;
	if(!((bo>>(4-2))&1)) {
		CTR-=1;
	}
	ctr_ok = ((bo>>(4-2))&1) | ((CTR !=0) ^ ((bo>>(4-3))&1));
	cond_ok = (bo& (1<<(4-0))) | (((CR>>(31-bi))&1)==((bo>>(4-1))&1));
	fprintf(stderr,"from CIA %08x \n",CIA);
	if(ctr_ok & cond_ok) {
		uint32_t tmp_lr = LR;
		if(lk) {
			LR=CIA+4;
		}
		NIA=tmp_lr & 0xfffffffc;
	}
	fprintf(stderr,"instr i_bclrx(%08x)  to NIA %08x\n",icode,NIA);
}

/*
 * -------------------------------------------
 * crnor  UISA Form XL 
 * v1
 * -------------------------------------------
 */
void 
i_crnor(uint32_t icode) {
	uint32_t crbD=31-((icode>>21)&0x1f);
	uint32_t crbA=31-((icode>>16)&0x1f);
	uint32_t crbB=31-((icode>>11)&0x1f);
	if((CR&(1<<crbA)) || (CR&(1<<crbB))) {
		CR=CR & ~(1<<crbD);
	} else {
		CR=CR | (1<<crbD);
	}
	dprintf("instr i_crnor(%08x)\n",icode);
}

/* 
 * -----------------------------------
 * rfi OEA Supervisor Form XL 
 * return from interrupt
 * incomplete v1
 * -----------------------------------
 */
void 
i_rfi(uint32_t icode) {
	uint32_t mask = 0x87c0ff73;
#if 0
	if(!supervisor) {
		fprintf(stderr,"Mist\n");
		Exception();
	}
#endif 
	PpcSetMsr((MSR & ~(mask|(1<<18))) | (SRR1 & mask));
	NIA = SRR0 & 0xfffffffc;	
	fprintf(stderr,"instr i_rfi(%08x) incomplete\n",icode);
}

/*
 * ------------------------------
 * crandc UISA Form XL
 * v1
 * ------------------------------
 */
void 
i_crandc(uint32_t icode) {
	uint32_t crbD=31-((icode>>21)&0x1f); 
	uint32_t crbA=31-((icode>>16)&0x1f);
	uint32_t crbB=31-((icode>>11)&0x1f);
	if((CR&(1<<crbA)) && !(CR&(1<<crbB))) {
		CR=CR | (1<<crbD);
	} else {
		CR=CR & ~(1<<crbD);
	}
	dprintf("instr i_crandc(%08x)\n",icode);
}

/* 
 * ------------------------------------------------------------------------------
 * isync VEA Form XL
 * 	Instruction Synchronize
 *	Currently does nothing because Instruction pipeline is not implemented
 * ------------------------------------------------------------------------------
 */
void 
i_isync(uint32_t icode) {
	dprintf("instr i_isync(%08x) ignored\n",icode);
}

/*
 * ----------------------------------------
 * crxor UISA Form XL
 * v1
 * ----------------------------------------
 */
void 
i_crxor(uint32_t icode) {
	uint32_t crbD=31-((icode>>21)&0x1f); 
	uint32_t crbA=31-((icode>>16)&0x1f);
	uint32_t crbB=31-((icode>>11)&0x1f);
	if((((CR>>crbA)&1) != ((CR>>crbB)&1))) {
		CR=CR | (1<<crbD);
	} else {
		CR=CR & ~(1<<crbD);
	}
	dprintf("instr i_crxor(%08x)\n",icode);
}

/*
 * ------------------------------
 * crnand UISA Form XL
 * v1
 * ------------------------------
 */
void 
i_crnand(uint32_t icode) {
	uint32_t crbD=31-((icode>>21)&0x1f); 
	uint32_t crbA=31-((icode>>16)&0x1f);
	uint32_t crbB=31-((icode>>11)&0x1f);
	if((CR&(1<<crbA)) && (CR&(1<<crbB))) {
		CR=CR & ~(1<<crbD);
	} else {
		CR=CR | (1<<crbD);
	}
	dprintf("instr i_crnand(%08x)\n",icode);
}

/*
 * ---------------------------
 * crand UISA Form XL
 * v1
 * ---------------------------
 */
void 
i_crand(uint32_t icode) {
	uint32_t crbD=31-((icode>>21)&0x1f); 
	uint32_t crbA=31-((icode>>16)&0x1f);
	uint32_t crbB=31-((icode>>11)&0x1f);
	if((CR&(1<<crbA)) && (CR&(1<<crbB))) {
		CR=CR | (1<<crbD);
	} else {
		CR=CR & ~(1<<crbD);
	}
	dprintf("instr i_crand(%08x)\n",icode);
}

/*
 * ---------------------------------------
 * creqv UISA Form XL
 * v1
 * ---------------------------------------
 */
void 
i_creqv(uint32_t icode) {
	uint32_t crbD=31-((icode>>21)&0x1f); 
	uint32_t crbA=31-((icode>>16)&0x1f);
	uint32_t crbB=31-((icode>>11)&0x1f);
	if(((CR>>crbA)&1) == ((CR>>crbB)&1)) {
		CR=CR | (1<<crbD);
	} else {
		CR=CR & ~(1<<crbD);
	}
	dprintf("instr i_creqv(%08x)\n",icode);
}

/*
 * --------------------
 * crorc UISA Form XL
 * v1
 * --------------------
 */
void 
i_crorc(uint32_t icode) {
	uint32_t crbD=31-((icode>>21)&0x1f); 
	uint32_t crbA=31-((icode>>16)&0x1f);
	uint32_t crbB=31-((icode>>11)&0x1f);
	if((CR&(1<<crbA)) ||  !(CR&(1<<crbB))) {
		CR=CR | (1<<crbD);
	} else {
		CR=CR & ~(1<<crbD);
	}
	dprintf("instr i_crorc(%08x)\n",icode);
}

/*
 * --------------------------
 * cror UISA Form XL
 * v1
 * --------------------------
 */
void 
i_cror(uint32_t icode) {
	uint32_t crbD=31-((icode>>21)&0x1f); 
	uint32_t crbA=31-((icode>>16)&0x1f);
	uint32_t crbB=31-((icode>>11)&0x1f);
	if((CR&(1<<crbA)) || (CR&(1<<crbB))) {
		CR=CR | (1<<crbD);
	} else {
		CR=CR & ~(1<<crbD);
	}
	dprintf("instr i_cror(%08x)\n",icode);
}

/*
 *--------------------------------
 * bcctrx UISA Form XL
 * 	Jump to count register
 * v1
 *--------------------------------
 */
void 
i_bcctrx(uint32_t icode) {
	uint32_t bo=(icode>>21)&0x1f;
	uint32_t bi=(icode>>16)&0x1f;
	int lk = icode &1;
	int cond_ok;
	cond_ok = (bo & (1<<4)) | (((CR>>(31-bi))&1)==((bo>>(4-1))&1));
	if(cond_ok) {
		if(lk) {
			LR=CIA+4;
		}
		NIA = CTR & 0xfffffffc;	
	}
	dprintf("instr i_bcctrx(%08x)\n",icode);
}

/* stolen from pearpc */
static inline uint32_t
mix_mask(int32_t mb,int32_t me) {
	uint32_t mask;
	if (mb <= me) {
                if (me-mb == 31) {
                        mask = 0xffffffff;
                } else {
                        mask = ((1<<(me-mb+1))-1)<<(31-me);
                }
        } else {
		int rot = 31-me;
		uint32_t w = (1<<(32-mb+me+1))-1;
		if(rot) {
			mask = (w>>rot)	| (w<<(32-rot));
		} else {
			mask = w;
		}
        }
	return mask;
}

/*
 * -----------------------------------------------
 * rlwimix UISA Form M
 * 	Rotate left immediate then mask insert
 * mix_mask not verified
 * -----------------------------------------------
 */
void 
i_rlwimix(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int sl = (icode>>11) & 0x1f;
	int mb = (icode>>6) & 0x1f;
	int me = (icode>>1) & 0x1f;
	int rc = icode & 1;
	uint32_t r;
	uint32_t mask;
	uint32_t result;
	r=(GPR(s)<<sl) | (GPR(s) >> (32-sl));
	mask = mix_mask(mb,me);
	result = GPR(a) = (r & mask) | (GPR(a) & ~mask); 			
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_rlwimix(%08x) not tested\n",icode);
}

/*
 * -------------------------------------------------------
 * rlwinmx 
 * 	Rotate left word immediate then and with mask
 * mix_mask not verified
 * -------------------------------------------------------
 */
void 
i_rlwinmx(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
        int a = (icode>>16) & 0x1f;
        int sl = (icode>>11) & 0x1f;
        int mb = (icode>>6) & 0x1f;
        int me = (icode>>1) & 0x1f;
        int rc = icode & 1;
        uint32_t r;
	uint32_t mask;
	uint32_t result;
        r=(GPR(s)<<sl) | (GPR(s) >> (32-sl));
        mask = mix_mask(mb,me);
        result = GPR(a) = (r & mask); 
        if(rc) {
		update_cr0(result);
        }
	dprintf("instr i_rlwinmx(%08x)\n",icode);
}

void 
i_rlwnmx(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
        int a = (icode>>16) & 0x1f;
        int b = (icode>>11) & 0x1f;
	int sl;
        int mb = (icode>>6) & 0x1f;
        int me = (icode>>1) & 0x1f;
        int rc = icode & 1;
	uint32_t r;
	uint32_t mask;
	uint32_t result;
	sl = GPR(b) & 0x1f;
        r=(GPR(s)<<sl) | (GPR(s) >> (32-sl));
        mask = mix_mask(mb,me);
        result = GPR(a) = (r & mask); 
        if(rc) {
		update_cr0(result);
        }
	fprintf(stderr,"instr i_rlwnmx(%08x) not tested\n",icode);
}

/* 
 * -----------------------------
 * ori UISA Form D 
 * OR immediate
 * v1	
 * -----------------------------
 */
void 
i_ori(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	uint16_t uimm = icode;
	GPR(a) = GPR(s) | uimm;
	/* no registers else are changed */
}
/*
 * ------------------------------------------
 * oris UISA Form D
 * OR immediate shifted
 * v1
 * ------------------------------------------
 */
void 
i_oris(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	uint32_t uimm = (icode & 0xffff)<<16;
	GPR(a) = GPR(s) | uimm;
	/* no registers else are changed */
}

/*
 * ---------------------------------------------------------------------
 * xori  UISA Form D
 * 	XOR immediate
 * v1
 * ---------------------------------------------------------------------
 */
void 
i_xori(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	uint16_t uimm = (icode & 0xffff);
	GPR(a) = GPR(s) ^ uimm;
	dprintf("instr i_xori(%08x)\n",icode);
}

/* 
 * -------------------------------------------------------------------
 * xoris UISA Form D
 *	XOR Immediate shifted
 * v1
 * -------------------------------------------------------------------
 */
void 
i_xoris(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	uint32_t uimm = (icode & 0xffff)<<16;
	GPR(a) = GPR(s) ^ uimm;
	/* no registers else are changed */
	dprintf("instr i_xoris(%08x)\n",icode);
}

/*
 * -----------------------
 * andi. UISA Form D 
 * v1
 * -----------------------
 */
void 
i_andi_(uint32_t icode) {
	uint32_t result;
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	uint16_t uimm = icode&0xffff;
	result = GPR(s) & uimm;
	update_cr0(result);
	GPR(a) = result;
	dprintf("instr i_andi(%08x)\n",icode);
}

/*
 * ----------------------------
 * andis. UISA Form D
 * v1
 * ----------------------------
 */
void 
i_andis_(uint32_t icode) {
	uint32_t result;
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	uint32_t uimm = (icode&0xffff)<<16;
	result = GPR(s) & uimm;
	GPR(a) = result;
	update_cr0(result);
	dprintf("instr i_andis(%08x)\n",icode);
}

void 
i_rldiclx(uint32_t icode) {
	fprintf(stderr,"instr i_rldclx(%08x) not implemented\n",icode);
}

void 
i_rldicrx(uint32_t icode) {
	fprintf(stderr,"instr i_rldicrx(%08x) not implemented\n",icode);
}

void 
i_rldicx(uint32_t icode) {
	fprintf(stderr,"instr i_rldicx(%08x) not implemented\n",icode);
}

void 
i_rldimix(uint32_t icode) {
	fprintf(stderr,"instr i_rldimix(%08x) not implemented\n",icode);
}

void 
i_rldclx(uint32_t icode) {
	fprintf(stderr,"instr i_rldclx(%08x) not implemented\n",icode);
}

void 
i_rldcrx(uint32_t icode) {
	fprintf(stderr,"instr i_rldcrx(%08x) not implemented\n",icode);
}

/* 
 * -----------------------------------------------------------
 * cmp UISA Form X
 * 866UM says on page 127 that the L bit is ignored
 * v1
 * -----------------------------------------------------------
 */
void 
i_cmp(uint32_t icode) {
	uint32_t crfd=7-((icode>>23)&7);
	uint32_t a=(icode>>16)&0x1f;
	uint32_t b=(icode>>11)&0x1f;
	int32_t Ra = GPR(a);
	int32_t Rb = GPR(b);
	uint32_t c;
#if 0
	int L=(icode>>21)&1;
	if(L) {
		fprintf(stderr,"Invalid instruction format for cmp icode %08x at %08x\n",icode,CIA);
	}
#endif
	if(Ra<Rb) {
		c=8;
	} else if(Ra>Rb) {
		c=4;
	} else {
		c=2;
	}
	if (XER & XER_SO)  {
		c |= 1;	
	}
	CR &= 0xffffffff ^ (0xf<<(crfd<<2));
	CR |= c<<(crfd<<2);
	dprintf("instr i_cmp(%08x)\n",icode);
}

/*
 * --------------------------------------------------------
 * tw UISA Form X
 * 	Trap Word
 * --------------------------------------------------------
 */
void 
i_tw(uint32_t icode) {
	int to = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	int32_t A = GPR(a);
	int32_t B = GPR(b);
	if(((int32_t)A < (int32_t)B) && (to & (1<<0))) {
		// Exception
	}
	if(((int32_t)A > (int32_t)B) && (to & (1<<1))) {
		// Exception
	}
	if((A==B) && (to & (1<<2))) {
		// Exception
	}
	if(((uint32_t)A < (uint32_t)B) && (to & (1<<3))) {
		// Exception
	}
	if(((uint32_t)A > (uint32_t)B) && (to & (1<<4))) {
		// Exception
	}
	fprintf(stderr,"instr i_tw(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------------------------
 * subfcx
 * 	Subtract from Carrying
 * ------------------------------------------------------------
 */
void 
i_subfcx(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	int rc = icode & 1;
	int oe = icode & (1<<10);
	uint32_t result;
	uint32_t op1 = GPR(b),op2 = GPR(a);
	GPR(d) = result = op1 - op2;
	if(sub_carry(op1,op2,result)) {
		XER = XER | XER_CA;
	} else {
		XER = XER & ~XER_CA;
	}
	if(oe) {
		if(sub_overflow(op1,op2,result)) {
			XER = XER | XER_OV | XER_SO;
		} else {
			XER = XER & ~XER_OV;
		}
	}
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_subfcx(%08x)\n",icode);
}

void 
i_mulhdux(uint32_t icode) {
	fprintf(stderr,"instr i_mulhdux(%08x) not implemented\n",icode);
}

/*
 * --------------------------------------
 * ADDCx UISA Form XO
 * v1
 * --------------------------------------
 */ 
void 
i_addcx(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	int oe = icode&(1<<10);
	int rc = icode&1;
	uint32_t result,op1,op2;
	op1=GPR(a); op2 = GPR(b); 		
	GPR(d) = result = op1+op2;
	if(add_carry(op1,op2,result)) {
		XER = XER | XER_CA;
	} else {
		XER = XER & ~XER_CA;
	}
	if(oe) {
		if(add_overflow( op1,op2,result)) {
			XER |= XER_SO | XER_OV;
    		} else {
			XER &= ~XER_OV;
    		}
	}
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_addcx(%08x)\n",icode);
}

/*
 * ---------------------------------------------------
 * mulhwux UISA Form XO
 * 	Multiply high word unsigned 
 * v1
 * ---------------------------------------------------
 */
void 
i_mulhwux(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;	
	int b = (icode>>11)&0x1f;	
	int rc = icode & 1;
	uint32_t result;
	uint64_t prod = (uint64_t)GPR(a) * (uint64_t)GPR(b);
	result = GPR(d) = (prod >> 32);
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_mulhwux(%08x)\n",icode);
}

/*
 * ---------------------------------------------------------------
 * mfcr  UISA Form X
 * Move from Condition register
 * v1 
 * ---------------------------------------------------------------
 */
void 
i_mfcr(uint32_t icode) {
	int d=(icode>>21)&0x1f;
#if 0
	check_illegal icode
#endif
	GPR(d) = CR;
	dprintf("instr i_mfcr(%08x)\n",icode);
}


/*
 * -------------------------------------------
 * lwarx UISA Form X
 * 	Load word and reserve indexed
 * v1 
 * -------------------------------------------
 */
void 
i_lwarx(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;	
	if(a==0) {
		ea=GPR(b);
	} else {
		ea=GPR(a)+GPR(b);
	}
	if(!(ea&3)) {
		fprintf(stderr,"DSI exception 0x00300 missing here\n");
		return;
		// Alignment exception
	}
	gcpu.reservation_valid=1;
	gcpu.reservation=ea;
	GPR(d)=MMU_Read32(ea);
	dprintf("instr i_lwarx(%08x)\n",icode);
}

void 
i_ldx(uint32_t icode) {
	fprintf(stderr,"instr i_ldx(%08x) not implemented\n",icode);
}

/*
 * ----------------------------------------------------
 * lwzx UISA Form X 
 *   Load Word and zero indexed
 * v1	
 * ----------------------------------------------------
 */
void 
i_lwzx(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	uint32_t ea;
	if(a==0) {
		ea=GPR(b);
	} else {
		ea=GPR(a)+GPR(b);
	}
	GPR(d)=MMU_Read32(ea);
	dprintf("instr i_lwzx(%08x)\n",icode);
}

/*
 * --------------------------------------------------------
 * slwx
 * Shift left word
 * --------------------------------------------------------
 */
void 
i_slwx(uint32_t icode) {
	int s=(icode>>21);
	int a=(icode>>16);
	int b=(icode>>11);
	int rc = icode & 1;	
	int sh = GPR(b) & 0x3f;
	uint32_t result;
	if(sh>31) {
		result = GPR(a) = 0;
	} else {
		result = GPR(a) = GPR(s) << sh;
	}
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_slwx(%08x)\n",icode);
}

/*
 * ----------------------------
 * cntlzw UISA Form X
 * v1
 * ----------------------------
 */
void 
i_cntlzwx(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	uint32_t result;
	int rc = icode&1;
	if(b) {
		fprintf(stderr,"Illegal instruction format\n");
		return;
	}
	result = GPR(a) = count_leading_zeros(GPR(s));
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_cntlzwx(%08x)\n",icode);
}

void 
i_sldx(uint32_t icode) {
	fprintf(stderr,"instr i_sldx(%08x) not implemented\n",icode);
}

/*
 * ----------------------------
 * andx UISA Form X
 * v1
 * ----------------------------
 */
void 
i_andx(uint32_t icode) {
	uint32_t result;
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	int rc=icode&1;
	result = GPR(s) & GPR(b);
	GPR(a) = result;
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_andx(%08x) not implemented\n",icode);
}

/*
 * --------------------------------------
 * cmpl UISA Form X 
 * v1
 * --------------------------------------
 */
void 
i_cmpl(uint32_t icode) {
	uint32_t crfd=7-((icode>>23)&7);
	uint32_t a=(icode>>16)&0x1f;
	uint32_t b=(icode>>11)&0x1f;
	uint32_t Ra = GPR(a);
	uint32_t Rb = GPR(b);
	int L=(icode>>21)&1;
	uint32_t c;
	if(L) {
		fprintf(stderr,"Invalid instruction for cmpl\n");
		return;
	}
	if(Ra<Rb) {
		c=8;
	} else if(Ra>Rb) {
		c=4;
	} else {
		c=2;
	}
	if (XER & XER_SO)  {
		c |= 1;	
	}
	CR &= 0xffffffff ^ (0xf<<(crfd<<2));
	CR |= c<<(crfd<<2);
	fprintf(stderr,"instr i_cmpl(%08x)\n",icode);
}

/*
 * -------------------------------------------------------
 * subfx UISA Form XO
 * 	Subtract from
 * -------------------------------------------------------
 */
void 
i_subfx(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	int rc = icode & 1;
	int oe = icode & (1<<10);
	uint32_t op1 = GPR(b),op2 = GPR(a);
	uint32_t result;
	result = GPR(d) = op1-op2;
	if(oe) {
		if(sub_overflow(op1,op2,result)) {
			XER = XER | XER_OV | XER_SO;	
		} else {
			XER = XER & ~XER_OV;
		}
	}
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_subfx(%08x)\n",icode);
}

/* Load doubleword with update index: 64 Bit impl. only */
void 
i_ldux(uint32_t icode) {
	fprintf(stderr,"instr i_ldux(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------------
 * dcbst VEA Form X
 * 	Data cache block store
 * 	Currently does nothing because no cache is
 *	emulated
 * v1
 * ------------------------------------------------
 */
void 
i_dcbst(uint32_t icode) {
	fprintf(stderr,"ignore i_dcbst(%08x)\n",icode);
}

/*
 * ------------------------------------------
 * lwzux UISA Form X
 * Load word and zero with update indexed
 * v1
 * ------------------------------------------
 */
void 
i_lwzux(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	uint32_t ea;
	ea=GPR(a)+GPR(b);
	GPR(d)=MMU_Read32(ea);
	GPR(a)=ea;
	fprintf(stderr,"instr i_lwzux(%08x)\n",icode);
}

void 
i_zntlzdx(uint32_t icode) {
	fprintf(stderr,"instr i_zntlzdx(%08x) not implemented\n",icode);
}

/*
 * ----------------------
 * andcx UISA Form X
 * v1
 * ----------------------
 */
void 
i_andcx(uint32_t icode) {
	uint32_t result;
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	int rc=icode&1;
	GPR(a) = result = GPR(s) & ~GPR(b);
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_andcx(%08x) not implemented\n",icode);
}

void 
i_td(uint32_t icode) {
	fprintf(stderr,"instr i_td(%08x) not implemented\n",icode);
}

void 
i_mulhdx(uint32_t icode) {
	fprintf(stderr,"instr i_mulhdx(%08x) not implemented\n",icode);
}

/*
 * -------------------------------------
 * mulhwx UISA Form XO
 *	Multiply high word
 * -------------------------------------
 */
void 
i_mulhwx(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;	
	int b = (icode>>11)&0x1f;	
	int rc = icode & 1;
	uint32_t result;
	int64_t prod = (int64_t)GPR(a) * (int64_t)GPR(b);
	result = GPR(d) = (prod >> 32);
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_mulhwx(%08x) not implemented\n",icode);
}

/*
 * --------------------------------------------
 * mfmsr UISA supervisor Form X
 *	Move from Machine status register
 * --------------------------------------------
 */
void 
i_mfmsr(uint32_t icode) {
	int d = (icode>>21)&0x1f;
#if 0
	if(!supervisor) {
		exception();
	}
#endif
	GPR(d) = MSR;
//	fprintf(stderr,"instr i_mfmsr(%08x) not implemented\n",icode);
}

void 
i_ldarx(uint32_t icode) {
	fprintf(stderr,"instr i_ldarx(%08x) not implemented\n",icode);
}

/*
 * ---------------------------------------------------------
 * dcbf VEA Form X
 * 	Data Cache Block flush
 * 	Do nothing because currently no cache is emulated 
 * v1
 * ---------------------------------------------------------
 */
void 
i_dcbf(uint32_t icode) {
	fprintf(stderr,"ignore i_dcbf(%08x)\n",icode);
}

/*
 * -----------------------------------------------
 * lbzx  UISA Form X
 * Load Byte and zero indexed 
 * -----------------------------------------------
 */
void 
i_lbzx(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	if(a==0) {
		ea=GPR(b);
	} else {
		ea=GPR(a)+GPR(b);
	}
	GPR(d) = MMU_Read8(ea);
	fprintf(stderr,"instr i_lbzx(%08x) not implemented\n",icode);
}

/*
 * ----------------------------------------------------
 * negx UISA Form XO
 * 	two's complement
 * v1
 * ----------------------------------------------------
 */
void 
i_negx(uint32_t icode) {
	int d = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int oe = (icode >> 10)&1;
	int rc = icode & 1;
	uint32_t result;
	result = GPR(d) = ~GPR(a) + 1;
	if(oe) {
		if(result == 0x80000000) {
			XER = XER | XER_SO | XER_OV;
		} else {
			XER = XER & ~XER_OV;
		}
	}
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_negx(%08x)\n",icode);
}

/*
 * ----------------------------------------------
 * lbzux UISA Form X
 * Load Byte and Zero with update indexed
 * v1
 * ----------------------------------------------
 */
void 
i_lbzux(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	if((a==0)||(a==d)) {
		/* Trigger exception here */
		fprintf(stderr,"illegal instruction format\n");
		return;
	}
	ea=GPR(a)+GPR(b);
	GPR(d) = MMU_Read8(ea);
	GPR(a) = ea;
	fprintf(stderr,"instr i_lbzux(%08x)\n",icode);
}

/*
 * ----------------------------------------------
 * norx UISA form X
 * v1
 * ----------------------------------------------
 */
void 
i_norx(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	int rc = icode & 1;
	uint32_t result;
	result = GPR(a) = ~(GPR(s) | GPR(b));
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_norx(%08x) not implemented\n",icode);
}

/*
 * -----------------------------------------------------------------
 * subfex UISA Form XO
 * 	Subtract from extended
 * -----------------------------------------------------------------
 */
void 
i_subfex(uint32_t icode) {
	int d = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	int rc = icode & 1;
	int oe = icode & (1<<10);
	uint32_t result;
	uint32_t op1 = GPR(b),op2 = GPR(a);
	if(XER & XER_CA) {
                GPR(d) = result=op1-op2;
        } else {
		GPR(d) = result=op1-op2-1;
        }
	if(sub_carry(op1,op2,result)) {
		XER=XER | XER_CA;
	} else {
		XER=XER & ~XER_CA;
	}
	if(oe) {
		if(sub_overflow(op1,op2,result)) {
			XER=XER | XER_OV | XER_SO;
		} else {
			XER=XER & ~XER_OV;
		}
	}
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_subfex(%08x)\n",icode);
}

/*
 * ----------------------------------------
 * addex UISA form XO 
 * v1
 * ----------------------------------------
 */
void 
i_addex(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	int oe = icode&(1<<10);
	int rc = icode&1;
	uint32_t result,op1,op2;
	op1=GPR(a); op2 = GPR(b); 		
	result = op1+op2;
	if(XER & XER_CA) {
		result++;
	}
	GPR(d)=result;
	if(add_carry(op1,op2,result)) {
		XER |= XER_CA;
	} else {
		XER &= ~XER_CA;
	}
	if(oe) {
		if(add_overflow(op1,op2,result)) {
			XER |= XER_SO | XER_OV;
    		} else {
			XER &= ~XER_OV;
    		}
	}
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_addex(%08x)\n",icode);
}

/*
 * ---------------------------------------------------------
 * mtcrf  UISA Form XFX
 *	Move to condition register fields
 * ---------------------------------------------------------
 */
void 
i_mtcrf(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int crm=(icode>>12)&0xff;
	uint32_t mask = 0;
	int i;
	for(i=0;i<8;i++	) {
		if(crm & (1<<i)) {
			mask |= (0xf<<(i*4));
		}
	}
	CR = (GPR(s) & mask) | (CR & ~mask);
	fprintf(stderr,"instr i_mtcrf(%08x)\n",icode);
}

/* 
 * --------------------------
 * OEA Supervisor Form X 
 * --------------------------
 */
void 
i_mtmsr(uint32_t icode) {
	int s = (icode>>21)&0x1f;	
#if 0
	if(icode&bla) {
		fprintf(stderr,"Illegal icode %08x\n",icode);
		Exception();
	}
	if(!oea supervisor) {
		Exception();
	}
#endif
	PpcSetMsr(GPR(s));
	fprintf(stderr,"instr i_mtmsr(%08x)\n",icode);
}

void 
i_stdx(uint32_t icode) {
	fprintf(stderr,"instr i_stdx(%08x) not implemented\n",icode);
}

/* 
 * --------------------------------------------
 * stwcx UISA Form x
 * 	Store Word Conditional indexed
 * v1
 * --------------------------------------------
 */
void 
i_stwcx_(uint32_t icode) {
	int s=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a)+GPR(b);
	} else {
		ea=GPR(b);
	}
	if(gcpu.reservation_valid) {
		gcpu.reservation_valid = 0;
		if(ea!= gcpu.reservation) {
			fprintf(stderr,"reservation for wrong address\n");
		}
		MMU_Write32(GPR(s),ea);		
		CR=(CR & ~(CR_LT | CR_GT | CR_SO)) |  CR_EQ; 
		if(XER & XER_SO) {
			CR |= CR_SO;
		}
	} else {
		CR=(CR & ~(CR_LT | CR_GT | CR_EQ | CR_SO)) ; 
		if(XER & XER_SO) {
			CR |= CR_SO;
		}
	}
	fprintf(stderr,"instr i_stwcx(%08x)\n",icode);
}
/*
 * -------------------------------------------------------------
 * stwx UISA Form X
 *	Store Word indexed
 * v1
 * -------------------------------------------------------------
 */
void 
i_stwx(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
	MMU_Write32(GPR(s),ea);	
	fprintf(stderr,"instr i_stwx(%08x)\n",icode);
}

void 
i_stdux(uint32_t icode) {
	fprintf(stderr,"instr i_stdux(%08x) not implemented\n",icode);
}

/*
 * -----------------------------------------------------------
 * stwux
 *	Store Word with update Indexed
 * -----------------------------------------------------------
 */
void 
i_stwux(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	uint32_t ea;
	ea = GPR(a) + GPR(b);
	MMU_Write32(GPR(s),ea);
	GPR(a)=ea;
	fprintf(stderr,"instr i_stwux(%08x)\n",icode);
}

/*
 * --------------------------------------------------
 * subfzex UISA Form XO
 *	Subtract from Zero extended
 * --------------------------------------------------
 */
void 
i_subfzex(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>21)&0x1f;
	int oe = icode & (1<<10);
	int rc = icode & 1;
	uint32_t result;
	if(XER & XER_CA) {
		result = 0-GPR(a);
	} else {
		result = 0-GPR(a)-1;
	}
	if(sub_carry(0,GPR(a),result)) {
		XER = XER | XER_CA;
	} else {
		XER = XER & ~XER_CA;
	}
	if(oe) {
		if(sub_overflow(0,GPR(a),result)) {
			XER = XER | XER_OV | XER_SO;
		} else {
			XER = XER & ~XER_OV;
		}
	}
	if(rc) {
		update_cr0(result);
	}
	GPR(d) = result;
	fprintf(stderr,"instr i_subfzex(%08x)\n",icode);
}

/*
 * -----------------------------
 * addzex UISA Form XO
 * -----------------------------
 */
void 
i_addzex(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;
	int oe = icode&(1<<10);
	int rc = icode&1;
	uint32_t result,op1;
	op1=GPR(a); 		
	result = op1;
	if(XER & XER_CA) {
		result++;
	}
	if(add_carry(op1,0,result)) {
		XER |= XER_CA;
	} else {
		XER &= ~XER_CA;
	}
	GPR(d) = result;
	if(oe) {
		if(add_overflow( op1,0,result)) {
			XER |= XER_SO | XER_OV;
    		} else {
			XER &= ~XER_OV;
    		}
	}
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_addzex(%08x) not implemented\n",icode);
}

/*
 * -------------------------------------
 * mtsr OEA Supervisor Form X
 * 	move to Segment Register
 * incomplete v1
 * -------------------------------------
 */
void 
i_mtsr(uint32_t icode) {
	// OEA check missing //
	int s = (icode>>21) & 0x1f;
	int sr = (icode>>16) &0xf;
#if 0
	if(!supervisor(x)) {
		Exception
	}
#endif
	SR(sr) = GPR(s);
	fprintf(stderr,"instr i_mtsr(%08x) not implemented\n",icode);
}

void 
i_stdcx(uint32_t icode) {
	fprintf(stderr,"instr i_stdcx(%08x) not implemented\n",icode);
}

/*
 * -----------------------------------------------------
 * Store Byte indexed
 * -----------------------------------------------------
 */
void 
i_stbx(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;	
	uint32_t ea;
	if(a) {
		ea=GPR(a) + GPR(b);
	} else {
		ea=GPR(b);
	}
	MMU_Write8(GPR(s)&0xff,ea);
	fprintf(stderr,"instr i_stbx(%08x)\n",icode);
}

/*
 * -------------------------------------------------------------------
 * subfmex UISA Form XO
 * 	Subtract from Minus one extended
 * -------------------------------------------------------------------
 */
void 
i_subfmex(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>21)&0x1f;
	int oe = icode & (1<<10);
	int rc = icode & 1;
	uint32_t result;
	if(XER & XER_CA) {
		result = -1-GPR(a);
	} else {
		result = -1-GPR(a)-1;
	}
	if(sub_carry((uint32_t)-1,GPR(a),result)) {
		XER = XER | XER_CA;
	} else {
		XER = XER & ~XER_CA;
	}
	if(oe) {
		if(sub_overflow((uint32_t)-1,GPR(a),result)) {
			XER = XER | XER_OV | XER_SO;
		} else {
			XER = XER & ~XER_OV;
		}
	}
	if(rc) {
		update_cr0(result);
	}
	GPR(d) = result;
	fprintf(stderr,"instr i_subfmex(%08x)\n",icode);
}

void 
i_mulld(uint32_t icode) {
	fprintf(stderr,"instr i_mulld(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------
 * addmex UISA Form XO
 * v1
 * ------------------------------------
 */
void 
i_addmex(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;
	int oe = icode&(1<<10);
	int rc = icode&1;
	uint32_t result,op1,op2;
	op1=GPR(a); op2 = (uint32_t)-1; 		
	result = op1 + op2;
	if(XER & XER_CA) {
		result++;
	}
	GPR(d) = result;
	if(add_carry(op1,op2,result)) {
		XER |= XER_CA;
	} else {
		XER &= ~XER_CA;
	}
	if(oe) {
		if(add_overflow( op1,op2,result)) {
			XER |= XER_SO | XER_OV;
    		} else {
			XER &= ~XER_OV;
    		}
	}
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_addmex(%08x)\n",icode);
}

/*
 * -----------------------------------------------------------------
 * mullwx UISA Form XO
 * 
 * -----------------------------------------------------------------
 */
void 
i_mullwx(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;	
	int b = (icode>>11)&0x1f;	
	int rc = icode & 1;
	int oe = (icode>>10) & 1;
	uint32_t low,high;
	uint64_t prod;
	low = GPR(d) = prod = (int64_t)GPR(a) * (int64_t)GPR(b);
	if(oe) {
		high=prod>>32;
		if((high==0) || (high == 0xffffffff)) {
			XER=XER & ~XER_OV;
		} else {
			XER=XER | XER_OV | XER_SO;
		}
	}
	if(rc) {
		update_cr0(low);
	}
	fprintf(stderr,"instr i_mullwx(%08x)\n",icode);
}

/*
 * ------------------------------------------
 * mtsrin OEA supervisor Form X
 * 	Move to segment register indirect
 * incomplete v1
 * ------------------------------------------
 */
void 
i_mtsrin(uint32_t icode) {
	/*  OEA Supervisor */
	int s = (icode>>21)&0x1f;
	int b = (icode>>11)&0x1f;	
	SR(GPR(b)>>28) = GPR(s);
	fprintf(stderr,"instr i_mtsrin(%08x) not implemented\n",icode);
}

/*
 *-----------------------------------------------------------
 * dcbtst VEA Form X
 * 	Data Block touch for store
 *	Currently ignored because no cache is emulated
 * v1
 *-----------------------------------------------------------
 */
void 
i_dcbtst(uint32_t icode) {
	int a = (icode>>16) & 0x1f;
	int b = (icode>>21) & 0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
	fprintf(stderr,"ignore i_dcbtst(%08x)\n",icode);
}

/*
 * ----------------------------------
 * stbux UISA Form X
 *  	Store byte with update indexed
 * v1
 * ----------------------------------
 */
void 
i_stbux(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;	
	uint32_t ea;
	ea = GPR(a) + GPR(b);
	MMU_Write8(GPR(s)&0xff,ea);
	GPR(a) = ea;
	fprintf(stderr,"instr i_stbux(%08x)\n",icode);
}

/*
 * ----------------------------------
 * ADDx UISA Form XO 
 * v1
 * ----------------------------------
 */
void 
i_addx(uint32_t icode) {
	int d = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	int oe = icode&(1<<10);
	int rc = icode&1;
	uint32_t result,op1,op2;
	op1=GPR(a); op2 = GPR(b); 		
	GPR(d) = result = op1+op2;
	if(oe) {
		if(add_overflow( op1,op2,result)) {
			XER |= XER_SO | XER_OV;
    		} else {
			XER &= ~XER_OV;
    		}
	}
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_addx(%08x)\n",icode);
}

/*
 * ---------------------------------------
 * dcbt VEA Form X
 * 	Data block touch
 * v1
 * ---------------------------------------
 */
void 
i_dcbt(uint32_t icode) {
	int a = (icode>>16) & 0x1f;
	int b = (icode>>21) & 0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
	fprintf(stderr,"ignore i_dcbt(%08x) not implemented\n",icode);
}

/*
 * --------------------------------------
 * lhzx UISA Form X
 * 	Load half word and Zero indexed
 * v1
 * --------------------------------------
 */
void 
i_lhzx(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	uint32_t ea;
	if(a==0) {
		ea=GPR(b);
	} else {
		ea=GPR(a)+GPR(b);
	}
	GPR(d) = MMU_Read16(ea);
	fprintf(stderr,"instr i_lhzx(%08x)\n",icode);
}

/*
 * ---------------------------------
 * eqvx  UISA Form X
 * v1
 * ---------------------------------
 */
void 
i_eqvx(uint32_t icode) {
	int s=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>10)&0x1f;
	int rc=icode&1;
	uint32_t result;
	result = GPR(a) = ~(GPR(s) ^ GPR(b));
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_eqvx(%08x)\n",icode);
}

/*
 * ------------------------------------------------------
 * tlbie OEA supervisor optional Form X
 * Translation Lookaside Buffer invalidate Entry
 * Currently invalidates all
 * ------------------------------------------------------
 */
void 
i_tlbie(uint32_t icode) {
	MMU_InvalidateTlb();
	fprintf(stderr,"instr i_tlbie(%08x)\n",icode);
}

/*
 * ------------------------------------------
 * eciwx
 * incomplete v1
 * ------------------------------------------
 */
void 
i_eciwx(uint32_t icode) {
	int d = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	uint32_t ea;	
	if(!(EAR & (1<<31))) {
		/* Exception */
		fprintf(stderr,"DSI Exception missing here\n");
		return;
	}
	if(a==0) {
		ea = GPR(b);
	} else {
		ea = GPR(a) + GPR(b);
	}
	GPR(d) = MMU_Read32(ea); /* Nocache */
	fprintf(stderr,"instr i_eciwx(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------------
 * lhzux UISA Form X
 * Load half word and zero with update indexed
 * ------------------------------------------------
 */
void 
i_lhzux(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	uint32_t ea;
	ea=GPR(a)+GPR(b);
	GPR(d) = MMU_Read16(ea);
	GPR(a)=ea;
	fprintf(stderr,"instr i_lhzux(%08x) not implemented\n",icode);
}

/*
 * --------------------------------------------------
 * xorx  UISA Form X
 * 	XOR
 * --------------------------------------------------
 */
void 
i_xorx(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	int rc = icode & 1;
	uint32_t result;
	result = GPR(a) = GPR(s) ^ GPR(b);	
	if(rc) {
		update_cr0(result);	
	}
	dprintf("instr i_xorx(%08x)\n",icode);
}

/*
 * ------------------------------------------------------------------
 * mfspr
 * incomplete v1
 * ------------------------------------------------------------------
 */
void 
i_mfspr(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int n = (((icode>>16)&0x1f)) | (((icode>>11)&0x1f)<<5);
	/* Check for Supervisor here ! */
	int oea = 1; 
	if(oea || (n==1) || (n==8)|| (n==9)) {
		if(HAS_SPR(n)) {
			GPR(d) = SPR(n);	
		} else if (HAS_SPR_READ(n)) {
			GPR(d) = SPR_READ(n);
		} else {
			/* Illegal instruction type or undefined */
			fprintf(stderr,"Mist, nonexisting SPR %d\n",n);
		}
	} else {
		fprintf(stderr,"Mist, illegal mfspr %d icode %d\n",n,icode);
	}	
	fprintf(stderr,"instr i_mfspr(%08x)\n",icode);
}

void 
i_lwax(uint32_t icode) {
	fprintf(stderr,"instr i_lwax(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------------
 * lhax UISA Form X
 *	Load Half word algebraic indexed	
 * v1
 * ------------------------------------------------
 */
void 
i_lhax(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	uint32_t result;
	if(a==0) {
		ea=GPR(b);
	} else {
		ea = GPR(a) + GPR(b);
	}
	result=MMU_Read16(ea);
	if(result&0x8000) {	
		GPR(d) = result|0xffff0000;
	} else {
		GPR(d)=result;
	}
	fprintf(stderr,"instr i_lhax(%08x)\n",icode);
}

/*
 * ----------------------------------------------------
 * tlbia OEA supervisor optional Form X
 * 	Translation Lookaside Buffer invalidate all
 * ----------------------------------------------------
 */
void 
i_tlbia(uint32_t icode) {
	MMU_InvalidateTlb();
	fprintf(stderr,"instr i_tlbia(%08x)\n",icode);
}

/*
 * ----------------------------------------------------------------------
 * mftb
 * move from timebase
 *
 * ----------------------------------------------------------------------
 */
void 
i_mftb(uint32_t icode) {
	/* VEA */
	int d = (icode>>21)&0x1f;
	int n = (((icode>>16)&0x1f)) | (((icode>>11)&0x1f)<<5);
	if(n==268) {
		GPR(d) = TBL;
	} else if (n==269) {
		GPR(d) = TBU;
	} else {
		fprintf(stderr,"Illegal time base register\n");
		// Exception illegal instruction
	}
	fprintf(stderr,"instr i_mftb(%08x)\n",icode);
}

void 
i_lwaux(uint32_t icode) {
	fprintf(stderr,"instr i_lwaux(%08x) not implemented\n",icode);
}

/*
 * -------------------------------------------------------
 * lhaux UISA Form X
 * 	Load half word algebraic with update indexed
 * v1
 * -------------------------------------------------------
 */
void 
i_lhaux(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	uint32_t result;
	if((a==0)||(a==d)) {
		fprintf(stderr,"Illegal instruction format\n");
		return;
	} 
	ea = GPR(a) + GPR(b);
	result=MMU_Read16(ea);
	if(result&0x8000) {	
		GPR(d) = result|0xffff0000;
	} else {
		GPR(d)=result;
	}
	GPR(a)=ea;
	fprintf(stderr,"instr i_lhaux(%08x) not implemented\n",icode);
}

/*
 * --------------------------------
 * sthx UISA Form X
 * 	Store Half Word Indexed
 * v1
 * --------------------------------
 */
void 
i_sthx(uint32_t icode) {
	int s=(icode >> 21)&0x1f;
	int a=(icode >> 16)&0x1f;
	int b=(icode >> 11)&0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
	MMU_Write16(GPR(s)&0xffff,ea);
	fprintf(stderr,"instr i_sthx(%08x)\n",icode);
}

/*
 * ----------------------------------------
 * orcx UISA Form X
 * OR with complement
 * ----------------------------------------
 */
void 
i_orcx(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	int rc = icode & 1;
	uint32_t result;
	result = GPR(a) = GPR(s) | ~GPR(b);
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_orcx(%08x)\n",icode);
}

void 
i_slbie(uint32_t icode) {
	fprintf(stderr,"instr i_slbie(%08x) not implemented\n",icode);
}

/*
 * ----------------------------------------------
 * ecowx 
 * External Control word out indexed
 * incomplete v1
 * ----------------------------------------------
 */
void 
i_ecowx(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	uint32_t ea;	
	if(!(EAR & (1<<31))) {
		fprintf(stderr,"exception missing here\n");
		return;
	}
	if(a==0) {
		ea = GPR(b);
	} else {
		ea = GPR(a) + GPR(b);
	}
	if(ea&3) {
		fprintf(stderr,"Alignment exception missing here\n");
		return;
	}
	MMU_Write32(GPR(s),ea); /* nochache */
	fprintf(stderr,"instr i_ecowx(%08x)\n",icode);
}

/*
 * ---------------------------------------------------------
 * sthux  UISA Form X
 * Store Half word with Update Indexed
 * v1
 * ---------------------------------------------------------
 */
void 
i_sthux(uint32_t icode) {
	int s=(icode >> 21)&0x1f;
	int a=(icode >> 16)&0x1f;
	int b=(icode >> 11)&0x1f;
	uint32_t ea;
	ea = GPR(a) + GPR(b);
	MMU_Write16(GPR(s)&0xffff,ea);
	GPR(a) = ea;
	fprintf(stderr,"instr i_sthux(%08x)\n",icode);
}

/*
 * --------------------------------
 * orx UISA Form X 
 * v1
 * --------------------------------
 */
void 
i_orx(uint32_t icode) {
	uint32_t result;
	int s=(icode >> 21)&0x1f;
	int a=(icode >> 16)&0x1f;
	int b=(icode >> 11)&0x1f;
	int rc=icode&1;
	result = GPR(a) = GPR(s) | GPR(b);
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_orx(%08x) at %08x\n",icode,CIA);
}

void 
i_divdux(uint32_t icode) {
	fprintf(stderr,"instr i_divdux(%08x) not implemented\n",icode);
}

/*
 * --------------------------------
 * divwux UISA Form XO
 * v1
 * --------------------------------
 */
void 
i_divwux(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	int oe =  (icode>>10)&1;
	int rc =  icode&1;
	uint32_t result;
	if(GPR(b)) {
		result =  GPR(a)/GPR(b);
	} else {
		fprintf(stderr,"Warning undefined result of division\n");
		result = 47110815; /* undefined */
	}
	if(oe) {
		XER=XER & ~XER_OV;
		if(GPR(b)==0) {
			XER |= XER_OV | XER_SO;
		}
	}
	if(rc) {
		update_cr0(result);
	}
	GPR(d) = result;
	fprintf(stderr,"instr i_divwux(%08x)\n",icode);
}

/*
 * ----------------------------------------------
 * mtspr UISA/OEA sometimes supervisor form XFX 
 * 	Move to special purpose register
 * incomplete v1
 * ----------------------------------------------
 */
void 
i_mtspr(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int n = (((icode>>16)&0x1f)) | (((icode>>11)&0x1f)<<5);
	/* Check for OEA here ! */
	int supervisor = 1; 
	if(supervisor || (n==1) || (n==8)|| (n==9)) {
		if(HAS_SPR(n)) {
			SPR(n) = GPR(s);	
			fprintf(stderr,"mtspr: SPR %d new value %08x from R%d\n",n,GPR(s),s);
		} else if (HAS_SPR_WRITE(n)) {
			SPR_WRITE(GPR(s),n);
		} else {
			fprintf(stderr,"mtspr: Mist, SPR %d does not exist, icode %08x\n",n,icode);
		}
		if(n==9) {
			fprintf(stderr,"Load spr(9) with %08x\n",GPR(s));
		}
	} else {
		fprintf(stderr,"Mist, mtspr not allowed %08x\n",icode);
#if 0	
		Exception();
#endif
		return;
	}	
	dprintf("instr i_mtspr(%08x) not implemented\n",icode);
}

/*
 * -------------------------------------------------------
 * dcbi VEA Form X 
 *	Data Cache Block Invalidate
 *	Is ignored because no data cache is emulated
 * v1
 * --------------------------------------------------------
 */
void 
i_dcbi(uint32_t icode) {
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
#if 0
	if(!translate_address(ea))  {
		PPC_Exception(DSI);
	}
#endif
	fprintf(stderr,"ignore i_dcbi(%08x)\n",icode);
}

/*
 * ---------------------------------------------------
 * nandx UISA Form X
 * 	NAND
 * v1
 * ---------------------------------------------------
 */
void 
i_nandx(uint32_t icode) {
	int s = (icode>>21)&0x1f;	
	int a = (icode>>16)&0x1f;	
	int b = (icode>>11)&0x1f;	
	int rc = icode & 1;
	uint32_t result;
	result = GPR(a) = ~(GPR(s) & GPR(b));
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_nandx(%08x) not implemented\n",icode);
}

void 
i_divdx(uint32_t icode) {
	fprintf(stderr,"instr i_divdx(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------
 * divwx UISA Form XO
 * v1
 * ------------------------------------
 */
void 
i_divwx(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	int oe =  (icode>>10)&1;
	int rc =  icode&1;
	int32_t result;
	if(GPR(b)) {
		result = (int32_t)GPR(a)/(int32_t)GPR(b);
	} else {
		fprintf(stderr,"Warning undefined result of division\n");
		result = 0x47110815; /* Manual says undefined */
	}
	if(oe) {
		XER=XER & ~XER_OV;
		if((GPR(a)==0x80000000) && (GPR(b)==0xffffffff)) {
			XER |= XER_OV | XER_SO;
		}
		if(GPR(b)==0) {
			XER |= XER_OV | XER_SO;
		}
	}
	if(rc) {
		update_cr0(result);
	}
	GPR(d) = result;
	fprintf(stderr,"instr i_divwx(%08x) not implemented\n",icode);
}

void 
i_slbia(uint32_t icode) {
	fprintf(stderr,"instr i_slbia(%08x) not implemented\n",icode);
}

/*
 * ---------------------------------------
 * mcrxr UISA Form X
 * Move to condition register from XER
 * ---------------------------------------
 */
void 
i_mcrxr(uint32_t icode) {
	int crfd = 7 - ((icode>>23) & 7);
	uint32_t mask = ~(0xf<<(4*crfd));
#if 0
	check_illegal icode	
#endif
	CR = (CR & mask) | (((XER & 0xf0000000)>>28)<<(4*crfd));
	XER = XER & 0x0fffffff;
	fprintf(stderr,"instr i_mcrxr(%08x)\n",icode);
}

/* 
 * --------------------------------------------------------------
 * lswx  UISA Form X
 * Load string word indexed
 * v1
 * --------------------------------------------------------------
 */
void 
i_lswx(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	int n=XER&0x7f;
	int r;
	int i;
	uint32_t ea;
	if(a==0) {
		ea =  GPR(b);
	} else {
		ea=GPR(b)+GPR(a);
	}
	r=GPR(d)-1;
	i=0;
	while(n>0) {
		if(i==0) {
			r=(r+1)&31;
			GPR(r)=0;
		}
		/* Create Exception on segment Boudary is missing here */
		GPR(r)=GPR(r) | (MMU_Read8(ea)<<(24-i));
		i=(i+8)&31;
		ea++;
		n--;
	}
	fprintf(stderr,"instr i_lswx(%08x)\n",icode);
}

/* 
 * ---------------------------------------------
 * lwbrx UISA Form X
 * Load  word byte reversed indexed
 * v1
 * ---------------------------------------------
 */
void 
i_lwbrx(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	if(a==0) {
		ea =  GPR(b);
	} else {
		ea=GPR(b)+GPR(a);
	}
	GPR(d) = swap32(MMU_Read32(ea));
	fprintf(stderr,"instr i_lwbrx(%08x)\n",icode);
}

void 
i_lfsx(uint32_t icode) {
	fprintf(stderr,"instr i_lfsx(%08x) not implemented\n",icode);
}

/*
 * --------------------------------------
 * srwx
 * Shift Right Word
 * --------------------------------------
 */

void 
i_srwx(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	int rc = icode & 1;
	int sh = GPR(b) & 0x3f;
	uint32_t result;
	if(sh>31) {
		result = GPR(a) = 0;
	} else {
		result = GPR(a) = GPR(s) >> sh;
	}
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_srwx(%08x) not implemented\n",icode);
}

void 
i_srdx(uint32_t icode) {
	fprintf(stderr,"instr i_srdx(%08x) not implemented\n",icode);
}

/*
 * -----------------------------------------------------------------------------
 * tlbsync
 *	Wait until all processors have invalidated the outstanding TLB-Entries
 * -----------------------------------------------------------------------------
 */
void 
i_tlbsync(uint32_t icode) {
	fprintf(stderr,"instr i_tlbsync(%08x) ignored\n",icode);
}

void 
i_lfsux(uint32_t icode) {
	fprintf(stderr,"instr i_lfsux(%08x) not implemented\n",icode);
}

void 
i_mfsr(uint32_t icode) {
	// OEA check missing //
	int d = (icode>>21) & 0x1f;
	int sr = (icode>>16) &0xf;
	GPR(d) = SR(sr);
	fprintf(stderr,"instr i_mfsr(%08x) not implemented\n",icode);
}

/*
 * -------------------------------------------------------
 * lswi UISA Form X
 * 	Load string word immediate
 * v1
 * -------------------------------------------------------
 */
void 
i_lswi(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int nb=(icode>>11)&0x1f;	
	int i;
	int n;
	int r;
	uint32_t ea;
	if(a==0) {
		ea=0;
	} else {
		ea=GPR(a);
	}
	if(nb==0) {
		n=32;
	} else {
		n=nb;
	}
	r=GPR(d)-1;
	i=0;
	while(n>0) {
		if(i==0) {
			r=(r+1)&31;
			GPR(r)=0;
		}
		GPR(r) = GPR(r) | MMU_Read8(ea)<<(24-i);
		i=(i+8)&31;
		ea++;
		n--;
	}
	fprintf(stderr,"instr i_lswi(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------------
 * sync
 *	Synchronize
 * ------------------------------------------------
 */
void 
i_sync(uint32_t icode) {
	fprintf(stderr,"instr i_sync(%08x) currently does nothing\n",icode);
}

void 
i_lfdx(uint32_t icode) {
	fprintf(stderr,"instr i_lfdx(%08x) not implemented\n",icode);
}

void 
i_lfdux(uint32_t icode) {
	fprintf(stderr,"instr i_lfdux(%08x) not implemented\n",icode);
}

/*
 * -----------------------------------------------
 * mfsrin Move from Segment register indirect
 * v1
 * -----------------------------------------------
 */
void 
i_mfsrin(uint32_t icode) {
	/*  OEA Supervisor */
	int d = (icode>>21)&0x1f;
	int b = (icode>>11)&0x1f;	
	GPR(d) = SR(GPR(b)>>28);
	fprintf(stderr,"instr i_mfsrin(%08x) not implemented\n",icode);
}

/*
 * -----------------------------------------------
 * stswx UISA Form X
 * 	Store String Word Indexed
 * v1
 * -----------------------------------------------
 */
void 
i_stswx(uint32_t icode) {
	int s=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	int n=XER&0x7f;
	int r;
	int i;
	uint32_t ea;
	if(a==0) {
		ea = GPR(b);
	} else {
		ea=GPR(b)+GPR(a);
	}
	r=GPR(s)-1;
	i=0;
	while(n>0) {
		if(i==0) {
			r=(r+1)&31;
		}
		MMU_Write8(GPR(r)>>(24-i),ea);
		i=(i+8)&31;
		ea++;
		n--;
	}
	fprintf(stderr,"instr i_stswx(%08x)\n",icode);
}

/*
 * ------------------------------------------------------------------
 * stwbrx UISA Form X
 * 	Store Word Byte Reversed Indexed
 * v1
 * ------------------------------------------------------------------
 */
void 
i_stwbrx(uint32_t icode) {
	int s=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
	MMU_Write32(swap32(GPR(s)),ea);	
	fprintf(stderr,"instr i_stwbrx(%08x)\n",icode);
}

void 
i_stfsx(uint32_t icode) {
	fprintf(stderr,"instr i_stfsx(%08x) not implemented\n",icode);
}

void 
i_stfsux(uint32_t icode) {
	fprintf(stderr,"instr i_stfsux(%08x) not implemented\n",icode);
}

/*
 * -------------------------------------------
 * stswi UISA Form X
 * 	Store String Word Immediate
 * v1
 * -------------------------------------------
 */ 
void 
i_stswi(uint32_t icode) {
	int s=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int nb=(icode>>11)&0x1f;	
	int i;
	int n;
	int r;
	uint32_t ea;
	if(a==0) {
		ea=0;
	} else {
		ea=GPR(a);
	}
	if(nb==0) {
		n=32;
	} else {
		n=nb;
	}
	r=GPR(s)-1;
	i=0;
	while(n>0) {
		if(i==0) {
			r=(r+1)&31;
		}
		MMU_Write8((GPR(r)>>(24-i))&0xff,ea);
		i=(i+8)&31;
		ea++;
		n--;
	}
	fprintf(stderr,"instr i_stswi(%08x)\n",icode);
}

/*
 *
 */
void 
i_stfdx(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
	MMU_Write64(FPR(s),ea);
	fprintf(stderr,"instr i_stfdx(%08x) not implemented\n",icode);
}

/*
 * -------------------------
 * dcba VEA Optional Form X
 * Currently does nothing
 * v1
 * -------------------------
 */
void 
i_dcba(uint32_t icode) {
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
	/* Write some random numbers to the memory block */
	fprintf(stderr,"instr i_dcba(%08x) ignored\n",icode);
}

void 
i_stfdux(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int b = (icode>>11) & 0x1f;
	uint32_t ea;
	ea = GPR(a) + GPR(b);
	MMU_Write64(FPR(s),ea);
	GPR(a) = ea;
	fprintf(stderr,"instr i_stfdux(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------
 * lhbrx
 * 	Load half word byte reversed indexed
 * v1
 * ------------------------------------------
 */
void 
i_lhbrx(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	if(a==0) {
		ea = GPR(b);
	} else {
		ea = GPR(a)+GPR(b);
	}
	GPR(d) = swap16(MMU_Read16(ea));	
	fprintf(stderr,"instr i_lhbrx(%08x)\n",icode);
}

/*
 * -------------------------------------------
 * srawx
 * 	Shift right algebraic word
 * carry untested
 * -------------------------------------------
 */
void 
i_srawx(uint32_t icode) {
	int s=(icode>>21);
	int a=(icode>>16);
	int b=(icode>>11);
	int rc = icode & 1;	
	int sh = GPR(b)&0x3f;	
	uint32_t result;
	XER = XER &  ~XER_CA;
	if(sh>31) {
		result = GPR(a) = 0;
	} else { 
		if(GPR(s) & 0x80000000) {
			if(((GPR(s) >> sh) << sh) != GPR(s)) {
				XER |= XER_CA;
			}
		}
		result = GPR(a) = ((int32_t)GPR(s)) >> sh;
	}
	if(rc) {
		update_cr0(result);
	}
	fprintf(stderr,"instr i_srawx(%08x)\n",icode);
}

void 
i_sradx(uint32_t icode) {
	fprintf(stderr,"instr i_sradx(%08x) not implemented\n",icode);
}

/*
 * ---------------------------------------------
 * srawix
 * 	Shift right algebraic word immediate
 * ---------------------------------------------
 */
void 
i_srawix(uint32_t icode) {
	int s=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int rc = icode & 1;	
	int sh = (icode>>11) & 0x1f;	
	uint32_t result;
	XER = XER &  ~XER_CA;
	if(sh>31) {
		result = GPR(a) = 0;
	} else { 
		if(GPR(s) & 0x80000000) {
			if(((GPR(s) >> sh) << sh) != GPR(s)) {
				XER |= XER_CA;
			}
		}
		result = GPR(a) = ((int32_t)GPR(s)) >> sh;
	}
	if(rc) {
		update_cr0(result);
	}
	dprintf("instr i_srawix(%08x)\n",icode);
}

/*
 * ------------------------------------------
 * eieio Enforce in order execution of IO
 * ------------------------------------------
 */
void 
i_eieio(uint32_t icode) {
	fprintf(stderr,"instr i_eieio(%08x) not implemented\n",icode);
}

/*
 * --------------------------------------
 * sthbrx UISA Form X
 * 	Store Half Word reverse Indexed
 * v1
 * --------------------------------------
 */
void 
i_sthbrx(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int b = (icode>>11)&0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
	MMU_Write16(swap16(GPR(s)&0xffff),ea);
	fprintf(stderr,"instr i_sthbrx(%08x)\n",icode);
}

/*
 * ----------------------------
 * extshx UISA Form X
 * v1
 * ----------------------------
 */
void 
i_extshx(uint32_t icode) {
	int s=(icode>>21) &0x1f;
        int a=(icode>>16) &0x1f;
        int b=(icode>>11) &0x1f;
	int rc = icode &1;
	int16_t imm;
	if(b!=0) {
		fprintf(stderr,"Illegal instruction format\n");
		return;
	}
	imm = GPR(s);
	GPR(a) = imm;
	if(rc) {
		update_cr0(GPR(a));
	}
	fprintf(stderr,"instr i_extshx(%08x) not implemented\n",icode);
}

/*
 * ----------------------------
 * extsbx UISA Form X
 * Extend Sign Byte
 * v1
 * ----------------------------
 */
void 
i_extsbx(uint32_t icode) {
	int s=(icode>>21) &0x1f;
	int a=(icode>>16) &0x1f;
	int b=(icode>>11) &0x1f;
	int rc = icode &1;
	int8_t imm;
	if(b!=0) {
		fprintf(stderr,"Illegal instruction format\n");
		return;
	}
	imm = GPR(s) & 0xff;
	GPR(a) = imm;
	if(rc) {
		update_cr0(GPR(a));
	}
	fprintf(stderr,"instr i_extsbx(%08x)\n",icode);
}

/*
 * ------------------------------------------------------------
 * icbi VEA Form X
 * Instruction Cache block invalidate
 * Currently does nothing because cache is not emulated
 * ------------------------------------------------------------
 */
void 
i_icbi(uint32_t icode) {
	fprintf(stderr,"instr i_icbi(%08x) ignored\n",icode);
}

void 
i_stfiwx(uint32_t icode) {
	int s=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + GPR(b);
	} else {
		ea = GPR(b);
	}
	MMU_Write32(FPR(s)&0xffffffff,ea);
	
	fprintf(stderr,"instr i_stfiwx(%08x) not implemented\n",icode);
}

void 
i_extsw(uint32_t icode) {
	fprintf(stderr,"instr i_extsw(%08x) not implemented\n",icode);
}

/*
 * -------------------------------------
 * dcbz
 * 	Data block clear to zero
 * v1
 * -------------------------------------
 */
void 
i_dcbz(uint32_t icode) {
	int i;
	int a=(icode>>16)&0x1f;
	int b=(icode>>11)&0x1f;
	uint32_t ea = GPR(b);
	if(a) {
		ea+=GPR(a);
	}
	ea = ea & ~0x1f;
	/* Exception Check Granularity ? */
	for(i=0;i<4;i++) {
		MMU_Write64(0,ea);
		ea+=8;
	}
	fprintf(stderr,"instr i_dcbz(%08x) not implemented\n",icode);
}

/*
 * ---------------------------------------------------------
 * lwz UISA Form D
 * 	Load word and zero
 * v1
 * ---------------------------------------------------------
 */
void 
i_lwz(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t offs=icode&0xffff;
	uint32_t ea;
	if(a==0) {
		ea=offs;
	} else {
		ea=GPR(a)+offs;
	}
	GPR(d)=MMU_Read32(ea);
	dprintf("instr i_lwz(%08x)\n",icode);
}

/*
 * ---------------------------------------------------------
 * lwzu UISA Form D
 * 	Load Word and Zero with Update
 * v1
 * ---------------------------------------------------------
 */
void 
i_lwzu(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t offs=icode&0xffff;
	uint32_t ea;
	ea=GPR(a)+offs;
	GPR(d)=MMU_Read32(ea);
	GPR(a)=ea;
	fprintf(stderr,"instr i_lwzu(%08x) not implemented\n",icode);
}

/*
 * -----------------------------
 * lbz UISA Form D
 * Load Byte and zero
 * -----------------------------
 */
void 
i_lbz(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	uint32_t ea;
	uint32_t offs;
	if(icode & 0x8000) {
		offs = icode | 0xffff0000;
	} else {
		offs = icode &0xffff;
	}
	if(a==0) {
		ea=offs;	
	} else {
		ea=GPR(a) + offs;
	}
	GPR(d) = MMU_Read8(ea);
	fprintf(stderr,"instr i_lbz(%08x)\n",icode);
}

/*
 * -----------------------------------------------------------
 * lbzu UISA Form D
 * 	Load Byte and Zero with update
 * v1 
 * -----------------------------------------------------------
 */
void 
i_lbzu(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	uint32_t ea;
	uint32_t offs;
	if(icode & 0x8000) {
		offs = icode | 0xffff0000;
	} else {
		offs = icode &0xffff;
	}
	if((a==0)||(a==d)) {
		fprintf(stderr,"illegal instruction format\n");
		return;
	}
	ea=GPR(a)+offs;
	GPR(d) = MMU_Read8(ea);
	GPR(a)=ea;
	fprintf(stderr,"instr i_lbzu(%08x) not implemented\n",icode);
}

/* 
 * -----------------------------------------
 * stw UISA Form D
 *	Store Word
 * -----------------------------------------
 */
void 
i_stw(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t imm = icode & 0xffff;
	uint32_t ea;	
	if(a) {
		ea = GPR(a) + imm;	
	} else {
		ea = imm;
	}	
	MMU_Write32(GPR(s),ea);
	dprintf("instr i_stw(%08x) not implemented\n",icode);
}
/*
 * ----------------------------------------------
 * stwu
 *	Store Word with update
 * v1
 * ----------------------------------------------
 */
void 
i_stwu(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t imm=0xffff;
	uint32_t ea;
	ea = GPR(a) + imm;
	MMU_Write32(GPR(s),ea);
	fprintf(stderr,"instr i_stwu(%08x), val %08x\n",icode,GPR(s));
	GPR(a)=ea;
}

/*
 * ---------------------------------
 * stb UISA Form D
 * v1
 * ---------------------------------
 */
void 
i_stb(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t imm = icode & 0xffff;
	uint32_t ea;
	if(a) {
		ea=GPR(a)+imm;
	} else {
		ea=imm;
	}
	MMU_Write8(GPR(s)&0xff,ea);
	fprintf(stderr,"instr i_stb(%08x)\n",icode);
}

/*
 * ---------------------------------
 * stbu UISA Form D
 *	Store Byte with update
 * v1
 * ---------------------------------
 */
void 
i_stbu(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t imm = icode & 0xffff;
	uint32_t ea;
	ea = GPR(a) + imm;	
	MMU_Write8(GPR(s)&0xff,ea);
	GPR(a) = ea;
	fprintf(stderr,"instr i_stbu(%08x)\n",icode);
}

/*
 * ---------------------------------
 * lhz UISA Form D
 * Load half word and zero
 * v1
 * ---------------------------------
 */
void 
i_lhz(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t ofs = icode & 0xffff;
	uint32_t ea;
	if(a==0) {
		ea=ofs;
	} else {
		ea=GPR(a)+ofs;
	}	
	GPR(d) = MMU_Read16(ea);
	fprintf(stderr,"instr i_lhz(%08x) not implemented\n",icode);
}

/*
 * -----------------------------------------
 * lhzu UISA Form D
 *  Load half word and zero with update
 * -----------------------------------------
 */
void 
i_lhzu(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t ofs = icode & 0xffff;
	uint32_t ea;
	ea=GPR(a)+ofs;
	GPR(d) = MMU_Read16(ea);
	GPR(a) = ea;
	fprintf(stderr,"instr i_lhzu(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------------------------
 * lha UISA Form D  
 *	Load Half word algebraic	
 * ------------------------------------------------------------
 */
void 
i_lha(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	uint32_t ea;
	uint32_t offs=  icode&0x8000 ? (icode | 0xffff0000):(icode &0xffff);
	uint32_t result=0xAffe;
	if(a==0) {
		ea=offs;
	} else {
		ea = GPR(a) + offs;
	}
	result=MMU_Read16(ea);
	if(result&0x8000) {	
		GPR(d) = result|0xffff0000;
	} else {
		GPR(d) = result;
	}
	fprintf(stderr,"instr i_lha(%08x)\n",icode);
}

/*
 * --------------------------------------------
 * lhau UISA Form D
 * 	Load Half word algebraic with update
 * v1
 * --------------------------------------------
 */
void 
i_lhau(uint32_t icode) {
	int d=(icode>>21)&0x1f;
	int a=(icode>>16)&0x1f;
	uint32_t ea;
	uint32_t offs=  icode&0x8000 ? (icode | 0xffff0000):(icode &0xffff);
	uint32_t result=0xAffe;
	if((a==0)||(a==d)) {
		fprintf(stderr,"Illegal instruction format\n");
		return;
	} else {
		ea = GPR(a) + offs;
	}
	result=MMU_Read16(ea);
	if(result&0x8000) {	
		GPR(d) = result|0xffff0000;
	} else {
		GPR(d)=result;
	}
	GPR(a)=ea;
	fprintf(stderr,"instr i_lhau(%08x)\n",icode);
}

/*
 * ----------------------------
 * sth UISA Form D
 * Store Half Word
 * v1
 * ----------------------------
 */
void 
i_sth(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t imm = icode & 0xffff;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + imm;
	} else {
		ea = imm;
	}
	MMU_Write16(GPR(s)&0xffff,ea);
	fprintf(stderr,"instr i_sth(%08x)\n",icode);
}

/*
 * ------------------------------------------------
 * sthu
 * Store Half Word with Update
 * v1
 * ------------------------------------------------
 */
void 
i_sthu(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t imm = icode & 0xffff;
	uint32_t ea;
	ea = GPR(a) + imm;
	MMU_Write16(GPR(s)&0xffff,ea);
	GPR(a) = ea;
	fprintf(stderr,"instr i_sthu(%08x)\n",icode);
}

/*
 * ---------------------------------
 * lmw UISA Form D
 * Load Multiple Word
 * v1
 * ---------------------------------
 */
void 
i_lmw(uint32_t icode) {
	int d = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t ofs=icode&0xffff;
	uint32_t ea;
	uint32_t r;
	if(a==0) {
		ea=ofs;
	} else {
		ea=GPR(a)+ofs;
	}
	r=GPR(d);
	while(r<=31) {
		GPR(r)=MMU_Read32(ea);
		ea+=4;
		r++;
	}	
	fprintf(stderr,"instr i_lmw(%08x)\n",icode);
}

/*
 * ---------------------------------------------
 * stmw UISA Form D
 * 	Store Multiple Word
 * v1
 * ---------------------------------------------
 */
void 
i_stmw(uint32_t icode) {
	int s = (icode>>21)&0x1f;
	int a = (icode>>16)&0x1f;
	int16_t ofs=icode&0xffff;
	uint32_t ea;
	uint32_t r;
	if(a==0) {
		ea=ofs;
	} else {
		ea=GPR(a)+ofs;
	}
	r=GPR(s);
	while(r<=31) {
		MMU_Write32(GPR(r),ea);
		r = r + 1;
		ea = ea + 4;
	}	
	fprintf(stderr,"instr i_stmw(%08x)\n",icode);
}

void 
i_lfs(uint32_t icode) {
	fprintf(stderr,"instr i_lfs(%08x) not implemented\n",icode);
}

void 
i_lfsu(uint32_t icode) {
	fprintf(stderr,"instr i_lfsu(%08x) not implemented\n",icode);
}

void 
i_lfd(uint32_t icode) {
	fprintf(stderr,"instr i_lfd(%08x) not implemented\n",icode);
}

void 
i_lfdu(uint32_t icode) {
	fprintf(stderr,"instr i_lfdu(%08x) not implemented\n",icode);
}

void 
i_stfs(uint32_t icode) {
	fprintf(stderr,"instr i_stfs(%08x) not implemented\n",icode);
}

void 
i_stfsu(uint32_t icode) {
	fprintf(stderr,"instr i_stfsu(%08x) not implemented\n",icode);
}

void 
i_stfd(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int16_t imm=icode&0xffff;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + imm;
	} else {
		ea = imm;
	}
	MMU_Write64(FPR(s),ea);
	fprintf(stderr,"instr i_stfd(%08x) not implemented\n",icode);
}

void 
i_stfdu(uint32_t icode) {
	int s = (icode>>21) & 0x1f;
	int a = (icode>>16) & 0x1f;
	int16_t imm=icode&0xffff;
	uint32_t ea;
	if(a) {
		ea = GPR(a) + imm;
	} else {
		ea = imm;
	}
	MMU_Write64(FPR(s),ea);
	GPR(a) = ea;
	fprintf(stderr,"instr i_stfdu(%08x) at %08x not implemented\n",icode,CIA);
}

/* Load Double word: 64 Bit impl. only */
void 
i_ld(uint32_t icode) {
	fprintf(stderr,"instr i_ld(%08x) not implemented\n",icode);
}

/* Load Double word with update: 64 Bit impl. only */
void 
i_ldu(uint32_t icode) {
	fprintf(stderr,"instr i_ldu(%08x) not implemented\n",icode);
}

void 
i_lwa(uint32_t icode) {
	fprintf(stderr,"instr i_lwa(%08x) not implemented\n",icode);
}

void 
i_fdivsx(uint32_t icode) {
	fprintf(stderr,"instr i_fdivsx(%08x) not implemented\n",icode);
}

void 
i_fsubsx(uint32_t icode) {
	fprintf(stderr,"instr i_fsubsx(%08x) not implemented\n",icode);
}

void 
i_faddsx(uint32_t icode) {
	fprintf(stderr,"instr i_faddsx(%08x) not implemented\n",icode);
}

void 
i_fsqrtsx(uint32_t icode) {
	fprintf(stderr,"instr i_fsqrtsx(%08x) not implemented\n",icode);
}

void 
i_fsresx(uint32_t icode) {
	fprintf(stderr,"instr i_fsresx(%08x) not implemented\n",icode);
}

void 
i_fmulsx(uint32_t icode) {
	fprintf(stderr,"instr i_fmulsx(%08x) not implemented\n",icode);
}

void 
i_fmsubsx(uint32_t icode) {
	fprintf(stderr,"instr i_fmsubsx(%08x) not implemented\n",icode);
}

void 
i_fmaddsx(uint32_t icode) {
	fprintf(stderr,"instr i_fmaddsx(%08x) not implemented\n",icode);
}

void 
i_fnmsubsx(uint32_t icode) {
	fprintf(stderr,"instr i_fnmsubsx(%08x) not implemented\n",icode);
}

void 
i_fnmaddsx(uint32_t icode) {
	fprintf(stderr,"instr i_fnmaddsx(%08x) not implemented\n",icode);
}

void 
i_std(uint32_t icode) {
	fprintf(stderr,"instr i_std(%08x) not implemented\n",icode);
}

void 
i_stdu(uint32_t icode) {
	fprintf(stderr,"instr i_stdu(%08x) not implemented\n",icode);
}

void 
i_fcmpu(uint32_t icode) {
	fprintf(stderr,"instr i_fcmpu(%08x) not implemented\n",icode);
}

void 
i_frspx(uint32_t icode) {
	fprintf(stderr,"instr i_frspx(%08x) not implemented\n",icode);
}

void 
i_fctiwx(uint32_t icode) {
	fprintf(stderr,"instr i_fctiwx(%08x) not implemented\n",icode);
}

void 
i_fctiwzx(uint32_t icode) {
	fprintf(stderr,"instr i_fctiwzx(%08x) not implemented\n",icode);
}

void 
i_fdivx(uint32_t icode) {
	fprintf(stderr,"instr i_fdivx(%08x) not implemented\n",icode);
}

void 
i_fsubx(uint32_t icode) {
	fprintf(stderr,"instr i_fsubx(%08x) not implemented\n",icode);
}

void 
i_faddx(uint32_t icode) {
	fprintf(stderr,"instr i_faddx(%08x) not implemented\n",icode);
}

void 
i_fsqrtx(uint32_t icode) {
	fprintf(stderr,"instr i_fsqrtx(%08x) not implemented\n",icode);
}

void 
i_fselx(uint32_t icode) {
	fprintf(stderr,"instr i_fselx(%08x) not implemented\n",icode);
}

void 
i_fmulx(uint32_t icode) {
	fprintf(stderr,"instr i_fmulx(%08x) not implemented\n",icode);
}

void 
i_fsqrtex(uint32_t icode) {
	fprintf(stderr,"instr i_fsqrtex(%08x) not implemented\n",icode);
}

void 
i_fmsubx(uint32_t icode) {
	fprintf(stderr,"instr i_fmsubx(%08x) not implemented\n",icode);
}

void 
i_fmaddx(uint32_t icode) {
	fprintf(stderr,"instr i_fmaddx(%08x) not implemented\n",icode);
}

void 
i_fnmsubx(uint32_t icode) {
	fprintf(stderr,"instr i_fnmsubx(%08x) not implemented\n",icode);
}

void 
i_fnmaddx(uint32_t icode) {
	fprintf(stderr,"instr i_fnmaddx(%08x) not implemented\n",icode);
}

void 
i_fcmpo(uint32_t icode) {
	fprintf(stderr,"instr i_fcmpo(%08x) not implemented\n",icode);
}

/*
 * ----------------------------------------------
 * mtfsb1x UISA Form X
 * Move to FPSCR Bit 1
 * ----------------------------------------------
 */
void 
i_mtfsb1x(uint32_t icode) {
	int crbd = 31-((icode>>21)&0x1f);
	int rc = icode&1;
	if((crbd == 29) || (crbd == 30)) {
		fprintf(stderr,"mtfsb1x geht net\n");
		return;
	}
	FPSCR = FPSCR | (1<<crbd);
	if(rc) {
		CR = (CR & 0xf0ffffff) | ((FPSCR>>4)&0x0f000000);
	}
	fprintf(stderr,"instr i_mtfsb1x(%08x) not implemented\n",icode);
}

void 
i_fnegx(uint32_t icode) {
	fprintf(stderr,"instr i_fnegx(%08x) not implemented\n",icode);
}

#define FPSCR_EXCEPTIONS (FPSCR_VXCVI | FPSCR_VXSQRT |FPSCR_VXSOFT | FPSCR_VXVC | FPSCR_VXIMZ \
		| FPSCR_VXZDZ | FPSCR_VXIDI | FPSCR_VXISI | FPSCR_VXSNAN \
		| FPSCR_XX | FPSCR_ZX | FPSCR_UX  | FPSCR_OX | FPSCR_FX)
 
/*
 * --------------------------------------------------------
 * mcrfs UISA Form X
 * Move to Condition register from FPSCR		
 * --------------------------------------------------------
 */
void 
i_mcrfs(uint32_t icode) {
	uint32_t mask;
	uint32_t clear;
	uint32_t setbits;
	int crfd = 7-((icode>>23)&7);
	int crfs = 7-((icode>>18)&7);
	clear = FPSCR & (0xf << (4*crfs)) & FPSCR_EXCEPTIONS;
	setbits=((FPSCR>>(4*crfs))&0xf) << (4*crfd);
	mask = ~(0xf << (4*crfd));
	CR = (CR & mask) | setbits;
	/* Clear the exception bits except FEX and VX */
	FPSCR = FPSCR ^ clear;	
	//update_fex_vx(); /* Summary bits */
	fprintf(stderr,"instr i_mcrfs(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------------------------------
 * mtfsb0x Move to FPSCR Bit 0 
 * ------------------------------------------------------------------
 */
void 
i_mtfsb0x(uint32_t icode) {
	int crbd = 31-((icode>>21)&0x1f);
	int rc = icode&1;
	if((crbd == 29) || (crbd == 30)) {
		fprintf(stderr,"mtfsb0x geht net\n");
		return;
	}
	FPSCR = FPSCR & ~(1<<crbd);
	if(rc) {
		CR = (CR & 0xf0ffffff) | ((FPSCR>>4)&0x0f000000);
	}
	fprintf(stderr,"instr i_mtfsb0x(%08x) not implemented\n",icode);
}

void 
i_fmrx(uint32_t icode) {
	fprintf(stderr,"instr i_fmrx(%08x) not implemented\n",icode);
}

/*
 * ----------------------------------------------
 * mtfsfix UISA Form X
 * 	Move to FPSCR field immediate
 * ----------------------------------------------
 */
void 
i_mtfsfix(uint32_t icode) {
	unsigned int crfd = 7-((icode>>23) & 0x7);
	unsigned int imm = (icode>>12) & 0xf;
	int rc = icode & 1;
	uint32_t mask = ~(0x0000000fU << (crfd*4));
	FPSCR = (FPSCR & mask)	 | (imm<<(crfd*4));
	if(rc) {
		CR = (CR & 0xf0ffffff) | ((FPSCR>>4)&0x0f000000);
	}
	fprintf(stderr,"instr i_mtfsfix(%08x)\n",icode);
}

void 
i_fnabsx(uint32_t icode) {
	fprintf(stderr,"instr i_fnabsx(%08x) not implemented\n",icode);
}

void 
i_fabsx(uint32_t icode) {
#if 0
	int frd=(icode>>21)&0xf;
	int frb=(icode>>11)&0xf;
	/* */		
	fprintf(stderr,"instr i_fabsx(%08x) not implemented\n",icode);
#endif
}

void 
i_mffsx(uint32_t icode) {
	int d = (icode>>21) & 0x1f;
	FPR(d) = (FPR(d) & 0xffffffff00000000ULL) | FPSCR;
	fprintf(stderr,"instr i_mffsx(%08x) not implemented\n",icode);
}

/*
 * ------------------------------------------------------------------
 * mtfsfx UISA Form XFL 
 * Move to FPSCR fields 
 * ------------------------------------------------------------------
 */
void 
i_mtfsfx(uint32_t icode) {
	int fm = (icode>>17)&0xff;
	int b = (icode>>11) & 0x1f;
	int rc = icode & 1;
	uint32_t mask = 0;
	unsigned int i;
	for(i=0;i<8;i++) {
		if(fm & (1<<i)) {
			mask |= (0xf<<(4*i));
		}
	}	
	FPSCR = (FPR(b) & mask) | (FPSCR & ~mask);
	if(rc) {
		CR = (CR & 0xf0ffffff) | ((FPSCR>>4)&0x0f000000);
	}
	fprintf(stderr,"instr i_mtfsfx(%08x)\n",icode);
}

void 
i_fctdix(uint32_t icode) {
	fprintf(stderr,"instr i_fctdix(%08x) not implemented\n",icode);
}

void 
i_fcfidx(uint32_t icode) {
	fprintf(stderr,"instr i_fcfidx(%08x) not implemented\n",icode);
}

void i_und(uint32_t icode) {
        fprintf(stderr,"Instruction not found for icode %08x at pc %08x\n",icode,CIA);
        //PPC_Exception(EX_UNDEFINED);
}

