/*
 * -----------------------------------------------------------------------------------
 *
 * Senseless poll detector used to lower cpu load when guest OS or bootloader
 * polls some IO registers like UART receive buffer or timer to fast.  
 * When some the rate to high this detector jumps over some cpu cycles 
 * and sleeps if enough cpu cycles are on the account.
 *
 * (C) 2006 Jochen Karrer
 *   Author: Jochen Karrer
 *
 *  State: implementation working with u-boot for 
 *  Atmel AT91RM9200 and Freescale i.MX21
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#include <configfile.h>
#include <cycletimer.h>
#include <senseless.h>
#include <fio.h>

typedef struct SenslessMonitor 
{
	CycleCounter_t last_senseless;
	CycleCounter_t saved_cycles;
	uint32_t nonsense_threshold;
	uint32_t jump_width;
	int64_t overjumped_nanoseconds; 
	uint32_t sensivity;
} SenselessMonitor;

static SenselessMonitor *smon;

/*
 * --------------------------------------------------------------------
 * save_cycles
 * 	Jump over jump_width cycles. Put the saved cycles onto a
 *	nanosecond account. If it is enough to sleep 10 ms
 *	then sleep and subtract the time from the account
 * --------------------------------------------------------------------
 */
static void 
save_cycles(SenselessMonitor *smon)
{
	struct timespec tvv,tvn;
	uint32_t jump_width = smon->jump_width;	
	CycleCounter_t cnow = CycleCounter_Get(); 
	if((cnow + jump_width) >= firstCycleTimerTimeout) {
		jump_width = firstCycleTimerTimeout - cnow;
		CycleCounter = firstCycleTimerTimeout;	
	} else {
		CycleCounter += jump_width;	
	}
	/* fprintf(stderr,"jump\n"); */
	smon->overjumped_nanoseconds += CyclesToNanoseconds(jump_width);
	smon->saved_cycles -= jump_width;
	while(smon->overjumped_nanoseconds > 11000000) {
		struct timespec tout;
		uint64_t nsecs;
		tout.tv_nsec=10000000; /* 10 ms */
                tout.tv_sec=0;
		clock_gettime(CLOCK_MONOTONIC,&tvv);
		FIO_WaitEventTimeout(&tout);
		clock_gettime(CLOCK_MONOTONIC,&tvn);
		nsecs=tvn.tv_nsec-tvv.tv_nsec + (int64_t)1000000000*(tvn.tv_sec - tvv.tv_sec);	
		smon->overjumped_nanoseconds -= nsecs*1.1;
	}
}
/*
 * ---------------------------------------------------------------------
 * Report a possibly senseless poll operation. A high
 * weight means that the operation needs to be executed less often
 * to be detected as senseless. The default weight is 100
 * ---------------------------------------------------------------------
 */
void
Senseless_Report(int weight) 
{
	uint64_t diffcycles = CycleCounter_Get() - smon->last_senseless;
	uint64_t consumed_cycles = 2 * diffcycles; 
	smon->last_senseless = CycleCounter_Get();
	smon->saved_cycles += NanosecondsToCycles(smon->sensivity*weight);
	if(consumed_cycles > smon->saved_cycles) {
		smon->saved_cycles = 0;
		return;
	}
	smon->saved_cycles -= consumed_cycles;
	if(smon->saved_cycles > smon->nonsense_threshold) {
		save_cycles(smon);
		if(smon->saved_cycles >  (smon->nonsense_threshold << 1)) {
			smon->saved_cycles = 0;
		}
	}
}

void
Senseless_Init() 
{
	smon = malloc(sizeof(*smon));
	if(!smon) {
		fprintf(stderr,"Out of memory for senslessity monitor\n");
		exit(1);
	}
	memset(smon,0,sizeof(*smon));
	smon->nonsense_threshold = CycleTimerRate_Get()/1000;
	smon->jump_width = smon->nonsense_threshold = CycleTimerRate_Get()/20000;
	smon->sensivity = 10;
	Config_ReadUInt32(&smon->sensivity,"poll_detector","sensivity");
	Config_ReadUInt32(&smon->jump_width,"poll_detector","jump_width");
	Config_ReadUInt32(&smon->nonsense_threshold,"poll_detector","threshold");
	fprintf(stderr,"Poll detector Sensivity %d\n",smon->sensivity);
}

