/*
 * ------------------------------------------------------------------------
 * 
 * Event driven	telnet server 
 * (C) 2004  Jochen Karrer
 *   Author: Jochen Karrer
 *   State: nothing is working
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ------------------------------------------------------------------------
 */

#if 0
#define dprintf(x...) { fprintf(stderr,x); }
#else
#define dprintf(x...)
#endif

#include <fio.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/socket.h>
#include <arpa/telnet.h>
#include <errno.h>

#include "telnetd.h"
#include "configfile.h"


#define MAX_CMDLINE_LEN (1024)
#define HISTORY_LINES (100)

struct TelnetSession {
	void *owner;
	TelnetServer *tserv;
	int rfh_is_active;
	FIO_FileHandler rfh;
	char cmdline[MAX_CMDLINE_LEN];
	int echo;
	int sga;
	int cmdline_wp;
	int state;
	int telnet_cmd;
	int fd;
};

struct TelnetServer {
	TelnetOps *telnet_ops;
	void *clientData;
	FIO_TcpServer tcpserv;
};

static void 
telnet_output(TelnetSession *ts,char *buf,int count) {
	int result;
	fcntl(ts->fd,F_SETFL,0);
	while(count) {
		result = write(ts->fd,buf,count);
		if(result<=0) {
			/* Can not delete the session now */ 
			return;
		}
		count-=result; buf+=result;
	}
	fcntl(ts->fd,F_SETFL,O_NONBLOCK);
}

static inline void 
telnet_outstr(TelnetSession *ts,char *buf) {
	telnet_output(ts,buf,strlen(buf));
}

static void
send_iac(TelnetSession *ts,uint8_t cmd,uint8_t opt) {
	char buf[3];
	buf[0]=IAC; buf[1] = cmd; buf[2] = opt;
	telnet_output(ts,buf,3); 
}

void
TS_Del(TelnetSession *ts) {
	TelnetServer *tserv = ts->tserv;
	TelnetOps *ops = tserv->telnet_ops;
	uint8_t c;
	if(ops->close) {
		ops->close(ts,ts->owner);
	}
	if(ts->rfh_is_active) {
		ts->rfh_is_active=0;
		dprintf("Remove the filehandler\n");
		FIO_RemoveFileHandler(&ts->rfh);
	}
	/* Don't close stdout stderr */
	shutdown(ts->fd,SHUT_RDWR);
	while(read(ts->fd,&c,1)>0) {
		dprintf("READEMPTY %02x\n",c);
	}
	if(ts->fd>2) {
		dprintf("close %d\n",ts->fd);
		if(close(ts->fd)<0) {
			perror("close fd failed");
		} else {
			ts->fd=-1;
		}
	} else {
		dprintf("Do not close %d\n",ts->fd);
	}
	free(ts);
}
#define STATE_RCV  	(0)
#define STATE_IAC  	(1)
#define STATE_SUB  	(2)
#define STATE_OPTION  	(3)

/* From RFC 854 */

static void
do_option(TelnetSession *ts,int option) 
{
	switch(option) {
		case TELOPT_ECHO:
			if(!ts->echo) {
				send_iac(ts,WILL,option);
				ts->echo=1;
			}
			break;

		case TELOPT_SGA:
			dprintf("accept SGA ????\n");
			if(!ts->sga) {
				send_iac(ts,WILL,option);
				ts->sga=1;
			}
			break;

		case TELOPT_BINARY:
			fprintf(stderr,"DO Binary\n");
			break;
		case TELOPT_LINEMODE:
			break;
		default:
			dprintf("refuse 0x%02x\n",option);
			send_iac(ts,WONT,option);
			break;
	}
}
static void
dont_option(TelnetSession *ts,int option) 
{
	switch(option) {
		case TELOPT_ECHO:
			dprintf("Telnet peer does not like echo\n");
			// give up ???
			if(ts->echo) {
				send_iac(ts,WONT,option);
				ts->echo=0;
			}
			break;
		default:
			send_iac(ts,WONT,option);
	}
}

static void
will_option(TelnetSession *ts,int option) 
{

}

static void
wont_option(TelnetSession *ts,int option) 
{
	switch(option) {
		case TELOPT_ECHO:
			dprintf("Peer won't echo\n");
			break;

		case TELOPT_SGA:
			dprintf("Peer refuses to suppress go ahead\n");
	}
}

static void 
prompt(TelnetSession *ts) {
	telnet_outstr(ts,"\r> ");
}

static int 
feed_state_machine (TelnetSession *ts,uint8_t c) {
	char reply[5];
	dprintf("feed 0x%02x\n",c);
	switch(ts->state) {
		case STATE_RCV:
			if(c==IAC) {
				ts->state=STATE_IAC;
			} else {
				if(c<=NSLC) {
				}
				switch(c) {
					case SLC_IP:
						dprintf("Interrupt\n");
						break;
					case '\r':
					case '\n':
						dprintf("Newline\n");
						if(ts->echo) {
							telnet_outstr(ts,"\n\r");
						}
						if(!strcmp(ts->cmdline,"quit")) {
							telnet_outstr(ts,"Bye\n\r");
							TS_Del(ts);
							return -1;
						}
						telnet_outstr(ts,"Executing commands is not implemented\n\r");
						prompt(ts);
						ts->cmdline_wp=0;
						ts->cmdline[0]=0;
						break;

					case 0x7f:
						if(ts->cmdline_wp>0) {
							ts->cmdline_wp--;
							ts->cmdline[ts->cmdline_wp]=0;
						}
						reply[0] = 0x1b;
						reply[1] = 0x5b;
						reply[2] = 0x44;
						reply[3] = 0x20;
						telnet_output(ts,reply,4);
						prompt(ts);
						telnet_outstr(ts,ts->cmdline);
						break;

					case 0:
						break;
					default:
						if(ts->echo) {
							telnet_output(ts,&c,1);
						}
						if(ts->cmdline_wp+1<MAX_CMDLINE_LEN) {
							ts->cmdline[ts->cmdline_wp]=c;
							ts->cmdline_wp++;
							ts->cmdline[ts->cmdline_wp]=0;
						}
					
				}	
			}
			break;
		case STATE_IAC: 
			switch(c) {
				case DONT: 
				case DO:
				case WONT:
				case WILL:
					ts->telnet_cmd = c;
					ts->state=STATE_OPTION;	
					break;

				case SB:
					ts->state=STATE_SUB;	
					break;
					
				case GA:
					ts->state=STATE_RCV;	
					break;
				case EL:
					ts->state=STATE_RCV;	
					break;
				case EC:
					ts->state=STATE_RCV;	
					break;
				case AYT:
					ts->state=STATE_RCV;	
					break;
				case AO:
					ts->state=STATE_RCV;	
					break;
				case IP:
					dprintf("Received Interrupt\n");
					ts->state=STATE_RCV;	
					break;
				case BREAK:
					ts->state=STATE_RCV;	
					break;
				case DM:
					ts->state=STATE_RCV;	
					break;
				case NOP:
					ts->state=STATE_RCV;	
					break;
				case SE:
					ts->state=STATE_RCV;	
					break;
			}
			break;

		case STATE_OPTION:
			switch(ts->telnet_cmd) {
				case DO:
					do_option(ts,c);
					ts->state = STATE_RCV;
					break;
					
				case DONT:
					dont_option(ts,c);
					ts->state = STATE_RCV;
					break;
				case WONT:
					dprintf("WONT 0x%02x\n",c);
					wont_option(ts,c);
					ts->state = STATE_RCV;
					break;
				case WILL:
					will_option(ts,c);
					ts->state = STATE_RCV;
					dprintf("WONT 0x%02x\n",c);
					break;
				default:
					fprintf(stderr,"Not understanding 0x%02x\n",c);
					ts->state = STATE_RCV;
					break;
			}
			break;
			
		case STATE_SUB: 
			if(c == SE) {
				ts->state=STATE_RCV;	
			} else {
				dprintf("sub %02x\n",c);
			}
			break;
		default:
			dprintf("unknown state %d\n",ts->state);
			break;	
	}
	return 0;
}

static int
telnet_input(void *cd,int mask) {
	int result;
	TelnetSession *ts=cd;
	uint8_t c;
	while((result=read(ts->fd,&c,1))==1) {
		if( feed_state_machine (ts, c)<0) {
			return 0;
		}
	}
	if((result<0) && (errno==EAGAIN))  {
		return 0;
	} else {
		dprintf("Connection lost\n");
		TS_Del(ts);
		return 0;
	}
}


static TelnetSession *
TS_New(int fd) {
	TelnetSession *ts=malloc(sizeof(TelnetSession));	 
	if(!ts) {
		fprintf(stderr,"Out of memory\n");
		exit(765);
	}
	memset(ts,0,sizeof(TelnetSession));
	ts->fd=fd;
	dprintf("New Telnet session fd %d\n",fd);
	fcntl(ts->fd,F_SETFL,O_NONBLOCK);
	FIO_AddFileHandler(&ts->rfh,ts->fd,FIO_READABLE,telnet_input,ts);
	ts->rfh_is_active=1;
	telnet_outstr(ts,"Welcome to Softgun Debug Shell\n");
	telnet_outstr(ts,"Nothing is implemented yet\n> ");
	ts->sga=1;
        send_iac(ts, WILL, TELOPT_SGA);
        send_iac(ts, DO, TELOPT_LFLOW); // remote flow control 
	ts->echo=1;
        send_iac(ts, WILL, TELOPT_ECHO);
        send_iac(ts, DONT, TELOPT_LINEMODE);
	return ts;
}



static void
tserv_connect(int sockfd,char *host,unsigned short port,void *cd) 
{
	TelnetServer *tserv = cd;
	TelnetOps *ops = tserv->telnet_ops;
        TelnetSession *ts;
	dprintf("Connection from %s port %d\n",host,port);
        ts = TS_New(sockfd);
	ts->tserv = tserv;
	if(ops->open) {
		ts->owner = ops->open(ts,tserv->clientData);
	} 
	return;
}


TelnetServer *
TelnetServer_New(char *host,int port,TelnetOps *ops,void *clientData)
{
	int fd;
        TelnetServer *tserv=malloc(sizeof(TelnetServer));
        if(!tserv)
                return NULL;
	
        if((fd=FIO_InitTcpServer(&tserv->tcpserv,tserv_connect,tserv,host,port))<0) {
                free(tserv);
                return NULL;
        }
	tserv->telnet_ops = ops;
	tserv->clientData=clientData;
	dprintf("Debug Server fd %d Listening on host \"%s\" port %d\n",fd,host,port);
        return tserv;
}
