/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif
#ifdef ENABLE_LAME

#include "../../player.h"
#include "../../other.h"
#include "../../audio.h"
#include "../../util.h"
#include "../../output.h"
#include "fconfig.h"
#include "common.h"
#include "lame.h"

audio_data_lame *
lame_audio_device (audio * output)
{

  audio_data_lame *lame;
  int ch = 0;
  int len;
  struct stat st;

  if (!(lame = (audio_data_lame *) malloc (sizeof (audio_data_lame))))
    fatal (_("Error: memory."));

  if (!lstat (output->audio_dev, &st))
    {
      if (lame_read_config (output->audio_dev, lame))
	fatal (_("Lame init error."));

    }
  else
    {

      lame->rate = 44100;
      lame->quality = 8;
      lame->bitrate = 96;
      lame->channels = OUT_STEREO;
      lame->lowpass = 0;
      lame->highpass = 0;
      lame->file = NULL;
      lame->fd = NULL;

      len = strlen (output->audio_dev);

      while (ch < len)
	{

	  switch (*(output->audio_dev + ch))
	    {

	    case 'b':
	      if (!strncmp (output->audio_dev + ch, "bitrate=", 6))
		{
		  ch += 8;
		  lame->bitrate =
		    audio_parse_int (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (lame);
		  return NULL;
		}

	      break;

	    case 'c':
	      if (!strncmp (output->audio_dev + ch, "channels=", 9))
		{
		  int a;

		  ch += 9;
		  a = audio_parse_int (output->audio_dev, &ch, len);

		  switch (a)
		    {
		    case 1:
		      lame->channels = OUT_MONO;
		      break;
		    case 2:
		      lame->channels = OUT_STEREO;
		      break;
		    default:
		      msg_error (_("Error in audiodevice parameter mode: %s"),
				 output->audio_dev);
		      free (lame);
		      return NULL;
		    }
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (lame);
		  return NULL;
		}

	      break;

	    case 'f':

	      if (!strncmp (output->audio_dev + ch, "file=", 5))
		{
		  ch += 5;
		  lame->file =
		    audio_parse_str (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (lame);
		  return NULL;
		}

	      break;

	    case 'h':

	      if (!strncmp (output->audio_dev + ch, "highpass=", 9))
		{
		  ch += 9;
		  lame->highpass =
		    audio_parse_int (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (lame);
		  return NULL;
		}

	      break;

	    case 'l':

	      if (!strncmp (output->audio_dev + ch, "lowpass=", 8))
		{
		  ch += 8;
		  lame->lowpass =
		    audio_parse_int (output->audio_dev, &ch, len);
		}

	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (lame);
		  return NULL;
		}

	      break;

	    case 'q':
	      if (!strncmp (output->audio_dev + ch, "quality=", 5))
		{
		  ch += 8;
		  lame->quality =
		    audio_parse_int (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (lame);
		  return NULL;
		}

	      break;

	    case 'r':

	      if (!strncmp (output->audio_dev + ch, "rate=", 5))
		{
		  ch += 5;
		  lame->rate =
		    audio_parse_int (output->audio_dev, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in audiodevice parameter: %s"),
			     output->audio_dev);
		  free (lame);
		  return NULL;
		}

	      break;

	    default:
	      msg_error (_("Error in audiodevice parameter: %s"),
			 output->audio_dev);
	      free (lame);
	      return NULL;
	    }

	  ch++;
	}
    }

  if (!lame->file)
    {
      msg_error (_("No file in lame config."));
      free (lame);
      return NULL;
    }

  return lame;
}

int
lame_start (audio * output, long rate, int channels, int bitrate)
{
  audio_data_lame *data = (audio_data_lame *) output->data;
  char *fl;

  if (!data)
    fatal (_("Internal error."));

  if (!
      (data->lame =
       lame_config (rate, data->rate, channels, data->channels, data->lowpass,
		    data->highpass, data->bitrate, data->quality)))
    return 1;

  fl = output_check_file (data->file);

  if (!(data->fd = fopen (fl, "wb")))
    {
      msg_error (_("Open error: %s."), fl);
      free (fl);

      lame_quit (data->lame);
      return 1;
    }

  free (fl);

  return 0;
}

int
lame_pre_write (audio * output, int channels, int bitrate, void *what, size_t size)
{

  audio_data_lame *data = (audio_data_lame *) output->data;

  unsigned char *output_buf = NULL;
  int output_len = 0;
  int ret;

  if (!data)
    fatal (_("Internal error."));

  lame_write (data->lame, channels, bitrate, what, size, &output_buf, &output_len);

  if (output_len < 0)
    msg (_("Error in encoding."));

  if (output_len < 1)
    return output_len;

  ret = fwrite (output_buf, 1, output_len, data->fd);

  free (output_buf);

  return ret;
}

void
lame_pre_close (audio * output)
{
  audio_data_lame *data = (audio_data_lame *) output->data;

  if (data->lame)
    {
      lame_quit (data->lame);
      fclose (data->fd);
    }
}

void
lame_info (audio * output)
{
  audio_data_lame *data = (audio_data_lame *) output->data;

  if (!data)
    fatal (_("Internal error."));

  msg (_("Lame %s - %s\n" "\tRate %d\n" "\tBitrate %d\n" "\tChannels %d\n" "\tQuality %d\n"
	 "\tFile %s\n"), get_lame_version (), get_lame_url (), data->rate,
       data->bitrate, data->channels, data->quality, data->file);

}
#endif
