#!/usr/local/bin/perl
#
# This class represents a list of services that run on a given host, allowing
# you to query individual services for status information.  This knows how to
# load the Service information from the database, and has the smarts to change
# the service colors based on updates times and Acks.
#
#   host          - string FQDN of the machine its associated with
#   service_hash  - a hash containing the services org by service names
#
# + new()         - constructor (sets instance vars to arguments passed in)
# + gets/sets()   - magical set/get functions (autoloaded based on func name)
# + display()     - output format and view
#
# + service()     - returns the specific Service object based on name
# + services()    - returns a list of all the Service objects
# + names()       - returns a list of all the Service object names.
# + color()       - returns the LCD color of all the services in the list
#
# History:
# (1) Cleaned up (Ed July 31, 1997);

use Spong::Service;
use Spong::AckList;
use Spong::Ack;

package Spong::ServiceList;

# Constructor.  This requires the FQDN of the host that the Services are
# associated with.  It goes through and loads the information from the database
# and also loads any Acks that might be associated with the host.  It then 
# computes the colors for each host and builds Service objects for each service
# running on that host.  If there are no services running on the host, then
# undef is returned.

sub new {
   my( $class, $host ) = @_;
   my( %service_hash, @acks, $service, $ack, $service_name, $got_one );
   my $self = {};

   return undef if ! -d "$main::SPONGDB/$host/services";

   # Go through the database, and grab the name and state of each service 
   # running on this host.
   
   if ( ! $main::SERVICELISTFIX ) {
      opendir( DIR, "$main::SPONGDB/$host/services" );
      while( defined( $_ = readdir( DIR ) ) ) {
         next unless /^(.*)-(green|yellow|red|purple|clear)$/ ;
         my( $name, $color ) = ($1, $2);

         # A hack to get around a local problem.
         if( $name eq "procs" ) { $name = "jobs"; }
      
         $got_one = 1;
         my( $rtime ) = (stat( "$main::SPONGDB/$host/services/$_" ))[9];
         my $service  = Spong::Service->new( $host, $name );
      
         $service->rtime( $rtime );
         if( $rtime < ( time() - (3 * $main::SPONGSLEEP) ) ) { 
            $service->rcolor( $color ); $service->color( "purple" );
         } else {
            $service->rcolor( $color ); $service->color( $color );
         }
         $service_hash{$name} = $service;
      }
      closedir( DIR );
   } else {
      my @files = ();
      {
         local $SIG{'CHLD'} = 'IGNORE';
         eval { @files = `cd $main::SPONGDB/$host/services/; ls *`; };
      }
      foreach ( @files ) {
         chomp;
         next unless /^(.*)-(green|yellow|red|purple|clear)$/ ;
         my( $name, $color ) = ($1, $2);

         # A hack to get around a local problem.
         if( $name eq "procs" ) { $name = "jobs"; }
      
         $got_one = 1;
         my( $rtime ) = (stat( "$main::SPONGDB/$host/services/$_" ))[9];
         my $service  = Spong::Service->new( $host, $name );
      
         $service->rtime( $rtime );
         if( $rtime < ( time() - (3 * $main::SPONGSLEEP) ) ) { 
            $service->rcolor( $color ); $service->color( "purple" );
         } else {
            $service->rcolor( $color ); $service->color( $color );
         }

         $service_hash{$name} = $service;
      }
   }

   return undef unless $got_one;

   # Go through the acknowledgments that we have for this host, and change the
   # state information on our Services if needed.

   $acklist = Spong::AckList->new( $host );
   if( $acklist ) {
      foreach $ack ( $acklist->acks() ) {
	 my $ack_services = $ack->services();

	 if( time() >= $ack->start() && time() <= $ack->end() ) {
	    foreach $service_name ( keys %service_hash ) {
	       if( $ack_services eq "all" || 
		   $ack_services =~ /\b$service_name\b/ ) {
		  
		  $service_hash{$service_name}->color( "blue" );
	       }
	    }
	 }
      }
   }

   $self->{'host'}          = $host;
   $self->{'service_hash'}  = \%service_hash;

   bless $self;
   return $self;
}

# Get/Set methods, nothing fancy here...

sub host { my $var = 'host';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }
sub service_hash { my $var = 'service_hash';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }


# Some utility functions that return information based on instance vars, and 
# the state of services that this object holds.

sub service  { return $_[0]->{'service_hash'}->{$_[1]}; }
sub services { return values %{$_[0]->{'service_hash'}}; }
sub names    { return keys %{$_[0]->{'service_hash'}}; }

sub color {
   my $self = shift;
   my( %state_hash ) = qw( red 5 blue 4 yellow 3 purple 2 green 1 clear 0);
   my $state = "green";

   foreach $service ( $self->services() ) {
      if( $state_hash{$service->color()} > $state_hash{$state} ) {
	 $state = $service->color(); } }

   return $state;
}


# Display summary.  Does both text and html, it relies on the Service objects
# to print out information about themselves, but this adds a little to by
# printing header information in cases.
#
# brief:      Just shows name of services and what color they are
# standard:   Shows a table of services (name,color,updated,summary)
# full:       Shows standard information about service along with messages
#
# ** full is not implemented yet.

sub display {
   my( $self, $type, $view ) = @_;

   if( $type eq "text" ) { return $self->display_text( $view ); }
   if( $type eq "html" ) { return $self->display_html( $view ); }
   if( $type eq "wml" ) { return $self->display_wml( $view ); }
}

sub display_text {
   my( $self, $format ) = @_;
   my( $service );

   if( $format eq "brief" ) {
      foreach $service ( $self->services() ) {
	 print $service->name, " "x(8-length( $service->name() ));
	 $service->display_text( "brief" );
      }
   } elsif( $format eq "standard" ) {
      print "Name    Color    Updated    Summary\n";
      print "------- -------- ---------- ", "-"x50, "\n";
      foreach( $self->services() ) { $_->display_text( "standard_table" ); }
   } elsif( $format eq "full" ) {
      foreach $service ( $self->services() ) {
	 $service->display_text( "full" );
	 print "\n\n";
      }
   }
}

sub display_html {
   my( $self, $format ) = @_;
   my( $event, $date );
   
   if( $format eq "brief" ) {
      print "<table border=0 cellspacing=1 cellpadding=1>\n";
      foreach $service ( $self->services() ) {
	 my $host = $service->host();
	 my $name = $service->name();

	 print "<tr><td align=center valign=center>";
	 $service->display_html( "brief" );
	 print "</td><td align=left valign=center>";
	 print "&nbsp<a href=\"!!WWWSPONG!!/service/$host/$name\">$name</a>";
	 print "</td></tr>\n";
      }
      print "</table>\n";

   } elsif( $format eq "standard" ) {
      print "<table width=100% border=1 cellspacing=1 cellpadding=1><tr>\n";
      print "<td width=60 align=center nowrap><b>Service</b></td>\n";
      print "<td width=1% nowrap>&nbsp</td>\n";
      print "<td width=60 align=center nowrap><b>Updated</b></td>\n";
      print "<td width=100% align=center><b>Summary</b></td></tr>\n";

      foreach( $self->services() ) { $_->display_html( "standard_table" ); }

      print "</table>\n";
   }
}

# This displays all services in the database for a host in a WML format
# suitable for displaying on a WAP enabled device

sub display_wml {
   my( $self, $format ) = @_;
   my( $service );
   my $host = $self->host();

   if( $format eq "standard" ) {
      foreach $service ( $self->services() ) {
         my $name = $service->name();
         my $color = substr($service->color(),0,2);
         $color =~ tr/a-z/A-Z/;

         print "<b><anchor title=\"$name\">$color";
         print "<go href=\"!!WAPSPONG!!/service/$host/$name\"/>";
         print "</anchor></b> $name<br/>\n";
      }
   }
}

1;
